<?php
session_start();
include(__DIR__ . '/../includes/koneksi.php');
include(__DIR__ . '/../includes/functions.php');

if (!isset($_SESSION['username'])) {
    header('Location: ../login.php');
    exit();
}

$username = $_SESSION['username'];
$stmt = $conn->prepare("SELECT id, tipe, nama, access_modul FROM users WHERE username = ?");
$stmt->bind_param("s", $username);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->fetch_assoc();
$stmt->close();

if (!$user || !in_array($user['tipe'], ['user', 'admin', 'editor'])) {
    header('Location: ../login.php');
    exit();
}

// compute approver status first so we can allow approvers to view even if module flag is off
$access_modul = json_decode($user['access_modul'], true);
$userId = (int)$user['id'];

// compute both types of subordinate/approval status: multi-level supervisor via letter_order and explicit approver via approval_chain
$subordinateIds = getSubordinatesForUser($conn, $userId);
$isSupervisor = !empty($subordinateIds);
$isApprover = isApprover($conn, $userId);

// If debug is requested, dump the current context and stop to help identify redirect reasons
if (defined('JASA_DEV') && JASA_DEV && !empty($_GET['debug']) && $_GET['debug'] === '1') {
    echo '<pre>SESSION=';
    print_r($_SESSION);
    echo "\nUSER=" . json_encode($user) . "\nACCESS_MODULE=" . json_encode($access_modul) . "\nIS_APPROVER=" . ($isApprover ? '1' : '0') . "\nIS_SUPERVISOR=" . ($isSupervisor ? '1' : '0') . "\n</pre>";
    exit();
}

// Izinkan akses jika:
// 1. User adalah tipe "user", ATAU
// 2. User adalah approver/supervisor
if ($user['tipe'] !== 'user' && !$isApprover && !$isSupervisor) {
    header('Location: login.php');
    exit();
}
$nama = htmlspecialchars($user['nama']);
$name = $nama; // for sidebar usage (gaji link)

date_default_timezone_set('Asia/Jakarta');
$today = date('Y-m-d');
$nowTime = date('H:i');

// Hitung agenda aktif
$countQuery = "SELECT COUNT(*) as jumlah FROM agenda WHERE tanggal > ? OR (tanggal = ? AND waktu >= ?)";
$countStmt = $conn->prepare($countQuery);
$countStmt->bind_param('sss', $today, $today, $nowTime);
$countStmt->execute();
$jumlahAgendaAktif = ($countStmt->get_result()->fetch_assoc()['jumlah'] ?? 0);
$countStmt->close();

// Prepare variables for approval badges; will be computed after $isAtasan is known
$jumlahApproveTodo = 0;

// $isApprover variable is set above; keep a boolean variable for later
$isAtasan = $isApprover;

// Build approval map for displaying review chain next to each row
list($map, $nameMap) = buildSupervisorMap($conn);

/**
 * Helper function to get approval chain status for a letter
 */
function getLetterApprovalChainStatus($conn, $officeId)
{
    $result = [
        'chain' => [],
        'current_step' => 0,
        'total_steps' => 0,
        'is_complete' => false,
        'paraf_display' => '',
        'next_approver_id' => null,
        'approved_by' => []
    ];

    // First, get dibuat_oleh to check if paraf should be displayed
    $stmt_dibuat = $conn->prepare("SELECT dibuat_oleh, penandatangan_id FROM office WHERE id = ?");
    $stmt_dibuat->bind_param("i", $officeId);
    $stmt_dibuat->execute();
    $res_dibuat = $stmt_dibuat->get_result();
    $dibuat_row = $res_dibuat->fetch_assoc();
    $stmt_dibuat->close();

    $dibuat_oleh = isset($dibuat_row['dibuat_oleh']) ? (int)$dibuat_row['dibuat_oleh'] : 0;
    $penandatangan_id = isset($dibuat_row['penandatangan_id']) ? (int)$dibuat_row['penandatangan_id'] : 0;

    // Don't show paraf if dibuat_oleh and penandatangan_id are the same person
    $shouldShowParaf = ($penandatangan_id > 0 && $penandatangan_id !== $dibuat_oleh);

    // Check if urutan_paraf exists
    $stmt = $conn->prepare("
        SELECT up.*, u.nama as approver_name, u.jabatan, i.inisial 
        FROM urutan_paraf up
        LEFT JOIN users u ON up.user_id = u.id
        LEFT JOIN inisial i ON up.user_id = i.user_id AND i.is_active = 1
        WHERE up.office_id = ?
        ORDER BY up.urutan ASC
    ");
    $stmt->bind_param("i", $officeId);
    $stmt->execute();
    $res = $stmt->get_result();

    $parafParts = [];
    $lastApproved = 0;
    while ($row = $res->fetch_assoc()) {
        $result['chain'][] = $row;
        $result['total_steps']++;

        if ($row['approved_at'] !== null) {
            $lastApproved = (int)$row['urutan'];
            $result['current_step'] = $lastApproved;
            $result['approved_by'][] = (int)$row['user_id'];
            // Only include inisial in paraf if paraf should be shown
            if ($shouldShowParaf && !empty($row['inisial'])) {
                $parafParts[] = $row['inisial'];
            }
        } else if ($result['next_approver_id'] === null) {
            $result['next_approver_id'] = (int)$row['user_id'];
        }
    }
    $stmt->close();

    // Only set paraf_display if paraf should be shown
    $result['paraf_display'] = $shouldShowParaf ? implode('/', $parafParts) : '';
    $result['is_complete'] = ($result['current_step'] === $result['total_steps'] && $result['total_steps'] > 0);

    return $result;
}

/**
 * Check if user can approve a specific letter
 */
function canUserApproveLetter($approvalStatus, $userId)
{
    return $approvalStatus['next_approver_id'] === $userId;
}

/**
 * Check if user is in the approval chain for a letter
 */
function isUserInApprovalChain($approvalStatus, $userId)
{
    foreach ($approvalStatus['chain'] as $step) {
        if ((int)$step['user_id'] === $userId) {
            return true;
        }
    }
    return false;
}

// ensure $subordinateIds contain union from approval_chain + letter_order (getSubordinatesForUser already did this)
// if $isAtasan is true we will also compute picsByLevel using approval_chain
if ($isAtasan) {
    // Add explicit approval_chain users to subordinate list in case they were not present in letter_order
    $stmt = $conn->prepare("SELECT DISTINCT user_id FROM approval_chain WHERE approver_id = ?");
    $stmt->bind_param('i', $userId);
    $stmt->execute();
    $res = $stmt->get_result();
    while ($r = $res->fetch_assoc()) $subordinateIds[] = (int)$r['user_id'];
    $stmt->close();
    $subordinateIds = array_values(array_unique($subordinateIds));

    // Compute jumlahApproveTodo now that we know the user is an approver
    $picsByLevel = [];
    $stmt = $conn->prepare("SELECT user_id, urutan FROM approval_chain WHERE approver_id = ?");
    $stmt->bind_param('i', $userId);
    $stmt->execute();
    $res = $stmt->get_result();
    while ($r = $res->fetch_assoc()) {
        $picsByLevel[(int)$r['urutan']][] = (int)$r['user_id'];
    }
    $stmt->close();

    $whereParts = [];
    foreach ($picsByLevel as $level => $pics) {
        if (empty($pics)) continue;
        // Validasi: maksimal level yang valid di todolist schema
        if ($level > 2) {
            error_log("Warning: approval_chain level {$level} melebihi batas maksimum. Menggunakan level 2 sebagai fallback.");
            $level = 2;
        }
        $picList = implode(',', array_map('intval', $pics));
        $cond = "t.pic IN ($picList) AND t.approval_status_level{$level} = 'menunggu'";
        if ($level > 1) {
            $prev = $level - 1;
            $cond .= " AND (t.approval_status_level{$prev} IS NULL OR t.approval_status_level{$prev} != 'menunggu')";
        }
        $whereParts[] = "($cond)";
    }

    if (!empty($whereParts)) {
        $sqlCountApprove = "SELECT COUNT(*) AS jumlah FROM todolist t WHERE " . implode(' OR ', $whereParts);
        $resCount = $conn->query($sqlCountApprove);
        $jumlahApproveTodo = $resCount ? (int)$resCount->fetch_assoc()['jumlah'] : 0;
    }
}

// Handle delete letter action (owner only)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_letter'])) {
    $letter_id = (int)$_POST['letter_id'];
    // Verify ownership and validation status
    $stmt_check = $conn->prepare("SELECT dibuat_oleh, IFNULL(is_validated,0) AS is_validated FROM office WHERE id = ?");
    $stmt_check->bind_param('i', $letter_id);
    $stmt_check->execute();
    $res_check = $stmt_check->get_result();
    $row_check = $res_check->fetch_assoc();
    $stmt_check->close();

    // Compare with current user id ($userId defined above)
    if ($row_check && (int)$row_check['dibuat_oleh'] === $userId) {
        // Allow deletion even if validated — owner may delete their own letter
        // First, fetch file paths to remove files
        $stmt_files = $conn->prepare("SELECT file_surat, lampiran_path, signature_path, qr_code FROM office WHERE id = ?");
        $stmt_files->bind_param('i', $letter_id);
        $stmt_files->execute();
        $res_files = $stmt_files->get_result();
        $files = $res_files ? $res_files->fetch_assoc() : null;
        $stmt_files->close();

        if ($files) {
            if (!empty($files['file_surat'])) {
                $filePath = __DIR__ . '/../' . ltrim($files['file_surat'], '/');
                if (is_file($filePath)) @unlink($filePath);
            }
            if (!empty($files['signature_path'])) {
                $sigPath = __DIR__ . '/../' . ltrim($files['signature_path'], '/');
                if (is_file($sigPath)) @unlink($sigPath);
            }
            if (!empty($files['qr_code'])) {
                $qrPath = __DIR__ . '/../' . ltrim($files['qr_code'], '/');
                if (is_file($qrPath)) @unlink($qrPath);
            }
            if (!empty($files['lampiran_path'])) {
                $paths = json_decode($files['lampiran_path'], true);
                if (is_array($paths)) {
                    foreach ($paths as $p) {
                        $abs = __DIR__ . '/../' . ltrim($p, '/');
                        if (is_file($abs)) @unlink($abs);
                    }
                }
            }
        }

        // Remove approval chain entries for this letter
        $del_chain = $conn->prepare("DELETE FROM urutan_paraf WHERE office_id = ?");
        $del_chain->bind_param('i', $letter_id);
        $del_chain->execute();
        $del_chain->close();

        // Finally delete the office record
        $stmt_del = $conn->prepare("DELETE FROM office WHERE id = ?");
        $stmt_del->bind_param('i', $letter_id);
        if ($stmt_del->execute()) {
            $_SESSION['success_msg'] = 'Surat berhasil dihapus.';
            error_log(sprintf('LETTER_DELETE: user_id=%d deleted office_id=%d', $userId, $letter_id));
        } else {
            $_SESSION['error_msg'] = 'Gagal menghapus surat.';
            error_log(sprintf('LETTER_DELETE_FAIL: user_id=%d office_id=%d', $userId, $letter_id));
        }
        $stmt_del->close();
    } else {
        $_SESSION['error_msg'] = 'Anda hanya dapat menghapus surat milik Anda sendiri.';
    }
    header('Location: record-letter.php');
    exit();
}

// View mode toggle
$viewMode = $_GET['view'] ?? ($isSupervisor || $isApprover ? 'bawahan' : 'own'); // 'own' or 'bawahan'
$search = $_GET['search'] ?? '';
$limitOptions = [20, 100, 'All'];
$selectedLimit = $_GET['limit'] ?? 20;
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;

// Build base query
$where = "WHERE 1=1";
if (!empty($search)) {
    $esc = $conn->real_escape_string($search);
    $where .= " AND (o.perihal LIKE '%$esc%' OR o.nomor_surat LIKE '%$esc%' OR u.nama LIKE '%$esc%')";
}
if ($viewMode === 'bawahan' && ($isSupervisor || $isApprover) && !empty($subordinateIds)) {
    $allIds = array_unique(array_merge([$userId], $subordinateIds));
    $where .= " AND dibuat_oleh IN (" . implode(',', $allIds) . ")";
} else {
    $where .= " AND dibuat_oleh = $userId";
}

// Count total
$sqlCount = "SELECT COUNT(*) as c FROM office o LEFT JOIN users u ON u.id = o.dibuat_oleh $where";
$resCount = $conn->query($sqlCount);
$total = $resCount ? (int)$resCount->fetch_assoc()['c'] : 0;

$offset = ($page - 1) * ($selectedLimit === 'All' ? $total : (int)$selectedLimit);
$limitClause = $selectedLimit === 'All' ? '' : " LIMIT $offset, " . (int)$selectedLimit;

$sql = "SELECT o.*, u.nama AS pembuat_nama, u.jabatan AS pembuat_jabatan, 
        signer.nama AS nama_penandatangan, signer.jabatan AS jabatan_penandatangan,
        lc.catatan, lc.user_id as catatan_user_id, lc.created_at as catatan_created_at
        FROM office o 
        LEFT JOIN users u ON u.id = o.dibuat_oleh 
        LEFT JOIN users signer ON signer.id = o.dari_pribadi
        LEFT JOIN letter_catatan lc ON lc.office_id = o.id AND lc.user_id = $userId
        $where ORDER BY o.id DESC" . $limitClause;
$res = $conn->query($sql);
$rows = $res ? $res->fetch_all(MYSQLI_ASSOC) : [];
$totalPages = ($selectedLimit === 'All') ? 1 : ceil($total / (int)$selectedLimit);

?>

<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="utf-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1" />
    <title>Surat Anda - Record</title>
    <link href="/jasa_marga/img/baru/logojp.png" rel="icon">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        .greeting {
            text-align: center;
            animation: fadeIn 2s ease-in-out;
        }

        .sidebar {
            background-color: #0c50e1;
            color: white;
            min-height: 100vh;
            padding: 15px;
            width: 250px;
            flex-shrink: 0;
            overflow-y: auto;
        }

        .sidebar a {
            color: white;
            text-decoration: none;
            display: block;
            padding: 10px;
            margin: 5px 0;
            border-radius: 5px;
            transition: all 0.3s ease;
        }

        .sidebar a:hover,
        .sidebar a.active {
            background-color: #ffc107;
            color: #0047ab;
            transform: translateX(5px);
        }

        .btn-success {
            background-color: #ffc107;
            border-color: #ffc107;
            color: #0047ab;
            font-weight: bold;
        }

        .btn-success:hover {
            background-color: #e0a800;
            border-color: #e0a800;
            color: #0047ab;
        }

        .table thead {
            background-color: #0047ab;
            color: #ffc107;
            text-transform: uppercase;
            font-weight: bold;
        }

        .table thead th {
            background-color: #0c50e1;
            color: #ffc107;
        }

        .card {
            border: 1px solid #dee2e6;
            box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
        }

        .submenu {
            padding-left: 20px;
        }

        .submenu a {
            font-size: 0.9rem;
            padding: 8px 10px;
        }

        .submenu-toggle:hover {
            background-color: rgba(255, 193, 7, 0.1);
        }

        .arrow-icon {
            margin-left: auto;
        }

        .flex-grow-1 {
            flex: 1;
        }
    </style>
</head>

<body>
    <div class="d-flex">
        <?php include __DIR__ . '/../includes/sidebar-user.php'; ?>
        <div class="flex-grow-1 p-3">
            <div class="d-flex justify-content-between align-items-center mb-3">
                <div class="d-flex align-items-center">
                    <h4 class="mb-0 me-3">Record Surat</h4>
                    <form method="get" class="me-2">
                        <input type="hidden" name="view" value="<?= htmlspecialchars($viewMode) ?>">
                        <label class="me-1 fw-bold">Show</label>
                        <select name="limit" onchange="this.form.submit()" class="form-select form-select-sm w-auto">
                            <?php foreach ($limitOptions as $option): ?>
                                <option value="<?= $option ?>" <?= $selectedLimit == $option ? 'selected' : '' ?>><?= $option ?></option>
                            <?php endforeach; ?>
                        </select>
                    </form>
                </div>
                <div>
                    <form method="get" class="d-flex align-items-center">
                        <input type="hidden" name="view" value="<?= htmlspecialchars($viewMode) ?>">
                        <input type="hidden" name="limit" value="<?= htmlspecialchars($selectedLimit) ?>">
                        <input type="text" name="search" value="<?= htmlspecialchars($search) ?>" class="form-control form-control-sm me-2" placeholder="Cari...">
                        <button class="btn btn-sm btn-primary" type="submit">Cari</button>
                        <?php if ((!empty($access_modul['surat']) && $access_modul['surat'] === 'on') || $isApprover || $isSupervisor): ?>
                            <a href="create-letter.php" class="btn btn-sm btn-primary ms-2">+ Buat Surat</a>
                        <?php endif; ?>
                    </form>
                </div>
            </div>

            <div class="mb-3">
                <a href="?view=own" class="btn btn-sm btn-outline-secondary <?= ($viewMode === 'own') ? 'active' : '' ?>">Surat Saya</a>
                <?php if ($isSupervisor || $isApprover): ?>
                    <a href="?view=bawahan" class="btn btn-sm btn-outline-warning <?= ($viewMode === 'bawahan') ? 'active' : '' ?>">Surat Bawahan</a>
                <?php endif; ?>
            </div>

            <table class="table table-bordered">
                <thead>
                    <tr>
                        <th>Tanggal</th>
                        <th>Nomor</th>
                        <th>Dibuat Oleh</th>
                        <th>Penandatangan</th>
                        <th>Perihal</th>
                        <th>Status Approval</th>
                        <th>Catatan</th>
                        <th>File</th>
                        <th>Aksi</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($rows)): ?>
                        <tr>
                            <td colspan="9" class="text-center">Belum ada surat.</td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($rows as $row): ?>
                            <tr>
                                <td><?= htmlspecialchars($row['tanggal_surat']) ?></td>
                                <td>
                                    <?php if (!empty($row['tipe']) && $row['tipe'] === 'Surat Keluar'): ?>
                                        <?= htmlspecialchars(getKodeFromJabatan($row['pembuat_jabatan'])) ?>
                                    <?php endif; ?>
                                    <?= htmlspecialchars($row['nomor_surat']) ?>
                                </td>
                                <td><?= htmlspecialchars($row['pembuat_nama']) ?><br><small class="text-muted"><?= (!empty($row['pembuat_jabatan']) ? htmlspecialchars($row['pembuat_jabatan']) : '-') ?></small></td>
                                <td>
                                    <?php if (!empty($row['nama_penandatangan'])): ?>
                                        <?= htmlspecialchars($row['nama_penandatangan']) ?><br><small class="text-muted"><?= htmlspecialchars($row['jabatan_penandatangan'] ?? '-') ?></small>
                                    <?php else: ?>
                                        <span class="text-muted">-</span>
                                    <?php endif; ?>
                                </td>
                                <td><?= htmlspecialchars($row['perihal']) ?></td>
                                <td>
                                    <?php
                                    // Get approval chain status for this letter
                                    $approvalStatus = getLetterApprovalChainStatus($conn, (int)$row['id']);
                                    $isUserInChain = isUserInApprovalChain($approvalStatus, $userId);
                                    $canUserApprove = canUserApproveLetter($approvalStatus, $userId);

                                    if ($approvalStatus['total_steps'] > 0):
                                    ?>
                                        <div class="approval-chain-status">
                                            <?php if ($approvalStatus['is_complete']): ?>
                                                <span class="badge bg-success mb-1">
                                                    <i class="fas fa-check-circle"></i> Approval Lengkap
                                                </span><br>
                                            <?php else: ?>
                                                <span class="badge bg-warning text-dark mb-1">
                                                    Step <?= $approvalStatus['current_step'] ?>/<?= $approvalStatus['total_steps'] ?>
                                                </span><br>
                                            <?php endif; ?>

                                            <?php if (!empty($approvalStatus['paraf_display'])): ?>
                                                <small class="text-success"><i class="fas fa-signature"></i> <?= htmlspecialchars($approvalStatus['paraf_display']) ?></small><br>
                                            <?php endif; ?>

                                            <small class="text-muted">
                                                <?php foreach ($approvalStatus['chain'] as $idx => $step): ?>
                                                    <span class="<?= $step['approved_at'] ? 'text-success' : 'text-secondary' ?>" title="<?= htmlspecialchars($step['approver_name'] ?? 'Unknown') ?>">
                                                        <?= $step['approved_at'] ? '✓' : '○' ?><?= htmlspecialchars($step['inisial'] ?? '?') ?>
                                                    </span>
                                                    <?php if ($idx < count($approvalStatus['chain']) - 1) echo '→'; ?>
                                                <?php endforeach; ?>
                                            </small>
                                        </div>
                                    <?php else: ?>
                                        <small class="text-muted">
                                            <i class="fas fa-clock"></i> Menunggu inisialisasi
                                        </small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php
                                    // Check if current user can add notes (supervisor/approver of letter creator)
                                    $canAddNote = false;
                                    $creatorId = (int)$row['dibuat_oleh'];
                                    if ($creatorId !== $userId && ($isSupervisor || $isApprover)) {
                                        // Check if current user is a supervisor or approver of the creator
                                        $canAddNote = in_array($creatorId, $subordinateIds);
                                    }
                                    ?>
                                    <div class="catatan-cell">
                                        <?php if (!empty($row['catatan'])): ?>
                                            <div class="alert alert-info mb-2 p-2" style="font-size: 0.9rem;">
                                                <small>
                                                    <strong>Catatan:</strong><br>
                                                    <?= htmlspecialchars($row['catatan']) ?>
                                                    <?php if (!empty($row['catatan_created_at'])): ?>
                                                        <br><em class="text-muted"><?= date('d/m/Y H:i', strtotime($row['catatan_created_at'])) ?></em>
                                                    <?php endif; ?>
                                                </small>
                                            </div>
                                        <?php endif; ?>
                                        <?php if ($canAddNote): ?>
                                            <button type="button" class="btn btn-sm btn-outline-primary btn-add-catatan"
                                                data-id="<?= (int)$row['id'] ?>"
                                                data-nomor="<?= htmlspecialchars($row['nomor_surat']) ?>"
                                                title="Tambah/Edit catatan untuk surat ini">
                                                <i class="fas fa-sticky-note"></i>
                                                <?= !empty($row['catatan']) ? 'Edit' : 'Tambah' ?>
                                            </button>
                                        <?php elseif (empty($row['catatan'])): ?>
                                            <small class="text-muted">-</small>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td>
                                    <?php if (!empty($row['file_surat'])): ?>
                                        <?php
                                        $fileUrl = $row['file_surat'];
                                        if (!preg_match('#^(https?://|/)#i', $fileUrl)) {
                                            $fileUrl = '/jasa_marga/' . ltrim($fileUrl, '/');
                                        }
                                        ?>
                                        <a href="<?= htmlspecialchars($fileUrl) ?>" target="_blank" class="btn btn-sm btn-outline-success" title="Preview"><i class="fas fa-eye"></i></a>
                                        <a href="<?= htmlspecialchars($fileUrl) ?>" download class="btn btn-sm btn-outline-primary" title="Unduh"><i class="fas fa-download"></i></a>
                                    <?php else: ?>
                                        <span class="text-muted">-</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php
                                    $creatorId = (int)$row['dibuat_oleh'];
                                    $isOwner = ($creatorId === $userId);
                                    $isValidated = ((int)($row['is_validated'] ?? 0) === 1);
                                    ?>
                                    <?php if ($isValidated): ?>
                                        <span class="badge bg-success me-1">Approved</span>
                                        <?php if ($isOwner): ?>
                                            <a href="edit-letter.php?id=<?= (int)$row['id'] ?>" class="btn btn-sm btn-outline-warning" title="Edit surat (akan membatalkan approval)">
                                                <i class="fas fa-edit"></i> Edit
                                            </a>
                                            <form method="post" style="display:inline" onsubmit="return confirm('Yakin ingin menghapus surat ini (termasuk yang sudah tervalidasi)?');">
                                                <input type="hidden" name="delete_letter" value="1">
                                                <input type="hidden" name="letter_id" value="<?= (int)$row['id'] ?>">
                                                <button type="submit" class="btn btn-sm btn-danger">Hapus</button>
                                            </form>
                                        <?php endif; ?>
                                    <?php else: ?>
                                        <?php // Sequential approval - show button based on chain position 
                                        ?>

                                        <?php if ($isOwner && $approvalStatus['total_steps'] === 0): ?>
                                            <!-- Owner: Initialize approval chain first -->
                                            <button type="button" class="btn btn-sm btn-info btn-init-approval"
                                                data-id="<?= (int)$row['id'] ?>"
                                                data-nomor="<?= htmlspecialchars($row['nomor_surat']) ?>"
                                                data-signer="<?= (int)($row['dari_pribadi'] ?? 0) ?>">
                                                <i class="fas fa-play-circle"></i> Mulai Approval
                                            </button>
                                            <a href="edit-letter.php?id=<?= (int)$row['id'] ?>" class="btn btn-sm btn-outline-secondary">Edit</a>
                                            <form method="post" style="display:inline" onsubmit="return confirm('Yakin ingin menghapus surat ini?');">
                                                <input type="hidden" name="delete_letter" value="1">
                                                <input type="hidden" name="letter_id" value="<?= (int)$row['id'] ?>">
                                                <button type="submit" class="btn btn-sm btn-danger">Hapus</button>
                                            </form>
                                        <?php elseif ($canUserApprove): ?>
                                            <!-- User can approve (their turn) -->
                                            <button type="button" class="btn btn-sm btn-warning btn-approve"
                                                data-id="<?= (int)$row['id'] ?>"
                                                data-nomor="<?= htmlspecialchars($row['nomor_surat']) ?>"
                                                data-step="<?= $approvalStatus['current_step'] + 1 ?>"
                                                data-total="<?= $approvalStatus['total_steps'] ?>">
                                                <i class="fas fa-check-circle"></i> Approval
                                            </button>
                                            <button type="button" class="btn btn-sm btn-outline-info btn-view-chain"
                                                data-id="<?= (int)$row['id'] ?>"
                                                data-nomor="<?= htmlspecialchars($row['nomor_surat']) ?>">
                                                <i class="fas fa-list-ol"></i> Detail
                                            </button>
                                            <?php if ($isOwner): ?>
                                                <a href="edit-letter.php?id=<?= (int)$row['id'] ?>" class="btn btn-sm btn-outline-secondary">Edit</a>
                                                <form method="post" style="display:inline" onsubmit="return confirm('Yakin ingin menghapus surat ini?');">
                                                    <input type="hidden" name="delete_letter" value="1">
                                                    <input type="hidden" name="letter_id" value="<?= (int)$row['id'] ?>">
                                                    <button type="submit" class="btn btn-sm btn-outline-danger" title="Hapus"><i class="fas fa-trash"></i></button>
                                                </form>
                                            <?php endif; ?>
                                        <?php elseif ($isUserInChain): ?>
                                            <!-- User is in chain but not their turn -->
                                            <span class="badge bg-secondary me-1">Menunggu giliran</span>
                                            <button type="button" class="btn btn-sm btn-outline-info btn-view-chain"
                                                data-id="<?= (int)$row['id'] ?>"
                                                data-nomor="<?= htmlspecialchars($row['nomor_surat']) ?>">
                                                <i class="fas fa-list-ol"></i> Detail
                                            </button>
                                        <?php elseif ($isOwner): ?>
                                            <!-- Owner but approval in progress -->
                                            <span class="badge bg-info me-1">Proses approval</span>
                                            <button type="button" class="btn btn-sm btn-outline-info btn-view-chain"
                                                data-id="<?= (int)$row['id'] ?>"
                                                data-nomor="<?= htmlspecialchars($row['nomor_surat']) ?>">
                                                <i class="fas fa-list-ol"></i> Detail
                                            </button>
                                            <a href="edit-letter.php?id=<?= (int)$row['id'] ?>" class="btn btn-sm btn-outline-secondary">Edit</a>
                                            <form method="post" style="display:inline" onsubmit="return confirm('Yakin ingin menghapus surat ini?');">
                                                <input type="hidden" name="delete_letter" value="1">
                                                <input type="hidden" name="letter_id" value="<?= (int)$row['id'] ?>">
                                                <button type="submit" class="btn btn-sm btn-danger">Hapus</button>
                                            </form>
                                        <?php else: ?>
                                            <!-- Not owner, not in chain - view only -->
                                            <button type="button" class="btn btn-sm btn-secondary btn-preview"
                                                data-id="<?= (int)$row['id'] ?>"
                                                data-nomor="<?= htmlspecialchars($row['nomor_surat']) ?>"
                                                data-pembuat="<?= htmlspecialchars($row['pembuat_nama']) ?>"
                                                data-kode="<?= htmlspecialchars(getKodeFromJabatan($row['pembuat_jabatan'])) ?>"
                                                data-perihal="<?= htmlspecialchars($row['perihal']) ?>"
                                                data-file="<?= htmlspecialchars($row['file_surat'] ?? '') ?>">
                                                Preview
                                            </button>
                                            <a href="verify-letter.php?id=<?= (int)$row['id'] ?>" target="_blank" class="btn btn-sm btn-primary">Check</a>
                                        <?php endif; ?>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
            <?php if ($totalPages > 1): ?>
                <nav aria-label="Page navigation">
                    <ul class="pagination justify-content-center mt-3">
                        <li class="page-item <?= ($page <= 1) ? 'disabled' : '' ?>"><a class="page-link" href="?limit=<?= urlencode($selectedLimit) ?>&search=<?= urlencode($search) ?>&view=<?= urlencode($viewMode) ?>&page=<?= max(1, $page - 1) ?>">Prev</a></li>
                        <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                            <li class="page-item <?= ($page == $i) ? 'active' : '' ?>"><a class="page-link" href="?limit=<?= urlencode($selectedLimit) ?>&search=<?= urlencode($search) ?>&view=<?= urlencode($viewMode) ?>&page=<?= $i ?>"><?= $i ?></a></li>
                        <?php endfor; ?>
                        <li class="page-item <?= ($page >= $totalPages) ? 'disabled' : '' ?>"><a class="page-link" href="?limit=<?= urlencode($selectedLimit) ?>&search=<?= urlencode($search) ?>&view=<?= urlencode($viewMode) ?>&page=<?= min($totalPages, $page + 1) ?>">Next</a></li>
                    </ul>
                </nav>
            <?php endif; ?>

        </div>
    </div>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>

    <!-- Validation Modal -->
    <div class="modal fade" id="validationModal" tabindex="-1" aria-labelledby="validationModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header bg-warning">
                    <h5 class="modal-title" id="validationModalLabel"><i class="fas fa-check-circle"></i> Approval Surat</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form id="validationForm">
                    <div class="modal-body">
                        <div class="alert alert-info">
                            <i class="fas fa-info-circle"></i>
                            Anda akan melakukan approval surat nomor: <strong id="validationLetterNumber"></strong>
                            <br><small class="text-muted">Step <span id="approvalStepInfo"></span></small>
                        </div>

                        <!-- Approval Chain Status -->
                        <div class="mb-3" id="approvalChainDisplay">
                            <h6><i class="fas fa-list-ol"></i> Status Approval Chain</h6>
                            <div id="approvalChainContent" class="p-2 border rounded bg-light">
                                <!-- Will be populated by JS -->
                            </div>
                        </div>

                        <h6 class="mb-3"><i class="fas fa-signature"></i> Tanda Tangan Digital (Opsional)</h6>
                        <p class="text-muted small">Anda dapat menambahkan tanda tangan dengan menggambar di area canvas atau mengunggah file gambar.</p>

                        <!-- Signature Canvas -->
                        <div class="mb-3">
                            <label class="form-label">Gambar Tanda Tangan:</label>
                            <div class="border border-2 border-dashed p-2" style="background: #f8f9fa;">
                                <canvas id="signatureCanvas" width="600" height="200" style="border: 1px solid #dee2e6; background: white; cursor: crosshair; width: 100%; height: 150px;"></canvas>
                            </div>
                            <button type="button" id="clearSignature" class="btn btn-sm btn-outline-danger mt-2">
                                <i class="fas fa-eraser"></i> Hapus Tanda Tangan
                            </button>
                        </div>

                        <!-- File Upload -->
                        <div class="mb-3">
                            <label for="signatureFile" class="form-label">Atau Unggah File Tanda Tangan:</label>
                            <input type="file" class="form-control" id="signatureFile" accept="image/*">
                            <div class="form-text">Format yang didukung: JPG, PNG, GIF (Maks. 5MB)</div>

                            <!-- Preview uploaded signature -->
                            <div id="signaturePreview" style="display: none;" class="mt-2">
                                <label class="form-label">Preview:</label><br>
                                <img id="signaturePreviewImg" src="" alt="Signature Preview" style="max-width: 300px; max-height: 100px; border: 1px solid #dee2e6;">
                            </div>
                        </div>

                        <!-- Comment/Catatan -->
                        <div class="mb-3">
                            <label for="approvalComment" class="form-label"><i class="fas fa-comment"></i> Catatan (Opsional)</label>
                            <textarea class="form-control" id="approvalComment" rows="2" placeholder="Tambahkan catatan untuk approval ini..."></textarea>
                        </div>

                        <div class="alert alert-warning">
                            <i class="fas fa-exclamation-triangle"></i>
                            <strong>Perhatian:</strong> Setelah Anda approve, giliran akan berpindah ke approver berikutnya dalam chain.
                        </div>
                    </div>
                    <div class="modal-footer">
                        <input type="hidden" id="validationLetterId" value="">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" class="btn btn-success">
                            <i class="fas fa-check-circle"></i> Approve Surat
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Initialize Approval Modal -->
    <div class="modal fade" id="initApprovalModal" tabindex="-1" aria-labelledby="initApprovalModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header bg-info text-white">
                    <h5 class="modal-title" id="initApprovalModalLabel"><i class="fas fa-play-circle"></i> Mulai Proses Approval</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form id="initApprovalForm">
                    <div class="modal-body">
                        <div class="alert alert-info">
                            <i class="fas fa-info-circle"></i>
                            Anda akan memulai proses approval untuk surat nomor: <br>
                            <strong id="initLetterNumber"></strong>
                        </div>

                        <p>Proses ini akan:</p>
                        <ul>
                            <li>Membuat urutan approval berdasarkan penandatangan yang dipilih</li>
                            <li>Memberikan giliran approval pertama kepada approver pertama dalam chain</li>
                        </ul>

                        <div class="alert alert-warning">
                            <i class="fas fa-exclamation-triangle"></i>
                            <strong>Catatan:</strong> Setelah proses approval dimulai, surat masih bisa diedit. Approval chain akan tetap aktif.
                        </div>
                    </div>
                    <div class="modal-footer">
                        <input type="hidden" id="initLetterId" value="">
                        <input type="hidden" id="initSignerId" value="">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" class="btn btn-info">
                            <i class="fas fa-play-circle"></i> Mulai Approval
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Approval Chain Detail Modal -->
    <div class="modal fade" id="chainDetailModal" tabindex="-1" aria-labelledby="chainDetailModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title" id="chainDetailModalLabel"><i class="fas fa-list-ol"></i> Detail Approval Chain</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-info mb-3">
                        Surat Nomor: <strong id="chainDetailLetterNumber"></strong>
                    </div>

                    <!-- Chain Timeline -->
                    <div id="chainTimeline" class="mb-4">
                        <!-- Will be populated by JS -->
                    </div>

                    <!-- Comments Section -->
                    <h6><i class="fas fa-comments"></i> Catatan/Komentar</h6>
                    <div id="commentsSection" class="border rounded p-2" style="max-height: 200px; overflow-y: auto;">
                        <!-- Will be populated by JS -->
                    </div>

                    <!-- Add Comment Form (for users in chain) -->
                    <div id="addCommentSection" class="mt-3" style="display: none;">
                        <div class="input-group">
                            <input type="text" class="form-control" id="newComment" placeholder="Tambah catatan...">
                            <button type="button" class="btn btn-primary" id="btnAddComment">
                                <i class="fas fa-paper-plane"></i>
                            </button>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <input type="hidden" id="chainDetailLetterId" value="">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Tutup</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Catatan Modal -->
    <div class="modal fade" id="catatanModal" tabindex="-1" aria-labelledby="catatanModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="catatanModalLabel">Catatan Surat</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Nomor Surat</label>
                        <div id="catatanNomor" class="form-control-plaintext fw-bold"></div>
                    </div>
                    <div class="mb-3">
                        <label for="catatanText" class="form-label">Catatan</label>
                        <textarea class="form-control" id="catatanText" rows="4" placeholder="Masukkan catatan untuk surat ini..."></textarea>
                        <small class="text-muted">Catatan hanya terlihat untuk Anda dan dapat diedit kapan saja.</small>
                    </div>
                    <input type="hidden" id="catatanOfficeId">
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                    <button type="button" class="btn btn-primary" id="btnSaveCatatan">Simpan Catatan</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Preview Modal -->
    <div class="modal fade" id="previewModal" tabindex="-1" aria-labelledby="previewModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="previewModalLabel">Preview Surat</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <dl class="row">
                        <dt class="col-sm-3">Nomor</dt>
                        <dd class="col-sm-9" id="previewNomor"></dd>
                        <dt class="col-sm-3">Pembuat</dt>
                        <dd class="col-sm-9" id="previewPembuat"></dd>
                        <dt class="col-sm-3">Kode Jabatan</dt>
                        <dd class="col-sm-9" id="previewKode"></dd>
                        <dt class="col-sm-3">Perihal</dt>
                        <dd class="col-sm-9" id="previewPerihal"></dd>
                        <dt class="col-sm-3">File</dt>
                        <dd class="col-sm-9" id="previewFile"></dd>
                    </dl>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>
    </div>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.all.min.js"></script>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css">
    <style>
        .approval-timeline {
            position: relative;
            padding-left: 30px;
        }

        .approval-timeline::before {
            content: '';
            position: absolute;
            left: 10px;
            top: 0;
            bottom: 0;
            width: 2px;
            background: #dee2e6;
        }

        .approval-step {
            position: relative;
            padding: 10px 0;
            border-bottom: 1px solid #f0f0f0;
        }

        .approval-step:last-child {
            border-bottom: none;
        }

        .approval-step::before {
            content: '';
            position: absolute;
            left: -24px;
            top: 14px;
            width: 12px;
            height: 12px;
            border-radius: 50%;
            background: #6c757d;
        }

        .approval-step.approved::before {
            background: #28a745;
        }

        .approval-step.current::before {
            background: #ffc107;
            animation: pulse 1.5s infinite;
        }

        @keyframes pulse {
            0% {
                box-shadow: 0 0 0 0 rgba(255, 193, 7, 0.7);
            }

            70% {
                box-shadow: 0 0 0 10px rgba(255, 193, 7, 0);
            }

            100% {
                box-shadow: 0 0 0 0 rgba(255, 193, 7, 0);
            }
        }
    </style>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const API_BASE = '../api/approval.php';
            const API_CATATAN = './api-catatan.php';

            // Catatan Modal Handler
            const catatanModal = new bootstrap.Modal(document.getElementById('catatanModal'));
            document.querySelectorAll('.btn-add-catatan').forEach(btn => {
                btn.addEventListener('click', function() {
                    const officeId = this.dataset.id;
                    const nomorSurat = this.dataset.nomor;

                    document.getElementById('catatanOfficeId').value = officeId;
                    document.getElementById('catatanNomor').textContent = nomorSurat;

                    // Load existing catatan
                    fetch(`${API_CATATAN}?action=get&office_id=${officeId}`)
                        .then(r => r.json())
                        .then(data => {
                            if (data.success) {
                                document.getElementById('catatanText').value = data.catatan || '';
                            }
                        })
                        .catch(e => console.error('Error loading catatan:', e));

                    catatanModal.show();
                });
            });

            // Save Catatan
            document.getElementById('btnSaveCatatan').addEventListener('click', function() {
                const officeId = document.getElementById('catatanOfficeId').value;
                const catatan = document.getElementById('catatanText').value.trim();

                const formData = new FormData();
                formData.append('action', 'save');
                formData.append('office_id', officeId);
                formData.append('catatan', catatan);

                fetch(API_CATATAN, {
                        method: 'POST',
                        body: formData
                    })
                    .then(r => r.json())
                    .then(data => {
                        if (data.success) {
                            Swal.fire({
                                icon: 'success',
                                title: 'Berhasil',
                                text: data.message || 'Catatan berhasil disimpan',
                                timer: 1500
                            }).then(() => {
                                catatanModal.hide();
                                // Refresh the page to show updated catatan
                                setTimeout(() => location.reload(), 500);
                            });
                        } else {
                            Swal.fire({
                                icon: 'error',
                                title: 'Gagal',
                                text: data.error || 'Gagal menyimpan catatan'
                            });
                        }
                    })
                    .catch(e => {
                        console.error('Error:', e);
                        Swal.fire({
                            icon: 'error',
                            title: 'Error',
                            text: 'Terjadi kesalahan saat menyimpan catatan'
                        });
                    });
            });

            // Preview Modal Handler
            const previewModal = new bootstrap.Modal(document.getElementById('previewModal'));
            document.querySelectorAll('.btn-preview').forEach(btn => {
                btn.addEventListener('click', function() {
                    document.getElementById('previewNomor').textContent = this.dataset.nomor || '';
                    document.getElementById('previewPembuat').textContent = this.dataset.pembuat || '';
                    document.getElementById('previewKode').textContent = this.dataset.kode || '';
                    document.getElementById('previewPerihal').textContent = this.dataset.perihal || '';
                    const file = this.dataset.file || '';
                    if (file) {
                        const abs = (/^(https?:\/\/|\/)/i.test(file) ? file : ('/jasa_marga/' + file.replace(/^\//, '')));
                        document.getElementById('previewFile').innerHTML = `<a href="${abs}" target="_blank">Open File</a>`;
                    } else {
                        document.getElementById('previewFile').textContent = 'Tidak ada';
                    }
                    previewModal.show();
                });
            });

            // Initialize Approval Modal Handler
            const initApprovalModal = new bootstrap.Modal(document.getElementById('initApprovalModal'));
            document.querySelectorAll('.btn-init-approval').forEach(btn => {
                btn.addEventListener('click', function() {
                    document.getElementById('initLetterId').value = this.dataset.id;
                    document.getElementById('initSignerId').value = this.dataset.signer;
                    document.getElementById('initLetterNumber').textContent = this.dataset.nomor;
                    initApprovalModal.show();
                });
            });

            // Handle Init Approval Form
            document.getElementById('initApprovalForm').addEventListener('submit', function(e) {
                e.preventDefault();

                const letterId = document.getElementById('initLetterId').value;
                const signerId = document.getElementById('initSignerId').value;

                Swal.fire({
                    title: 'Memproses...',
                    text: 'Menginisialisasi approval chain',
                    allowOutsideClick: false,
                    showConfirmButton: false,
                    didOpen: () => Swal.showLoading()
                });

                const formData = new FormData();
                formData.append('action', 'initialize_approval_chain');
                formData.append('office_id', letterId);
                formData.append('signer_id', signerId);

                fetch(API_BASE, {
                        method: 'POST',
                        body: formData
                    })
                    .then(r => r.json())
                    .then(data => {
                        if (data.success) {
                            Swal.fire({
                                icon: 'success',
                                title: 'Berhasil!',
                                text: data.message || 'Approval chain berhasil diinisialisasi',
                                confirmButtonColor: '#28a745'
                            }).then(() => location.reload());
                        } else {
                            Swal.fire({
                                icon: 'error',
                                title: 'Gagal!',
                                text: data.message || 'Gagal menginisialisasi approval chain',
                                confirmButtonColor: '#dc3545'
                            });
                        }
                    })
                    .catch(err => {
                        Swal.fire({
                            icon: 'error',
                            title: 'Error!',
                            text: err.message,
                            confirmButtonColor: '#dc3545'
                        });
                    });
            });

            // Approval Button Handler - Redirect to edit-letter.php with approval mode
            document.querySelectorAll('.btn-approve').forEach(btn => {
                btn.addEventListener('click', function() {
                    const letterId = this.dataset.id;
                    const nomorSurat = this.dataset.nomor;
                    const step = this.dataset.step;
                    const total = this.dataset.total;

                    // Show confirmation before redirecting to edit page
                    Swal.fire({
                        title: 'Approval Surat',
                        html: `<div class="text-start">
                            <p>Anda akan melakukan approval untuk surat:</p>
                            <p><strong>${nomorSurat}</strong></p>
                            <hr>
                            <p><i class="fas fa-list-ol"></i> Step: <strong>${step} dari ${total}</strong></p>
                            <p class="text-muted small">Anda akan diarahkan ke halaman edit untuk mereview surat sebelum approval.</p>
                        </div>`,
                        icon: 'info',
                        showCancelButton: true,
                        confirmButtonText: '<i class="fas fa-arrow-right"></i> Lanjutkan',
                        cancelButtonText: 'Batal',
                        confirmButtonColor: '#ffc107',
                        cancelButtonColor: '#6c757d'
                    }).then((result) => {
                        if (result.isConfirmed) {
                            // Redirect to edit-letter.php with approval mode
                            window.location.href = `edit-letter.php?id=${letterId}&mode=approval`;
                        }
                    });
                });
            });

            function loadApprovalChain(letterId) {
                fetch(`${API_BASE}?action=get_approval_status&office_id=${letterId}`)
                    .then(r => r.json())
                    .then(data => {
                        if (data.success && data.data.approvals) {
                            const container = document.getElementById('approvalChainContent');
                            container.innerHTML = data.data.approvals.map((a, i) => `
                            <div class="d-flex align-items-center mb-1">
                                <span class="badge ${a.approved_at ? 'bg-success' : 'bg-secondary'} me-2">
                                    ${a.approved_at ? '✓' : (i === data.data.last_approved_step ? '◉' : '○')}
                                </span>
                                <span class="${a.approved_at ? 'text-success' : ''}">${a.approver_name || 'Unknown'}</span>
                                <span class="ms-2 text-muted small">(${a.inisial || '?'})</span>
                                ${a.approved_at ? `<span class="ms-auto text-muted small">${new Date(a.approved_at).toLocaleString('id-ID')}</span>` : ''}
                            </div>
                        `).join('');
                        }
                    });
            }

            // Signature Canvas
            const canvas = document.getElementById('signatureCanvas');
            const ctx = canvas.getContext('2d');
            let isDrawing = false;

            canvas.addEventListener('mousedown', (e) => {
                isDrawing = true;
                const rect = canvas.getBoundingClientRect();
                const scaleX = canvas.width / rect.width;
                const scaleY = canvas.height / rect.height;
                ctx.beginPath();
                ctx.moveTo((e.clientX - rect.left) * scaleX, (e.clientY - rect.top) * scaleY);
            });

            canvas.addEventListener('mousemove', (e) => {
                if (isDrawing) {
                    const rect = canvas.getBoundingClientRect();
                    const scaleX = canvas.width / rect.width;
                    const scaleY = canvas.height / rect.height;
                    ctx.lineTo((e.clientX - rect.left) * scaleX, (e.clientY - rect.top) * scaleY);
                    ctx.stroke();
                }
            });

            canvas.addEventListener('mouseup', () => isDrawing = false);
            canvas.addEventListener('mouseout', () => isDrawing = false);

            // Touch support for mobile
            canvas.addEventListener('touchstart', (e) => {
                e.preventDefault();
                isDrawing = true;
                const rect = canvas.getBoundingClientRect();
                const touch = e.touches[0];
                const scaleX = canvas.width / rect.width;
                const scaleY = canvas.height / rect.height;
                ctx.beginPath();
                ctx.moveTo((touch.clientX - rect.left) * scaleX, (touch.clientY - rect.top) * scaleY);
            });

            canvas.addEventListener('touchmove', (e) => {
                e.preventDefault();
                if (isDrawing) {
                    const rect = canvas.getBoundingClientRect();
                    const touch = e.touches[0];
                    const scaleX = canvas.width / rect.width;
                    const scaleY = canvas.height / rect.height;
                    ctx.lineTo((touch.clientX - rect.left) * scaleX, (touch.clientY - rect.top) * scaleY);
                    ctx.stroke();
                }
            });

            canvas.addEventListener('touchend', () => isDrawing = false);

            document.getElementById('clearSignature').addEventListener('click', function() {
                ctx.clearRect(0, 0, canvas.width, canvas.height);
            });

            document.getElementById('signatureFile').addEventListener('change', function(e) {
                const file = e.target.files[0];
                if (file) {
                    const reader = new FileReader();
                    reader.onload = function(event) {
                        document.getElementById('signaturePreviewImg').src = event.target.result;
                        document.getElementById('signaturePreview').style.display = 'block';
                    };
                    reader.readAsDataURL(file);
                }
            });

            // Handle Approval Form Submission
            document.getElementById('validationForm').addEventListener('submit', function(e) {
                e.preventDefault();

                const letterId = document.getElementById('validationLetterId').value;
                const comment = document.getElementById('approvalComment').value;
                const fileInput = document.getElementById('signatureFile');

                Swal.fire({
                    title: 'Memproses...',
                    text: 'Melakukan approval surat',
                    allowOutsideClick: false,
                    showConfirmButton: false,
                    didOpen: () => Swal.showLoading()
                });

                const formData = new FormData();
                formData.append('action', 'submit_approval');
                formData.append('office_id', letterId);
                if (comment) formData.append('comment', comment);

                // Check for signature
                const isCanvasEmpty = isCanvasBlank(canvas);

                if (!isCanvasEmpty) {
                    canvas.toBlob(blob => {
                        if (blob) formData.append('signature', blob, 'signature.png');
                        sendApprovalRequest(formData);
                    });
                } else if (fileInput.files[0]) {
                    formData.append('signature', fileInput.files[0]);
                    sendApprovalRequest(formData);
                } else {
                    sendApprovalRequest(formData);
                }
            });

            function isCanvasBlank(canvas) {
                const ctx = canvas.getContext('2d');
                const pixelBuffer = new Uint32Array(
                    ctx.getImageData(0, 0, canvas.width, canvas.height).data.buffer
                );
                return !pixelBuffer.some(color => color !== 0);
            }

            function sendApprovalRequest(formData) {
                fetch(API_BASE, {
                        method: 'POST',
                        body: formData
                    })
                    .then(r => r.json())
                    .then(data => {
                        if (data.success) {
                            Swal.fire({
                                icon: 'success',
                                title: 'Berhasil!',
                                text: data.message || 'Approval berhasil disimpan',
                                confirmButtonColor: '#28a745'
                            }).then(() => location.reload());
                        } else {
                            Swal.fire({
                                icon: 'error',
                                title: 'Gagal!',
                                text: data.message || 'Terjadi kesalahan',
                                confirmButtonColor: '#dc3545'
                            });
                        }
                    })
                    .catch(err => {
                        Swal.fire({
                            icon: 'error',
                            title: 'Error!',
                            text: err.message,
                            confirmButtonColor: '#dc3545'
                        });
                    });
            }

            // Chain Detail Modal Handler
            const chainDetailModal = new bootstrap.Modal(document.getElementById('chainDetailModal'));
            document.querySelectorAll('.btn-view-chain').forEach(btn => {
                btn.addEventListener('click', function() {
                    const letterId = this.dataset.id;
                    const nomorSurat = this.dataset.nomor;

                    document.getElementById('chainDetailLetterId').value = letterId;
                    document.getElementById('chainDetailLetterNumber').textContent = nomorSurat;

                    loadChainDetail(letterId);
                    loadComments(letterId);

                    chainDetailModal.show();
                });
            });

            function loadChainDetail(letterId) {
                fetch(`${API_BASE}?action=get_approval_status&office_id=${letterId}`)
                    .then(r => r.json())
                    .then(data => {
                        if (data.success && data.data.approvals) {
                            const timeline = document.getElementById('chainTimeline');
                            timeline.innerHTML = '<div class="approval-timeline">' +
                                data.data.approvals.map((a, i) => {
                                    const isApproved = a.approved_at !== null;
                                    const isCurrent = !isApproved && i === data.data.last_approved_step;
                                    return `
                                    <div class="approval-step ${isApproved ? 'approved' : ''} ${isCurrent ? 'current' : ''}">
                                        <div class="d-flex justify-content-between">
                                            <div>
                                                <strong>${a.approver_name || 'Unknown'}</strong>
                                                <span class="badge bg-secondary ms-1">${a.inisial || '?'}</span>
                                                ${isCurrent ? '<span class="badge bg-warning text-dark ms-1">Giliran</span>' : ''}
                                            </div>
                                            <div>
                                                ${isApproved ? `<span class="text-success"><i class="fas fa-check-circle"></i> ${new Date(a.approved_at).toLocaleString('id-ID')}</span>` : '<span class="text-muted">Menunggu</span>'}
                                            </div>
                                        </div>
                                        ${a.comment ? `<div class="text-muted small mt-1"><i class="fas fa-comment"></i> ${a.comment}</div>` : ''}
                                    </div>
                                `;
                                }).join('') +
                                '</div>';

                            // Show/hide add comment section based on user's position
                            const addCommentSection = document.getElementById('addCommentSection');
                            addCommentSection.style.display = data.data.user_position ? 'block' : 'none';
                        }
                    });
            }

            function loadComments(letterId) {
                fetch(`${API_BASE}?action=get_comments&office_id=${letterId}`)
                    .then(r => r.json())
                    .then(data => {
                        const container = document.getElementById('commentsSection');
                        if (data.success && data.data && data.data.length > 0) {
                            container.innerHTML = data.data.map(c => `
                            <div class="mb-2 p-2 bg-white rounded border-start border-3 border-primary">
                                <div class="d-flex justify-content-between">
                                    <strong>${c.user_name || 'Unknown'}</strong>
                                    <small class="text-muted">${new Date(c.created_at).toLocaleString('id-ID')}</small>
                                </div>
                                <div>${c.comment}</div>
                            </div>
                        `).join('');
                        } else {
                            container.innerHTML = '<div class="text-muted text-center">Belum ada catatan</div>';
                        }
                    });
            }

            // Add Comment Handler
            document.getElementById('btnAddComment').addEventListener('click', function() {
                const letterId = document.getElementById('chainDetailLetterId').value;
                const comment = document.getElementById('newComment').value.trim();

                if (!comment) {
                    Swal.fire({
                        icon: 'warning',
                        title: 'Perhatian',
                        text: 'Masukkan catatan terlebih dahulu'
                    });
                    return;
                }

                const formData = new FormData();
                formData.append('action', 'add_comment');
                formData.append('office_id', letterId);
                formData.append('comment', comment);

                fetch(API_BASE, {
                        method: 'POST',
                        body: formData
                    })
                    .then(r => r.json())
                    .then(data => {
                        if (data.success) {
                            document.getElementById('newComment').value = '';
                            loadComments(letterId);
                            Swal.fire({
                                icon: 'success',
                                title: 'Berhasil',
                                text: 'Catatan ditambahkan',
                                timer: 1500,
                                showConfirmButton: false
                            });
                        } else {
                            Swal.fire({
                                icon: 'error',
                                title: 'Gagal',
                                text: data.message
                            });
                        }
                    });
            });

            // Legacy validation button (for old records)
            document.querySelectorAll('.btn-validate').forEach(button => {
                button.addEventListener('click', function() {
                    const letterId = this.dataset.id;
                    const nomorSurat = this.dataset.nomor;

                    document.getElementById('validationLetterId').value = letterId;
                    document.getElementById('validationLetterNumber').textContent = nomorSurat;
                    document.getElementById('approvalStepInfo').textContent = '-';
                    document.getElementById('approvalChainContent').innerHTML = '<div class="text-muted">Legacy approval (tanpa chain)</div>';

                    const canvas = document.getElementById('signatureCanvas');
                    const ctx = canvas.getContext('2d');
                    ctx.clearRect(0, 0, canvas.width, canvas.height);
                    document.getElementById('signatureFile').value = '';
                    document.getElementById('signaturePreview').style.display = 'none';
                    document.getElementById('approvalComment').value = '';

                    validationModal.show();
                });
            });
        });
    </script>
</body>

</html>