<?php
session_start();
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

include(__DIR__ . '/../includes/koneksi.php');
include(__DIR__ . '/../includes/functions.php');

if (!isset($_SESSION['username'])) {
    header('Location: ../login.php');
    exit();
}

$officeId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if ($officeId <= 0) {
    header('Location: record-letter.php?error=invalid_id');
    exit();
}

$sqlOffice = "
        SELECT 
            o.id, o.kepentingan, o.tipe_surat_id, o.jenis_surat_id, o.tempat_dibuat, 
            o.klasifikasi_arsip_id, o.subjenis_surat_id,
            o.nomor_surat, o.sequence_number, o.tanggal_surat, o.dari_tipe, o.dari_pribadi,
            o.kepada, o.perihal, o.paraf, o.naskah,
            o.lampiran_path, o.lampiran_orig_name, o.lampiran_mime, o.lampiran_size,
            o.file_surat, o.qr_code, o.signature_path,
            o.editor_history, o.dibuat_oleh, o.penandatangan_id, o.is_validated,
            ts.kode as tipe_surat_kode,
            ka.kode as klasifikasi_kode,
            js.kode as jenis_kode,
            ss.kode as subjenis_kode
        FROM office o
        LEFT JOIN tipe_surat ts ON ts.id = o.tipe_surat_id
        LEFT JOIN klasifikasi_arsip ka ON ka.id = o.klasifikasi_arsip_id
        LEFT JOIN jenis_surat js ON js.id = o.jenis_surat_id
        LEFT JOIN subjenis_surat ss ON ss.id = o.subjenis_surat_id
        WHERE o.id = ?
        LIMIT 1
    ";
$stOffice = $conn->prepare($sqlOffice);
$stOffice->bind_param('i', $officeId);
$stOffice->execute();
$resOffice = $stOffice->get_result();
$surat = $resOffice ? $resOffice->fetch_assoc() : null;
$resOffice?->free();
$stOffice->close();

if (!$surat) {
    header('Location: record-letter.php?error=not_found');
    exit();
}

// NOTE: Validation check moved AFTER user is fetched (see below)

// Initialize sequence_number if null/empty
if (empty($surat['sequence_number']) && !empty($surat['nomor_surat']) && !empty($surat['tipe_surat_kode'])) {
    $extractedSequence = extractSequenceFromNomorSurat($surat['nomor_surat'], $surat['tipe_surat_kode']);
    if ($extractedSequence !== null) {
        $surat['sequence_number'] = $extractedSequence;
        // Update database with extracted sequence
        $updateSeq = $conn->prepare("UPDATE office SET sequence_number = ? WHERE id = ?");
        $updateSeq->bind_param('ii', $extractedSequence, $officeId);
        $updateSeq->execute();
        $updateSeq->close();
    }
}

$oldFileSurat    = $surat['file_surat'] ?? '';
$oldLampiranPath = $surat['lampiran_path'] ?? null;

$oldEditorHistory = $surat['editor_history'] ?? null;

$currentLampiranList = [];
if ($oldLampiranPath) {
    $decoded = json_decode($oldLampiranPath, true);
    if (is_array($decoded)) {
        $currentLampiranList = $decoded;
    } else {
        $currentLampiranList = [$oldLampiranPath];
    }
}

// Ambil user & check user role
$username = $_SESSION['username'];
$qUser = $conn->prepare("SELECT id, tipe, nama, jabatan, access_modul FROM users WHERE username = ?");
$qUser->bind_param("s", $username);
$qUser->execute();
$res  = $qUser->get_result();
$user = $res ? $res->fetch_assoc() : null;
$res?->free();
$qUser->close();

if (!$user || !in_array($user['tipe'], ['user', 'admin', 'editor'])) {
    header('Location: login.php');
    exit();
}

$access_modul = json_decode($user['access_modul'], true);
$userId = (int)$user['id'];

// Check if user is owner of this letter or admin - only they can edit validated letters
$isOwner = ((int)($surat['dibuat_oleh'] ?? 0) === $userId);
$isAdmin = ($user['tipe'] === 'admin');

// Determine if PARAF should be shown: only when penandatangan is different from dibuat_oleh
$dibuatOlehId = (int)($surat['dibuat_oleh'] ?? 0);
$penandatanganId = (int)($surat['penandatangan_id'] ?? 0);
$shouldShowParaf = ($penandatanganId > 0 && $penandatanganId !== $dibuatOlehId);

// Get signature path (from final approver)
$signaturePath = $surat['signature_path'] ?? '';

// If letter is validated, only owner or admin can edit (editing will auto-unvalidate)
if ((int)($surat['is_validated'] ?? 0) === 1 && !$isOwner && !$isAdmin) {
    $_SESSION['error_msg'] = 'Surat telah di-approve dan hanya pembuat atau admin yang dapat mengeditnya.';
    header('Location: record-letter.php');
    exit();
}

// Check approver/supervisor status
$subordinateIds = getSubordinatesForUser($conn, $userId);
$isSupervisor = !empty($subordinateIds);
$isApprover = isApprover($conn, $userId);

// Allow access if user has surat module enabled OR they are an approver OR they are supervisor
if ((empty($access_modul['surat']) || $access_modul['surat'] !== 'on') && !$isApprover && !$isSupervisor) {
    header('Location: login.php');
    exit();
}

// ============ APPROVAL MODE HANDLING ============
$approvalMode = isset($_GET['mode']) && $_GET['mode'] === 'approval';

// For approval mode: get signer's info if dari_pribadi is set
$signerNama = '';
$signerJabatan = '';
if ($approvalMode && !empty($surat['dari_pribadi'])) {
    $stmtSigner = $conn->prepare("SELECT nama, jabatan FROM users WHERE id = ?");
    $stmtSigner->bind_param("i", $surat['dari_pribadi']);
    $stmtSigner->execute();
    $signerInfo = $stmtSigner->get_result()->fetch_assoc();
    $stmtSigner->close();
    
    if ($signerInfo) {
        $signerNama = $signerInfo['nama'] ?? '';
        $signerJabatan = $signerInfo['jabatan'] ?? '';
    }
} else if ($approvalMode && $surat['dari_tipe'] === 'sendiri') {
    // If from_tipe is 'sendiri', use the creator's info
    $stmtCreator = $conn->prepare("SELECT nama, jabatan FROM users WHERE id = ?");
    $stmtCreator->bind_param("i", $surat['dibuat_oleh']);
    $stmtCreator->execute();
    $creatorInfo = $stmtCreator->get_result()->fetch_assoc();
    $stmtCreator->close();
    
    if ($creatorInfo) {
        $signerNama = $creatorInfo['nama'] ?? '';
        $signerJabatan = $creatorInfo['jabatan'] ?? '';
    }
}

$approvalInfo = null;
$canApprove = false;
$isFinalApprover = false;
$currentParaf = '';

if ($approvalMode) {
    // Get approval chain info for this letter
    $stmtApproval = $conn->prepare("
        SELECT up.*, u.nama as approver_name, i.inisial
        FROM urutan_paraf up
        LEFT JOIN users u ON up.user_id = u.id
        LEFT JOIN inisial i ON up.user_id = i.user_id AND i.is_active = 1
        WHERE up.office_id = ?
        ORDER BY up.urutan ASC
    ");
    $stmtApproval->bind_param("i", $officeId);
    $stmtApproval->execute();
    $resApproval = $stmtApproval->get_result();

    $approvalChain = [];
    $lastApprovedStep = 0;
    $parafParts = [];
    $userStepInChain = null;
    $totalSteps = 0;

    while ($row = $resApproval->fetch_assoc()) {
        $approvalChain[] = $row;
        $totalSteps++;

        if ($row['approved_at'] !== null) {
            $lastApprovedStep = (int)$row['urutan'];
            if (!empty($row['inisial'])) {
                $parafParts[] = $row['inisial'];
            }
        }

        // Check if current user is in chain
        if ((int)$row['user_id'] === $userId) {
            $userStepInChain = $row;
        }
    }
    $stmtApproval->close();

    // Build current paraf string (for display in PDF)
    $currentParaf = implode('/', $parafParts);

    // Check if user can approve (their turn)
    if ($userStepInChain) {
        $userUrutan = (int)$userStepInChain['urutan'];
        $canApprove = ($userUrutan === $lastApprovedStep + 1) && ($userStepInChain['approved_at'] === null);
        $isFinalApprover = ($userUrutan === $totalSteps);

        $approvalInfo = [
            'chain' => $approvalChain,
            'user_step' => $userStepInChain,
            'current_step' => $lastApprovedStep + 1,
            'total_steps' => $totalSteps,
            'can_approve' => $canApprove,
            'is_final' => $isFinalApprover,
            'current_paraf' => $currentParaf,
            'user_inisial' => $userStepInChain['inisial'] ?? ''
        ];

        // Build the expected paraf for this approval (include current user's initials)
        $userInisial = $userStepInChain['inisial'] ?? '';
        if (!empty($userInisial)) {
            if (!empty($currentParaf)) {
                $expectedParafForApproval = $userInisial . '/' . $currentParaf;
            } else {
                $expectedParafForApproval = $userInisial;
            }
        } else {
            $expectedParafForApproval = $currentParaf;
        }
    }

    // Allow viewing approval page if user is in approval chain
    // User can view but may not be able to approve again if already approved
    $isInApprovalChain = ($userStepInChain !== null);

    // If user is not in approval chain at all, redirect with error
    if (!$isInApprovalChain) {
        $_SESSION['error_msg'] = 'Anda tidak memiliki akses untuk melakukan approval surat ini.';
        header('Location: record-letter.php');
        exit();
    }

    // If user is in chain but already approved, set canApprove to false but allow viewing
    if (!$canApprove && $userStepInChain && $userStepInChain['approved_at'] !== null) {
        // User already approved - allow viewing but disable approval actions
        $alreadyApproved = true;
    } else if (!$canApprove) {
        // User is in chain but not their turn yet
        $_SESSION['error_msg'] = 'Menunggu giliran Anda untuk melakukan approval.';
        header('Location: record-letter.php');
        exit();
    }

    // Check if PDF exists for approval mode
    $pdfMissing = false;
    $fileSurat = $surat['file_surat'] ?? '';
    if (empty($fileSurat)) {
        $pdfMissing = true;
    } else {
        $pdfFullPath = __DIR__ . '/../' . ltrim($fileSurat, '/');
        if (!file_exists($pdfFullPath)) {
            $pdfMissing = true;
        }
    }
}

$idLogin      = $user['id'] ?? 0;
$namaLogin    = $user['nama'] ?: $username;
$jabatanLogin = $user['jabatan'] ?? '';

date_default_timezone_set("Asia/Jakarta");
$today   = date('Y-m-d');
$nowTime = date('H:i');

$jumlahAgendaAktif = 0;
if ($s = $conn->prepare("SELECT COUNT(*) FROM agenda WHERE tanggal > ? OR (tanggal = ? AND waktu >= ?)")) {
    $s->bind_param("sss", $today, $today, $nowTime);
    $s->execute();
    $s->bind_result($jumlah);
    if ($s->fetch()) $jumlahAgendaAktif = (int)$jumlah;
    $s->close();
}

// Load master data
$jenisSurat = [];
$klasifikasi = [];
$usersList = [];
$tipeSurat = [];
if ($r = $conn->query("SELECT id, kode, nama FROM jenis_surat WHERE is_active=1 ORDER BY nama")) {
    $jenisSurat = $r->fetch_all(MYSQLI_ASSOC);
    $r->free();
}
if ($r = $conn->query("SELECT id, kode, nama FROM klasifikasi_arsip WHERE is_active=1 ORDER BY nama")) {
    $klasifikasi = $r->fetch_all(MYSQLI_ASSOC);
    $r->free();
}
if ($r = $conn->query("SELECT id, nama, jabatan FROM users WHERE tipe IN ('user', 'admin') ORDER BY nama")) {
    $usersList = $r->fetch_all(MYSQLI_ASSOC);
    $r->free();
}
if ($r = $conn->query("SELECT id, kode, nama FROM tipe_surat ORDER BY nama")) {
    $tipeSurat = $r->fetch_all(MYSQLI_ASSOC);
    $r->free();
}

// Helper function for jabatan mapping
function getKodeJabatanFromJSON($jabatan)
{
    if (empty($jabatan)) return null;

    $jsonFile = __DIR__ . '/../data/kode_jabatan.json';
    if (!file_exists($jsonFile)) return null;

    $jsonContent = file_get_contents($jsonFile);
    $jabatanList = json_decode($jsonContent, true);

    if (!is_array($jabatanList)) return null;

    $jabatanNormalized = strtolower(trim($jabatan));

    foreach ($jabatanList as $item) {
        if (!isset($item['jabatan']) || !isset($item['kode'])) continue;

        // Check exact match
        if (strtolower(trim($item['jabatan'])) === $jabatanNormalized) {
            return $item['kode'];
        }

        // Check aliases if available
        if (isset($item['aliases']) && is_array($item['aliases'])) {
            foreach ($item['aliases'] as $alias) {
                if (strtolower(trim($alias)) === $jabatanNormalized) {
                    return $item['kode'];
                }
            }
        }
    }

    return null;
}

function get_next_office_id(mysqli $conn): int
{
    $next = 0;
    $sqlAI = "SELECT AUTO_INCREMENT AS ai FROM information_schema.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office' LIMIT 1";
    if ($res = $conn->query($sqlAI)) {
        if ($row = $res->fetch_assoc()) $next = (int)$row['ai'];
        $res->free();
    }
    if ($next <= 0) {
        if ($res2 = $conn->query("SELECT IFNULL(MAX(id),0)+1 AS next_id FROM office")) {
            $row2 = $res2->fetch_assoc();
            $next = (int)($row2['next_id'] ?? 1);
            $res2->free();
        }
        if ($next <= 0) $next = 1;
    }
    return $next;
}
function format_nomor_surat(int $jenisId, int $klasId, int $officeNextId): string
{
    return sprintf('%02d.%02d.%04d', $jenisId, $klasId, $officeNextId);
}

// Helper function to extract sequence number from nomor_surat
function extractSequenceFromNomorSurat($nomorSurat, $tipeSuratKode)
{
    switch ($tipeSuratKode) {
        case 'SK': // Surat Keluar
            // Format: [Kode].[Klasifikasi].[Jenis].[Subjenis].[SequenceNumber]
            // Example: DKU.JPT.PG.04.03.90 → extract 90
            $parts = explode('.', $nomorSurat);
            return isset($parts[count($parts) - 1]) ? (int)$parts[count($parts) - 1] : 1;

        case 'KPTS': // Surat Keputusan
            // Format: 092/KPTS-JPT/2025 → extract 92
            if (preg_match('/^(\d+)\/KPTS/', $nomorSurat, $matches)) {
                return (int)$matches[1];
            }
            break;

        case 'SE': // Surat Edaran
            // Format: 091/SE-DIR/2025 → extract 91
            if (preg_match('/^(\d+)\/SE/', $nomorSurat, $matches)) {
                return (int)$matches[1];
            }
            break;
    }

    return 1; // Default fallback
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    // Debug: Check what's in tipe_surat_id
    dev_log("EDIT LETTER - tipe_surat_id from DB: " . var_export($surat['tipe_surat_id'], true));

    $_POST['kepentingan']          = $surat['kepentingan'] ?? '';
    // If tipe_surat_id is NULL, try to guess from tipe_surat_kode
    $tipeSuratId = $surat['tipe_surat_id'];
    if (empty($tipeSuratId)) {
        // Guess based on kode if available
        $kode = $surat['tipe_surat_kode'] ?? '';
        dev_log("EDIT LETTER - Fallback detecting kode: '$kode'");
        if ($kode === 'SK') $tipeSuratId = '1';
        elseif ($kode === 'KPTS') $tipeSuratId = '2';
        elseif ($kode === 'SE') $tipeSuratId = '3';
        // Don't set default - leave empty if can't detect
    }
    $_POST['tipe_surat_id']        = $tipeSuratId;
    dev_log("EDIT LETTER - Final tipe_surat_id set to: " . var_export($tipeSuratId, true));
    $_POST['tempat_dibuat']        = $surat['tempat_dibuat'] ?? '';
    $_POST['nomor_surat']          = $surat['nomor_surat'] ?? '';
    $_POST['tanggal_surat']        = $surat['tanggal_surat'] ?? '';
    $_POST['kepada']               = $surat['kepada'] ?? '';
    $_POST['perihal']              = $surat['perihal'] ?? '';

    // Build paraf from urutan_paraf if office.paraf is empty
    $parafFromDb = $surat['paraf'] ?? '';
    if (empty($parafFromDb)) {
        // Try to build paraf from urutan_paraf table (already approved steps)
        $stmtParaf = $conn->prepare("
            SELECT i.inisial 
            FROM urutan_paraf up
            LEFT JOIN inisial i ON up.user_id = i.user_id AND i.is_active = 1
            WHERE up.office_id = ? AND up.approved_at IS NOT NULL
            ORDER BY up.urutan DESC
        ");
        $stmtParaf->bind_param("i", $officeId);
        $stmtParaf->execute();
        $resParaf = $stmtParaf->get_result();
        $parafInitials = [];
        while ($rowParaf = $resParaf->fetch_assoc()) {
            if (!empty($rowParaf['inisial'])) {
                $parafInitials[] = $rowParaf['inisial'];
            }
        }
        $stmtParaf->close();
        $parafFromDb = implode('/', $parafInitials);
    }
    $_POST['paraf']                = $parafFromDb;
    $_POST['file_surat']           = $surat['file_surat'] ?? '';

    // Handle dual classification system based on tipe surat
    $tipeSuratKode = $surat['tipe_surat_kode'] ?? '';
    if ($tipeSuratKode === 'SK') {
        // New system for Surat Keluar
        $_POST['klasifikasi_arsip_new'] = $surat['klasifikasi_arsip_id'] ?? '';
        $_POST['jenis_surat_new']       = $surat['jenis_surat_id'] ?? '';
        $_POST['subjenis_surat_id']     = $surat['subjenis_surat_id'] ?? '';
    } else {
        // Legacy system for KPTS/SE
        $_POST['jenis_surat_id']        = $surat['jenis_surat_id'] ?? '';
        $_POST['klasifikasi_arsip_id']  = $surat['klasifikasi_arsip_id'] ?? '';
    }

    // Handle dari_pribadi - convert ID to user selection
    $dariPribadiId = $surat['dari_pribadi'] ?? '';
    $dariTipe = $surat['dari_tipe'] ?? '';

    // Debug log
    dev_log("EDIT LETTER - dari_tipe from DB: '$dariTipe', dari_pribadi from DB: '$dariPribadiId'");

    // Always set these values from DB first
    $_POST['dari_tipe'] = $dariTipe;
    $_POST['dari_pribadi'] = $dariPribadiId;

    // If dari_pribadi has a value, determine the correct dari_tipe
    if (!empty($dariPribadiId) && $dariPribadiId !== '0') {
        // Check if it's the logged-in user
        if ((int)$dariPribadiId === $idLogin) {
            $_POST['dari_tipe'] = 'sendiri';
        } else {
            $_POST['dari_tipe'] = 'pribadi';
        }
        $_POST['dari_pribadi'] = $dariPribadiId;
    }

    dev_log("EDIT LETTER - Final dari_tipe: '{$_POST['dari_tipe']}', dari_pribadi: '{$_POST['dari_pribadi']}'");
}

$errors = [];
$success = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['__final_submit']) && $_POST['__final_submit'] === '1') {
    $kepentingan       = $_POST['kepentingan'] ?? '';
    $tipe_surat_id     = (int)($_POST['tipe_surat_id'] ?? 0);
    $tempat_dibuat     = trim($_POST['tempat_dibuat'] ?? '');
    $tanggal_surat     = $_POST['tanggal_surat'] ?? '';
    $dari_tipe         = $_POST['dari_tipe'] ?? '';
    $dari_pribadi      = trim($_POST['dari_pribadi'] ?? '');
    $kepada            = trim($_POST['kepada'] ?? '');
    $perihal           = trim($_POST['perihal'] ?? '');
    $fileSuratPath     = trim($_POST['file_surat'] ?? '');
    $paraf             = trim($_POST['paraf'] ?? '');
    $naskah_html       = $_POST['naskah_html'] ?? ''; // TinyMCE content

    // Get tipe surat kode to determine which classification system to use
    $tipeSuratKode = '';
    if ($tipe_surat_id > 0) {
        $stmtTipe = $conn->prepare("SELECT kode FROM tipe_surat WHERE id = ?");
        $stmtTipe->bind_param('i', $tipe_surat_id);
        $stmtTipe->execute();
        $resTipe = $stmtTipe->get_result();
        if ($rowTipe = $resTipe->fetch_assoc()) {
            $tipeSuratKode = $rowTipe['kode'];
        }
        $stmtTipe->close();
    }

    dev_log("EDIT LETTER DEBUG - tipe_surat_id: $tipe_surat_id, tipeSuratKode: $tipeSuratKode");
    dev_log("EDIT LETTER DEBUG - kepada value: '" . $kepada . "', tipeSuratKode: '" . $tipeSuratKode . "'");

    // Determine which fields to use based on tipe surat
    $jenis_surat_id = 0;
    $klasifikasi_arsip_id = 0;
    $subjenis_surat_id = 0;

    // STRICT: Only SK (Surat Keluar) uses classification system
    if ($tipeSuratKode === 'SK') {
        // New classification system for Surat Keluar (SK)
        $klasifikasi_arsip_id = (int)($_POST['klasifikasi_arsip_new'] ?? 0);
        $jenis_surat_id = (int)($_POST['jenis_surat_new'] ?? 0);
        $subjenis_surat_id = (int)($_POST['subjenis_surat_id'] ?? 0);
    }
    // KPTS and SE: NO classification - all fields remain 0 (will be NULL in DB)

    $enumKepentingan = ['internal', 'eksternal'];
    $enumDari        = ['internal', 'eksternal', 'pribadi', 'sendiri'];

    // Validation
    if (!in_array($kepentingan, $enumKepentingan, true)) $errors[] = 'Kepentingan tidak valid.';
    if ($tipe_surat_id <= 0)                             $errors[] = 'Tipe surat wajib diisi.';
    if ($tempat_dibuat === '')                           $errors[] = 'Tempat surat dibuat wajib diisi.';

    // Conditional validation for SK type
    if ($tipeSuratKode === 'SK') {
        if ($klasifikasi_arsip_id <= 0) $errors[] = 'Klasifikasi wajib diisi untuk Surat Keluar.';
        if ($jenis_surat_id <= 0)       $errors[] = 'Jenis surat wajib diisi untuk Surat Keluar.';
    }
    if ($tanggal_surat === '')                           $errors[] = 'Tanggal surat wajib diisi.';
    if (!in_array($dari_tipe, $enumDari, true))          $errors[] = 'Dari (tipe) tidak valid.';
    if ($dari_tipe === 'pribadi' && $dari_pribadi === '') $errors[] = 'Nama atas nama pribadi tidak boleh kosong.';
    if ($dari_tipe === 'sendiri' && $dari_pribadi === '') $dari_pribadi = (string)$idLogin;
    // Only require 'Kepada' for Surat Keluar (SK), not for Surat Keputusan or Surat Edaran
    if ($tipeSuratKode === 'SK' && $kepada === '')       $errors[] = 'Kepada wajib diisi.';
    if ($perihal === '')                                 $errors[] = 'Perihal wajib diisi.';
    if ($fileSuratPath === '')                           $errors[] = 'File surat belum diupload.';

    $lampiran_path = $lampiran_orig = $lampiran_mime = null;
    $lampiran_size = null;
    $newLampiranUploaded = false;

    if (!$errors && isset($_FILES['lampiran']) && $_FILES['lampiran']['error'] !== UPLOAD_ERR_NO_FILE) {
        if ($_FILES['lampiran']['error'] !== UPLOAD_ERR_OK) {
            $errors[] = 'Gagal mengunggah lampiran.';
        } else {
            $tmp  = $_FILES['lampiran']['tmp_name'];
            $mime = @mime_content_type($tmp) ?: ($_FILES['lampiran']['type'] ?? '');
            $size = (int)$_FILES['lampiran']['size'];
            $allowed = ['application/pdf', 'image/jpeg', 'image/png'];
            if (!in_array($mime, $allowed, true)) {
                $errors[] = 'Tipe lampiran tidak diizinkan. Gunakan PDF/JPG/PNG.';
            } elseif ($size > 25 * 1024 * 1024) {
                $errors[] = 'Lampiran melebihi 25 MB.';
            } else {
                $dir = __DIR__ . '/letter/uploads/lampiran/';
                if (!is_dir($dir)) mkdir($dir, 0775, true);
                $orig = $_FILES['lampiran']['name'];
                $ext  = strtolower(pathinfo($orig, PATHINFO_EXTENSION));
                $safe = preg_replace('/[^a-zA-Z0-9_\-\.]/', '_', pathinfo($orig, PATHINFO_FILENAME));
                $fname = $safe . '_' . time() . '.' . $ext;
                $dest  = $dir . $fname;
                if (!move_uploaded_file($tmp, $dest)) {
                    $errors[] = 'Tidak dapat menyimpan lampiran.';
                } else {
                    $lampiran_path = 'letter/uploads/lampiran/' . $fname;
                    $lampiran_orig = $orig;
                    $lampiran_mime = $mime;
                    $lampiran_size = $size;
                    $newLampiranUploaded = true;
                }
            }
        }
    }

    if (!$errors) {
        $nomor_surat = trim($_POST['nomor_surat'] ?? '');
        if ($nomor_surat === '') {
            $errors[] = 'Nomor surat tidak boleh kosong.';
        } else {
            // Extract sequence number from nomor_surat
            $sequence_number = extractSequenceFromNomorSurat($nomor_surat, $tipeSuratKode);
        }
    }

    if (!$errors) {
        $conn->begin_transaction();
        try {
            $dari_tipe_db = ($dari_tipe === 'pribadi' || $dari_tipe === 'sendiri') ? 'internal' : $dari_tipe;

            // Handle dari_pribadi based on type
            if ($dari_tipe === 'pribadi') {
                $dari_pribadi_db = (int)$dari_pribadi;
            } elseif ($dari_tipe === 'sendiri') {
                // For "Diri Sendiri", always use logged-in user's ID
                $dari_pribadi_db = $idLogin;
            } else {
                $dari_pribadi_db = null;
            }

            if (!$newLampiranUploaded) {
                $lampiran_path = $oldLampiranPath;
                $lampiran_orig = $surat['lampiran_orig_name'] ?? null;
                $lampiran_mime = $surat['lampiran_mime'] ?? null;
                $lampiran_size = $surat['lampiran_size'] ?? null;
            }

            $lampiran_path_json = $lampiran_path
                ? (is_array(json_decode($lampiran_path, true))
                    ? $lampiran_path // sudah JSON
                    : json_encode([$lampiran_path], JSON_UNESCAPED_SLASHES))
                : json_encode([], JSON_UNESCAPED_SLASHES);

            $newFileSurat = $fileSuratPath !== '' ? $fileSuratPath : $oldFileSurat;

            // Set fields to NULL if not applicable (for KPTS/SE)
            $jenis_surat_id_db = ($jenis_surat_id > 0) ? $jenis_surat_id : null;
            $klasifikasi_arsip_id_db = ($klasifikasi_arsip_id > 0) ? $klasifikasi_arsip_id : null;
            $subjenis_surat_id_db = ($subjenis_surat_id > 0) ? $subjenis_surat_id : null;

            // build editor history JSON
            $editorHistoryArr = [];
            if (!empty($oldEditorHistory)) {
                $maybe = json_decode($oldEditorHistory, true);
                if (is_array($maybe)) {
                    $editorHistoryArr = $maybe;
                } else {
                    // fallback: csv name list
                    $parts = array_map('trim', explode(',', $oldEditorHistory));
                    foreach ($parts as $p) {
                        if (empty($p)) continue;
                        $editorHistoryArr[] = ['name' => $p, 'edited_at' => null];
                    }
                }
            }
            // Append current editor, but first deduplicate and keep the latest edited_at per name
            $nowTs = date('Y-m-d H:i:s');
            // Normalize existing entries via name => entry mapping to deduplicate
            $uniqueMap = [];
            foreach ($editorHistoryArr as $ent) {
                $nm = trim((string)($ent['name'] ?? ''));
                $eid = isset($ent['id']) ? (string)$ent['id'] : null;
                if ($nm === '') continue;
                // Prefer dedupe by ID; if no ID, dedupe by normalized name
                $key = $eid ? ('id:' . $eid) : ('name:' . mb_strtolower($nm, 'UTF-8'));
                $existing = $uniqueMap[$key] ?? null;
                $existingAt = $existing['edited_at'] ?? null;
                $thisAt = $ent['edited_at'] ?? null;
                // keep the latest timestamp if both present
                if ($existingAt && $thisAt) {
                    $keepAt = (strtotime($existingAt) > strtotime($thisAt) ? $existingAt : $thisAt);
                    $uniqueMap[$key] = ['id' => $eid ?: ($existing['id'] ?? null), 'name' => $nm, 'edited_at' => $keepAt];
                } elseif ($thisAt) {
                    $uniqueMap[$key] = ['id' => $eid ?: ($existing['id'] ?? null), 'name' => $nm, 'edited_at' => $thisAt];
                } elseif ($existingAt) {
                    // keep existing
                    $uniqueMap[$key] = $existing;
                } else {
                    $uniqueMap[$key] = ['id' => $eid ?: ($existing['id'] ?? null), 'name' => $nm, 'edited_at' => null];
                }
            }

            // Add or update current editor
            $currentKey = $idLogin ? ('id:' . (string)$idLogin) : ('name:' . mb_strtolower(trim($namaLogin), 'UTF-8'));
            $uniqueMap[$currentKey] = ['id' => $idLogin, 'name' => $namaLogin, 'edited_at' => $nowTs];

            // Convert back to array and sort by edited_at (oldest -> newest) so end() returns most recent
            $editorHistoryArr = array_values($uniqueMap);
            usort($editorHistoryArr, function ($a, $b) {
                $ta = $a['edited_at'] ? strtotime($a['edited_at']) : 0;
                $tb = $b['edited_at'] ? strtotime($b['edited_at']) : 0;
                return $ta <=> $tb;
            });
            $editorHistoryJson = json_encode($editorHistoryArr, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

            // BUG FIX: Determine penandatangan_id from dari_pribadi when editing
            $penandatangan_id = null;
            if ($dari_tipe === 'pribadi') {
                $penandatangan_id = (int)$dari_pribadi;
            } elseif ($dari_tipe === 'sendiri') {
                $penandatangan_id = $idLogin;
            }

            $sql = "UPDATE office SET
                            kepentingan = ?,
                            tipe_surat_id = ?,
                            jenis_surat_id = ?,
                            tempat_dibuat = ?,
                            klasifikasi_arsip_id = ?,
                            subjenis_surat_id = ?,
                            nomor_surat = ?,
                            sequence_number = ?,
                            tanggal_surat = ?,
                            dari_tipe = ?,
                            dari_pribadi = ?,
                            kepada = ?,
                            perihal = ?,
                            naskah = ?,
                            paraf = '',
                            penandatangan_id = ?,
                            lampiran_path = ?,
                            lampiran_orig_name = ?,
                            lampiran_mime = ?,
                            lampiran_size = ?,
                            editor_history = ?,
                            file_surat = ?,
                            is_validated = 0,
                            qr_code = NULL,
                            validation_token = NULL,
                            validated_at = NULL,
                            updated_at = NOW()
                        WHERE id = ?";

            $st = $conn->prepare($sql);
            $lampiran_size_int = $lampiran_size !== null ? (int)$lampiran_size : 0;

            // Binding params with correct types for integers
            // Integer fields: tipe_surat_id, jenis_surat_id_db, klasifikasi_arsip_id_db, 
            // subjenis_surat_id_db, sequence_number, dari_pribadi_db, penandatangan_id, 
            // lampilan_size_int, officeId
            $bindTypes = 'siisiisississsisssissi';
            $st->bind_param(
                $bindTypes,
                $kepentingan,
                $tipe_surat_id,
                $jenis_surat_id_db,
                $tempat_dibuat,
                $klasifikasi_arsip_id_db,
                $subjenis_surat_id_db,
                $nomor_surat,
                $sequence_number,
                $tanggal_surat,
                $dari_tipe_db,
                $dari_pribadi_db,
                $kepada,
                $perihal,
                $naskah_html,
                $penandatangan_id,
                $lampiran_path_json,
                $lampiran_orig,
                $lampiran_mime,
                $lampiran_size_int,
                $editorHistoryJson,
                $newFileSurat,
                $officeId
            );

            if (!$st->execute()) {
                throw new Exception($st->error);
            }
            $st->close();

            // OPTION A: Reset approval chain completely when editing an approved letter
            // This ensures approvals must restart from the beginning
            $resetSql = "UPDATE urutan_paraf SET approved_at = NULL, signature_path = NULL WHERE office_id = ?";
            $resetStmt = $conn->prepare($resetSql);
            if (!$resetStmt) {
                throw new Exception("Failed to prepare reset SQL: " . $conn->error);
            }
            $resetStmt->bind_param('i', $officeId);
            if (!$resetStmt->execute()) {
                throw new Exception("Failed to reset urutan_paraf: " . $resetStmt->error);
            }
            $resetStmt->close();
            dev_log("✅ Reset approval chain (urutan_paraf) for office_id: $officeId");

            // Delete old QR code file if letter was validated
            if (!empty($surat['qr_code'])) {
                $qrCodePath = __DIR__ . '/' . ltrim($surat['qr_code'], '/');
                if (is_file($qrCodePath)) {
                    dev_log('🗑️ Deleting QR code file on edit: ' . $qrCodePath);
                    @unlink($qrCodePath);
                }
            }

            if ($oldFileSurat && $newFileSurat && $newFileSurat !== $oldFileSurat) {
                $oldPathAbs = __DIR__ . '/' . ltrim($oldFileSurat, '/');
                if (is_file($oldPathAbs)) {
                    @unlink($oldPathAbs);
                }
            }

            if ($newLampiranUploaded && $oldLampiranPath) {
                $oldJson = json_decode($oldLampiranPath, true);

                if (is_array($oldJson)) {
                    foreach ($oldJson as $p) {
                        $abs = __DIR__ . '/' . ltrim($p, '/');
                        if (is_file($abs)) @unlink($abs);
                    }
                } else {
                    $abs = __DIR__ . '/' . ltrim($oldLampiranPath, '/');
                    if (is_file($abs)) @unlink($abs);
                }
            }

            $conn->commit();

            header('Location: record-letter.php?updated=1');
            exit();
        } catch (Exception $ex) {
            $conn->rollback();

            if (isset($finalPathAbs) && is_file($finalPathAbs)) @unlink($finalPathAbs);
            if (isset($lampiran_path) && $lampiran_path && $lampiran_path !== $oldLampiranPath) {
                $absLampiran = __DIR__ . '/' . ltrim($lampiran_path, '/');
                if (is_file($absLampiran)) @unlink($absLampiran);
            }

            dev_log('SURAT_UPDATE_FAIL: ' . $ex->getMessage());
            $errors[] = "Gagal menyimpan perubahan: " . $ex->getMessage();
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <!-- Favicon -->
    <link href="/jasa_marga/img/baru/logojp.png" rel="icon">

    <title>Edit Surat (User)</title>

    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">

    <style>
        .greeting {
            text-align: center;
            color: #0c50e1;
            animation: fadeIn 2s ease-in-out;
        }

        .icon {
            margin-top: 40px;
            font-size: 50px;
            margin-bottom: 0px;
            animation: bounce 1.5s infinite;
        }

        @keyframes bounce {

            0%,
            100% {
                transform: translateY(0);
            }

            50% {
                transform: translateY(-10px);
            }
        }

        .sidebar {
            background-color: #0c50e1;
            color: white;
            min-height: 100vh;
            padding: 15px;
        }

        .sidebar a {
            color: white;
            text-decoration: none;
            display: block;
            padding: 10px;
            margin: 5px 0;
            border-radius: 5px;
        }

        .sidebar a:hover,
        .sidebar a.active {
            background-color: #ffc107;
            color: #0047ab;
        }

        .btn-success {
            background-color: #ffc107;
            /* Kuning mencolok */
            border-color: #ffc107;
            color: #0047ab;
            font-weight: bold;
        }

        /* Warna untuk header tabel */
        .table thead {
            background-color: #0047ab;
            /* Biru Jasamarga */
            color: #0c50e1;
            text-transform: uppercase;
            font-weight: bold;
            color: #ffc107;
        }

        .table thead th {
            background-color: #0c50e1;
            /* Biru Jasamarga */
            color: #ffc107;
        }

        .sidebar-animated {
            background-color: #001eff;
            ;
            padding: 20px 10px;
            font-family: Arial, sans-serif;
            height: 100vh;
            width: 250px;
            transition: all 0.3s ease;
            max-height: 100vh;
            /* Limits sidebar height to viewport */
            overflow-y: auto;
            /* Enables vertical scroll */
            overflow-x: hidden;
            /* Hides horizontal scroll */
            padding-bottom: 20px;
            /* Adds extra padding to avoid cut-off */
            margin-bottom: 0;
            /* Ensures no unwanted margin at the bottom */
            scrollbar-width: thin;
        }

        /* Optional: Styling untuk scrollbar pada Chrome, Edge, dan Safari */
        .sidebar-animated::-webkit-scrollbar {
            width: 8px;
            /* Mengatur lebar scrollbar */
        }

        .sidebar-animated::-webkit-scrollbar-thumb {
            background-color: #e1b12c;
            /* Warna scrollbar */
            border-radius: 10px;
            /* Membuat sudut melengkung */
        }

        .sidebar-animated::-webkit-scrollbar-thumb:hover {
            background-color: #ffc107;
            /* Warna scrollbar saat di-hover */
        }

        .sidebar-animated a {
            color: #FFFFFFFF;
            text-decoration: none;
            font-size: 18px;
            display: flex;
            align-items: center;
            padding: 10px;
            border-radius: 5px;
            transition: background-color 0.3s ease, transform 0.3s ease;
        }

        .sidebar-animated a:hover {
            background-color: #ffc107;
            color: #fff;
            transform: translateX(5px);
        }

        .sidebar-animated a i {
            margin-right: 10px;
        }

        .sidebar-animated .submenu-toggle {
            cursor: pointer;
            transition: color 0.3s;
        }

        .sidebar-animated .submenu-toggle:hover .arrow-icon {
            transform: rotate(90deg);
            color: #001eff;
        }

        .sidebar-animated .submenu {
            padding-left: 1px;
            overflow: hidden;
            transition: max-height 0.4s ease;
        }

        .sidebar-animated .collapse.show {
            max-height: 300px;
            /* adjust as needed */
            transition: max-height 0.4s ease-in;
        }

        .arrow-icon {
            margin-left: auto;
            transition: transform 0.3s;
        }

        /* Hover and animation effects for submenu */
        .submenu a {
            font-size: 16px;
            color: #FFFFFFFF;
            transition: color 0.3s ease, transform 0.3s ease;
        }

        .submenu a:hover {
            color: #001eff;
            transform: translateX(5px);
        }

        /* Sidebar active link style */
        .sidebar-animated a.active {
            background-color: #ffc107;
            /* Warna latar belakang khusus untuk menu aktif */
            color: #fff;
            /* Warna teks untuk menu aktif */
            font-weight: bold;
        }

        .sidebar-animated a.active i {
            color: #fff;
            /* Warna ikon untuk menu aktif */
        }

        /* Navbar custom styling */
        .navbar-custom {
            background-color: #001eff;
            /* Same as sidebar background color */
            transition: background-color 0.3s ease;
            padding: 15px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            /* Adds a subtle shadow */
        }

        /* Logo and Sidebar Toggle */
        .navbar-custom .navbar-brand img {
            transition: transform 0.3s ease;
        }

        .navbar-custom .navbar-brand:hover img {
            transform: scale(1.05);
        }

        /* Toggle Sidebar button */
        .toggle-sidebar-btn {
            color: #fff;
            font-size: 1.3rem;
            cursor: pointer;
            margin-left: 10px;
            transition: color 0.3s ease;
        }

        .toggle-sidebar-btn:hover {
            color: #ffc107;
        }

        /* Animasi untuk kartu */
        @keyframes fadeInUp {
            0% {
                opacity: 0;
                transform: translateY(20px);
            }

            100% {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .highlight-card {
            border-radius: 15px;
            /* Membuat sudut melengkung */
            transition: transform 0.3s ease, box-shadow 0.3s ease;
            animation: fadeInUp 0.8s ease-in-out;
        }

        /* Hover efek */
        .highlight-card:hover {
            transform: translateY(-10px);
            box-shadow: 0 10px 20px rgba(0, 0, 0, 0.2);
        }

        /* Gaya untuk teks */
        .highlight-card h3 {
            font-size: 2.5rem;
            font-weight: bold;
            margin: 0;
        }

        .highlight-card p {
            font-size: 1.1rem;
            margin-top: 5px;
            opacity: 0.9;
        }

        /* Footer styling */
        .highlight-card .card-footer {
            background: rgba(0, 0, 0, 0.1);
            border-top: 1px solid rgba(255, 255, 255, 0.2);
        }

        .highlight-card .card-footer a {
            text-decoration: none;
            font-weight: bold;
            transition: color 0.3s ease;
        }

        .highlight-card .card-footer a:hover {
            color: #ffd700;
            /* Warna emas saat hover */
        }

        /* Animasi untuk ikon */
        .highlight-card i {
            animation: bounce 1.5s infinite;
        }

        /* Bounce animasi untuk ikon */
        @keyframes bounce {

            0%,
            20%,
            50%,
            80%,
            100% {
                transform: translateY(0);
            }

            40% {
                transform: translateY(-10px);
            }

            60% {
                transform: translateY(-5px);
            }
        }

        .pagination .page-link {
            color: #007bff;
        }

        .pagination .active .page-link {
            background-color: #007bff;
            border-color: #007bff;
            color: white;
        }


        .search-container-new {
            position: relative;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .search-box-new {
            display: flex;
            align-items: center;
            transition: all 0.3s ease;
        }

        .search-input-new {
            border-radius: 25px;
            padding: 8px 12px;
            border: 1px solid #ced4da;
            background-color: #ffc107;
            color: #001eff;
            transition: width 0.4s ease, box-shadow 0.3s ease;
            width: 184px;
        }

        .search-input-new::placeholder {
            color: #001eff;
            font-style: italic;
            opacity: 0.8;
        }

        .search-input-new:focus {
            width: 250px;
            background-color: #ffffff;
            border-color: #ffc107;
            box-shadow: 0 0 8px rgba(255, 193, 7, 0.6);
            /* Glow kuning */
            outline: none;
        }

        .search-btn-new {
            background-color: #001eff;
            color: #fff;
            border: none;
            border-radius: 20px;
            padding: 6px 20px;
            font-size: 14px;
            cursor: pointer;
            transition: background-color 0.3s ease, transform 0.3s ease;
            display: none;
        }

        .search-btn-new.show {
            display: inline-block;
        }

        /* ---------- CSS Tambahan ---------- */
        .stepper {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 14px;
            margin: 25px 0 24px;
        }

        .stepper .step {
            display: flex;
            align-items: center;
            gap: 10px;
            position: relative;
        }

        .stepper .step:not(:last-child)::after {
            content: "";
            width: 300px;
            height: 4px;
            background: #0c50e1;
            opacity: .35;
            border-radius: 4px;
            margin-left: 12px;
            margin-right: 4px;
        }

        .stepper .step.completed:not(:last-child)::after {
            background: #ffc107;
            opacity: 1;
        }

        .step-circle {
            width: 36px;
            height: 36px;
            border-radius: 50%;
            display: grid;
            place-items: center;
            font-weight: 700;
            border: 3px solid #0c50e1;
            color: #0c50e1;
            background: #fff;
        }

        .step.active .step-circle {
            background: #ffc107;
            border-color: #ffc107;
            color: #001eff;
            box-shadow: 0 0 0 4px rgba(255, 193, 7, .25);
        }

        .step.completed .step-circle {
            background: #0c50e1;
            border-color: #0c50e1;
            color: #ffc107;
        }

        .step-label {
            font-weight: 600;
            color: #001eff;
        }

        .step-desc {
            font-size: .85rem;
            color: #6c757d;
            margin-top: -2px;
        }

        .card-header {
            background: #0c50e1;
            color: #ffc107;
            border-bottom: 0;
            font-weight: 700;
        }

        .card {
            border: 1px solid rgba(0, 0, 0, .06);
            box-shadow: 0 6px 18px rgba(0, 0, 0, .06);
            border-radius: 16px;
        }

        .btn-print-custom {
            background: #ffc107;
            color: #0c50e1;
            font-weight: 700;
        }

        .btn-print-custom:hover {
            background: #0c50e1;
            color: #ffc107;
            border-color: #0c50e1;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(12, 80, 225, 0.45);
        }

        .btn-save-custom {
            background: #0c50e1;
            color: #ffc107;
            font-weight: 700;
        }

        .btn-save-custom:hover {
            background: #ffc107;
            color: #001eff;
            border-color: #ffc107;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(255, 193, 7, 0.45);
        }

        .btn-print-custom,
        .btn-save-custom {
            transition: all 0.25s ease;
        }

        .form-control,
        .form-select {
            border-radius: 10px;
        }

        .badge-kode {
            font-size: .75rem;
            background: #e9f1ff;
            color: #0c50e1;
            border: 1px solid #0c50e1;
            border-radius: 8px;
            padding: .25rem .5rem;
        }

        /* Safety: ensure if cards are direct children of the top-level .d-flex (markup bug),
           they will stack vertically and not display side-by-side. Scoped to letter-edit pages only. */
        .main-edit-safety .d-flex {
            flex-wrap: wrap;
        }

        .main-edit-safety .d-flex>.card {
            flex: 0 0 100% !important;
            max-width: 100% !important;
        }

        /* Ensure the main row layout does not collapse to vertical (prevent stacking under sidebar) */
        .main-edit-row {
            display: flex !important;
            flex-wrap: nowrap !important;
            align-items: flex-start !important;
        }
    </style>
</head>

<body>
    <?php if (!empty($errors)) {
        echo '<pre style="color:#b00;background:#fee;padding:8px;border:1px solid #f99;">'
            . htmlspecialchars(print_r($errors, true), ENT_QUOTES, 'UTF-8') . '</pre>';
    } ?>

    <?php if ((int)($surat['is_validated'] ?? 0) === 1): ?>
        <script>
            document.addEventListener('DOMContentLoaded', function() {
                Swal.fire({
                    icon: 'warning',
                    title: 'Surat Sudah Di-approve',
                    html: 'Surat ini <strong>sudah di-approve</strong>. Jika Anda menyimpan perubahan, approval akan <strong>otomatis dibatalkan</strong> dan QR code akan dihapus.',
                    confirmButtonText: 'Mengerti',
                    confirmButtonColor: '#0c50e1'
                });
            });
        </script>
    <?php endif; ?>

    <?php if (isset($_GET['updated'])): ?>
        <script>
            Swal.fire({
                icon: 'success',
                title: 'Berhasil!',
                text: 'Surat berhasil diperbarui.',
                showConfirmButton: true
            });
        </script>

    <?php elseif (isset($_GET['deleted'])): ?>
        <script>
            Swal.fire({
                icon: 'success',
                title: 'Berhasil!',
                text: 'Surat berhasil dihapus.',
                showConfirmButton: true
            });
        </script>
    <?php endif; ?>

    <div class="d-flex">
        <?php include __DIR__ . '/../includes/sidebar-user.php'; ?>

        <!-- Main Content -->
        <div id="mainContent" class="container-fluid py-3">

            <?php if ($approvalMode && $approvalInfo): ?>
                <!-- Approval Mode Banner -->
                <div class="alert alert-warning alert-dismissible fade show mb-3" role="alert">
                    <div class="d-flex align-items-center">
                        <i class="fas fa-check-circle fa-2x me-3"></i>
                        <div class="flex-grow-1">
                            <h5 class="alert-heading mb-1">
                                <i class="fas fa-stamp"></i> Mode Approval - Step <?= $approvalInfo['current_step'] ?> dari <?= $approvalInfo['total_steps'] ?>
                            </h5>
                            <p class="mb-1">
                                Anda sedang melakukan approval untuk surat ini.
                                <?php if ($approvalInfo['is_final']): ?>
                                    <strong class="text-danger">Anda adalah approver terakhir (Penandatangan). QR Code akan di-generate setelah approval.</strong>
                                <?php else: ?>
                                    Setelah Anda approve, giliran akan berpindah ke approver berikutnya.
                                <?php endif; ?>
                            </p>
                            <div class="mt-2">
                                <strong>Approval Chain:</strong>
                                <?php foreach ($approvalInfo['chain'] as $idx => $step): ?>
                                    <span class="badge <?= $step['approved_at'] ? 'bg-success' : ((int)$step['urutan'] === $approvalInfo['current_step'] ? 'bg-warning text-dark' : 'bg-secondary') ?> me-1">
                                        <?= $step['approved_at'] ? '✓' : ((int)$step['urutan'] === $approvalInfo['current_step'] ? '◉' : '○') ?>
                                        <?= htmlspecialchars($step['inisial'] ?? '?') ?>
                                    </span>
                                    <?php if ($idx < count($approvalInfo['chain']) - 1) echo '→'; ?>
                                <?php endforeach; ?>
                            </div>
                            <?php if (!empty($approvalInfo['current_paraf'])): ?>
                                <div class="mt-1"><small>Paraf saat ini: <strong><?= htmlspecialchars($approvalInfo['current_paraf']) ?></strong></small></div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>

                <?php if (!empty($pdfMissing)): ?>
                    <!-- PDF Missing Warning -->
                    <div class="alert alert-danger mb-3" role="alert">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        <strong>PDF Surat Tidak Ditemukan!</strong>
                        <p class="mb-0 mt-1">File PDF untuk surat ini belum ada atau tidak ditemukan.
                            Silakan klik tombol <strong>"Cetak & Upload PDF"</strong> di bawah untuk generate ulang PDF sebelum melakukan approval.</p>
                    </div>
                <?php endif; ?>
            <?php endif; ?>

            <div class="d-flex align-items-center justify-content-between mb-2">
                <h1 class="h3 m-0"><?= $approvalMode ? 'Approval Surat' : 'Edit Surat' ?></h1>
                <span class="badge-kode">Tanggal:
                    <?php
                    $dateValue = $_POST['tanggal_surat'] ?? date('Y-m-d');
                    if ($dateValue) {
                        // Convert to Indonesian format "20 November 2025"
                        $timestamp = strtotime($dateValue);
                        if ($timestamp !== false) {
                            $bulanIndonesia = [
                                1 => 'Januari',
                                2 => 'Februari',
                                3 => 'Maret',
                                4 => 'April',
                                5 => 'Mei',
                                6 => 'Juni',
                                7 => 'Juli',
                                8 => 'Agustus',
                                9 => 'September',
                                10 => 'Oktober',
                                11 => 'November',
                                12 => 'Desember'
                            ];
                            $hari = date('j', $timestamp);
                            $bulan = $bulanIndonesia[date('n', $timestamp)];
                            $tahun = date('Y', $timestamp);
                            echo htmlspecialchars("$hari $bulan $tahun", ENT_QUOTES, 'UTF-8');
                        } else {
                            echo htmlspecialchars($dateValue, ENT_QUOTES, 'UTF-8');
                        }
                    } else {
                        $timestamp = time();
                        $bulanIndonesia = [
                            1 => 'Januari',
                            2 => 'Februari',
                            3 => 'Maret',
                            4 => 'April',
                            5 => 'Mei',
                            6 => 'Juni',
                            7 => 'Juli',
                            8 => 'Agustus',
                            9 => 'September',
                            10 => 'Oktober',
                            11 => 'November',
                            12 => 'Desember'
                        ];
                        $hari = date('j', $timestamp);
                        $bulan = $bulanIndonesia[date('n', $timestamp)];
                        $tahun = date('Y', $timestamp);
                        echo htmlspecialchars("$hari $bulan $tahun", ENT_QUOTES, 'UTF-8');
                    }
                    ?></span>
            </div>

            <!-- STEPPER -->
            <nav class="stepper" aria-label="Progress">
                <div class="step active" data-step="1">
                    <div class="step-circle">1</div>
                    <div>
                        <div class="step-label">Formulir Surat</div>
                    </div>
                </div>
                <div class="step" data-step="2">
                    <div class="step-circle">2</div>
                    <div>
                        <div class="step-label">Konten Surat</div>
                    </div>
                </div>
            </nav>

            <form method="post" action="" enctype="multipart/form-data" autocomplete="off" id="formNaskah">
                <!-- ============ STEP 1 ============ -->
                <section class="step-pane" data-step="1">

                    <!-- CARD 1 (IDENTITAS SURAT) -->
                    <div class="card mb-4">
                        <div class="card-header">Identitas Surat</div>
                        <div class="card-body row g-3">
                            <!-- Kepentingan -->
                            <div class="col-12">
                                <label class="form-label d-block">Kepentingan <span class="text-danger">*</span></label>
                                <div class="d-flex gap-4">
                                    <label class="form-check">
                                        <input type="radio" class="form-check-input" name="kepentingan" value="internal"
                                            <?php echo (($_POST['kepentingan'] ?? '') === 'internal') ? 'checked' : ''; ?>>
                                        <span class="form-check-label">Internal</span>
                                    </label>
                                    <label class="form-check">
                                        <input type="radio" class="form-check-input" name="kepentingan"
                                            value="eksternal"
                                            <?php echo (($_POST['kepentingan'] ?? '') === 'eksternal') ? 'checked' : ''; ?>>
                                        <span class="form-check-label">Eksternal</span>
                                    </label>
                                </div>
                            </div>

                            <!-- Tipe Surat -->
                            <div class="col-12">
                                <label for="tipe_surat_id" class="form-label">Tipe Surat <span
                                        class="text-danger">*</span></label>
                                <select id="tipe_surat_id" name="tipe_surat_id" class="form-select" required>
                                    <option value="" disabled selected>-- Pilih Tipe Surat --</option>
                                    <!-- Will be populated via JavaScript -->
                                </select>
                            </div>

                            <!-- Classification Fields (Only for Surat Keluar) -->
                            <div id="classification-fields" class="d-none">
                                <!-- Klasifikasi -->
                                <div class="col-12 mb-3">
                                    <label for="klasifikasi_arsip_new" class="form-label">Klasifikasi <span
                                            class="text-danger">*</span></label>
                                    <select id="klasifikasi_arsip_new" name="klasifikasi_arsip_new" class="form-select">
                                        <option value="" disabled selected>-- Pilih Klasifikasi --</option>
                                    </select>
                                </div>

                                <!-- Jenis Surat -->
                                <div class="col-12 mb-3">
                                    <label for="jenis_surat_new" class="form-label">Jenis Surat <span
                                            class="text-danger">*</span></label>
                                    <select id="jenis_surat_new" name="jenis_surat_new" class="form-select">
                                        <option value="" disabled selected>-- Pilih Klasifikasi Terlebih Dahulu --
                                        </option>
                                    </select>
                                </div>

                                <!-- Subjenis Surat -->
                                <div class="col-12 mb-3">
                                    <label for="subjenis_surat_id" class="form-label">Subjenis Surat <span
                                            class="text-danger">*</span></label>
                                    <select id="subjenis_surat_id" name="subjenis_surat_id" class="form-select">
                                        <option value="" disabled selected>-- Pilih Jenis Surat Terlebih Dahulu --
                                        </option>
                                    </select>
                                </div>
                            </div>

                            <!-- Legacy Jenis Surat (Hidden by default) -->
                            <div id="legacy-jenis-field" class="col-12 d-none">
                                <label for="jenis_surat_id" class="form-label">Jenis Surat (Legacy) <span
                                        class="text-danger">*</span></label>
                                <select id="jenis_surat_id" name="jenis_surat_id" class="form-select" required>
                                    <option value="" disabled
                                        <?php echo empty($_POST['jenis_surat_id']) ? 'selected' : ''; ?>>-- Pilih Jenis
                                        Surat --</option>
                                    <?php foreach ($jenisSurat as $row): ?>
                                        <option value="<?php echo (int)$row['id']; ?>"
                                            data-kode="<?php echo htmlspecialchars($row['kode'], ENT_QUOTES, 'UTF-8'); ?>"
                                            <?php echo ((string)($_POST['jenis_surat_id'] ?? '') === (string)$row['id']) ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($row['nama'], ENT_QUOTES, 'UTF-8'); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>

                        <!-- Legacy Klasifikasi (Hidden by default) -->
                        <div class="row mb-3">
                            <div id="legacy-klasifikasi-field" class="col-12 d-none">
                                <label for="klasifikasi_arsip_id" class="form-label">Klasifikasi (Legacy) <span
                                        class="text-danger">*</span></label>
                                <select id="klasifikasi_arsip_id" name="klasifikasi_arsip_id" class="form-select">
                                    <option value="" disabled
                                        <?php echo empty($_POST['klasifikasi_arsip_id']) ? 'selected' : ''; ?>>-- Pilih
                                        Klasifikasi --</option>
                                    <?php foreach ($klasifikasi as $row): ?>
                                        <option value="<?php echo (int)$row['id']; ?>"
                                            data-kode="<?php echo htmlspecialchars($row['kode'], ENT_QUOTES, 'UTF-8'); ?>"
                                            <?php echo ((string)($_POST['klasifikasi_arsip_id'] ?? '') === (string)$row['id']) ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($row['nama'], ENT_QUOTES, 'UTF-8'); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>

                        <div class="row mb-3">
                            <!-- Tempat -->
                            <div class="col-12">
                                <label for="tempat_dibuat" class="form-label">Tempat Surat Dibuat <span
                                        class="text-danger">*</span></label>
                                <input type="text" id="tempat_dibuat" name="tempat_dibuat" class="form-control" required
                                    value="<?php echo htmlspecialchars($_POST['tempat_dibuat'] ?? '', ENT_QUOTES, 'UTF-8'); ?>">
                            </div>

                            <!-- Pejabat yang Menandatangani (moved from Subyek card) -->
                            <div class="col-12">
                                <label class="form-label d-block">Pejabat yang Menandatangani <span
                                        class="text-danger">*</span></label>
                                <?php if ($approvalMode): ?>
                                    <div class="alert alert-info py-2 mb-2"><small><i class="fas fa-lock"></i> Tidak dapat diubah saat approval</small></div>
                                <?php endif; ?>
                                <div class="d-flex flex-wrap gap-4">
                                    <label class="form-check">
                                        <input type="radio" class="form-check-input" name="dari_tipe" value="pribadi"
                                            <?php echo (($_POST['dari_tipe'] ?? '') === 'pribadi') ? 'checked' : ''; ?>
                                            <?php echo $approvalMode ? 'disabled' : ''; ?>>
                                        <span class="form-check-label">Pejabat</span>
                                    </label>
                                    <label class="form-check">
                                        <input type="radio" class="form-check-input" name="dari_tipe" value="sendiri" id="dariTipeSendiri"
                                            <?php echo (($_POST['dari_tipe'] ?? '') === 'sendiri') ? 'checked' : ''; ?>
                                            <?php echo $approvalMode ? 'disabled' : ''; ?>>
                                        <span class="form-check-label">Diri Sendiri</span>
                                    </label>
                                </div>
                                <?php if ($approvalMode): ?>
                                    <!-- Hidden inputs to preserve values when disabled -->
                                    <input type="hidden" name="dari_tipe" value="<?php echo htmlspecialchars($_POST['dari_tipe'] ?? $surat['dari_tipe'] ?? '', ENT_QUOTES); ?>">
                                    <input type="hidden" name="dari_pribadi" value="<?php echo htmlspecialchars($_POST['dari_pribadi'] ?? $surat['dari_pribadi'] ?? '', ENT_QUOTES); ?>">
                                <?php endif; ?>
                            </div>

                            <!-- Nama Yang Menandatangani Dropdown (filtered to 16 authorized signers) -->
                            <div class="col-12" id="wrapPribadi" style="display:none;">
                                <label for="dari_pribadi" class="form-label">Nama Yang Menandatangani</label>
                                <select id="dari_pribadi" name="dari_pribadi" class="form-select" <?php echo $approvalMode ? 'disabled' : ''; ?>>
                                    <option value="" disabled selected>-- Pilih Pejabat --</option>
                                    <!-- Will be populated via JavaScript with authorized signers only -->
                                </select>
                            </div>
                            <input type="hidden" id="user_nama" value="<?= htmlspecialchars($namaLogin, ENT_QUOTES) ?>">
                            <input type="hidden" id="user_jabatan" value="<?= htmlspecialchars($jabatanLogin, ENT_QUOTES) ?>">
                            <input type="hidden" id="user_id" value="<?= (int)$idLogin ?>">
                            <input type="hidden" id="user_tipe" value="<?= htmlspecialchars($user['tipe'] ?? '', ENT_QUOTES) ?>">
                            <input type="hidden" id="current_dari_pribadi" value="<?= htmlspecialchars($_POST['dari_pribadi'] ?? '', ENT_QUOTES) ?>">

                            <!-- Backdate Checkbox -->
                            <div class="col-12">
                                <div class="form-check">
                                    <input type="checkbox" class="form-check-input" id="backdate_checkbox" name="is_backdate">
                                    <label class="form-check-label" for="backdate_checkbox">Backdate</label>
                                </div>
                            </div>

                            <!-- Nomor Surat with Previous Number Indicator -->
                            <div class="col-12">
                                <label for="nomor_surat" class="form-label">
                                    Nomor Surat <span class="text-danger">*</span>
                                    <small id="prev_nomor_indicator" class="text-muted ms-2" style="font-weight: normal;"></small>
                                </label>
                                <input type="text" id="nomor_surat" name="nomor_surat" class="form-control" required
                                    readonly
                                    value="<?php echo htmlspecialchars($_POST['nomor_surat'] ?? $surat['nomor_surat'] ?? '', ENT_QUOTES, 'UTF-8'); ?>">

                                <!-- Backdate Edit Fields (hidden by default) -->
                                <div id="backdate_fields" class="d-none mt-2">
                                    <div class="row g-2">
                                        <div class="col-md-4">
                                            <label for="backdate_sequence" class="form-label small">Nomor Urut</label>
                                            <input type="number" id="backdate_sequence" name="backdate_sequence"
                                                class="form-control form-control-sm" min="1" placeholder="Urutan"
                                                value="<?php echo htmlspecialchars($surat['sequence_number'] ?? '', ENT_QUOTES, 'UTF-8'); ?>">
                                        </div>
                                        <div class="col-md-4" id="backdate_year_field" style="display:none;">
                                            <label for="backdate_year" class="form-label small">Tahun</label>
                                            <input type="number" id="backdate_year" name="backdate_year"
                                                class="form-control form-control-sm" min="2020" max="2099"
                                                value="<?= date('Y') ?>" placeholder="Tahun">
                                        </div>
                                    </div>
                                    <small class="text-muted" id="backdate_hint">Masukkan nomor urut untuk backdate</small>
                                </div>
                            </div>

                            <!-- Tanggal Surat -->
                            <div class="col-12">
                                <label for="tanggal_surat" class="form-label">Tanggal Surat <span
                                        class="text-danger">*</span></label>
                                <input type="date" id="tanggal_surat" name="tanggal_surat" class="form-control" required
                                    value="<?php echo htmlspecialchars($_POST['tanggal_surat'] ?? $surat['tanggal_surat'] ?? '', ENT_QUOTES, 'UTF-8'); ?>">
                            </div>

                            <!-- Paraf field removed - now auto-generated from approval chain -->
                        </div>
                    </div>

                    <!-- Hidden field to preserve sequence number for API calls -->
                    <input type="hidden" id="sequence_number" name="sequence_number"
                        value="<?php echo htmlspecialchars($surat['sequence_number'] ?? '', ENT_QUOTES, 'UTF-8'); ?>">

                    <!-- CARD 2 (SUBYEK) - Only Kepada field remains here -->
                    <div class="card mb-4" id="subyek-card">
                        <div class="card-header">Subyek</div>
                        <div class="card-body row g-3">
                            <!-- Kepada -->
                            <div class="col-12" id="kepada-field">
                                <label for="kepada" class="form-label">Kepada <span class="text-danger">*</span></label>
                                <input type="text" id="kepada" name="kepada" class="form-control"
                                    value="<?php echo htmlspecialchars($_POST['kepada'] ?? $surat['kepada'] ?? '', ENT_QUOTES, 'UTF-8'); ?>">
                            </div>
                        </div>
                    </div>

                    <!-- CARD 3 (INFORMASI SURAT) -->
                    <div class="card mb-3">
                        <div class="card-header">Informasi Surat</div>
                        <div class="card-body row g-3">
                            <!-- Perihal -->
                            <div class="col-12">
                                <label for="perihal" class="form-label">Perihal <span
                                        class="text-danger">*</span></label>
                                <input type="text" id="perihal" name="perihal" class="form-control" required
                                    value="<?php echo htmlspecialchars($_POST['perihal'] ?? $surat['perihal'] ?? '', ENT_QUOTES, 'UTF-8'); ?>">
                            </div>

                            <!-- Lampiran -->
                            <div class="col-12">
                                <label for="lampiran" class="form-label">Lampiran</label>
                                <input type="file" id="lampiran" name="lampiran" class="form-control"
                                    accept=".pdf,.jpg,.jpeg,.png">

                                <input type="hidden" id="lampiran_url" name="lampiran_url" value="">
                                <input type="hidden" id="lampiran_mime" name="lampiran_mime" value="">

                                <?php if (!empty($currentLampiranList)): ?>
                                    <small class="text-muted d-block mt-1">
                                        Lampiran saat ini:
                                        <?php foreach ($currentLampiranList as $lp): ?>
                                            <?php $url = htmlspecialchars($lp, ENT_QUOTES, 'UTF-8'); ?>
                                            <a href="<?= $url ?>" target="_blank">
                                                <?= htmlspecialchars($surat['lampiran_orig_name'] ?? basename($lp), ENT_QUOTES, 'UTF-8'); ?>
                                            </a>
                                        <?php endforeach; ?>
                                    </small>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                    <!-- TOMBOL -->
                    <div class="d-flex justify-content-end gap-2 mt-3">
                        <button type="button" class="btn btn-save-custom" id="toStep2">Lanjutkan</button>
                    </div>
                </section>

                <!-- ============ STEP 2 ============ -->
                <section class="step-pane d-none" data-step="2">
                    <div class="card mb-3">
                        <div class="card-header d-flex align-items-center justify-content-between">
                            <span style="font-size: 18px;">Isi Surat (Mode Edit)</span>
                        </div>
                        <div class="card-body p-0">
                            <textarea id="editorNaskah" name="naskah_html"></textarea>
                        </div>
                    </div>

                    <input type="hidden" name="__final_submit" value="1" />
                    <?php if ($approvalMode): ?>
                        <input type="hidden" name="approval_mode" value="1" />
                        <input type="hidden" name="is_final_approver" value="<?= $isFinalApprover ? '1' : '0' ?>" />
                    <?php endif; ?>

                    <div class="d-flex justify-content-between align-items-center flex-wrap gap-2 mt-3">
                        <button type="button" class="btn btn-danger" id="backTo1">Kembali</button>

                        <div class="d-flex gap-2">
                            <button type="button" id="btnUploadPdf" class="btn btn-print-custom">
                                Cetak & Upload PDF
                            </button>
                            <?php if ($approvalMode): ?>
                                <?php if (isset($alreadyApproved) && $alreadyApproved): ?>
                                    <button type="button" id="btnApprove" class="btn btn-success fw-bold" disabled>
                                        <i class="fas fa-check-circle"></i> ✓ Sudah Di-Approve
                                    </button>
                                <?php else: ?>
                                    <button type="button" id="btnApprove" class="btn btn-warning fw-bold">
                                        <i class="fas fa-check-circle"></i> <?= $isFinalApprover ? 'Approve & Generate QR' : 'Approve' ?>
                                    </button>
                                <?php endif; ?>
                            <?php else: ?>
                                <button type="submit" id="btnSave" class="btn btn-save-custom">
                                    Perbarui
                                </button>
                            <?php endif; ?>
                        </div>
                    </div>

                    <input type="hidden" name="file_surat" id="fileSuratPath"
                        value="<?php echo htmlspecialchars($_POST['file_surat'] ?? '', ENT_QUOTES, 'UTF-8'); ?>">
                    <input type="hidden" name="paraf" id="paraf"
                        value="<?php echo htmlspecialchars($approvalMode ? ($expectedParafForApproval ?? $currentParaf ?? '') : ($_POST['paraf'] ?? ''), ENT_QUOTES, 'UTF-8'); ?>">
                    <input type="hidden" name="path_ttd" id="path_ttd"
                        value="<?php echo htmlspecialchars($signaturePath, ENT_QUOTES, 'UTF-8'); ?>">
                    <input type="hidden" name="qr_code" id="qr_code"
                        value="<?php echo htmlspecialchars($surat['qr_code'] ?? '', ENT_QUOTES, 'UTF-8'); ?>">
                    <input type="hidden" name="nama_pejabat" id="nama_pejabat" value="<?php echo htmlspecialchars($signerNama, ENT_QUOTES, 'UTF-8'); ?>">
                    <input type="hidden" name="jabatan_pejabat" id="jabatan_pejabat" value="<?php echo htmlspecialchars($signerJabatan, ENT_QUOTES, 'UTF-8'); ?>">
                </section>
            </form>
        </div>
    </div>

    <!-- JS -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

    <!-- JS Surat -->
    <script>
        // Office ID from server for reliable sequence ID preservation
        const OFFICE_ID = <?php echo json_encode($officeId); ?>;

        // PARAF display control - only show when penandatangan != dibuat_oleh
        const SHOULD_SHOW_PARAF = <?php echo json_encode($shouldShowParaf); ?>;

        // Signature path for final approver
        const SIGNATURE_PATH = <?php echo json_encode($signaturePath); ?>;

        // Surat type code (SK, KPTS, SE) for validation
        const TIPE_SURAT_KODE = <?php echo json_encode($surat['tipe_surat_kode'] ?? ''); ?>;
        console.log('📋 Surat type loaded:', TIPE_SURAT_KODE);

        // Saved naskah content from database (if any)
        const SAVED_NASKAH = <?php echo json_encode($surat['naskah'] ?? ''); ?>;

        // Extract body content from saved naskah, removing any old CSS
        // We always use fresh template CSS for consistency
        let NASKAH_BODY_ONLY = SAVED_NASKAH;

        if (SAVED_NASKAH && SAVED_NASKAH.includes('<style>')) {
            // Remove old style block - we'll use fresh template CSS instead
            NASKAH_BODY_ONLY = SAVED_NASKAH.replace(/<style[\s\S]*?<\/style>/i, '').trim();
            console.log('📄 Removed old CSS from saved naskah - will use fresh template CSS');
        }

        // ===== GLOBAL FUNCTIONS (accessible from all scopes) =====
        /**
         * Extract edited field values from TinyMCE editor and sync to form inputs
         */
        function extractValuesFromEditor() {
            if (!window.tinymce || !tinymce.get('editorNaskah')) {
                console.warn('⚠️ TinyMCE not ready for extraction');
                return;
            }

            const editor = tinymce.get('editorNaskah');
            const content = editor.getContent();

            // Create a temporary DOM to parse the content
            const tempDiv = document.createElement('div');
            tempDiv.innerHTML = content;

            // Use global TIPE_SURAT_KODE for nomor_surat validation
            const tipeSuratKode = TIPE_SURAT_KODE || '';
            console.log(`   📋 Using surat type: ${tipeSuratKode}`);

            // Map of data-field values to form input IDs
            const fieldMapping = {
                'TANGGAL_SURAT': 'tanggal_surat',
                'KOTA': 'tempat_dibuat',
                'KEPADA': 'kepada',
                'PERIHAL': 'perihal',
                'NOMOR_SURAT': 'nomor_surat'
            };

            let extractedCount = 0;

            console.log('🔍 DEBUG extractValuesFromEditor():');
            console.log('   Editor content length:', content.length);

            // Find all elements with data-field attribute
            const editableFields = tempDiv.querySelectorAll('[data-field]');
            console.log('   Found', editableFields.length, 'editable fields with data-field attribute');

            editableFields.forEach(el => {
                const fieldName = el.getAttribute('data-field');
                const inputId = fieldMapping[fieldName];

                console.log(`   Processing field: ${fieldName} → ${inputId}`);

                if (inputId) {
                    const formInput = document.getElementById(inputId);
                    if (formInput) {
                        let value = el.textContent.trim();
                        console.log(`     Extracted text: "${value}"`);

                        // Skip if value is still a placeholder
                        if (value.startsWith('[[') && value.endsWith(']]')) {
                            console.log(`     ⏭️ Skipping placeholder`);
                            return;
                        }

                        // Special handling for date field - convert Indonesian date to YYYY-MM-DD
                        if (fieldName === 'TANGGAL_SURAT') {
                            const convertedDate = convertIndonesianDateToISO(value);
                            if (convertedDate) {
                                value = convertedDate;
                                console.log(`     📅 Converted to ISO: "${convertedDate}"`);
                            } else {
                                console.warn(`     ⚠️ Could not convert date: "${value}"`);
                                return; // Don't update if we can't convert
                            }
                        }

                        // Special handling for nomor_surat - validate based on surat type
                        if (fieldName === 'NOMOR_SURAT') {
                            const validatedValue = validateNomorSurat(value, formInput.value, tipeSuratKode);
                            if (validatedValue === null && value !== formInput.value) {
                                // Invalid edit - skip this update
                                console.warn(`     ❌ Invalid nomor_surat edit - NOT updating`);
                                return;
                            }
                            if (validatedValue === null) {
                                // No change or invalid
                                console.log(`     ℹ️ No valid changes to nomor_surat`);
                                return;
                            }
                            value = validatedValue;
                        }

                        // Only update if value is different
                        if (formInput.value !== value) {
                            console.log(`     ✅ UPDATING: "${formInput.value}" → "${value}"`);
                            formInput.value = value;
                            extractedCount++;
                        } else {
                            console.log(`     ℹ️ Value unchanged`);
                        }
                    } else {
                        console.log(`     ❌ Form input #${inputId} not found!`);
                    }
                }
            });

            console.log(`   ✅ Extraction complete: ${extractedCount} field(s) updated`);
            console.log('   Form values after extraction:');
            for (const [field, inputId] of Object.entries(fieldMapping)) {
                const inp = document.getElementById(inputId);
                if (inp) {
                    console.log(`     #${inputId} = "${inp.value}"`);
                }
            }
            return extractedCount;
        }

        /**
         * Convert Indonesian date format to ISO format (YYYY-MM-DD)
         * Handles formats like: "19 Desember 2025", "Minggu, 7 Desember 2025", "19-12-2025", "19/12/2025"
         */
        function convertIndonesianDateToISO(dateStr) {
            if (!dateStr) return null;

            // Already in ISO format?
            if (/^\d{4}-\d{2}-\d{2}$/.test(dateStr)) {
                return dateStr;
            }

            // Indonesian month names
            const months = {
                'januari': '01',
                'februari': '02',
                'maret': '03',
                'april': '04',
                'mei': '05',
                'juni': '06',
                'juli': '07',
                'agustus': '08',
                'september': '09',
                'oktober': '10',
                'november': '11',
                'desember': '12'
            };

            // Indonesian day names (to remove if present)
            const dayNames = ['senin', 'selasa', 'rabu', 'kamis', 'jumat', 'sabtu', 'minggu'];

            // Try format: "Minggu, 7 Desember 2025" (with day name)
            const match3 = dateStr.match(/^(\w+),?\s+(\d{1,2})\s+(\w+)\s+(\d{4})$/i);
            if (match3) {
                const dayName = match3[1].toLowerCase();
                const day = match3[2].padStart(2, '0');
                const monthName = match3[3].toLowerCase();
                const year = match3[4];
                const month = months[monthName];
                if (month && dayNames.includes(dayName)) {
                    console.log(`   📅 Parsed format with day name: "${dateStr}" → "${year}-${month}-${day}"`);
                    return `${year}-${month}-${day}`;
                }
            }

            // Try format: "19 Desember 2025"
            const match = dateStr.match(/^(\d{1,2})\s+(\w+)\s+(\d{4})$/i);
            if (match) {
                const day = match[1].padStart(2, '0');
                const monthName = match[2].toLowerCase();
                const year = match[3];
                const month = months[monthName];
                if (month) {
                    return `${year}-${month}-${day}`;
                }
            }

            // Try format: "19-12-2025" or "19/12/2025" (DD-MM-YYYY)
            const match2 = dateStr.match(/^(\d{1,2})[-\/](\d{1,2})[-\/](\d{4})$/);
            if (match2) {
                const day = match2[1].padStart(2, '0');
                const month = match2[2].padStart(2, '0');
                const year = match2[3];
                return `${year}-${month}-${day}`;
            }
            return null;
        }

        /**
         * Validate and extract editable parts of nomor_surat based on surat type
         * Returns the edited number if valid, null if invalid
         * 
         * Patterns:
         * - KPTS: "001/KPTS-JPT/2025" → editable parts: sequence (001) and year (2025)
         * - SE: "001/SE-DIR/2025" → editable parts: sequence (001) and year (2025)
         * - SK: "DKU.JPT.HK.09.03.8" → editable part: only the last number (8)
         */
        function validateNomorSurat(editedValue, originalValue, tipeSuratKode) {
            if (!editedValue || editedValue === originalValue) {
                return null; // No change, skip update
            }

            console.log(`   🔢 Validating nomor_surat for tipeSuratKode="${tipeSuratKode}": "${originalValue}" → "${editedValue}"`);

            if (tipeSuratKode === 'KPTS') {
                // Format: 001/KPTS-JPT/2025
                // Only sequence (001) and year (2025) should change
                // Middle part KPTS-JPT MUST NOT change
                const regex = /^(\d+)\/KPTS-JPT\/(\d{4})$/i;
                const editMatch = editedValue.match(regex);
                const origMatch = originalValue.match(regex);

                if (!editMatch) {
                    console.warn(`     ❌ Invalid KPTS format in edited value. Expected: ###/KPTS-JPT/YYYY, got: "${editedValue}"`);
                    return null;
                }
                if (!origMatch) {
                    console.warn(`     ❌ Invalid KPTS format in original value. Expected: ###/KPTS-JPT/YYYY, got: "${originalValue}"`);
                    return null;
                }

                const editSeq = editMatch[1];
                const editYear = editMatch[2];
                const origSeq = origMatch[1];
                const origYear = origMatch[2];

                // The middle part must remain unchanged
                if (editMatch[0].toUpperCase() === origMatch[0].toUpperCase()) {
                    console.log(`     ℹ️ No actual change in nomor_surat`);
                    return null;
                }

                console.log(`     ✅ Valid KPTS edit: sequence ${origSeq}→${editSeq}, year ${origYear}→${editYear}`);
                return editedValue;

            } else if (tipeSuratKode === 'SE') {
                // Format: 001/SE-DIR/2025
                // Only sequence (001) and year (2025) should change
                // Middle part SE-DIR MUST NOT change
                const regex = /^(\d+)\/SE-DIR\/(\d{4})$/i;
                const editMatch = editedValue.match(regex);
                const origMatch = originalValue.match(regex);

                if (!editMatch) {
                    console.warn(`     ❌ Invalid SE format in edited value. Expected: ###/SE-DIR/YYYY, got: "${editedValue}"`);
                    return null;
                }
                if (!origMatch) {
                    console.warn(`     ❌ Invalid SE format in original value. Expected: ###/SE-DIR/YYYY, got: "${originalValue}"`);
                    return null;
                }

                const editSeq = editMatch[1];
                const editYear = editMatch[2];
                const origSeq = origMatch[1];
                const origYear = origMatch[2];

                // The middle part must remain unchanged
                if (editMatch[0].toUpperCase() === origMatch[0].toUpperCase()) {
                    console.log(`     ℹ️ No actual change in nomor_surat`);
                    return null;
                }

                console.log(`     ✅ Valid SE edit: sequence ${origSeq}→${editSeq}, year ${origYear}→${editYear}`);
                return editedValue;

            } else if (tipeSuratKode === 'SK') {
                // Format: DKU.JPT.HK.09.03.8
                // Only the last number (urutan) should change
                const parts = editedValue.split('.');
                const origParts = originalValue.split('.');

                if (parts.length !== 6 || origParts.length !== 6) {
                    console.warn(`     ❌ Invalid SK format. Expected 6 parts separated by dots, got: "${editedValue}"`);
                    return null;
                }

                // Check if all parts except last are the same (case-insensitive for prefix)
                const prefixMatch = parts.slice(0, 5).every((p, i) => p.toUpperCase() === origParts[i].toUpperCase());

                if (!prefixMatch) {
                    console.warn(`     ❌ Invalid SK edit. Cannot change the prefix (DKU.JPT.HK.09.03). Tried to change from: "${originalValue}" to: "${editedValue}"`);
                    return null;
                }

                if (isNaN(parts[5]) || isNaN(origParts[5])) {
                    console.warn(`     ❌ Invalid SK format. Last part must be a number. Got: "${parts[5]}"`);
                    return null;
                }

                if (parts[5] === origParts[5]) {
                    console.log(`     ℹ️ No actual change in urutan number`);
                    return null;
                }

                console.log(`     ✅ Valid SK edit: urutan ${origParts[5]}→${parts[5]}`);
                return editedValue;

            } else {
                console.warn(`     ⚠️ Unknown surat type: "${tipeSuratKode}". Cannot validate nomor_surat. Accepting as-is.`);
                // For unknown types, don't allow changes to nomor_surat
                return null;
            }

            return null;
        }

        /**
         * Helper functions for form value collection
         */
        function get(id) {
            return document.getElementById(id);
        }

        function getVal(id) {
            return (get(id)?.value || '').trim();
        }

        function safe(v) {
            return String(v ?? '');
        }

        function toIDDate(str) {
            if (!str) return '';
            const m = /^(\d{4})-(\d{2})-(\d{2})$/.exec(str);
            if (!m) return str;

            const months = [
                'Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni',
                'Juli', 'Agustus', 'September', 'Oktober', 'November', 'Desember'
            ];

            const day = parseInt(m[3]);
            const monthIndex = parseInt(m[2]) - 1;
            const year = m[1];

            return `${day} ${months[monthIndex]} ${year}`;
        }

        function todayID() {
            const d = new Date();
            const months = [
                'Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni',
                'Juli', 'Agustus', 'September', 'Oktober', 'November', 'Desember'
            ];

            const day = d.getDate();
            const month = months[d.getMonth()];
            const year = d.getFullYear();
            return `${day} ${month} ${year}`;
        }

        /**
         * Parse nomor surat into editable and non-editable parts based on letter type
         * - SK (Surat Keluar): Format KD.KL.JE.SJ.01 - only sequence editable
         * - KPTS (Surat Keputusan): Format 001/KPTS-JPT/2025 - sequence and year editable
         * - SE (Surat Edaran): Format 001/SE-DIR/2025 - sequence and year editable
         */
        function parseNomorSurat(nomorSurat, tipeSuratKode) {
            const result = {
                full: nomorSurat,
                prefix: '',
                sequence: '',
                middle: '',
                year: '',
                prefixReadonly: '',
                middleReadonly: ''
            };

            if (!nomorSurat) return result;

            if (tipeSuratKode === 'SK') {
                // SK format: KD.KL.JE.SJ.01 or KL.JE.SJ.01
                const parts = nomorSurat.split('.');
                if (parts.length >= 2) {
                    result.sequence = parts[parts.length - 1]; // Last part is sequence
                    result.prefix = parts.slice(0, -1).join('.'); // Everything except last part
                    result.prefixReadonly = result.prefix + '.';
                } else {
                    result.sequence = nomorSurat;
                }
            } else if (tipeSuratKode === 'KPTS') {
                // KPTS format: 001/KPTS-JPT/2025
                const match = nomorSurat.match(/^(\d+)(\/KPTS-JPT\/)(\d{4})$/);
                if (match) {
                    result.sequence = match[1];
                    result.middle = match[2];
                    result.year = match[3];
                    result.middleReadonly = match[2];
                } else {
                    result.sequence = nomorSurat;
                }
            } else if (tipeSuratKode === 'SE') {
                // SE format: 001/SE-DIR/2025
                const match = nomorSurat.match(/^(\d+)(\/SE-DIR\/)(\d{4})$/);
                if (match) {
                    result.sequence = match[1];
                    result.middle = match[2];
                    result.year = match[3];
                    result.middleReadonly = match[2];
                } else {
                    result.sequence = nomorSurat;
                }
            } else {
                // Default: entire nomor is the sequence
                result.sequence = nomorSurat;
            }

            return result;
        }

        /**
         * Collect all form values for placeholder replacement
         */
        function collectFormValues() {
            const jenisOpt = get('jenis_surat_id')?.selectedOptions?.[0];
            const klasOpt = get('klasifikasi_arsip_id')?.selectedOptions?.[0];
            const dariTipe = (document.querySelector('input[name="dari_tipe"]:checked')?.value || '');

            const namaLogin = getVal('user_nama');
            const jabatanUser = getVal('user_jabatan');

            let namaPejabat = getVal('nama_pejabat') || '';
            let jabatanPejabat = getVal('jabatan_pejabat') || '';

            if (dariTipe === 'pribadi') {
                const selectedOption = get('dari_pribadi')?.selectedOptions?.[0];
                if (selectedOption) {
                    // Extract nama from "Nama - Jabatan" format
                    const optionText = selectedOption.textContent.trim();
                    namaPejabat = optionText.split('-')[0].trim();
                    jabatanPejabat = selectedOption?.dataset?.jabatan || jabatanUser || '';
                } else {
                    namaPejabat = namaLogin;
                    jabatanPejabat = jabatanUser || '';
                }
            } else if (dariTipe === 'sendiri') {
                namaPejabat = namaLogin;
                jabatanPejabat = jabatanUser || '';
            }

            const tglInput = getVal('tanggal_surat');
            const tglID = toIDDate(tglInput) || todayID();

            // Determine document title based on tipe_surat
            const tipeSuratSelect = document.getElementById('tipe_surat_id');
            let documentTitle = 'Surat Keluar'; // default
            let tipeSuratKode = '';
            if (tipeSuratSelect) {
                const selectedOption = tipeSuratSelect.options[tipeSuratSelect.selectedIndex];
                if (selectedOption) {
                    tipeSuratKode = selectedOption.dataset?.kode || '';
                    const tipeSuratText = selectedOption.textContent.trim();
                    if (tipeSuratText.includes('Keputusan')) {
                        documentTitle = 'KEPUTUSAN DIREKSI PT JASAMARGA PANDAAN TOL';
                    } else if (tipeSuratText.includes('Edaran')) {
                        documentTitle = 'SURAT EDARAN';
                    } else {
                        // For regular Surat Keluar, always use "Surat Keluar"
                        documentTitle = 'Surat Keluar';
                    }
                }
            }

            // Parse nomor surat into editable parts based on letter type
            const nomorSurat = getVal('nomor_surat');
            const parsedNomor = parseNomorSurat(nomorSurat, tipeSuratKode);

            // Get PARAF value - read directly from input field
            const parafValue = getVal('paraf');
            console.log('🔍 [collectFormValues] PARAF raw from #paraf input:', parafValue);

            return {
                JENIS_SURAT: safe(jenisOpt ? jenisOpt.textContent.trim() : ''),
                DOCUMENT_TITLE: safe(documentTitle),
                NOMOR_SURAT: safe(nomorSurat),

                // Parsed nomor surat parts for backdate editing
                NOMOR_SEQUENCE: safe(parsedNomor.sequence),
                NOMOR_YEAR: safe(parsedNomor.year),
                NOMOR_PREFIX_READONLY: safe(parsedNomor.prefixReadonly),
                NOMOR_MIDDLE_READONLY: safe(parsedNomor.middleReadonly),

                KOTA: safe(getVal('tempat_dibuat')),
                TANGGAL_SURAT: safe(tglID),

                // Surat Keluar specific fields
                LAMPIRAN: safe(getVal('lampiran_count') || '-'),
                PERIHAL: safe(getVal('perihal') || ''),
                KEPADA: safe(getVal('kepada') || ''),
                JABATAN_TUJUAN: safe(getVal('jabatan_tujuan') || ''),
                ALAMAT_TUJUAN: safe(getVal('alamat_tujuan') || ''),

                // Reference surat fields (for PKL letters)
                NOMOR_SURAT_REF: safe(getVal('nomor_surat_ref') || '....../....../.......'),
                TANGGAL_SURAT_REF: safe(getVal('tanggal_surat_ref') || '...............'),

                // Signatory info
                NAMA_PENANDATANGAN: safe(namaPejabat),
                JABATAN_PENANDATANGAN: safe(jabatanPejabat),

                // Legacy fields (for old templates)
                NAMA_PEJABAT: safe(namaPejabat),
                NIK_PEJABAT: safe(getVal('nik_pejabat') || ''),
                JABATAN_PEJABAT: safe(jabatanPejabat),
                DEPARTEMEN: safe(getVal('departemen') || ''),
                NAMA_PERUSAHAAN: safe(getVal('nama_perusahaan') || 'PT Jasamarga Pandaan Tol'),

                NAMA_MAHASISWA: safe(getVal('kepada') || '...............'),
                NIM: safe(getVal('nim') || '...............'),
                JURUSAN: safe(getVal('jurusan') || ''),
                PROGRAM_STUDI: safe(getVal('program_studi') || '...............'),
                UNIVERSITAS: safe(getVal('universitas') || ''),

                TANGGAL_MULAI: getVal('tanggal_mulai') || '...............',
                TANGGAL_SELESAI: getVal('tanggal_selesai') || '...............',

                PATH_TANDA_TANGAN: safe(getVal('path_ttd') || SIGNATURE_PATH || ''),
                QR_CODE: safe(getVal('qr_code') || ''),
                // PATH_STEMPEL:      safe(getVal('path_stempel') || ''),

                PARAF: safe(parafValue),

                DARI_TIPE: dariTipe
            };
        }
        // ===== END GLOBAL FUNCTIONS =====

        (function() {
            const jenisSelect = document.getElementById('jenis_surat_id');
            const klasSelect = document.getElementById('klasifikasi_arsip_id');
            const nomorInput = document.getElementById('nomor_surat');

            async function updateNomor() {
                if (!jenisSelect || !klasSelect || !nomorInput) return;
                const jenisId = parseInt(jenisSelect.value || '0', 10);
                const klasId = parseInt(klasSelect.value || '0', 10);
                if (!jenisId || !klasId) return;

                try {
                    const url =
                        `/jasa_marga/api/letter-autofill.php?jenis_id=${encodeURIComponent(jenisId)}&klas_id=${encodeURIComponent(klasId)}`;
                    const res = await fetch(url, {
                        credentials: 'same-origin'
                    });

                    const ctype = res.headers.get('content-type') || '';
                    if (!ctype.includes('application/json')) {
                        console.warn('Autofill nomor: respons bukan JSON. Status:', res.status);
                        return;
                    }

                    const data = await res.json();
                    if (data && data.ok && data.nomor) {
                        nomorInput.value = data.nomor;
                    } else {
                        console.warn('Autofill nomor: data tidak ok', data);
                    }
                } catch (e) {
                    console.warn('Gagal auto nomor_surat', e);
                }
            }

            if (jenisSelect) jenisSelect.addEventListener('change', updateNomor);
            if (klasSelect) klasSelect.addEventListener('change', updateNomor);
        })();

        (function() {
            const get = (id) => document.getElementById(id);
            const fileInput = document.getElementById('lampiran');
            const hidUrl = document.getElementById('lampiran_url');
            const hidMime = document.getElementById('lampiran_mime');

            let lastBlobUrl = null;

            fileInput?.addEventListener('change', (e) => {
                const f = e.target.files && e.target.files[0] ? e.target.files[0] : null;

                if (lastBlobUrl) {
                    URL.revokeObjectURL(lastBlobUrl);
                    lastBlobUrl = null;
                }

                if (!f) {
                    hidUrl.value = '';
                    hidMime.value = '';
                    return;
                }

                const blobUrl = URL.createObjectURL(f);
                lastBlobUrl = blobUrl;

                hidUrl.value = blobUrl;
                hidMime.value = f.type || '';
            });
        })();
    </script>

    <!-- TinyMCE CDN -->
    <script src="https://cdn.tiny.cloud/1/cmsiwdrufuu76fz8pr2le38xwjjm6np15k9gz62jfvaespms/tinymce/7/tinymce.min.js"
        referrerpolicy="origin"></script>
    <script>
        // Global function to determine template URL based on tipe_surat
        function getTemplateUrl() {
            const tipeSelect = document.getElementById('tipe_surat_id');
            if (!tipeSelect) return 'template-letter.php';

            const selectedOption = tipeSelect.options[tipeSelect.selectedIndex];
            if (!selectedOption) return 'template-letter.php';

            // Use data-kode attribute (same as create-letter.php)
            const kode = selectedOption.dataset.kode || 'SK';

            if (kode === 'SE') {
                return 'template-letter.php?type=edaran';
            } else if (kode === 'KPTS') {
                return 'template-letter.php?type=keputusan';
            }
            return 'template-letter.php';
        }

        async function fetchTemplateStyles() {
            const res = await fetch(getTemplateUrl(), {
                cache: 'no-store'
            });
            const html = await res.text();
            const m = html.match(/<style[^>]*>([\s\S]*?)<\/style>/i);
            return m ? m[1] : '';
        }

        // Dynamic Classification System JavaScript
        (function() {
            const tipeSuratSelect = document.getElementById('tipe_surat_id');
            const classificationFields = document.getElementById('classification-fields');
            const legacyJenisField = document.getElementById('legacy-jenis-field');
            const legacyKlasifikasiField = document.getElementById('legacy-klasifikasi-field');

            const klasifikasiSelect = document.getElementById('klasifikasi_arsip_new');
            const jenisSelect = document.getElementById('jenis_surat_new');
            const subjenisSelect = document.getElementById('subjenis_surat_id');
            const nomorInput = document.getElementById('nomor_surat');
            const dariPribadiSelect = document.getElementById('dari_pribadi');

            // Store extracted ID to preserve across dropdown changes
            let preservedSequenceId = null;

            // Initialize form
            initializeForm();

            // Event listeners
            if (tipeSuratSelect) {
                tipeSuratSelect.addEventListener('change', handleTipeSuratChange);
            }
            if (klasifikasiSelect) {
                klasifikasiSelect.addEventListener('change', handleKlasifikasiChange);
            }
            if (jenisSelect) {
                jenisSelect.addEventListener('change', handleJenisChange);
            }
            if (subjenisSelect) {
                subjenisSelect.addEventListener('change', handleSubjenisChange);
            }
            if (dariPribadiSelect) {
                dariPribadiSelect.addEventListener('change', handleDariPribadiChange);
            }

            async function handleDariPribadiChange() {
                // Generate nomor surat when pejabat changes
                await generateNomorSurat();
            }

            async function initializeForm() {
                await loadTipeSurat();
                await loadKlasifikasi();
            }

            async function loadTipeSurat() {
                try {
                    const response = await fetch('api_classification.php?action=get_tipe_surat');
                    const data = await response.json();

                    console.log('Tipe Surat API Response:', data);

                    if (data.success && tipeSuratSelect) {
                        tipeSuratSelect.innerHTML =
                            '<option value="" disabled>-- Pilih Tipe Surat --</option>';
                        data.data.forEach(item => {
                            const option = document.createElement('option');
                            option.value = item.id;
                            option.textContent = item.nama;
                            option.dataset.kode = item.kode;
                            tipeSuratSelect.appendChild(option);
                        });

                        // Set selected value from PHP if exists
                        const selectedValue = '<?php echo addslashes($_POST["tipe_surat_id"] ?? ""); ?>';
                        console.log('Selected Tipe Surat ID from PHP:', selectedValue);
                        console.log('Available options:', Array.from(tipeSuratSelect.options).map(o => ({
                            value: o.value,
                            text: o.textContent
                        })));

                        if (selectedValue && selectedValue !== '') {
                            // Wait a bit for options to be fully rendered
                            setTimeout(async () => {
                                tipeSuratSelect.value = selectedValue;
                                console.log('Tipe Surat value after setting:', tipeSuratSelect.value);
                                console.log('Selected option:', tipeSuratSelect.selectedOptions[0]
                                    ?.textContent);
                                // Trigger change to show appropriate fields
                                await handleTipeSuratChange();

                                // Load classification data if SK is selected
                                const kode = tipeSuratSelect.selectedOptions[0]?.dataset.kode;
                                if (kode === 'SK') {
                                    // Load saved classification values
                                    const klasifikasiValue =
                                        '<?php echo addslashes($_POST["klasifikasi_arsip_new"] ?? ""); ?>';
                                    const jenisValue =
                                        '<?php echo addslashes($_POST["jenis_surat_new"] ?? ""); ?>';
                                    const subjenisValue =
                                        '<?php echo addslashes($_POST["subjenis_surat_id"] ?? ""); ?>';

                                    console.log('Loading SK classification values:', {
                                        klasifikasiValue,
                                        jenisValue,
                                        subjenisValue
                                    });

                                    if (klasifikasiValue) {
                                        await new Promise(resolve => setTimeout(resolve, 100));
                                        klasifikasiSelect.value = klasifikasiValue;
                                        await handleKlasifikasiChange();

                                        if (jenisValue) {
                                            await new Promise(resolve => setTimeout(resolve, 150));
                                            jenisSelect.value = jenisValue;
                                            console.log(
                                                'Jenis value set, calling handleJenisChange...');
                                            await handleJenisChange();

                                            if (subjenisValue) {
                                                await new Promise(resolve => setTimeout(resolve, 300));
                                                console.log('Setting subjenis value:', subjenisValue);
                                                console.log('Available subjenis options:', Array.from(
                                                    subjenisSelect.options).map(o => ({
                                                    value: o.value,
                                                    text: o.textContent
                                                })));
                                                subjenisSelect.value = subjenisValue;
                                                console.log('Subjenis selected:', subjenisSelect.value);
                                                console.log('Subjenis selectedIndex:', subjenisSelect
                                                    .selectedIndex);

                                                // Trigger subjenis change to generate nomor_surat with all fields set
                                                // The sequence ID is preserved via existing_sequence_id parameter
                                                await handleSubjenisChange();
                                            }
                                        }
                                    }
                                }
                            }, 100);
                        } else {
                            console.warn('No tipe surat value from PHP');
                            // No value selected, set first option as placeholder
                            if (tipeSuratSelect.options.length > 0) {
                                tipeSuratSelect.options[0].selected = true;
                            }
                        }
                    }
                } catch (error) {
                    console.error('Error loading tipe surat:', error);
                }
            }

            async function loadKlasifikasi() {
                try {
                    const response = await fetch('api_classification.php?action=get_klasifikasi');
                    const data = await response.json();

                    if (data.success && klasifikasiSelect) {
                        klasifikasiSelect.innerHTML =
                            '<option value="" disabled selected>-- Pilih Klasifikasi --</option>';
                        data.data.forEach(item => {
                            const option = document.createElement('option');
                            option.value = item.id;
                            option.textContent = `${item.kode} - ${item.nama}`;
                            option.dataset.kode = item.kode;
                            klasifikasiSelect.appendChild(option);
                        });
                    }
                } catch (error) {
                    console.error('Error loading klasifikasi:', error);
                }
            }

            async function handleTipeSuratChange() {
                const selectedOption = tipeSuratSelect.selectedOptions[0];
                if (!selectedOption) return;

                const tipeSuratKode = selectedOption.dataset.kode;
                const kepadaField = document.getElementById('kepada-field');
                const kepadaInput = document.getElementById('kepada');
                const subyekCard = document.getElementById('subyek-card');

                // Show/hide fields based on tipe surat
                if (tipeSuratKode === 'SK') { // Surat Keluar
                    classificationFields.classList.remove('d-none');
                    legacyJenisField.classList.add('d-none');
                    legacyKlasifikasiField.classList.add('d-none');
                    if (subyekCard) subyekCard.classList.remove('d-none');
                    if (kepadaInput) kepadaInput.required = true;

                    // Make new fields required
                    setFieldsRequired(['klasifikasi_arsip_new', 'jenis_surat_new', 'subjenis_surat_id'], true);
                    setFieldsRequired(['jenis_surat_id', 'klasifikasi_arsip_id'], false);
                } else { // Keputusan Direksi atau Edaran Direksi
                    classificationFields.classList.add('d-none');
                    legacyJenisField.classList.add('d-none');
                    legacyKlasifikasiField.classList.add('d-none');
                    if (subyekCard) subyekCard.classList.add('d-none');
                    if (kepadaInput) kepadaInput.required = false;

                    // Remove required from all classification fields
                    setFieldsRequired(['klasifikasi_arsip_new', 'jenis_surat_new', 'subjenis_surat_id'], false);
                    setFieldsRequired(['jenis_surat_id', 'klasifikasi_arsip_id'], false);
                }

                // Generate nomor surat
                await generateNomorSurat();
            }

            async function handleKlasifikasiChange() {
                const klasifikasiId = klasifikasiSelect.value;
                if (!klasifikasiId) return;

                // Clear dependent fields
                jenisSelect.innerHTML = '<option value="" disabled selected>Loading...</option>';
                subjenisSelect.innerHTML =
                    '<option value="" disabled selected>-- Pilih Jenis Surat Terlebih Dahulu --</option>';

                try {
                    const response = await fetch(
                        `api_classification.php?action=get_jenis_by_klasifikasi&klasifikasi_id=${klasifikasiId}`
                    );
                    const data = await response.json();

                    if (data.success) {
                        jenisSelect.innerHTML =
                            '<option value="" disabled selected>-- Pilih Jenis Surat --</option>';
                        data.data.forEach(item => {
                            const option = document.createElement('option');
                            option.value = item.id;
                            option.textContent = `${item.kode} - ${item.nama}`;
                            option.dataset.kode = item.kode;
                            jenisSelect.appendChild(option);
                        });
                    } else {
                        jenisSelect.innerHTML =
                            '<option value="" disabled selected>-- Tidak ada jenis surat --</option>';
                    }
                } catch (error) {
                    console.error('Error loading jenis surat:', error);
                    jenisSelect.innerHTML = '<option value="" disabled selected>-- Error loading data --</option>';
                }
            }

            async function handleJenisChange() {
                const jenisId = jenisSelect.value;
                if (!jenisId) return;

                // Clear dependent fields
                subjenisSelect.innerHTML = '<option value="" disabled selected>Loading...</option>';

                try {
                    const response = await fetch(
                        `api_classification.php?action=get_subjenis_by_jenis&jenis_id=${jenisId}`);
                    const data = await response.json();

                    if (data.success) {
                        subjenisSelect.innerHTML =
                            '<option value="" disabled selected>-- Pilih Subjenis Surat --</option>';
                        data.data.forEach(item => {
                            const option = document.createElement('option');
                            option.value = item.id;
                            option.textContent = `${item.kode} - ${item.nama}`;
                            option.dataset.kode = item.kode;
                            subjenisSelect.appendChild(option);
                        });
                    } else {
                        subjenisSelect.innerHTML =
                            '<option value="" disabled selected>-- Tidak ada subjenis surat --</option>';
                    }
                } catch (error) {
                    console.error('Error loading subjenis surat:', error);
                    subjenisSelect.innerHTML =
                        '<option value="" disabled selected>-- Error loading data --</option>';
                }

                // Generate nomor surat
                await generateNomorSurat();
            }

            async function handleSubjenisChange() {
                // Generate nomor surat when subjenis changes
                await generateNomorSurat();
            }

            async function generateNomorSurat() {
                if (!nomorInput) return;

                const tipeSuratId = tipeSuratSelect.value;
                if (!tipeSuratId) {
                    nomorInput.value = '';
                    return;
                }

                // Use reliable office ID from server instead of extracting from nomor surat
                // This prevents ID duplication issues when nomor surat gets corrupted
                // Get current sequence number from input field (not original office ID)
                const sequenceInput = document.getElementById('sequence_number');
                let existingSequenceId = sequenceInput ? sequenceInput.value : null;
                const currentNomor = nomorInput.value.trim(); // Define currentNomor for later use

                console.log('DEBUG - Using reliable office ID:', existingSequenceId);
                console.log('DEBUG - Current nomor surat:', currentNomor);
                console.log('DEBUG - Tipe surat ID:', tipeSuratId);

                // Store the reliable sequence ID for future use
                if (existingSequenceId) {
                    preservedSequenceId = existingSequenceId;
                    console.log('DEBUG - Stored preserved sequence ID:', preservedSequenceId);
                }

                try {
                    const params = new URLSearchParams({
                        action: 'generate_nomor_surat',
                        tipe_surat_id: tipeSuratId
                    });

                    console.log('=== NOMOR SURAT REGENERATION DEBUG ===');
                    console.log('Office ID from server:', OFFICE_ID);
                    console.log('Tipe surat ID:', tipeSuratId);
                    console.log('Final existingSequenceId:', existingSequenceId);

                    // Pass existing sequence ID to preserve it
                    if (existingSequenceId) {
                        params.append('existing_sequence_id', existingSequenceId);
                        console.log('✓ Added existing_sequence_id to params:', existingSequenceId);
                    } else {
                        console.log('⚠️ No existingSequenceId to preserve');
                    }

                    // Add classification data for Surat Keluar
                    const selectedTipeOption = tipeSuratSelect.selectedOptions[0];
                    console.log('Selected tipe option:', selectedTipeOption);
                    console.log('Tipe kode:', selectedTipeOption?.dataset?.kode);

                    if (selectedTipeOption && selectedTipeOption.dataset.kode === 'SK') {
                        console.log('=== SURAT KELUAR (SK) DEBUG ===');

                        const klasifikasiOption = klasifikasiSelect.selectedOptions[0];
                        const jenisOption = jenisSelect.selectedOptions[0];
                        const subjenisId = subjenisSelect.value;

                        console.log('Klasifikasi element:', klasifikasiSelect);
                        console.log('Klasifikasi option:', klasifikasiOption);
                        console.log('Klasifikasi kode:', klasifikasiOption?.dataset?.kode);
                        console.log('Jenis element:', jenisSelect);
                        console.log('Jenis option:', jenisOption);
                        console.log('Jenis kode:', jenisOption?.dataset?.kode);
                        console.log('Subjenis element:', subjenisSelect);
                        console.log('Subjenis ID:', subjenisId);

                        if (klasifikasiOption && jenisOption && subjenisId) {
                            console.log('✓ All required SK data available');
                            const klasifikasiKode = klasifikasiOption.dataset.kode;
                            const jenisKode = jenisOption.dataset.kode;

                            console.log('Adding params - Klasifikasi kode:', klasifikasiKode);
                            console.log('Adding params - Jenis kode:', jenisKode);
                            console.log('Adding params - Subjenis ID:', subjenisId);

                            params.append('klasifikasi_kode', klasifikasiKode);
                            params.append('jenis_kode', jenisKode);
                            params.append('subjenis_id', subjenisId);

                            // Get jabatan from dropdown "Nama Yang Mentandatangani"
                            let jabatan = '';
                            const dariTipeChecked = document.querySelector('input[name="dari_tipe"]:checked');

                            console.log('Dari tipe checked element:', dariTipeChecked);
                            console.log('Dari tipe value:', dariTipeChecked?.value);
                            console.log('Dari pribadi select:', dariPribadiSelect);
                            console.log('Dari pribadi value:', dariPribadiSelect?.value);

                            if (dariTipeChecked && dariTipeChecked.value === 'pribadi' && dariPribadiSelect &&
                                dariPribadiSelect.value) {
                                // Get jabatan from selected pejabat
                                const selectedOption = dariPribadiSelect.selectedOptions[0];
                                jabatan = selectedOption?.dataset?.jabatan || '';
                                console.log('Jabatan from pribadi:', jabatan);
                            } else if (dariTipeChecked && dariTipeChecked.value === 'sendiri') {
                                // Get jabatan from logged-in user (hidden field)
                                jabatan = document.getElementById('user_jabatan')?.value || '';
                                console.log('Jabatan from user_jabatan field:', jabatan);

                                // If jabatan still empty, try to get from selected option in dropdown
                                if (!jabatan && dariPribadiSelect && dariPribadiSelect.value) {
                                    const selectedOption = dariPribadiSelect.selectedOptions[0];
                                    jabatan = selectedOption?.dataset?.jabatan || '';
                                    console.log('Jabatan from dropdown fallback:', jabatan);
                                }
                            }

                            // Jabatan is optional - if empty, nomor will be generated without kode jabatan prefix
                            // This allows users without jabatan mapping to still create letters
                            console.log('Final jabatan:', jabatan);
                            if (!jabatan) {
                                console.info(
                                    '⚠️ Jabatan kosong - nomor surat akan dibuat tanpa kode jabatan prefix');
                            }

                            params.append('jabatan', jabatan || '');
                            console.log('✓ SK params complete');
                        } else {
                            // Not all fields selected yet
                            console.log('❌ SK: Not all required fields selected');
                            nomorInput.value = '';
                            return;
                        }
                    } else {
                        console.log('Non-SK surat type, skipping additional params');
                    }

                    console.log('Final API params:', params.toString());

                    const response = await fetch(`api_classification.php?${params}`);
                    console.log('API response status:', response.status);

                    if (!response.ok) {
                        console.error('API response not ok:', response.status, response.statusText);
                        nomorInput.value = '';
                        return;
                    }

                    const data = await response.json();
                    console.log('API response data:', data);

                    if (data.success) {
                        let generatedNomor = data.data.nomor_surat;
                        console.log('✓ API Success - Generated nomor:', generatedNomor);

                        // API now returns complete nomor surat, no need for post-processing
                        console.log('✓ Using complete nomor from API:', generatedNomor);
                        nomorInput.value = generatedNomor;
                        console.log('✓ Nomor surat updated in input field');
                    } else {
                        console.error('❌ API Error generating nomor surat:', data.message);
                        console.error('Full error data:', data);
                        nomorInput.value = '';
                    }
                } catch (error) {
                    console.error('Error generating nomor surat:', error);
                    nomorInput.value = '';
                }
            }

            function setFieldsRequired(fieldIds, required) {
                fieldIds.forEach(fieldId => {
                    const field = document.getElementById(fieldId);
                    if (field) {
                        if (required) {
                            field.setAttribute('required', '');
                        } else {
                            field.removeAttribute('required');
                        }
                    }
                });
            }
        })();

        (function() {
            const pribadiWrap = document.getElementById('wrapPribadi');
            const pribadiSelect = document.getElementById('dari_pribadi');
            const userNama = document.getElementById('user_nama')?.value || '';
            const userJabatan = document.getElementById('user_jabatan')?.value || '';
            const userId = document.getElementById('user_id')?.value || '';
            let currentUserAuthorized = false;
            const currentDariPribadi = document.getElementById('current_dari_pribadi')?.value || '';
            const dariTipeSendiri = document.getElementById('dariTipeSendiri');

            // Load authorized signers from API
            async function loadAuthorizedSigners() {
                try {
                    const response = await fetch(`api_classification.php?action=get_authorized_signers&current_user_id=${encodeURIComponent(userId)}`);
                    const data = await response.json();

                    if (data.success && data.data) {
                        // Clear existing options
                        pribadiSelect.innerHTML = '<option value="" disabled selected>-- Pilih Pejabat --</option>';

                        // Add authorized signers
                        data.data.forEach(signer => {
                            const option = document.createElement('option');
                            option.value = signer.user_id;
                            option.dataset.jabatan = signer.jabatan || '';
                            option.textContent = signer.nama + (signer.jabatan ? ' - ' + signer.jabatan : '');

                            // Select if matches current value
                            if (currentDariPribadi && String(signer.user_id) === String(currentDariPribadi)) {
                                option.selected = true;
                            }

                            pribadiSelect.appendChild(option);
                        });

                        // Check if current user is authorized signer
                        currentUserAuthorized = !!data.current_user_authorized;
                        checkDiriSendiriEligibility();
                    }
                } catch (error) {
                    console.error('Error loading authorized signers:', error);
                }
            }

            // Check if current user can use "Diri Sendiri" option
            function checkDiriSendiriEligibility() {
                if (!dariTipeSendiri || !userId) return;

                const userTipe = document.getElementById('user_tipe')?.value || '';
                if (userTipe === 'admin') return; // allow admin

                if (!currentUserAuthorized) {
                    dariTipeSendiri.disabled = true;
                    dariTipeSendiri.parentElement.classList.add('text-muted');
                    dariTipeSendiri.parentElement.title = 'Anda bukan penandatangan terdaftar';
                }
            }

            // Function to select current user in dropdown by matching nama
            function selectCurrentUser() {
                const options = pribadiSelect.options;
                for (let i = 0; i < options.length; i++) {
                    const optionText = options[i].textContent.trim();
                    if (optionText.startsWith(userNama)) {
                        pribadiSelect.selectedIndex = i;
                        return;
                    }
                }
            }

            function updateSignerFields() {
                // Update hidden nama_pejabat and jabatan_pejabat fields based on current selection
                const checkedRadio = document.querySelector('input[name="dari_tipe"]:checked');
                const namaField = document.getElementById('nama_pejabat');
                const jabatanField = document.getElementById('jabatan_pejabat');

                if (!namaField || !jabatanField) return;

                if (checkedRadio && checkedRadio.value === 'pribadi') {
                    // Get from selected option in dari_pribadi dropdown
                    const selectedOption = pribadiSelect.selectedOptions[0];
                    if (selectedOption) {
                        const optionText = selectedOption.textContent.trim();
                        const nama = optionText.split('-')[0].trim();
                        const jabatan = selectedOption.dataset.jabatan || '';
                        namaField.value = nama;
                        jabatanField.value = jabatan;
                        console.log('✅ Signer fields updated (pribadi):', nama, '-', jabatan);
                    }
                } else if (checkedRadio && checkedRadio.value === 'sendiri') {
                    // Get from user_nama and user_jabatan
                    const userNamaField = document.getElementById('user_nama');
                    const userJabatanField = document.getElementById('user_jabatan');
                    namaField.value = userNamaField?.value || '';
                    jabatanField.value = userJabatanField?.value || '';
                    console.log('✅ Signer fields updated (sendiri):', namaField.value, '-', jabatanField.value);
                } else {
                    // Clear fields
                    namaField.value = '';
                    jabatanField.value = '';
                }
            }

            function handleDariTipeChange() {
                const checkedRadio = document.querySelector('input[name="dari_tipe"]:checked');
                if (!checkedRadio) return;

                if (checkedRadio.value === 'pribadi') {
                    pribadiWrap.style.display = 'block';
                    pribadiSelect.disabled = false;
                } else if (checkedRadio.value === 'sendiri') {
                    pribadiWrap.style.display = 'block';
                    pribadiSelect.disabled = true;
                    selectCurrentUser();
                } else {
                    pribadiWrap.style.display = 'none';
                    pribadiSelect.disabled = false;
                }

                // Update signer fields
                updateSignerFields();
            }

            document.querySelectorAll('input[name="dari_tipe"]').forEach(r => {
                r.addEventListener('change', handleDariTipeChange);
            });

            // Also update signer fields when dari_pribadi dropdown changes
            if (pribadiSelect) {
                pribadiSelect.addEventListener('change', updateSignerFields);
            }

            // Initialize: load authorized signers then handle dari_tipe
            loadAuthorizedSigners().then(() => {
                handleDariTipeChange();
            });

            // Backdate functionality
            const backdateCheckbox = document.getElementById('backdate_checkbox');
            const backdateFields = document.getElementById('backdate_fields');
            const backdateSequence = document.getElementById('backdate_sequence');
            const backdateYear = document.getElementById('backdate_year');
            const backdateYearField = document.getElementById('backdate_year_field');
            const backdateHint = document.getElementById('backdate_hint');
            const nomorInput = document.getElementById('nomor_surat');
            const tipeSuratSelect = document.getElementById('tipe_surat_id');

            if (backdateCheckbox) {
                backdateCheckbox.addEventListener('change', handleBackdateToggle);
            }
            if (backdateSequence) {
                backdateSequence.addEventListener('input', handleBackdateInput);
            }
            if (backdateYear) {
                backdateYear.addEventListener('input', handleBackdateInput);
            }

            function handleBackdateToggle() {
                if (backdateCheckbox.checked) {
                    backdateFields.classList.remove('d-none');
                    nomorInput.removeAttribute('readonly');
                    updateBackdateFieldsVisibility();
                } else {
                    backdateFields.classList.add('d-none');
                    nomorInput.setAttribute('readonly', 'readonly');
                }
            }

            function updateBackdateFieldsVisibility() {
                const selectedOption = tipeSuratSelect?.selectedOptions[0];
                const tipeSuratKode = selectedOption?.dataset.kode;

                if (tipeSuratKode === 'SK') {
                    backdateYearField.style.display = 'none';
                    backdateHint.textContent = 'Masukkan nomor urut untuk backdate';
                } else {
                    backdateYearField.style.display = 'block';
                    backdateHint.textContent = 'Masukkan nomor urut dan tahun untuk backdate';
                }
            }

            function handleBackdateInput() {
                if (!backdateCheckbox.checked) return;

                const selectedOption = tipeSuratSelect?.selectedOptions[0];
                const tipeSuratKode = selectedOption?.dataset.kode;
                const sequence = backdateSequence.value;
                const year = backdateYear.value || new Date().getFullYear();

                if (!sequence) return;

                // For backdate, we rebuild the nomor based on type
                if (tipeSuratKode === 'SK') {
                    // SK format: [KodeJabatan].[Klasifikasi].[Jenis].[Subjenis].[Sequence]
                    const currentNomor = nomorInput.value;
                    if (currentNomor) {
                        const parts = currentNomor.split('.');
                        if (parts.length >= 5) {
                            parts[parts.length - 1] = sequence;
                            nomorInput.value = parts.join('.');
                        }
                    }
                } else if (tipeSuratKode === 'KPTS') {
                    // KPTS format: 001/KPTS-JPT/2025
                    const paddedSeq = String(sequence).padStart(3, '0');
                    nomorInput.value = `${paddedSeq}/KPTS-JPT/${year}`;
                } else if (tipeSuratKode === 'SE') {
                    // SE format: 001/SE-DIR/2025
                    const paddedSeq = String(sequence).padStart(3, '0');
                    nomorInput.value = `${paddedSeq}/SE-DIR/${year}`;
                }
            }

            // Update backdate fields visibility when tipe surat changes
            if (tipeSuratSelect) {
                tipeSuratSelect.addEventListener('change', () => {
                    if (backdateCheckbox && backdateCheckbox.checked) {
                        updateBackdateFieldsVisibility();
                    }
                });
            }

            const to2 = document.getElementById('toStep2');
            const back1 = document.getElementById('backTo1');
            const panes = document.querySelectorAll('.step-pane');
            const steps = document.querySelectorAll('.stepper .step');
            const form = document.getElementById('formNaskah');

            function setStep(n) {
                panes.forEach(p => p.classList.toggle('d-none', p.dataset.step !== String(n)));
                steps.forEach(s => {
                    const cur = Number(s.dataset.step);
                    s.classList.toggle('active', cur === n);
                    s.classList.toggle('completed', cur < n);
                });
                // Scroll to mainContent so the editor appears in view on Step 2
                const mainEl = document.getElementById('mainContent') || document.querySelector('.content-container');
                const targetTop = mainEl ? (mainEl.getBoundingClientRect().top + window.pageYOffset - 8) : 0;
                window.scrollTo({
                    top: targetTop,
                    behavior: 'smooth'
                });

                if (String(n) === '2') {
                    setTimeout(() => {
                        const editor = window.tinymce?.get('editorNaskah');
                        if (editor) {
                            try {
                                editor.focus();
                            } catch (e) {}
                            const ta = document.getElementById('editorNaskah');
                            if (ta) ta.scrollIntoView({
                                behavior: 'smooth',
                                block: 'start'
                            });
                        }
                    }, 250);
                }
            }
            back1.addEventListener('click', () => setStep(1));

            async function applyTemplateStyleToEditor(editorInstance) {
                if (!editorInstance) editorInstance = tinymce.get('editorNaskah');
                if (!editorInstance) return;
                const tplUrl = getTemplateUrl();
                try {
                    const doc = editorInstance.getDoc();
                    if (!doc) return;
                    // Ensure relative URLs inside templates resolve correctly by adding a base href
                    let baseEl = doc.getElementById('editorTemplateBase');
                    if (!baseEl) {
                        baseEl = doc.createElement('base');
                        baseEl.id = 'editorTemplateBase';
                        baseEl.href = '/jasa_marga/';
                        doc.head.appendChild(baseEl);
                    } else {
                        baseEl.href = '/jasa_marga/';
                    }

                    // ALWAYS fetch fresh CSS from template for consistency with create-letter
                    const cssText = await fetchTemplateStyles();
                    console.log('✅ Using fresh template CSS (consistent with create-letter)');

                    let styleEl = doc.getElementById('editorTemplateStyle');
                    if (!styleEl) {
                        styleEl = doc.createElement('style');
                        styleEl.id = 'editorTemplateStyle';
                        doc.head.appendChild(styleEl);
                    }
                    // Apply CSS + page defaults for consistent display
                    styleEl.textContent = cssText + '\n.page{ width:210mm !important; min-height:297mm !important; padding:12mm 14mm 42mm 12mm !important; margin:0 auto !important; background:#fff !important; }';
                    console.log('✅ Applied styles to editor');
                } catch (err) {
                    console.error('applyTemplateStyleToEditor error', err);
                }
            }

            async function fetchTemplate() {
                const res = await fetch(getTemplateUrl(), {
                    cache: 'no-store'
                });
                const txt = await res.text();
                const bodyOnly = txt.replace(/^[\s\S]*?<body[^>]*>/i, '').replace(/<\/body>[\s\S]*$/i, '').trim();
                return bodyOnly;
            }

            function fillPlaceholders(html, data) {
                let out = html;

                for (const [key, val] of Object.entries(data)) {
                    if (val === null) continue;
                    const re = new RegExp(`\\[\\[${key}\\]\\]`, 'g');
                    out = out.replace(re, val);
                }

                // Keep signature img element but set src to placeholder for later replacement
                if (!data.PATH_TANDA_TANGAN) {
                    out = out.replace(
                        /(<img[^>]*class=["'][^"']*signature[^"']*["'][^>]*src=")([^"]*)(")/gi,
                        '$1[[PATH_TANDA_TANGAN]]$3'
                    );
                }
                if (!data.PATH_STEMPEL) {
                    out = out.replace(/<img[^>]*class=["']stamp["'][^>]*>/gi, '');
                }
                // Prevent editor trying to fetch placeholder QR_CODE as a resource
                out = out.replace(/src="\[\[QR_CODE\]\]"/g, 'src=""');

                return out;
            }

            let editorReady = false;

            function initTinyMCE() {
                if (editorReady && tinymce.get('editorNaskah')) return Promise.resolve();
                return new Promise((resolve) => {
                    tinymce.init({
                        selector: '#editorNaskah',
                        height: 780,
                        menubar: true,
                        branding: false,
                        convert_urls: false,
                        remove_script_host: false,
                        entity_encoding: 'raw',
                        automatic_uploads: true,
                        images_reuse_filename: false,
                        paste_data_images: false,
                        file_picker_types: 'image',
                        images_upload_url: '/jasa_marga/api/upload/image.php',
                        images_upload_credentials: true,
                        images_upload_handler: (blobInfo, progress) => {
                            const fd = new FormData();
                            fd.append('image', blobInfo.blob(), blobInfo.filename());

                            return fetch('/jasa_marga/api/upload/image.php', {
                                    method: 'POST',
                                    credentials: 'same-origin',
                                    body: fd
                                })
                                .then(r => {
                                    if (!r.ok) throw new Error('HTTP ' + r.status);
                                    return r.json();
                                })
                                .then(json => {
                                    if (json && json.location) {
                                        return json.location;
                                    }
                                    throw new Error(json?.error || 'Upload gagal');
                                });
                        },
                        protect: [/\[\[[\s\S]*?\]\]/g],
                        extended_valid_elements: '*[*]',
                        valid_children: '+body[style],+div[div|section|header|footer|img]',
                        verify_html: false,
                        plugins: 'lists table link image code preview autoresize',
                        toolbar: 'undo redo | styles | bold italic underline | alignleft aligncenter alignright alignjustify | bullist numlist outdent indent | table link image | removeformat | preview | code',
                        init_instance_callback: function(editor) {
                            editorReady = true;
                            applyTemplateStyleToEditor(editor).catch(e => console.error('applyTemplateStyleToEditor:', e));
                            try {
                                resolve();
                            } catch (e) {
                                /* ignore */
                            }
                        },

                        content_style: `
                        * { box-sizing: border-box; }
                        html, body { margin:0; padding:0; }
                        body {
                            font-family: Arial, Helvetica, sans-serif;
                            color: #111;
                            background: #fff;
                        }
                        @page{
                            size: A4;
                            margin: 12mm 14mm 28mm 12mm;
                        }
                        :root{
                            --page-w:210mm;
                            --page-h:297mm;
                            --pad-top:12mm;
                            --pad-right:14mm;
                            --pad-bottom:42mm;
                            --pad-left:12mm;
                        }
                        .page{
                            width: var(--page-w);
                            min-height: var(--page-h);
                            padding: var(--pad-top) var(--pad-right) var(--pad-bottom) var(--pad-left);
                            margin: 0 auto;
                            position: relative;
                            background: #fff;
                        }
                        .html2pdf__page-break{
                            height:0; border:0; page-break-after:always;
                        }
                        .header {
                            display:flex; align-items:flex-start; justify-content:space-between;
                            gap: 16px; margin-bottom: 10mm;
                        }
                        .brand { display:flex; align-items:center; gap:12px; }
                        .brand img.logo { height:70px; width:auto; object-fit:contain; }
                        /* Header-left / Header-right placeholder to match template-surat-edaran */
                        .header-left { flex: 1; }
                        .header-left img.logo { height: 90px; width: 360px; object-fit: contain; display: block; }
                        .header-right { flex: 1; text-align: right; font-size:13px; color:#333; line-height:1.4; }
                        .brand .title { line-height:1.1; font-weight:700; letter-spacing:.3px; }
                        .brand .title .main { font-size:22px; color:#0c50e1; }
                        .brand .title .sub { font-size:18px; color:#0c50e1; text-transform:uppercase; }
                        .header .topnote { text-align:right; font-size:13px; color:#333; max-width:300px; margin-top: 20px; }
                        .doc-title {
                            text-align:center; font-weight:600; font-size:18px;
                        }
                        .doc-number {
                            text-align:center; font-size:18px; font-weight:600;
                        }
                        .doc-subject {
                            text-align:center; font-size:18px; font-weight:600;
                        }
                        .divider { height:1px; background:#e6e9ef; margin: 2mm 0 8mm; }
                        .section { margin-bottom: 6mm; font-size: 14px !important; }
                        .lead { margin-bottom: 4mm; }
                        table.kv { border-collapse: collapse; width:100%; font-size: 14px; }
                        table.kv td { padding: 2px 0; vertical-align: top; }
                        table.kv td.col1 { width:120px; } table.kv td.col2 { width:10px; } table.kv td.col3 { width:auto; }
                        span.highlight { font-weight: 700 !important; }
                        .sign-block {
                            margin-top: 14mm;
                            width: 100%;
                            display:flex; justify-content:flex-end;
                            position: relative;
                        }
                        .sign {
                            width: 72mm;
                            text-align:left; font-size:14px;
                            position: relative; padding-top: 0;
                        }
                        .sign .place-date { margin-bottom: 15mm; }
                        .sign .stamp { position:absolute; right: -8mm; top: 18mm; opacity:.85; width: 34mm; height:auto; }
                        .sign .signature { height: 22mm; width:auto; display:block; margin: -6mm 0 0 0; }
                        .sign .signature-info { display: block; margin-top: 20mm; }
                        .sign .name { font-weight:700; margin: 0; }
                        .sign .sub-name {
                        font-weight: 700 !important;
                    }
                        .sign .title { margin-top: 1mm; }
                        .sign-block .paraf {
                            position: absolute;
                            left: 0;
                            bottom: 0;
                            font-size: 14px;
                            color: #333;
                            font-weight: 500;
                        }
                        .footer {
                            position: absolute;
                            right: 14mm;
                            bottom: 12mm;
                            left: auto;
                            width: auto;
                            text-align: right;
                            font-size: 10px;
                            color: #333;
                            line-height: 1.4;
                        }
                        .page .footer{ display:none !important; }
                        .page:first-of-type .footer{ display:block !important; }
                        .sign-block, .footer, .doc-title, .doc-number, table.kv {
                        page-break-inside: avoid;
                        break-inside: avoid;
                        }
                        .page:not(:last-child){ page-break-after: always; }
                        
                        /* Two-column section for Surat Keputusan */
                        .section-title-2 {
                            font-weight: bold;
                            font-size: 16px;
                            margin-bottom: 12px;
                            text-align: center;
                        }
                        .two-column-section {
                            display: flex;
                        }
                        .two-column-section .column {
                            flex: 1;
                        }
                        .two-column-section .column-2 {
                            flex: 8;
                        }
                        .two-column-section .column-2 p {
                            margin: 0;
                        }
                        .page-break {
                            page-break-before: always;
                            break-before: page;
                        }
                        .signature-section {
                            margin-top: 20mm;
                        }
                        .signature-info-table {
                            margin-left: auto;
                            margin-right: 0;
                            width: auto;
                            font-size: 14px;
                        }
                        .signature-info-table td {
                            padding: 2px 0;
                            vertical-align: top;
                        }
                        .signature-info-table .label {
                            font-weight: 600;
                        }
                        .signature-info-table .colon {
                            font-weight: 600;
                        }
                        .company-title {
                            text-align: right;
                            font-weight: 600;
                            font-size: 14px;
                            margin-bottom: 24px;
                        }
                        .signature-table {
                            margin-left: auto;
                            margin-right: 32px;
                            text-align: left;
                            font-size: 14px;
                            margin-bottom: 3mm;
                        }
                        .signature-table td {
                            font-weight: 600;
                            padding: 0;
                        }
                        .signature-section-container {
                            position: relative;
                            text-align: right;
                        }
                        .signature-paraf {
                            position: absolute;
                            left: 0;
                            top: 48mm;
                            font-size: 16px;
                            color: #333;
                        }
                        .signature-qr-container {
                            margin-right: 80px;
                        }
                        .signature-img {
                            height: 22mm;
                            width: auto;
                            display: block;
                            margin: -4mm 0 0 auto;
                        }
                        .signature-name {
                            font-weight: 600;
                            margin-top: 5mm;
                            margin-right: 92px;
                        }
                        .signature-sub-name {
                            margin-right: 32px;
                            font-weight: 700;
                        }
                        .sign-block-keputusan {
                            display: flex;
                            justify-content: center;
                            position: relative;
                        }
                        .sign-paraf-keputusan {
                            position: absolute;
                            left: 0;
                            top: 60mm;
                            font-size: 12px;
                            color: #333;
                        }
                        .numbered-list {
                            padding-left: 4px;
                            list-style: none;
                            margin: 0;
                        }
                        .numbered-list li {
                            margin-bottom: 5px;
                            margin-top: 0px;
                            position: relative;
                        }

                        .lampiran-box{
                            border:1px solid #e0e0e0;
                            padding:0;
                            margin: 0 auto;
                        }
                        .lampiran-slot{
                            width: calc(var(--page-w) - var(--pad-left) - var(--pad-right));
                            height: calc(var(--page-h) - var(--pad-top) - var(--pad-bottom) - 26mm);
                            margin: 0 auto;
                            display:flex; align-items:center; justify-content:center;
                            overflow:hidden;
                            page-break-inside: avoid;
                            break-inside: avoid;
                        }
                        .lampiran-slot > img.lampiran-img{
                            max-width: 100%;
                            max-height: 100%;
                            width: auto; height: auto;
                            object-fit: contain;
                            display: block; margin: 0;
                        }
                        .lampiran-slot > iframe.lampiran-frame{
                            width: 100%; height: 100%; border: 0;
                        }
                        img.signature[src=""], img.stamp[src=""] { display:none !important; }
                        table, th, td { border: none !important; outline: none !important; }
                        .mce-item-table, .mce-item-table td, .mce-item-table th { border: none !important; outline: none !important; }
                        
                        /* Nomor Surat editable parts styling */
                        .nomor-surat-wrapper {
                            display: inline;
                        }
                        .editable-sequence, .editable-year {
                            background-color: #fff3cd;
                            border: 1px dashed #ffc107;
                            border-radius: 3px;
                            padding: 0 4px;
                            min-width: 30px;
                            display: inline-block;
                            cursor: text;
                        }
                        .editable-sequence:focus, .editable-year:focus {
                            background-color: #fff8e1;
                            border-color: #ff9800;
                            outline: none;
                        }
                        .editable-sequence:hover, .editable-year:hover {
                            background-color: #ffecb3;
                        }
                        
                        @media print {
                            @page{ size:A4; margin:12mm 20mm 28mm; }
                            .page{ -webkit-print-color-adjust: exact; print-color-adjust: exact; }
                            .editable-sequence, .editable-year {
                                background-color: transparent;
                                border: none;
                                padding: 0;
                            }
                        }
                    `,
                        setup(ed) {
                            ed.on('init', () => {
                                editorReady = true;

                                // Add listeners for editable nomor surat parts
                                const doc = ed.getDoc();
                                if (doc) {
                                    // Sync nomor surat when editable parts are changed
                                    const syncNomorSuratFromEditor = () => {
                                        const wrapper = doc.querySelector('.nomor-surat-wrapper');
                                        if (wrapper) {
                                            const tipeSurat = wrapper.dataset.tipe;
                                            const seqEl = wrapper.querySelector('.editable-sequence');
                                            const yearEl = wrapper.querySelector('.editable-year');

                                            const sequence = seqEl ? seqEl.textContent.trim() : '';
                                            const year = yearEl ? yearEl.textContent.trim() : '';

                                            let newNomorSurat = '';

                                            if (tipeSurat === 'SK') {
                                                // SK format: rebuild from prefix + sequence
                                                const nomorInput = document.getElementById('nomor_surat');
                                                const currentNomor = nomorInput ? nomorInput.value : '';
                                                const parts = currentNomor.split('.');
                                                if (parts.length >= 2) {
                                                    parts[parts.length - 1] = sequence;
                                                    newNomorSurat = parts.join('.');
                                                } else {
                                                    newNomorSurat = sequence;
                                                }
                                            } else if (tipeSurat === 'KPTS') {
                                                newNomorSurat = sequence + '/KPTS-JPT/' + year;
                                            } else if (tipeSurat === 'SE') {
                                                newNomorSurat = sequence + '/SE-DIR/' + year;
                                            }

                                            if (newNomorSurat) {
                                                const nomorInput = document.getElementById('nomor_surat');
                                                if (nomorInput) {
                                                    nomorInput.value = newNomorSurat;
                                                    console.log('📝 Nomor surat updated from editor:', newNomorSurat);
                                                }
                                            }
                                        }
                                    };

                                    // Listen for input changes on editable elements
                                    doc.addEventListener('input', (e) => {
                                        if (e.target.classList.contains('editable-sequence') ||
                                            e.target.classList.contains('editable-year')) {
                                            syncNomorSuratFromEditor();
                                        }
                                    });

                                    // Also sync on blur
                                    doc.addEventListener('blur', (e) => {
                                        if (e.target.classList.contains('editable-sequence') ||
                                            e.target.classList.contains('editable-year')) {
                                            syncNomorSuratFromEditor();
                                        }
                                    }, true);
                                }

                                resolve();
                            });
                        }
                    });
                });
            }

            async function buildEditorFromTemplate() {
                if (!form.checkValidity()) {
                    form.reportValidity();
                    return;
                }

                // Update signer fields before building editor
                updateSignerFields();

                await initTinyMCE();
                const editor = tinymce.get('editorNaskah');

                // Check if we have saved content from database
                if (NASKAH_BODY_ONLY && NASKAH_BODY_ONLY.trim().length > 0) {
                    console.log('📄 Loading saved naskah from database');
                    editor.setContent(NASKAH_BODY_ONLY);
                } else {
                    // Generate from template if no saved content
                    console.log('📄 Generating from template');
                    const tpl = await fetchTemplate();
                    const vals = collectFormValues();
                    const html = fillPlaceholders(tpl, vals);
                    editor.setContent(html);
                }

                await applyTemplateStyleToEditor(editor);
                setStep(2);
            }

            to2.addEventListener('click', buildEditorFromTemplate);

            const syncBtn = document.getElementById('syncFromForm');
            if (syncBtn) {
                syncBtn.addEventListener('click', async () => {
                    const tpl = await fetchTemplate();
                    const vals = collectFormValues();
                    const html = fillPlaceholders(tpl, vals);
                    if (editorReady) {
                        const editor = tinymce.get('editorNaskah');
                        editor.setContent(html);
                        await applyTemplateStyleToEditor(editor);
                    }
                });
            }

            const printBtn = document.getElementById('printDoc');
            if (printBtn) {
                printBtn.addEventListener('click', () => {
                    if (editorReady) {
                        const ed = tinymce.get('editorNaskah');
                        ed.execCommand('mcePrint');
                    }
                });
            }
        })();
    </script>
    <script>
        (function() {
            const form = document.getElementById('formNaskah');
            form.addEventListener('submit', function(e) {
                if (window.tinymce) tinymce.triggerSave();
                const html = (document.getElementById('editorNaskah').value || '');
                const m = html.match(/<img[^>]+src=["']([^"']+)["'][^>]*>/gi) || [];
                const hasBase64 = /src\s*=\s*["']data:image\//i.test(html);
                console.log('BASE64_DETECTED:', hasBase64, 'IMG_TAGS_COUNT:', m.length);
            });
        })();
    </script>

    <!-- Generate PDF → Upload -->
    <script>
        // Ensure listener exists early and will enable the button immediately if html2pdf is already present
        document.addEventListener('html2pdf:loaded', function() {
            try {
                window._html2pdf_loaded = true;
                const b = document.getElementById('btnUploadPdf');
                if (b) b.disabled = false;
            } catch (e) {
                console.error(e);
            }
        });
    </script>
    <script>
        (function() {
            function loadScript(src, onload, onerror) {
                var s = document.createElement('script');
                s.src = src;
                s.async = false;
                s.onload = onload;
                s.onerror = onerror;
                document.head.appendChild(s);
            }
            // Try local first, then CDN
            loadScript('/jasa_marga/node_modules/html2pdf.js/dist/html2pdf.bundle.min.js', function() {
                console.log('html2pdf loaded (local)');
                document.dispatchEvent(new Event('html2pdf:loaded'));
            }, function() {
                console.warn('local html2pdf not found, loading CDN');
                loadScript('https://cdnjs.cloudflare.com/ajax/libs/html2pdf.js/0.10.1/html2pdf.bundle.min.js', function() {
                    console.log('html2pdf loaded (CDN)');
                    document.dispatchEvent(new Event('html2pdf:loaded'));
                }, function() {
                    console.error('Failed to load html2pdf from both local and CDN');
                });
            });
        })();
    </script>
    <script>
        function collectStep1Values() {
            const get = id => (document.getElementById(id)?.value || '').trim();

            const jenisOpt = document.getElementById('jenis_surat_id')?.selectedOptions?.[0];
            const klasOpt = document.getElementById('klasifikasi_arsip_id')?.selectedOptions?.[0];

            const dariTipe = (document.querySelector('input[name="dari_tipe"]:checked')?.value || '');
            const namaLogin = (document.getElementById('user_nama')?.value || '').trim();
            const jabatanLogin = (document.getElementById('user_jabatan')?.value || '').trim();

            let namaPejabat = '';
            let jabatanPejabat = '';

            if (dariTipe === 'pribadi') {
                const selectedOption = document.getElementById('dari_pribadi')?.selectedOptions?.[0];
                if (selectedOption) {
                    // Extract nama from "Nama - Jabatan" format
                    const optionText = selectedOption.textContent.trim();
                    namaPejabat = optionText.split('-')[0].trim();
                    jabatanPejabat = selectedOption?.dataset?.jabatan || jabatanLogin || '';
                } else {
                    namaPejabat = namaLogin;
                    jabatanPejabat = jabatanLogin;
                }
            } else if (dariTipe === 'sendiri') {
                namaPejabat = namaLogin;
                jabatanPejabat = jabatanLogin;
            } else {
                namaPejabat = get('nama_pejabat') || '';
                jabatanPejabat = get('jabatan_pejabat') || '';
            }

            const tglRaw = get('tanggal_surat');
            const tglID = (() => {
                if (!tglRaw) return '';
                const m = /^(\d{4})-(\d{2})-(\d{2})$/.exec(tglRaw);
                return m ? `${m[3]}-${m[2]}-${m[1]}` : tglRaw;
            })();

            const fallbackPerusahaan = 'PT Jasamarga Pandaan Tol';

            const parafValue = get('paraf') || '';
            console.log('🔍 collectStep1Values - Reading PARAF:');
            console.log('   - Element exists:', !!document.getElementById('paraf'));
            console.log('   - Raw value:', document.getElementById('paraf')?.value);
            console.log('   - After get():', get('paraf'));
            console.log('   - Final parafValue:', parafValue);

            return {
                JENIS_SURAT: (jenisOpt ? jenisOpt.textContent.trim() : ''),
                NOMOR_SURAT: get('nomor_surat'),
                KOTA: get('tempat_dibuat'),
                TANGGAL_SURAT: tglID,

                NAMA_PEJABAT: namaPejabat,
                NIK_PEJABAT: get('nik_pejabat'),
                JABATAN_PEJABAT: jabatanPejabat,
                DEPARTEMEN: get('departemen'),
                NAMA_PERUSAHAAN: get('nama_perusahaan') || fallbackPerusahaan,

                NAMA_MAHASISWA: get('kepada'),
                NIM: get('nim'),
                JURUSAN: get('jurusan'),
                PROGRAM_STUDI: get('program_studi'),
                UNIVERSITAS: get('universitas'),

                TANGGAL_MULAI: (get('tanggal_mulai') || '[[TANGGAL_MULAI]]'),
                TANGGAL_SELESAI: (get('tanggal_selesai') || '[[TANGGAL_SELESAI]]'),

                PATH_TANDA_TANGAN: (get('path_ttd') || SIGNATURE_PATH || ''),
                QR_CODE: (get('qr_code') || ''),
                PARAF: parafValue
            };
        }

        function replacePlaceholders(html, data) {
            let out = html;
            console.log('🔍 replacePlaceholders called');
            console.log('   - Input HTML length:', html.length);
            console.log('   - Data keys:', Object.keys(data).join(', '));
            console.log('   - PARAF value:', data.PARAF);
            console.log('   - PATH_TANDA_TANGAN value:', data.PATH_TANDA_TANGAN);
            console.log('   - QR_CODE value:', data.QR_CODE);

            for (const [k, v] of Object.entries(data)) {
                const re = new RegExp(`\\[\\[${k}\\]\\]`, 'g');
                const matches = html.match(re);
                if (matches) {
                    console.log(`   ✅ Found ${matches.length} occurrences of [[${k}]]`);
                    out = out.replace(re, String(v ?? ''));
                    console.log(`   ✅ Replaced [[${k}]] with: ${String(v ?? '').substring(0, 50)}`);
                } else {
                    console.log(`   ℹ️  No [[${k}]] found in HTML`);
                }
            }

            // Remove any remaining unreplaced placeholders
            const unreplaced = out.match(/\[\[[A-Z0-9_]+\]\]/g);
            if (unreplaced) {
                console.log('⚠️ Unreplaced placeholders found:', unreplaced);
                out = out.replace(/\[\[[A-Z0-9_]+\]\]/g, '');
            }

            // Keep signature img element but set src to placeholder for later replacement
            if (!data.PATH_TANDA_TANGAN) {
                const beforeImg = out.length;
                out = out.replace(
                    /(<img[^>]*class=["'][^"']*signature[^"']*["'][^>]*src=")([^"]*)(")/gi,
                    '$1[[PATH_TANDA_TANGAN]]$3'
                );
                if (out.length !== beforeImg) {
                    console.log('   ✅ Reset signature img src to placeholder (PATH_TANDA_TANGAN was empty)');
                }
            }

            console.log('   - Output HTML length:', out.length);
            console.log('🔍 replacePlaceholders completed');

            console.log('🔍 Final HTML length:', out.length);
            return out;
        }

        /**
         * Consolidated helper: Replace PARAF placeholders with actual/fallback value
         * Handles both [[PARAF]] placeholder replacement and approval-mode PARAF injection
         * @param {string} content - The HTML content to process
         * @param {string} parafValue - The PARAF value to inject (if empty, uses fallback)
         * @param {boolean} isApprovalMode - Whether in approval mode (affects replacement strategy)
         * @returns {object} Result object with {content, parafUsed, hadFallback}
         */
        function replaceParafInContent(content, parafValue, isApprovalMode = false) {
            let parafUsed = parafValue ? parafValue.trim() : '';
            let hadFallback = false;
            let result = content;
            let replacementCount = 0;

            // Check if PARAF should be shown (penandatangan != dibuat_oleh)
            if (!SHOULD_SHOW_PARAF) {
                console.log('📝 PARAF hidden: penandatangan sama dengan dibuat_oleh');
                // Remove PARAF placeholder and Paraf: text completely
                result = result.replace(/\[\[PARAF\]\]/g, '');
                result = result.replace(/\|\s*Paraf\s*:\s*(?:\[\[PARAF\]\])?/gi, '');
                result = result.replace(/<span[^>]*>Paraf:\s*<\/span>/gi, '');
                result = result.replace(/<div[^>]*class=["'][^"']*signature-paraf[^"']*["'][^>]*>.*?<\/div>/gis, '');
                return {
                    content: result,
                    parafUsed: '',
                    hadFallback: false,
                    replacementCount: 0,
                    parafHidden: true
                };
            }

            if (!parafUsed) {
                parafUsed = '';
                hadFallback = true;
            }

            if (isApprovalMode) {
                // Approval mode: inject PARAF in specific locations
                console.log('📝 Approval Mode - Replacing PARAF:');
                console.log('   PARAF value (raw):', parafValue ? parafValue.trim() : '[empty]');
                console.log('   PARAF used:', parafUsed);
                console.log('   Content length before replacements:', content.length);

                // Debug: Show current Paraf content
                const parafMatch = result.match(/Paraf\s*:[^<]*/i);
                console.log('   Current Paraf content:', parafMatch ? parafMatch[0] : 'not found');
                console.log('   Has [[PARAF]] before:', result.includes('[[PARAF]]'));
                console.log('   Has signature-paraf div before:', result.includes('signature-paraf'));

                // CRITICAL FIX: First, replace [[PARAF]] placeholder (most reliable method)
                // This ensures nested placeholders like <span>[[PARAF]]</span> get replaced
                const beforePlaceholder = result.length;
                result = result.replace(/\[\[PARAF\]\]/g, parafUsed);
                if (result.length !== beforePlaceholder) {
                    replacementCount++;
                    console.log('   ✅ Replaced [[PARAF]] placeholder');
                    console.log('   Has [[PARAF]] after:', result.includes('[[PARAF]]'));
                }

                // Replace "| Paraf:" with "| Paraf: VALUE"
                const beforeLen = result.length;
                result = result.replace(/(\|\s*Paraf\s*:)(\s*)([^<]*)?(<)/gi, `$1 ${parafUsed}$4`);
                if (result.length !== beforeLen) {
                    replacementCount++;
                    console.log('   ✅ Replaced PARAF in NOMOR section');
                }

                // Also try replacing standalone "Paraf:" (without |)
                const beforeLen2 = result.length;
                result = result.replace(/(>Paraf\s*:)(\s*)(<)/gi, `$1 ${parafUsed}$3`);
                if (result.length !== beforeLen2) {
                    replacementCount++;
                    console.log('   ✅ Replaced standalone Paraf label');
                }

                // Replace signature-paraf div content - handle ANY existing content
                const beforeLen3 = result.length;
                const parafDivBefore = result.match(/<div[^>]*class=["'][^"']*signature-paraf[^"']*["'][^>]*>[\s\S]*?<\/div>/i)?.[0];
                console.log('   Signature-paraf div BEFORE:', parafDivBefore ? parafDivBefore.substring(0, 100) : 'NOT FOUND');

                result = result.replace(
                    /(<div[^>]*class=["'][^"']*signature-paraf[^"']*["'][^>]*>)[\s\S]*?(<\/div>)/gi,
                    `$1${parafUsed}$2`
                );

                const parafDivAfter = result.match(/<div[^>]*class=["'][^"']*signature-paraf[^"']*["'][^>]*>[\s\S]*?<\/div>/i)?.[0];
                console.log('   Signature-paraf div AFTER:', parafDivAfter ? parafDivAfter.substring(0, 100) : 'NOT FOUND');

                if (result.length !== beforeLen3) {
                    replacementCount++;
                    console.log('   ✅ Replaced signature-paraf div content');
                }

                // Debug: Verify replacement worked
                const afterMatch = result.match(/Paraf\s*:[^<]*/i);
                console.log('   After replacement Paraf content:', afterMatch ? afterMatch[0] : 'not found');
                console.log('   Content length after replacements:', result.length);

                if (hadFallback) {
                    console.log('   ⚠️ Note: PARAF was empty and fallback was applied.');
                }

                console.log('✅ PARAF replacement complete (', replacementCount, 'replacements)');
            } else {
                // Normal mode: replace [[PARAF]] placeholder
                console.log('📝 Reading PARAF directly from #paraf input:', parafUsed);
                result = result.replace(/\[\[PARAF\]\]/g, parafUsed);
                console.log('✅ Replaced [[PARAF]] with:', parafUsed);
                replacementCount = (content.match(/\[\[PARAF\]\]/g) || []).length;
            }

            return {
                content: result,
                parafUsed,
                hadFallback,
                replacementCount
            };
        }

        window.addEventListener('load', function() {
            console.log('✅ Halaman sudah dimuat, script siap.');

            const uploadBtn = document.getElementById('btnUploadPdf');
            const saveBtn = document.getElementById('btnSave');
            const fileInput = document.getElementById('fileSuratPath');
            const tipeSuratSelect = document.getElementById('tipe_surat_id');

            if (!uploadBtn) {
                console.error('❌ Tombol Cetak & Upload PDF tidak ditemukan setelah load!');
                return;
            }
            if (!fileInput) {
                console.error('❌ Elemen fileSuratPath tidak ditemukan!');
                return;
            }

            // Enable upload only if html2pdf already loaded, otherwise wait for event
            uploadBtn.disabled = !window._html2pdf_loaded;
            document.addEventListener('html2pdf:loaded', function() {
                try {
                    uploadBtn.disabled = false;
                } catch (e) {}
            });

            console.log('✅ Semua elemen ditemukan, mengatur event listener...');

            // saveBtn only exists in normal edit mode, not in approval mode
            if (saveBtn) {
                saveBtn.addEventListener('click', async function(e) {
                    // Prevent default form submission - we'll submit manually after sync
                    e.preventDefault();

                    // Extract values from TinyMCE editor back to form fields
                    // This ensures changes made directly in the editor are saved to the database
                    extractValuesFromEditor();

                    // Sync TinyMCE content to textarea before form submission
                    if (window.tinymce && tinymce.get('editorNaskah')) {
                        const editor = tinymce.get('editorNaskah');
                        const editorContent = editor.getContent() || '';

                        // ALWAYS fetch fresh CSS from template for consistency with newly created letters
                        let cssToUse = '';
                        try {
                            cssToUse = await fetchTemplateStyles();
                            console.log('✅ Fetched fresh CSS for save (consistent with create-letter)');
                            console.log('📊 Raw CSS length:', cssToUse.length);
                            console.log('📊 Template URL:', getTemplateUrl());
                        } catch (err) {
                            console.error('Failed to fetch CSS:', err);
                        }

                        // Clean CSS for mPDF compatibility
                        const cleanedCss = cssToUse
                            .replace(/:root\s*\{[\s\S]*?\}/g, '')
                            .replace(/@media[\s\S]*?\{[\s\S]*?\}/g, '')
                            .replace(/@page\s*\{[\s\S]*?\}/g, '')
                            .replace(/var\(--page-w\)/g, '210mm')
                            .replace(/var\(--page-h\)/g, '297mm')
                            .replace(/var\(--pad-top\)/g, '12mm')
                            .replace(/var\(--pad-right\)/g, '14mm')
                            .replace(/var\(--pad-bottom\)/g, '42mm')
                            .replace(/var\(--pad-left\)/g, '12mm')
                            .replace(/var\(--content-h\)/g, '243mm')
                            .replace(/var\(--content-w\)/g, '182mm')
                            .replace(/var\(--slot-gap\)/g, '6mm')
                            .replace(/var\(--slot-h\)/g, '237mm')
                            .replace(/calc\s*\(\s*297mm\s*-\s*12mm\s*-\s*42mm\s*\)/gi, '243mm')
                            .replace(/calc\s*\(\s*210mm\s*-\s*(?:var\(--pad-left\)|12mm)\s*-\s*(?:var\(--pad-right\)|14mm)\s*\)/gi, '182mm')
                            .replace(/calc\s*\(\s*(?:var\(--content-h\)|243mm)\s*-\s*(?:var\(--slot-gap\)|6mm)\s*\)/gi, '237mm')
                            .replace(/calc\s*\([^)]*\)/gi, 'auto')
                            .replace(/var\s*\([^)]*\)/gi, 'auto')
                            .replace(/\.page\s*:\s*not\(\s*:last-child\s*\)\s*\{\s*page-break-after\s*:\s*always;?\s*\}/gi, '')
                            .replace(/\.page\s*\{[^}]*\}/gi, m => m.replace(/page-break-(after|before)\s*:\s*always;?/gi, ''));

                        // Combine CSS with content
                        const finalNaskah = cleanedCss.trim() ?
                            '<style>' + cleanedCss + '</style>' + editorContent :
                            editorContent;

                        document.getElementById('editorNaskah').value = finalNaskah;
                        console.log('✅ TinyMCE content WITH CSS synced to textarea before save');
                        console.log('📄 Final naskah length:', finalNaskah.length, 'chars');
                        console.log('📊 Cleaned CSS length:', cleanedCss.length, 'chars');
                        console.log('📊 CSS contains padding:', cleanedCss.includes('padding') ? 'YES' : 'NO');
                    }

                    // Check if PDF has been uploaded
                    if (!fileInput.value.trim()) {
                        alert('⚠️ Harap klik "Cetak & Upload PDF" terlebih dahulu sebelum memperbarui!');
                        return; // Don't submit form
                    }

                    // Submit the form
                    document.getElementById('formNaskah').submit();
                });
            }

            if (tipeSuratSelect) {
                tipeSuratSelect.addEventListener('change', async () => {
                    if (editorReady && window.tinymce) {
                        await applyTemplateStyleToEditor(tinymce.get('editorNaskah'));
                    }
                });
            }

            uploadBtn.addEventListener('click', async function(e) {
                e.preventDefault();
                console.log('🔘 Tombol Cetak & Upload PDF diklik!');
                try {
                    if (!window.html2pdf) {
                        console.error('❌ html2pdf tidak ditemukan!');
                        alert('html2pdf.js belum tersedia. Sistem sedang mencoba memuat library dari CDN. Silakan coba lagi sebentar.');
                        return;
                    }
                    if (!window.tinymce || !tinymce.get('editorNaskah')) {
                        console.error('❌ TinyMCE tidak siap!');
                        alert('Editor TinyMCE belum siap.');
                        return;
                    }

                    console.log('✅ Semua dependency siap, mulai generate PDF...');

                    let content = tinymce.get('editorNaskah').getContent() || '';
                    if (!content.trim()) {
                        alert('Konten surat masih kosong!');
                        return;
                    }

                    // Determine approval mode and get PARAF value
                    const isApprovalMode = <?= json_encode($approvalMode) ?>;
                    const parafFromInput = (document.getElementById('paraf')?.value || '').trim();

                    // Check if PARAF is empty and ask user to confirm
                    if (!parafFromInput) {
                        const proceed = await Swal.fire({
                            title: 'Paraf kosong',
                            text: 'Nilai PARAF kosong — lanjutkan membuat PDF tanpa Paraf? (Rekomendasi: lengkapi paraf terlebih dahulu.)',
                            icon: 'warning',
                            showCancelButton: true,
                            confirmButtonText: 'Lanjutkan',
                            cancelButtonText: 'Batal'
                        });
                        if (!proceed.isConfirmed) {
                            console.log('📛 Pengguna membatalkan pembuatan PDF karena PARAF kosong');
                            return;
                        }
                    }

                    // Use consolidated helper to replace PARAF
                    const parafResult = replaceParafInContent(content, parafFromInput, isApprovalMode);
                    content = parafResult.content;

                    // CRITICAL FIX: Inject signature section if missing (same as approval mode)
                    const hasSignatureSection = content.includes('signature-paraf') ||
                        content.includes('sign-section') ||
                        content.includes('sign-block');

                    if (!hasSignatureSection) {
                        console.log('⚠️ Signature section NOT FOUND - Injecting it now...');

                        // Get signature data from form
                        const namaPejabat = document.getElementById('nama_pejabat')?.value || '';
                        const jabatanPejabat = document.getElementById('jabatan_pejabat')?.value || '';
                        let signaturePath = document.getElementById('path_ttd')?.value || '';

                        // Add prefix if needed
                        if (signaturePath && !signaturePath.startsWith('/')) {
                            signaturePath = '/jasa_marga/' + signaturePath;
                        }

                        // Build generic signature block HTML
                        const signatureBlockHTML = `
                            <div class="sign-block" style="margin-top: 20mm; padding-top: 10mm; border-top: 1px solid #ccc; display: flex; justify-content: space-between;">
                                <div style="flex: 1; font-size: 12px;">
                                    <div style="font-weight: bold;">${parafFromInput}</div>
                                </div>
                                <div style="flex: 2; text-align: center;">
                                    <div style="font-weight: bold; margin-bottom: 3mm;">PT. Jasamarga Pandaan Tol</div>
                                    ${signaturePath ? `<img class="signature" src="${signaturePath}" alt="Tanda Tangan" style="height: 20mm; width: auto; display: block; margin: 0 auto 2mm;">` : '<div style="height: 20mm;"></div>'}
                                    <div style="font-weight: bold; margin-top: 2mm; text-decoration: underline;">${namaPejabat}</div>
                                    <div style="font-size: 11px; margin-top: 1mm;">${jabatanPejabat}</div>
                                </div>
                            </div>
                        `;

                        // Find the last </div> and insert before it
                        const lastDivIndex = content.lastIndexOf('</div>');
                        if (lastDivIndex > -1) {
                            content = content.substring(0, lastDivIndex) + signatureBlockHTML + content.substring(lastDivIndex);
                            console.log('✅ Signature section injected successfully');
                        } else {
                            content += signatureBlockHTML;
                            console.log('✅ Signature section appended at end');
                        }
                    } else {
                        console.log('✅ Signature section already exists in content');
                    }

                    // Replace other placeholders
                    if (/\[\[[A-Z0-9_]+\]\]/i.test(content)) {
                        const vals = collectFormValues();
                        console.log('📝 Replacing other placeholders...');
                        content = replacePlaceholders(content, vals);
                    }

                    content = content
                        .replace(
                            /<div[^>]*class=["'][^"']*html2pdf__page-break[^"']*["'][^>]*>\s*<\/div>/gi,
                            '')
                        .replace(
                            /<div[^>]*class=["'][^"']*(page-appendix|lampiran-page)[^"']*["'][^>]*>[\s\S]*?<\/div>/gi,
                            '');

                    // ALWAYS fetch fresh CSS from template for consistency with create-letter
                    const styleText = await fetchTemplateStyles();
                    console.log('✅ PDF: Using fresh template CSS (consistent with create-letter)');

                    // Clean CSS for mPDF compatibility (minimal approach - keep styling, remove problematic CSS)
                    const cleanedStyle = styleText
                        // Only remove the problematic parts that mPDF can't handle
                        // Remove :root blocks (CSS variable definitions)
                        .replace(/:root\s*\{[\s\S]*?\}/g, '')
                        // Remove @media blocks (media queries not supported)
                        .replace(/@media[\s\S]*?\{[\s\S]*?\}/g, '')
                        // Remove @page blocks (handled by mPDF config)
                        .replace(/@page\s*\{[\s\S]*?\}/g, '')
                        // Replace CSS variables with hardcoded values
                        .replace(/var\(--page-w\)/g, '210mm')
                        .replace(/var\(--page-h\)/g, '297mm')
                        .replace(/var\(--pad-top\)/g, '12mm')
                        .replace(/var\(--pad-right\)/g, '14mm')
                        .replace(/var\(--pad-bottom\)/g, '42mm')
                        .replace(/var\(--pad-left\)/g, '12mm')
                        .replace(/var\(--content-h\)/g, '243mm')
                        .replace(/var\(--content-w\)/g, '182mm')
                        .replace(/var\(--slot-gap\)/g, '6mm')
                        .replace(/var\(--slot-h\)/g, '237mm')
                        // Replace calc() with approximate values
                        .replace(/calc\s*\(\s*297mm\s*-\s*12mm\s*-\s*42mm\s*\)/gi, '243mm')
                        .replace(/calc\s*\(\s*210mm\s*-\s*(?:var\(--pad-left\)|12mm)\s*-\s*(?:var\(--pad-right\)|14mm)\s*\)/gi, '182mm')
                        .replace(/calc\s*\(\s*(?:var\(--content-h\)|243mm)\s*-\s*(?:var\(--slot-gap\)|6mm)\s*\)/gi, '237mm')
                        // Remove remaining calc() and var() - replace with auto
                        .replace(/calc\s*\([^)]*\)/gi, 'auto')
                        .replace(/var\s*\([^)]*\)/gi, 'auto')
                        // Original cleanups for page-break properties
                        .replace(
                            /\.page\s*:\s*not\(\s*:last-child\s*\)\s*\{\s*page-break-after\s*:\s*always;?\s*\}/gi,
                            '')
                        .replace(/\.page\s*\{[^}]*\}/gi, m => m.replace(
                            /page-break-(after|before)\s*:\s*always;?/gi, ''));

                    const appendixCss = `
                    .page{
                        width: 210mm !important;
                        min-height: 297mm !important;
                        padding: 12mm 14mm 42mm 12mm !important;
                        margin: 0 auto !important;
                        position: relative !important;
                        background: #fff !important;
                    }
                    .sign-block {
                        margin-top: 8mm !important;
                        width: 100% !important;
                        display: flex !important;
                        justify-content: flex-end !important;
                        page-break-inside: avoid !important;
                        position: relative !important;
                    }
                    .sign {
                        width: 72mm !important;
                        text-align: left !important;
                        font-size: 14px !important;
                        position: relative !important;
                    }
                    .sign .place-date {
                        margin-bottom: 15mm !important;
                    }
                    .sign .signature-info {
                        display: block !important;
                        margin-top: 20mm !important;
                    }
                    .sign .name {
                        font-weight: 700 !important;
                        margin: 0 !important;
                    }
                    .sign .sub-name {
                        font-weight: 700 !important;
                    }
                    .sign .title {
                        margin-top: 1mm !important;
                    }
                    .sign-block .paraf {
                        position: absolute !important;
                        left: 0 !important;
                        bottom: 0 !important;
                        font-size: 14px !important;
                        color: #333 !important;
                        text-align: left !important;
                        font-weight: 500 !important;
                        white-space: nowrap !important;
                    }
                    .company-title {
                        text-align: right;
                        font-weight: 600;
                        font-size: 14px;
                        margin-bottom: 24px;
                    }
                    .footer {
                        position: absolute !important;
                        right: 14mm !important;
                        bottom: 8mm !important;
                        text-align: right !important;
                        font-size: 10px !important;
                    }
                    .page-appendix{
                        break-before: page !important;
                        page-break-before: always !important;
                    }
                    .lampiran-box{ 
                        display:flex; 
                        align-items:center; 
                        justify-content:center; 
                        border:0 !important; 
                    }
                    .lampiran-img,
                    .lampiran-frame{
                        width:100% !important; 
                        max-width:100% !important;
                        object-fit: contain; 
                        border:0 !important;
                    }
                    .html2pdf__page-break{ 
                        break-before: page !important; 
                        page-break-before: always !important; 
                    }
                    `;

                    const styles = [
                        cleanedStyle,
                        appendixCss,
                    ].join('\n');

                    (function appendLampiran() {
                        const inp = document.getElementById('lampiran');
                        if (!inp || !inp.files || !inp.files[0]) return;

                        const f = inp.files[0];
                        const type = (f.type || '').toLowerCase();
                        const url = URL.createObjectURL(f);

                        const lampiranInner = type.startsWith('image/') ?
                            `<img src="${url}" class="lampiran-img" alt="Lampiran">` :
                            (type === 'application/pdf' ?
                                `<iframe src="${url}" class="lampiran-frame" title="Lampiran PDF"></iframe>` :
                                `<p style="font-size:12px;text-align:center;">(Lampiran ${type || 'unknown'} tidak bisa dipratinjau)</p>`
                            );

                        content = content
                            .replace(
                                /<div[^>]*class=["'][^"']*html2pdf__page-break[^"']*["'][^>]*>\s*<\/div>/gi,
                                '')
                            .replace(
                                /<div[^>]*class=["'][^"']*(page-appendix|lampiran-page)[^"']*["'][^>]*>[\s\S]*?<\/div>/gi,
                                '');

                        content += `
                        <div class="html2pdf__page-break"></div>
                        <div class="page page-appendix">
                        <div class="lampiran-box">
                            <div class="lampiran-slot">${lampiranInner}</div>
                        </div>
                        </div>`;
                    })();

                    const finalHtml = [
                        '<!DOCTYPE html>',
                        '<html lang="id">',
                        '<head>',
                        '<meta charset="utf-8">',
                        '<meta name="viewport" content="width=device-width, initial-scale=1">',
                        '<style>',
                        styles,
                        '</style>',
                        '</head>',
                        '<body>',
                        content,
                        '</body>',
                        '</html>'
                    ].join('');

                    let safeHtml = finalHtml
                        .replace(/\sdata-mce-[a-z-]+="[^"]*"/gi, '');

                    const opt = {
                        margin: [0, 0, 0, 0],
                        filename: 'surat_temp.pdf',
                        image: {
                            type: 'png',
                            quality: 0.98
                        },
                        html2canvas: {
                            scale: 2,
                            useCORS: true,
                            allowTaint: true
                        },
                        jsPDF: {
                            unit: 'mm',
                            format: 'a4',
                            orientation: 'portrait',
                            compress: false
                        },
                        pagebreak: {
                            mode: ['css']
                        }
                    };

                    console.log('🔄 Membuat PDF dengan html2pdf.js…');
                    const worker = html2pdf().set(opt).from(safeHtml).toPdf();
                    const pdf = await worker.get('pdf');

                    let total = pdf.internal.getNumberOfPages();
                    while (total > 1) {
                        const ops = pdf.internal.pages[total];
                        const opCount = Array.isArray(ops) ? ops.filter(Boolean).length : 0;

                        if (opCount <= 3) {
                            pdf.deletePage(total);
                            total--;
                        } else {
                            break;
                        }
                    }

                    const blobPdf = pdf.output('blob');

                    const fd = new FormData();
                    fd.append('pdf_file', blobPdf, `surat_${Date.now()}.pdf`);

                    console.log('⬆️ Upload ke server…');
                    const res = await fetch('/jasa_marga/api/upload/pdf.php', {
                        method: 'POST',
                        body: fd
                    });
                    const data = await res.json();

                    if (data && data.ok && data.path) {
                        fileInput.value = data.path;
                        console.log('✅ PDF berhasil diupload! Path:', data.path);
                        alert('✅ PDF berhasil dibuat dan disimpan!');
                    } else {
                        console.error('Upload response:', data);
                        alert('❌ Gagal upload PDF: ' + (data?.error || 'Tidak diketahui.'));
                    }

                } catch (err) {
                    console.error('❌ Error saat generate/upload PDF:', err);
                    alert('Terjadi kesalahan saat membuat atau mengupload PDF!');
                }
            });
        });
    </script>

    <!-- Preview Hasil Surat JS -->
    <!-- <script>
        (function(){
        const btn = document.getElementById('printDoc');

        const PRINT_CSS = `
            @page { size: A4; margin: 12mm 20mm 28mm; }

            :root{
                --page-w:210mm;
                --page-h:297mm;
                --pad-top:12mm;
                --pad-bottom:28mm;
            }

            html, body { background:#fff; }
            body { -webkit-print-color-adjust: exact; print-color-adjust: exact; }
            *{ box-sizing:border-box; }
            .page{
                width: var(--page-w);
                min-height: var(--page-h);
                padding: var(--pad-top) 20mm var(--pad-bottom);
                margin: 0 auto;
                background:#fff;
                position: relative;
                overflow: hidden;
                page-break-after: always;
                break-inside: avoid;
            }
            .page:last-child{ page-break-after: auto; }
            .header { display:flex; align-items:flex-start; justify-content:space-between; gap:16px; margin-bottom:10mm; }
            .brand { display:flex; align-items:center; gap:12px; }
            .brand img.logo { height:70px; width:auto; object-fit:contain; }
            .brand .title { line-height:1.1; font-weight:700; letter-spacing:.3px; }
            .brand .title .main { font-size:22px; color:#0c50e1; }
            .brand .title .sub  { font-size:18px; color:#0c50e1; text-transform:uppercase; }
            .topnote{ text-align:right; font-size:13px; color:#333; max-width:300px; margin-top:20px; }
            .doc-title { text-align:center; margin:2mm 0 .8mm; font-weight:900; font-size:20px; text-transform: uppercase; }
            .doc-number{ text-align:center; font-size:18px; margin-bottom:10mm; font-weight:700; }
            .divider { height:1px; background:#e6e9ef; margin:2mm 0 8mm; }
            .section { margin-bottom:6mm; font-size:15px; }
            .lead { margin-bottom:4mm; }
            table.kv { border-collapse:collapse; width:100%; font-size:15px; }
            table.kv td { padding:2px 0; vertical-align:top; }
            table.kv td.col1 { width:120px; } table.kv td.col2 { width:10px; } table.kv td.col3 { width:auto; }
            .sign-block{ margin-top:14mm; width:100%; display:flex; justify-content:flex-end; }
            .sign { width:72mm; text-align:left; font-size:15px; position:relative; padding-top:0; }
            .sign .place-date { margin-bottom:14mm; }
            .sign .stamp{ position:absolute; right:-8mm; top:18mm; opacity:.85; width:34mm; height:auto; }
            .sign .signature{ height:30mm; width:auto; display:block; margin:-6mm 0 0 0; }
            .sign .name{ font-weight:700; margin-top:2mm; }
            .sign .title{ margin-top:1mm; }
            .footer{
                position: absolute;
                left:20mm; right:20mm; bottom:12mm;
                font-size:10px; color:#333;
            }
            .footer .address{ text-align:right; line-height:1.4; }
            .page:not(:first-child) .footer{ display:none !important; }
            .lampiran-frame { width:100%; border:0; 
                height: calc(var(--page-h) - var(--pad-top) - var(--pad-bottom) - 26mm);
            }
            .lampiran-img { max-width:100%; height:auto; display:block; margin:0 auto;
                max-height: calc(var(--page-h) - var(--pad-top) - var(--pad-bottom) - 26mm);
            }
            table, th, td { border: none !important; outline: none !important; }
            .mce-item-table, .mce-item-table td, .mce-item-table th { border: none !important; outline: none !important; }
            img.signature[src=""], img.stamp[src=""] { display:none !important; }
        `;

        function openPrintPreview(htmlBody){
            const w = window.open('', '_blank');
            if(!w){ alert('Popup diblokir. Izinkan pop-up untuk melihat preview.'); return; }
            w.document.open();
            w.document.write(`<!DOCTYPE html>
            <html lang="id">
            <head>
            <meta charset="utf-8">
            <title>Preview Surat</title>
            <meta name="viewport" content="width=device-width, initial-scale=1">
            <style>${PRINT_CSS}</style>
            </head>
            <body>${htmlBody}</body>
            </html>`);

            // Ambil lampiran dari input file
            const inp = document.getElementById('lampiran');
            if (inp && inp.files && inp.files[0]) {
            const f = inp.files[0];
            const mime = (f.type || '').toLowerCase();
            const url = URL.createObjectURL(f);

            let lampiranBlock = `
                <div style="page-break-before:always"></div>
                <div class="page page-appendix">
                <h3 style="text-align:center;margin:0 0 6mm;">Lampiran</h3>
                <div style="font-size:14px;margin-bottom:4mm;">Nomor: ${
                    (document.getElementById('nomor_surat')?.value || '')
                }</div>
            `;

            if (mime.startsWith('image/')) {
                lampiranBlock += `
                <div style="border:1px solid #e0e0e0; padding:6mm; text-align:center;">
                    <img src="${url}" class="lampiran-img" alt="Lampiran">
                </div>`;
            } else if (mime === 'application/pdf') {
                lampiranBlock += `
                <div style="border:1px solid #e0e0e0; padding:0;">
                    <iframe src="${url}" class="lampiran-frame"></iframe>
                </div>`;
            } else {
                lampiranBlock += `<p><b>Lampiran (${mime || 'unknown'}) tidak dapat dipratinjau.</b></p>`;
            }
            lampiranBlock += `</div>`;

            // selipkan ke body preview
            w.document.body.insertAdjacentHTML('beforeend', lampiranBlock);
            }

            w.document.close();
            w.focus();
            setTimeout(()=>{ w.print(); }, 600);
        }

        btn?.addEventListener('click', function(){
            let html = '';
            if (window.tinymce && tinymce.get('editorNaskah')) {
            html = tinymce.get('editorNaskah').getContent({ format: 'html' });
            } else {
            const el = document.getElementById('editorNaskah');
            html = el ? (el.value || el.innerHTML || '') : '';
            }
            if(!html.trim()){
            alert('Konten surat masih kosong. Silakan isi/Generate terlebih dahulu.');
            return;
            }
            html = html
            .replace(/\sdata-mce-[a-z-]+="[^"]*"/g, '')
            .replace(/\sclass="([^"]*)"/g, (m, cls) => {
                const cleaned = cls.split(/\s+/).filter(c => !/^mce-/.test(c)).join(' ');
                return cleaned ? ` class="${cleaned}"` : '';
            })
            .replace(/\scontenteditable="[^"]*"/g, '')
            .replace(/\srole="application"/g, '');
            openPrintPreview(html);
        });
        })();
    </script> -->

    <script>
        // Event listener untuk memperlihatkan tombol ketika input diklik
        const searchInputNew = document.getElementById('searchInputNew');
        const searchButtonNew = document.getElementById('searchButtonNew');

        if (searchInputNew && searchButtonNew) {
            searchInputNew.addEventListener('focus', function() {
                searchButtonNew.classList.add('show');
                searchInputNew.placeholder = '';
            });

            searchInputNew.addEventListener('blur', function() {
                if (!searchInputNew.value.trim()) {
                    searchButtonNew.classList.remove('show');
                    searchInputNew.placeholder = 'Tap untuk mencari...';
                }
            });
        }

        document.querySelectorAll('.btn-delete').forEach(button => {
            button.addEventListener('click', function() {
                document.getElementById('deleteUserId').value = this.dataset.id;
                new bootstrap.Modal(document.getElementById('deleteUserModal')).show();
            });
        });

        document.querySelectorAll('.btn-edit').forEach(button => {
            button.addEventListener('click', function() {
                document.getElementById('editUserId').value = this.dataset.id;
                document.getElementById('editUsername').value = this.dataset.username;
                document.getElementById('editNama').value = this.dataset.nama;
                document.getElementById('editEmail').value = this.dataset.email;
                document.getElementById('editTipe').value = this.dataset.tipe;
                document.getElementById('editJabatan').value = this.dataset.jabatan;
                document.getElementById('editGender').value = this.dataset.gender;
                document.getElementById('editSubUnit').value = this.dataset.sub_unit_id;

                // Reset dan centang akses
                document.getElementById('aksesBerita').checked = false;
                document.getElementById('aksesSosial').checked = false;
                document.getElementById('aksesGaji').checked = false;
                document.getElementById('aksesSurat').checked = false;
                document.getElementById('aksesAgenda').checked = false;

                try {
                    const akses = JSON.parse(this.dataset.access);
                    if (akses.berita === 'on') document.getElementById('aksesBerita').checked = true;
                    if (akses.sosial === 'on') document.getElementById('aksesSosial').checked = true;
                    if (akses.gaji === 'on') document.getElementById('aksesGaji').checked = true;
                    if (akses.surat === 'on') document.getElementById('aksesSurat').checked = true;
                    if (akses.agenda === 'on') document.getElementById('aksesAgenda').checked = true;
                    if (akses.todolist === 'on') document.getElementById('aksesTodolist').checked = true;
                } catch (e) {
                    console.error("Gagal parse access_modul");
                }

                new bootstrap.Modal(document.getElementById('editUserModal')).show();
            });
        });

        document.addEventListener('DOMContentLoaded', function() {
            var editButtons = document.querySelectorAll('.btn-edit');
            var modal = new bootstrap.Modal(document.getElementById('editUserModal'));

            editButtons.forEach(function(button) {
                button.addEventListener('click', function() {
                    var userId = this.getAttribute('data-id');
                    var username = this.getAttribute('data-username');
                    document.getElementById('editUserId').value = userId;
                    document.getElementById('editUsername').value = username;
                    modal.show();
                });
            });
        });

        // Handle sequence number changes
        document.addEventListener('DOMContentLoaded', function() {
            const sequenceInput = document.getElementById('sequence_number');
            const regenerateBtn = document.getElementById('regenerateNumber');
            const nomorSuratInput = document.getElementById('nomor_surat');
            const tipeSuratSelect = document.getElementById('tipe_surat_id');

            if (regenerateBtn) {
                regenerateBtn.addEventListener('click', async function() {
                    const sequenceNumber = sequenceInput.value;
                    const tipeSuratId = tipeSuratSelect.value;

                    if (!sequenceNumber || !tipeSuratId) {
                        alert('Please fill in sequence number and tipe surat');
                        return;
                    }

                    // Get original classification data from hidden fields
                    const originalKlasifikasiKode = document.getElementById('original_klasifikasi_kode')
                        ?.value || '';
                    const originalJenisKode = document.getElementById('original_jenis_kode')?.value ||
                        '';
                    const originalSubjenisId = document.getElementById('original_subjenis_id')?.value ||
                        '';
                    const jabatanInput = document.getElementById('user_jabatan');

                    try {
                        const params = new URLSearchParams({
                            action: 'generate_nomor_surat',
                            tipe_surat_id: tipeSuratId,
                            existing_sequence_id: sequenceNumber,
                            klasifikasi_kode: originalKlasifikasiKode,
                            jenis_kode: originalJenisKode,
                            subjenis_id: originalSubjenisId,
                            jabatan: jabatanInput?.value || ''
                        });

                        const response = await fetch(`api_classification.php?${params}`);
                        const data = await response.json();

                        if (data.success) {
                            nomorSuratInput.value = data.data.nomor_surat;

                            // Show success message
                            if (window.Swal) {
                                Swal.fire({
                                    icon: 'success',
                                    title: 'Success!',
                                    text: 'Letter number regenerated successfully',
                                    timer: 2000,
                                    showConfirmButton: false
                                });
                            } else {
                                alert('Letter number regenerated successfully');
                            }
                        } else {
                            throw new Error(data.message || 'Failed to regenerate letter number');
                        }
                    } catch (error) {
                        console.error('Error regenerating letter number:', error);
                        if (window.Swal) {
                            Swal.fire({
                                icon: 'error',
                                title: 'Error!',
                                text: error.message
                            });
                        } else {
                            alert('Error: ' + error.message);
                        }
                    }
                });
            }

            // ============ APPROVAL MODE HANDLER ============
            const btnApprove = document.getElementById('btnApprove');
            const isApprovalMode = <?= json_encode($approvalMode) ?>;
            const isFinalApprover = <?= json_encode($isFinalApprover) ?>;
            const approvalInfo = <?= json_encode($approvalInfo) ?>;
            const alreadyApproved = <?= json_encode(isset($alreadyApproved) && $alreadyApproved) ?>;

            console.log('🔍 Approval Mode Debug:');
            console.log('  - isApprovalMode:', isApprovalMode);
            console.log('  - alreadyApproved:', alreadyApproved);
            console.log('  - approvalInfo:', approvalInfo);

            // In approval mode, update the paraf field to include the current user's initials
            if (isApprovalMode && approvalInfo) {
                const currentParafInput = document.getElementById('paraf');
                console.log('  - currentParafInput found:', !!currentParafInput);
                console.log('  - user_inisial:', approvalInfo.user_inisial);
                console.log('  - current_paraf (from previous approvers):', approvalInfo.current_paraf);

                if (currentParafInput && approvalInfo.user_inisial) {
                    // Build expected paraf: previous paraf + current user's initials
                    let expectedParaf = approvalInfo.current_paraf || '';
                    if (expectedParaf && approvalInfo.user_inisial) {
                        expectedParaf = approvalInfo.user_inisial + '/' + expectedParaf;
                    } else if (approvalInfo.user_inisial) {
                        expectedParaf = approvalInfo.user_inisial;
                    }
                    currentParafInput.value = expectedParaf;
                    console.log('✅ Updated paraf for approval mode:', expectedParaf);
                } else {
                    console.log('⚠️ Cannot update paraf - missing input or user_inisial');
                }
            } else {
                console.log('ℹ️ Not in approval mode or no approvalInfo');
            }

            if (btnApprove && isApprovalMode) {
                btnApprove.addEventListener('click', async function(e) {
                    e.preventDefault();

                    // If already approved, show message instead
                    if (alreadyApproved) {
                        Swal.fire({
                            icon: 'info',
                            title: 'Sudah Di-Approve',
                            text: 'Anda telah melakukan approval untuk surat ini. Tidak dapat melakukan approval lebih dari sekali.',
                            confirmButtonColor: '#28a745'
                        });
                        return;
                    }

                    const fileInput = document.getElementById('fileSuratPath');

                    // Check if PDF has been generated
                    if (!fileInput || !fileInput.value.trim()) {
                        Swal.fire({
                            icon: 'warning',
                            title: 'PDF Belum Di-generate',
                            html: 'Harap klik <strong>"Cetak & Upload PDF"</strong> terlebih dahulu sebelum melakukan approval.',
                            confirmButtonColor: '#ffc107'
                        });
                        return;
                    }

                    if (isFinalApprover) {
                        // Final approver - show signature modal first
                        showSignatureModal();
                    } else {
                        // Non-final approver - confirm and submit approval directly
                        const result = await Swal.fire({
                            title: 'Konfirmasi Approval',
                            html: `<p>Anda akan menyetujui surat ini.</p>
                                   <p>Paraf Anda (<strong>${approvalInfo.user_inisial || '?'}</strong>) akan ditambahkan ke surat.</p>`,
                            icon: 'question',
                            showCancelButton: true,
                            confirmButtonText: '<i class="fas fa-check"></i> Approve',
                            cancelButtonText: 'Batal',
                            confirmButtonColor: '#28a745',
                            cancelButtonColor: '#6c757d'
                        });

                        if (result.isConfirmed) {
                            await submitApproval(null); // No signature for non-final approver
                        }
                    }
                });
            }

            // Signature Modal for final approver
            function showSignatureModal() {
                Swal.fire({
                    title: 'Tanda Tangan Final',
                    html: `
                        <div class="text-start">
                            <p class="text-muted small">Anda adalah approver terakhir. Tanda tangan Anda akan disertakan dan QR Code akan di-generate.</p>
                            <div class="mb-3">
                                <label class="form-label fw-bold">Gambar Tanda Tangan:</label>
                                <div class="border rounded p-2" style="background: #f8f9fa;">
                                    <canvas id="swalSignatureCanvas" width="400" height="150" style="border: 1px solid #dee2e6; background: white; cursor: crosshair; width: 100%;"></canvas>
                                </div>
                                <button type="button" id="swalClearSignature" class="btn btn-sm btn-outline-danger mt-2">
                                    <i class="fas fa-eraser"></i> Hapus
                                </button>
                            </div>
                            <div class="mb-3">
                                <label class="form-label fw-bold">Atau unggah file:</label>
                                <input type="file" id="swalSignatureFile" class="form-control form-control-sm" accept="image/*">
                            </div>
                        </div>
                    `,
                    showCancelButton: true,
                    confirmButtonText: '<i class="fas fa-check-circle"></i> Approve & Generate QR',
                    cancelButtonText: 'Batal',
                    confirmButtonColor: '#28a745',
                    cancelButtonColor: '#6c757d',
                    width: '500px',
                    didOpen: () => {
                        // Initialize signature canvas
                        const canvas = document.getElementById('swalSignatureCanvas');
                        const ctx = canvas.getContext('2d');
                        let isDrawing = false;

                        canvas.addEventListener('mousedown', (e) => {
                            isDrawing = true;
                            const rect = canvas.getBoundingClientRect();
                            const scaleX = canvas.width / rect.width;
                            const scaleY = canvas.height / rect.height;
                            ctx.beginPath();
                            ctx.moveTo((e.clientX - rect.left) * scaleX, (e.clientY - rect.top) * scaleY);
                        });

                        canvas.addEventListener('mousemove', (e) => {
                            if (isDrawing) {
                                const rect = canvas.getBoundingClientRect();
                                const scaleX = canvas.width / rect.width;
                                const scaleY = canvas.height / rect.height;
                                ctx.lineTo((e.clientX - rect.left) * scaleX, (e.clientY - rect.top) * scaleY);
                                ctx.stroke();
                            }
                        });

                        canvas.addEventListener('mouseup', () => isDrawing = false);
                        canvas.addEventListener('mouseout', () => isDrawing = false);

                        document.getElementById('swalClearSignature').addEventListener('click', () => {
                            ctx.clearRect(0, 0, canvas.width, canvas.height);
                        });
                    },
                    preConfirm: async () => {
                        const canvas = document.getElementById('swalSignatureCanvas');
                        const fileInput = document.getElementById('swalSignatureFile');

                        let signatureBlob = null;

                        // Check if canvas has drawing
                        const isCanvasEmpty = isCanvasBlank(canvas);

                        if (!isCanvasEmpty) {
                            signatureBlob = await new Promise(resolve => canvas.toBlob(resolve, 'image/png'));
                        } else if (fileInput.files[0]) {
                            signatureBlob = fileInput.files[0];
                        }

                        // Signature is optional but recommended
                        return {
                            signatureBlob
                        };
                    }
                }).then(async (result) => {
                    if (result.isConfirmed) {
                        await submitApproval(result.value.signatureBlob);
                    }
                });
            }

            function isCanvasBlank(canvas) {
                const ctx = canvas.getContext('2d');
                const pixelBuffer = new Uint32Array(
                    ctx.getImageData(0, 0, canvas.width, canvas.height).data.buffer
                );
                return !pixelBuffer.some(color => color !== 0);
            }

            // Upload signature blob and get the file path
            async function uploadSignatureBlob(blob) {
                if (!blob) {
                    console.warn('⚠️ No blob to upload');
                    return '';
                }

                console.log('🔧 Uploading signature blob - Type:', blob.type, 'Size:', blob.size);

                const fd = new FormData();
                fd.append('signature_file', blob, `signature_${Date.now()}.png`);

                try {
                    const response = await fetch('/jasa_marga/api/upload/pdf.php', {
                        method: 'POST',
                        body: fd
                    });
                    const data = await response.json();

                    if (data && data.ok && data.path) {
                        console.log('✅ Signature uploaded! Path:', data.path);
                        return data.path;
                    } else {
                        console.warn('⚠️ Signature upload failed:', data);
                        return '';
                    }
                } catch (err) {
                    console.error('❌ Error uploading signature:', err);
                    return '';
                }
            }

            async function regenerateAndUploadPDFWithSignature(signatureBlob) {
                // Regenerate PDF with signature embedded BEFORE approval submission
                // This ensures the signature appears in the PDF

                console.log('🔘 Regenerating PDF with signature before approval...');
                console.log('📊 Input signatureBlob:', signatureBlob);

                try {
                    // First, upload the signature blob to get a file path
                    let signaturePath = '';
                    if (signatureBlob) {
                        signaturePath = await uploadSignatureBlob(signatureBlob);
                        if (!signaturePath) {
                            console.warn('⚠️ Could not upload signature, continuing without it');
                        } else {
                            // Add prefix if path doesn't start with / (make it absolute)
                            if (!signaturePath.startsWith('/')) {
                                signaturePath = '/jasa_marga/' + signaturePath;
                            }
                            console.log('✅ Signature path with prefix:', signaturePath);
                        }
                    } else {
                        console.warn('⚠️ No signature blob provided');
                    }

                    // Get fresh content from editor
                    const editor = tinymce.get('editorNaskah');
                    if (!editor) {
                        throw new Error('Editor not found');
                    }

                    let content = editor.getContent() || '';
                    console.log('📄 Editor content length:', content.length);
                    console.log('📄 Has signature section:', content.includes('signature-paraf') || content.includes('sign-section') || content.includes('sign-block'));

                    if (!content.trim()) {
                        throw new Error('Konten surat kosong');
                    }

                    // Get signature data from form
                    const namaPenandatangan = document.getElementById('nama_pejabat')?.value ||
                        document.querySelector('[name="nama_pejabat"]')?.value || '';
                    const jabatanPenandatangan = document.getElementById('jabatan_pejabat')?.value ||
                        document.querySelector('[name="jabatan_pejabat"]')?.value || '';

                    // Get paraf value (which includes approver's initials)
                    const isApprovalMode = <?= json_encode($approvalMode) ?>;
                    const parafFromInput = (document.getElementById('paraf')?.value || '').trim();

                    console.log('📝 Signature injection data:');
                    console.log('   - Nama Penandatangan:', namaPenandatangan);
                    console.log('   - Jabatan:', jabatanPenandatangan);
                    console.log('   - Paraf:', parafFromInput);
                    console.log('   - Signature path:', signaturePath);

                    console.log('✅ isApprovalMode:', isApprovalMode);
                    console.log('✅ Paraf from input:', parafFromInput);
                    console.log('✅ Paraf input element exists:', !!document.getElementById('paraf'));

                    // Use consolidated helper to replace PARAF
                    const parafResult = replaceParafInContent(content, parafFromInput, isApprovalMode);
                    content = parafResult.content;
                    console.log('✅ PARAF replaced - replacements made:', parafResult.replacementCount);
                    console.log('✅ Content after PARAF replacement length:', content.length);
                    console.log('✅ Content still has [[PARAF]]:', content.includes('[[PARAF]]'));
                    console.log('✅ Content now has paraf value:', content.includes(parafFromInput));

                    // CRITICAL FIX: Inject signature section if missing
                    const hasSignatureSection = content.includes('signature-paraf') ||
                        content.includes('sign-section') ||
                        content.includes('sign-block');

                    if (!hasSignatureSection) {
                        console.log('⚠️ Signature section NOT FOUND - Injecting it now...');

                        // Build generic signature block HTML
                        const signatureBlockHTML = `
                            <div class="sign-block" style="margin-top: 20mm; padding-top: 10mm; border-top: 1px solid #ccc; display: flex; justify-content: space-between;">
                                <div style="flex: 1; font-size: 12px;">
                                    <div style="font-weight: bold;">${parafFromInput}</div>
                                </div>
                                <div style="flex: 2; text-align: center;">
                                    <div style="font-weight: bold; margin-bottom: 3mm;">PT. Jasamarga Pandaan Tol</div>
                                    ${signaturePath ? `<img class="signature" src="${signaturePath}" alt="Tanda Tangan" style="height: 20mm; width: auto; display: block; margin: 0 auto 2mm;">` : '<div style="height: 20mm;"></div>'}
                                    <div style="font-weight: bold; margin-top: 2mm; text-decoration: underline;">${namaPenandatangan}</div>
                                    <div style="font-size: 11px; margin-top: 1mm;">${jabatanPenandatangan}</div>
                                </div>
                            </div>
                        `;

                        // Find the last </div> and insert before it
                        const lastDivIndex = content.lastIndexOf('</div>');
                        if (lastDivIndex > -1) {
                            content = content.substring(0, lastDivIndex) + signatureBlockHTML + content.substring(lastDivIndex);
                            console.log('✅ Signature section injected successfully');
                        } else {
                            content += signatureBlockHTML;
                            console.log('✅ Signature section appended at end');
                        }
                    } else {
                        console.log('✅ Signature section already exists in content');
                    }

                    // Use the uploaded signature path (server-side file) instead of data URL
                    // This works better with html2pdf

                    // Replace other placeholders with override for signature
                    const vals = collectFormValues();
                    console.log('✅ Form values collected');
                    console.log('   - PARAF value:', vals.PARAF);
                    console.log('   - PATH_TANDA_TANGAN value:', vals.PATH_TANDA_TANGAN);
                    console.log('   - QR_CODE value:', vals.QR_CODE);

                    // Override with the uploaded signature path if available
                    if (signaturePath) {
                        vals.PATH_TANDA_TANGAN = signaturePath;
                        console.log('✅ Signature path set in placeholders:', signaturePath);
                    } else {
                        console.warn('⚠️ No signature path available for placeholders');
                    }

                    // Always call replacePlaceholders
                    console.log('🔄 Calling replacePlaceholders...');
                    content = replacePlaceholders(content, vals);
                    console.log('✅ replacePlaceholders completed');
                    console.log('✅ Content after replacePlaceholders length:', content.length);

                    // DIRECT INJECTION: Update any signature img src directly with the uploaded path
                    if (signaturePath) {
                        // Replace signature img src with server path
                        const beforeDirectInject = content.length;
                        content = content.replace(
                            /(<img[^>]*class=["'][^"']*signature[^"']*["'][^>]*src=")([^"]*)(")/gi,
                            `$1${signaturePath}$3`
                        );
                        console.log('✅ Direct inject signature img src - length changed:', beforeDirectInject !== content.length);

                        // Also replace [[PATH_TANDA_TANGAN]] if it still exists
                        const beforePathReplace = content.length;
                        content = content.replace(/\[\[PATH_TANDA_TANGAN\]\]/g, signaturePath);
                        console.log('✅ Replaced [[PATH_TANDA_TANGAN]] - length changed:', beforePathReplace !== content.length);

                        // If no signature img exists, inject one before signature-name
                        if (!content.includes('class="signature') && !content.includes("class='signature")) {
                            const beforeInject = content.length;
                            content = content.replace(
                                /(<div[^>]*class=["'][^"']*signature-name[^"']*["'][^>]*>)/gi,
                                `<img class="signature signature-img" src="${signaturePath}" alt="Tanda Tangan" style="height:22mm;width:auto;display:block;margin:-4mm 0 0 auto;">$1`
                            );
                            console.log('✅ Injected signature img before signature-name - length changed:', beforeInject !== content.length);
                        } else {
                            console.log('✅ Signature img already exists in content');
                        }
                        console.log('✅ Signature image directly injected into content with path:', signaturePath);
                    }

                    // Final content check before PDF generation
                    console.log('📋 FINAL CONTENT CHECK BEFORE PDF:');
                    console.log('   - Has signature-paraf:', content.includes('signature-paraf'));
                    console.log('   - Has signature img:', content.includes('class="signature'));
                    console.log('   - Has paraf value in content:', content.includes(parafFromInput));
                    console.log('   - Signature img src sample:', content.match(/<img[^>]*class="[^"]*signature[^"]*"[^>]*>/)?.[0] || 'NOT FOUND');

                    content = content
                        .replace(
                            /<div[^>]*class=["'][^"']*html2pdf__page-break[^"']*["'][^>]*>\s*<\/div>/gi,
                            '')
                        .replace(
                            /<div[^>]*class=["'][^"']*(page-appendix|lampiran-page)[^"']*["'][^>]*>[\s\S]*?<\/div>/gi,
                            '');

                    // Fetch template styles
                    const styleText = await fetchTemplateStyles();
                    console.log('✅ PDF: Using fresh template CSS with signature');

                    // Clean CSS
                    const cleanedStyle = styleText
                        .replace(/:root\s*\{[\s\S]*?\}/g, '')
                        .replace(/@media[\s\S]*?\{[\s\S]*?\}/g, '')
                        .replace(/@page\s*\{[\s\S]*?\}/g, '')
                        .replace(/var\(--page-w\)/g, '210mm')
                        .replace(/var\(--page-h\)/g, '297mm')
                        .replace(/var\(--pad-top\)/g, '12mm')
                        .replace(/var\(--pad-right\)/g, '14mm')
                        .replace(/var\(--pad-bottom\)/g, '42mm')
                        .replace(/var\(--pad-left\)/g, '12mm')
                        .replace(/var\(--content-h\)/g, '243mm')
                        .replace(/var\(--content-w\)/g, '182mm')
                        .replace(/var\(--slot-gap\)/g, '6mm')
                        .replace(/var\(--slot-h\)/g, '237mm')
                        .replace(/calc\s*\(\s*297mm\s*-\s*12mm\s*-\s*42mm\s*\)/gi, '243mm')
                        .replace(/calc\s*\(\s*210mm\s*-\s*(?:var\(--pad-left\)|12mm)\s*-\s*(?:var\(--pad-right\)|14mm)\s*\)/gi, '182mm')
                        .replace(/calc\s*\(\s*(?:var\(--content-h\)|243mm)\s*-\s*(?:var\(--slot-gap\)|6mm)\s*\)/gi, '237mm')
                        .replace(/calc\s*\([^)]*\)/gi, 'auto')
                        .replace(/var\s*\([^)]*\)/gi, 'auto')
                        .replace(/\.page\s*:\s*not\(\s*:last-child\s*\)\s*\{\s*page-break-after\s*:\s*always;?\s*\}/gi, '')
                        .replace(/\.page\s*\{[^}]*\}/gi, m => m.replace(/page-break-(after|before)\s*:\s*always;?/gi, ''));

                    const appendixCss = `
                    .page{
                        width: 210mm !important;
                        min-height: 297mm !important;
                        padding: 12mm 14mm 42mm 12mm !important;
                        margin: 0 auto !important;
                        position: relative !important;
                        background: #fff !important;
                    }
                    .sign-block {
                        margin-top: 8mm !important;
                        width: 100% !important;
                        display: flex !important;
                        justify-content: flex-end !important;
                        page-break-inside: avoid !important;
                        position: relative !important;
                    }
                    .sign {
                        width: 72mm !important;
                        text-align: left !important;
                        font-size: 14px !important;
                        position: relative !important;
                    }
                    .sign .place-date {
                        margin-bottom: 15mm !important;
                    }
                    .sign .signature-info {
                        display: block !important;
                        margin-top: 20mm !important;
                    }
                    .sign .name {
                        font-weight: 700 !important;
                        margin: 0 !important;
                    }
                    .sign .sub-name {
                        font-weight: 700 !important;
                    }
                    .sign .title {
                        margin-top: 1mm !important;
                    }
                    .sign-block .paraf {
                        position: absolute !important;
                        left: 0 !important;
                        bottom: 0 !important;
                        font-size: 14px !important;
                        color: #333 !important;
                        text-align: left !important;
                        font-weight: 500 !important;
                        white-space: nowrap !important;
                    }
                    .signature-final {
                        max-width: 150px !important;
                        height: auto !important;
                        display: block !important;
                        margin-top: 10px !important;
                    }
                    .company-title {
                        text-align: right;
                        font-weight: 600;
                        font-size: 14px;
                        margin-bottom: 24px;
                    }
                    .footer {
                        position: absolute !important;
                        right: 14mm !important;
                        bottom: 8mm !important;
                        text-align: right !important;
                        font-size: 10px !important;
                    }
                    .page-appendix{
                        break-before: page !important;
                        page-break-before: always !important;
                    }
                    .lampiran-box{ 
                        display:flex; 
                        align-items:center; 
                        justify-content:center; 
                        border:0 !important; 
                    }
                    .lampiran-img,
                    .lampiran-frame{
                        width:100% !important; 
                        max-width:100% !important;
                        object-fit: contain; 
                        border:0 !important;
                    }
                    .html2pdf__page-break{ 
                        break-before: page !important; 
                        page-break-before: always !important; 
                    }`;

                    const styles = [
                        cleanedStyle,
                        appendixCss,
                    ].join('\n');

                    // Add lampiran if exists
                    (function appendLampiran() {
                        const inp = document.getElementById('lampiran');
                        if (!inp || !inp.files || !inp.files[0]) return;

                        const f = inp.files[0];
                        const type = (f.type || '').toLowerCase();
                        const url = URL.createObjectURL(f);

                        const lampiranInner = type.startsWith('image/') ?
                            `<img src="${url}" class="lampiran-img" alt="Lampiran">` :
                            (type === 'application/pdf' ?
                                `<iframe src="${url}" class="lampiran-frame" title="Lampiran PDF"></iframe>` :
                                `<p style="font-size:12px;text-align:center;">(Lampiran ${type || 'unknown'} tidak bisa dipratinjau)</p>`
                            );

                        content = content
                            .replace(
                                /<div[^>]*class=["'][^"']*html2pdf__page-break[^"']*["'][^>]*>\s*<\/div>/gi,
                                '')
                            .replace(
                                /<div[^>]*class=["'][^"']*(page-appendix|lampiran-page)[^"']*["'][^>]*>[\s\S]*?<\/div>/gi,
                                '');

                        content += `
                        <div class="html2pdf__page-break"></div>
                        <div class="page page-appendix">
                        <div class="lampiran-box">
                            <div class="lampiran-slot">${lampiranInner}</div>
                        </div>
                        </div>`;
                    })();

                    const finalHtml = [
                        '<!DOCTYPE html>',
                        '<html lang="id">',
                        '<head>',
                        '<meta charset="utf-8">',
                        '<meta name="viewport" content="width=device-width, initial-scale=1">',
                        '<style>',
                        styles,
                        '</style>',
                        '</head>',
                        '<body>',
                        content,
                        '</body>',
                        '</html>'
                    ].join('');

                    let safeHtml = finalHtml
                        .replace(/\sdata-mce-[a-z-]+="[^"]*"/gi, '');

                    const opt = {
                        margin: [0, 0, 0, 0],
                        filename: 'surat_approval.pdf',
                        image: {
                            type: 'png',
                            quality: 0.98
                        },
                        html2canvas: {
                            scale: 2,
                            useCORS: true,
                            allowTaint: true
                        },
                        jsPDF: {
                            unit: 'mm',
                            format: 'a4',
                            orientation: 'portrait',
                            compress: false
                        },
                        pagebreak: {
                            mode: ['css']
                        }
                    };

                    console.log('🔄 Regenerating PDF with html2pdf.js and signature…');

                    // COMPREHENSIVE DEBUG: Check the final HTML before PDF generation
                    console.log('📋 === COMPREHENSIVE DEBUG CHECK ===');
                    console.log('🔍 safeHtml length:', safeHtml.length);
                    console.log('🔍 Has signature-paraf div:', safeHtml.includes('signature-paraf'));
                    console.log('🔍 Has [[PARAF]]:', safeHtml.includes('[[PARAF]]'));
                    console.log('🔍 Has paraf value text:', safeHtml.includes(parafFromInput));

                    // Check for signature img
                    console.log('🔍 Has any img with class="signature:', safeHtml.includes('class="signature'));
                    console.log('🔍 Has img[class*="signature"]:', /<img[^>]*class=["\'][^"\']*signature[^"\']*["\'][^>]*>/.test(safeHtml));

                    // Find all img tags
                    const allImgs = safeHtml.match(/<img[^>]*>/gi) || [];
                    console.log('🔍 Total img tags found:', allImgs.length);
                    allImgs.forEach((img, idx) => {
                        console.log(`   [Img ${idx}]:`, img.substring(0, 150));
                    });

                    // Check for signature-paraf div specifically
                    const parafDivMatch = safeHtml.match(/<div[^>]*class=["\'][^"\']*signature-paraf[^"\']*["\'][^>]*>[\s\S]*?<\/div>/i);
                    if (parafDivMatch) {
                        console.log('✅ Found signature-paraf div:', parafDivMatch[0]);
                    } else {
                        console.log('❌ signature-paraf div NOT FOUND');
                    }

                    // Check for signature-img specifically
                    const sigImgMatch = safeHtml.match(/<img[^>]*class=["\'][^"\']*signature[^"\']*["\'][^>]*>/i);
                    if (sigImgMatch) {
                        console.log('✅ Found signature img:', sigImgMatch[0].substring(0, 200));
                    } else {
                        console.log('❌ Signature img NOT FOUND');
                    }

                    // Sample content around signature
                    const sigIndex = safeHtml.indexOf('signature-paraf');
                    if (sigIndex > -1) {
                        const start = Math.max(0, sigIndex - 200);
                        const end = Math.min(safeHtml.length, sigIndex + 500);
                        console.log('📍 Context around signature-paraf:', safeHtml.substring(start, end));
                    }

                    console.log('📋 === END DEBUG CHECK ===');

                    // Log a sample of signature img if it exists
                    const sigMatch = safeHtml.match(/<img[^>]*class="[^"]*signature[^"]*"[^>]*>/i);
                    if (sigMatch) {
                        console.log('🔍 DEBUG - Signature img found:', sigMatch[0].substring(0, 150));
                    } else {
                        console.log('🔍 DEBUG - No signature img found in HTML!');
                    }

                    const worker = html2pdf().set(opt).from(safeHtml).toPdf();
                    const pdf = await worker.get('pdf');

                    console.log('✅ PDF generated successfully');
                    console.log('   - Number of pages:', pdf.internal.getNumberOfPages());
                    console.log('   - PDF internal state:', Object.keys(pdf.internal).join(', '));

                    let total = pdf.internal.getNumberOfPages();
                    while (total > 1) {
                        const ops = pdf.internal.pages[total];
                        const opCount = Array.isArray(ops) ? ops.filter(Boolean).length : 0;

                        if (opCount <= 3) {
                            pdf.deletePage(total);
                            total--;
                        } else {
                            break;
                        }
                    }

                    const blobPdf = pdf.output('blob');
                    console.log('✅ PDF blob created - Size:', blobPdf.size, 'bytes');

                    const fd = new FormData();
                    fd.append('pdf_file', blobPdf, `surat_approval_${Date.now()}.pdf`);

                    console.log('⬆️ Uploading regenerated PDF with signature to server…');
                    const res = await fetch('/jasa_marga/api/upload/pdf.php', {
                        method: 'POST',
                        body: fd
                    });
                    const data = await res.json();

                    if (data && data.ok && data.path) {
                        const fileInput = document.getElementById('pdf_surat');
                        if (fileInput) {
                            fileInput.value = data.path;
                        }
                        console.log('✅ PDF dengan signature berhasil diupload! Path:', data.path);
                        return true;
                    } else {
                        console.error('Upload response:', data);
                        throw new Error('Gagal upload PDF dengan signature: ' + (data?.error || 'Tidak diketahui'));
                    }
                } catch (err) {
                    console.error('❌ Error saat regenerate/upload PDF dengan signature:', err);
                    throw err;
                }
            }

            async function submitApproval(signatureBlob) {
                Swal.fire({
                    title: 'Memproses Approval...',
                    html: isFinalApprover ? 'Regenerating PDF dengan signature, generating QR Code...' : 'Regenerating PDF dengan signature...',
                    allowOutsideClick: false,
                    showConfirmButton: false,
                    didOpen: () => Swal.showLoading()
                });

                try {
                    // First, regenerate and upload PDF with signature if signature provided
                    if (signatureBlob) {
                        await regenerateAndUploadPDFWithSignature(signatureBlob);
                    }

                    // Now submit approval with the regenerated PDF
                    const formData = new FormData();
                    formData.append('action', 'submit_approval');
                    formData.append('office_id', OFFICE_ID);
                    formData.append('is_final', isFinalApprover ? '1' : '0');

                    if (signatureBlob) {
                        formData.append('signature', signatureBlob, 'signature.png');
                    }

                    const response = await fetch('../api/approval.php', {
                        method: 'POST',
                        body: formData
                    });

                    const data = await response.json();

                    if (data.success) {
                        await Swal.fire({
                            icon: 'success',
                            title: 'Approval Berhasil!',
                            html: isFinalApprover ?
                                `<p>Surat telah di-approve dan QR Code berhasil di-generate.</p>
                                   <p>PDF telah diperbarui dengan signature dan paraf Anda.</p>
                                   ${data.data && data.data.verification_url ? `<p><small>URL Verifikasi: <a href="${data.data.verification_url}" target="_blank">${data.data.verification_url}</a></small></p>` : ''}` : '<p>Approval Anda telah disimpan dengan PDF yang telah diperbarui. Giliran berpindah ke approver berikutnya.</p>',
                            confirmButtonColor: '#28a745'
                        });

                        // Redirect back to record page
                        window.location.href = 'record-letter.php?approval_success=1';
                    } else {
                        throw new Error(data.message || 'Gagal melakukan approval');
                    }
                } catch (error) {
                    console.error('Approval error:', error);
                    Swal.fire({
                        icon: 'error',
                        title: 'Error!',
                        text: error.message,
                        confirmButtonColor: '#dc3545'
                    });
                }
            }
        });
    </script>

</body>

</html>