<?php
session_start();
include(__DIR__ . '/../includes/koneksi.php');
include(__DIR__ . '/../includes/functions.php');

/**
 * Helper function to initialize approval chain when letter is created
 * For users: Include creator + signer (0/2 approval)
 * Creator approves first, then signer approves
 */
function initializeApprovalChainForLetter($conn, $officeId, $creatorUserId, $signerId)
{
    // Get the signer info from authorized_signers
    $stmt = $conn->prepare("
        SELECT id, user_id, nama
        FROM authorized_signers 
        WHERE user_id = ? AND is_active = 1
        LIMIT 1
    ");
    $stmt->bind_param("i", $signerId);
    $stmt->execute();
    $signer = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if (!$signer) {
        // Try finding by authorized_signers.id
        $stmt = $conn->prepare("
            SELECT id, user_id, nama
            FROM authorized_signers 
            WHERE id = ? AND is_active = 1
            LIMIT 1
        ");
        $stmt->bind_param("i", $signerId);
        $stmt->execute();
        $signer = $stmt->get_result()->fetch_assoc();
        $stmt->close();
    }

    if (!$signer) {
        return false; // No signer found
    }

    // Get creator name
    $creatorStmt = $conn->prepare("SELECT nama FROM users WHERE id = ?");
    $creatorStmt->bind_param("i", $creatorUserId);
    $creatorStmt->execute();
    $creatorRow = $creatorStmt->get_result()->fetch_assoc();
    $creatorStmt->close();
    $creatorName = $creatorRow ? $creatorRow['nama'] : '';

    error_log("DEBUG initializeApprovalChainForLetter (USER) - Office ID: $officeId, Creator ID: $creatorUserId, Signer ID: " . $signer['user_id']);

    // Build approval chain with creator + signer (0/2 approval)
    // User creates letter, user can approve, then signer approves
    $approvalChain = [];

    // Add creator first (will be urutan=1)
    $approvalChain[] = [
        'user_id' => $creatorUserId,
        'nama' => $creatorName
    ];

    // Add signer second (will be urutan=2), but only if different from creator
    if ($creatorUserId !== (int)$signer['user_id']) {
        $approvalChain[] = [
            'user_id' => (int)$signer['user_id'],
            'nama' => $signer['nama']
        ];
    }

    error_log("DEBUG Approval chain created with " . count($approvalChain) . " step(s)");

    // Delete existing chain entries for this letter
    $stmt = $conn->prepare("DELETE FROM urutan_paraf WHERE office_id = ?");
    $stmt->bind_param("i", $officeId);
    $stmt->execute();
    error_log("DEBUG DELETE from urutan_paraf: office_id=$officeId, affected_rows=" . $stmt->affected_rows);
    $stmt->close();

    // Insert approval chain into urutan_paraf
    $urutan = 1;
    foreach ($approvalChain as $member) {
        $stmt = $conn->prepare("INSERT INTO urutan_paraf (office_id, user_id, urutan) VALUES (?, ?, ?)");
        $stmt->bind_param("iii", $officeId, $member['user_id'], $urutan);
        if ($stmt->execute()) {
            error_log("DEBUG Inserted urutan_paraf: office_id=$officeId, user_id=" . $member['user_id'] . ", urutan=$urutan");
        } else {
            error_log("DEBUG FAILED to insert urutan_paraf: " . $stmt->error);
        }
        $stmt->close();
        $urutan++;
    }

    // Verify what was inserted
    $verifyStmt = $conn->prepare("SELECT COUNT(*) as total FROM urutan_paraf WHERE office_id = ?");
    $verifyStmt->bind_param("i", $officeId);
    $verifyStmt->execute();
    $verifyResult = $verifyStmt->get_result()->fetch_assoc();
    $verifyStmt->close();
    error_log("DEBUG After insert, urutan_paraf count for office_id=$officeId: " . $verifyResult['total']);

    // Update office with penandatangan_id
    $signerUserId = (int)$signer['user_id'];
    $stmt = $conn->prepare("UPDATE office SET penandatangan_id = ? WHERE id = ?");
    $stmt->bind_param("ii", $signerUserId, $officeId);
    $stmt->execute();
    $stmt->close();

    return true;
}

if (!isset($_SESSION['username'])) {
    header('Location: ../login.php');
    exit();
}

$username = $_SESSION['username'];
$stmt = $conn->prepare("SELECT id, tipe, nama, access_modul, jabatan FROM users WHERE username = ?");
$stmt->bind_param("s", $username);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->fetch_assoc();
$stmt->close();

if (!$user || !in_array($user['tipe'], ['user', 'admin', 'editor'])) {
    header('Location: ../login.php');
    exit();
}

$access_modul = json_decode($user['access_modul'], true);
$userId = (int)$user['id'];

// Check approver/supervisor status
$subordinateIds = getSubordinatesForUser($conn, $userId);
$isSupervisor = !empty($subordinateIds);
$isApprover = isApprover($conn, $userId);

// Allow access if user has surat module enabled OR they are an approver OR they are supervisor
if ((empty($access_modul['surat']) || $access_modul['surat'] !== 'on') && !$isApprover && !$isSupervisor) {
    header('Location: ../login.php');
    exit();
}

$userId = (int)$user['id'];
$name = htmlspecialchars($user['nama']);

// Load tipe_surat options
// Load tipe_surat options and classification data
$tipeSuratList = [];
$jenisSurat = [];
$klasifikasi = [];
$subjenisList = [];
$usersList = [];
if ($r = $conn->query("SELECT id, kode, nama FROM tipe_surat ORDER BY nama")) {
    $tipeSuratList = $r->fetch_all(MYSQLI_ASSOC);
    $r->free();
}
// Note: Office AUTO_INCREMENT preview removed - numbering is generated via API (letter_sequences)
$skTipeId = 0;
foreach ($tipeSuratList as $tt) {
    if (isset($tt['kode']) && trim($tt['kode']) === 'SK') {
        $skTipeId = (int)$tt['id'];
        break;
    }
}
if ($r = $conn->query("SELECT id, kode, nama, id_klasifikasi FROM jenis_surat WHERE is_active=1 ORDER BY nama")) {
    $jenisSurat = $r->fetch_all(MYSQLI_ASSOC);
    $r->free();
}
if ($r = $conn->query("SELECT id, kode, nama FROM klasifikasi_arsip WHERE is_active=1 ORDER BY kode, nama")) {
    $klasifikasi = $r->fetch_all(MYSQLI_ASSOC);
    $r->free();
}
if ($r = $conn->query("SELECT id, kode, nama, id_jenis FROM subjenis_surat WHERE 1=1 ORDER BY nama")) {
    $subjenisList = $r->fetch_all(MYSQLI_ASSOC);
    $r->free();
}
if ($r = $conn->query("SELECT id, nama, jabatan FROM users WHERE tipe IN ('user', 'admin') ORDER BY nama")) {
    $usersList = $r->fetch_all(MYSQLI_ASSOC);
    $r->free();
}

// load kode jabatan mapping for client-side preview
$kodeJabatanMap = [];
if (file_exists(__DIR__ . '/../data/kode_jabatan.json')) {
    $kb = json_decode(file_get_contents(__DIR__ . '/../data/kode_jabatan.json'), true);
    if (isset($kb['kodeJabatan'])) $kodeJabatanMap = $kb['kodeJabatan'];
}

// helper: load kodeJabatan mapping
function getKodeJabatanFromJSON($jabatan)
{
    $jsonFile = __DIR__ . '/../data/kode_jabatan.json';
    if (!file_exists($jsonFile)) return null;
    $jsonContent = file_get_contents($jsonFile);
    $data = json_decode($jsonContent, true);
    if (!$data || !isset($data['kodeJabatan'])) return null;
    $jabatanNormalized = strtolower(trim(preg_replace('/\s+/', ' ', $jabatan)));
    foreach ($data['kodeJabatan'] as $item) {
        $itemJabatanNormalized = strtolower(trim(preg_replace('/\s+/', ' ', $item['jabatan'])));
        if ($itemJabatanNormalized === $jabatanNormalized) return $item['kode'];
        if (isset($item['aliases']) && is_array($item['aliases'])) {
            foreach ($item['aliases'] as $alias) {
                $aliasNormalized = strtolower(trim(preg_replace('/\s+/', ' ', $alias)));
                if ($aliasNormalized === $jabatanNormalized) return $item['kode'];
            }
        }
    }
    return null;
}

$errors = [];
$success = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $kepentingan = $_POST['kepentingan'] ?? 'internal';
    $tipe_surat_id = (int)($_POST['tipe_surat_id'] ?? 0);
    $tempat_dibuat = trim($_POST['tempat_dibuat'] ?? '');
    $tanggal_surat = $_POST['tanggal_surat'] ?? '';
    $kepada = trim($_POST['kepada'] ?? '');
    $perihal = trim($_POST['perihal'] ?? '');
    $paraf = trim($_POST['paraf'] ?? '');
    $naskah_html = $_POST['naskah_html'] ?? ''; // TinyMCE content with CSS
    // additional fields
    $jenis_surat_id = (int)($_POST['jenis_surat_new'] ?? 0);
    $klasifikasi_arsip_id = (int)($_POST['klasifikasi_arsip_new'] ?? 0);
    $subjenis_surat_id = (int)($_POST['subjenis_surat_id'] ?? 0);
    $dari_tipe = $_POST['dari_tipe'] ?? 'internal';
    $dari_pribadi = (int)($_POST['dari_pribadi'] ?? 0);

    if (!in_array($kepentingan, ['internal', 'eksternal'])) $kepentingan = 'internal';
    if ($tipe_surat_id <= 0) $errors[] = 'Tipe surat harus dipilih.';
    if ($tempat_dibuat === '') $errors[] = 'Tempat dibuat wajib diisi.';
    if ($tanggal_surat === '') $errors[] = 'Tanggal surat wajib diisi.';
    // Get tipe surat code for conditional validations
    $tipeSuratKode = '';
    if ($tipe_surat_id > 0) {
        $stmtT = $conn->prepare("SELECT kode FROM tipe_surat WHERE id = ?");
        $stmtT->bind_param('i', $tipe_surat_id);
        $stmtT->execute();
        $resT = $stmtT->get_result();
        if ($rowT = $resT->fetch_assoc()) {
            $tipeSuratKode = $rowT['kode'] ?? '';
        }
        $stmtT->close();
    }
    // 'Kepada' is required only for SK; for KPTS/SE it's optional
    if ($tipeSuratKode === 'SK' && $kepada === '') $errors[] = 'Kepada wajib diisi untuk SK.';
    if ($perihal === '') $errors[] = 'Perihal wajib diisi.';
    $enumDari = ['internal', 'eksternal', 'pribadi', 'sendiri'];
    if (!in_array($dari_tipe, $enumDari, true)) $errors[] = 'Dari (tipe) tidak valid.';
    if ($dari_tipe === 'pribadi' && $dari_pribadi <= 0) $errors[] = 'Pilih nama untuk Atas Nama (Pribadi).';

    // Handle file upload (PDF or image)
    $fileSuratPath = null;

    // Check if file path came from AJAX upload (hidden input)
    if (!empty($_POST['file_surat'])) {
        $fileSuratPath = trim($_POST['file_surat']);
    }

    // Also check for direct file upload
    if (isset($_FILES['file_surat']) && $_FILES['file_surat']['error'] !== UPLOAD_ERR_NO_FILE) {
        if ($_FILES['file_surat']['error'] !== UPLOAD_ERR_OK) {
            $errors[] = 'Upload file surat gagal.';
        } else {
            $tmp = $_FILES['file_surat']['tmp_name'];
            $mime = @mime_content_type($tmp) ?: ($_FILES['file_surat']['type'] ?? '');
            $allowed = ['application/pdf', 'image/png', 'image/jpeg'];
            if (!in_array($mime, $allowed, true)) {
                $errors[] = 'Tipe file tidak diizinkan. Gunakan PDF/JPG/PNG.';
            } else {
                $dir = __DIR__ . '/letter/print/';
                if (!is_dir($dir)) mkdir($dir, 0775, true);
                $orig = $_FILES['file_surat']['name'];
                $ext  = strtolower(pathinfo($orig, PATHINFO_EXTENSION));
                $fname = 'user_' . $userId . '_' . time() . '.' . $ext;
                $dest  = $dir . $fname;
                if (!move_uploaded_file($tmp, $dest)) {
                    $errors[] = 'Tidak dapat menyimpan file surat.';
                } else {
                    $fileSuratPath = 'letter/print/' . $fname;
                }
            }
        }
    }

    // Handle lampiran upload (optional)
    $lampiran_path = null;
    $lampiran_orig = null;
    $lampiran_mime = null;
    $lampiran_size = null;
    if (isset($_FILES['lampiran']) && $_FILES['lampiran']['error'] !== UPLOAD_ERR_NO_FILE) {
        if ($_FILES['lampiran']['error'] !== UPLOAD_ERR_OK) {
            $errors[] = 'Upload lampiran gagal.';
        } else {
            $tmp  = $_FILES['lampiran']['tmp_name'];
            $mime = @mime_content_type($tmp) ?: ($_FILES['lampiran']['type'] ?? '');
            $size = (int)$_FILES['lampiran']['size'];
            $allowed = ['application/pdf', 'image/jpeg', 'image/png'];
            if (!in_array($mime, $allowed, true)) {
                $errors[] = 'Tipe lampiran tidak diizinkan. Gunakan PDF/JPG/PNG.';
            } elseif ($size > 25 * 1024 * 1024) {
                $errors[] = 'Lampiran melebihi 25 MB.';
            } else {
                $dir = __DIR__ . '/letter/uploads/lampiran/';
                if (!is_dir($dir)) mkdir($dir, 0775, true);
                $orig = $_FILES['lampiran']['name'];
                $ext  = strtolower(pathinfo($orig, PATHINFO_EXTENSION));
                $safe = preg_replace('/[^a-zA-Z0-9_\-\.]/', '_', pathinfo($orig, PATHINFO_FILENAME));
                $fname = $safe . '_' . time() . '.' . $ext;
                $dest  = $dir . $fname;
                if (!move_uploaded_file($tmp, $dest)) {
                    $errors[] = 'Tidak dapat menyimpan lampiran.';
                } else {
                    $lampiran_path = 'letter/uploads/lampiran/' . $fname;
                    $lampiran_orig = $orig;
                    $lampiran_mime = $mime;
                    $lampiran_size = $size;
                }
            }
        }
    }

    if (!$errors) {
        // Include API functions for proper sequence numbering
        require_once(__DIR__ . '/../api/classification.php');

        // Resolve tipe surat code
        $stmtTipe = $conn->prepare("SELECT kode FROM tipe_surat WHERE id = ?");
        $stmtTipe->bind_param('i', $tipe_surat_id);
        $stmtTipe->execute();
        $resTipe = $stmtTipe->get_result();
        $tipeRow = $resTipe->fetch_assoc();
        $tipeSuratKode = $tipeRow['kode'] ?? '';
        $stmtTipe->close();

        // Get proper sequence number from letter_sequences table (reserve=true to persist)
        $sequence_number = getNextSequenceNumber($conn, $tipe_surat_id, true);

        if ($tipeSuratKode === 'SK') {
            if ($jenis_surat_id > 0 && $klasifikasi_arsip_id > 0 && $subjenis_surat_id > 0) {
                // Get klasifikasi kode
                $stmtK = $conn->prepare("SELECT kode FROM klasifikasi_arsip WHERE id = ?");
                $stmtK->bind_param('i', $klasifikasi_arsip_id);
                $stmtK->execute();
                $resK = $stmtK->get_result();
                $klasRow = $resK->fetch_assoc();
                $klasifikasiKode = $klasRow['kode'] ?? '';
                $stmtK->close();

                // Get jenis kode
                $stmtJ = $conn->prepare("SELECT kode FROM jenis_surat WHERE id = ?");
                $stmtJ->bind_param('i', $jenis_surat_id);
                $stmtJ->execute();
                $resJ = $stmtJ->get_result();
                $jenisRow = $resJ->fetch_assoc();
                $jenisKode = $jenisRow['kode'] ?? '';
                $stmtJ->close();

                // Get subjenis kode
                $stmtS = $conn->prepare("SELECT kode FROM subjenis_surat WHERE id = ?");
                $stmtS->bind_param('i', $subjenis_surat_id);
                $stmtS->execute();
                $resS = $stmtS->get_result();
                $subRow = $resS->fetch_assoc();
                $subjenisKode = $subRow['kode'] ?? '';
                $stmtS->close();

                // Get jabatan based on dari_tipe
                $jabatan = '';
                if ($dari_tipe === 'pribadi' && $dari_pribadi > 0) {
                    $stmtJab = $conn->prepare("SELECT jabatan FROM users WHERE id = ?");
                    $stmtJab->bind_param('i', $dari_pribadi);
                    $stmtJab->execute();
                    $resJab = $stmtJab->get_result();
                    if ($rowJab = $resJab->fetch_assoc()) {
                        $jabatan = $rowJab['jabatan'] ?? '';
                    }
                    $stmtJab->close();
                } elseif ($dari_tipe === 'sendiri') {
                    $jabatan = $user['jabatan'] ?? '';
                }

                // Get kode jabatan from JSON mapping
                $kodeJabatan = getKodeJabatanFromJSON($jabatan);

                // Format: KODE.KLASIFIKASI.JENIS.SUBJENIS.SEQ or KLASIFIKASI.JENIS.SUBJENIS.SEQ
                $formattedSeq = str_pad($sequence_number, 2, '0', STR_PAD_LEFT);
                if ($kodeJabatan) {
                    $nomor_surat = "{$kodeJabatan}.{$klasifikasiKode}.{$jenisKode}.{$subjenisKode}.{$formattedSeq}";
                } else {
                    $nomor_surat = "{$klasifikasiKode}.{$jenisKode}.{$subjenisKode}.{$formattedSeq}";
                }
            } else {
                $errors[] = 'Jenis, Klasifikasi, dan Subjenis harus diisi untuk SK.';
                $nomor_surat = 'DRAFT-' . date('YmdHis') . '-' . $userId;
            }
        } elseif ($tipeSuratKode === 'KPTS') {
            $nomor_surat = str_pad($sequence_number, 3, '0', STR_PAD_LEFT) . '/KPTS-JPT/' . date('Y');
        } elseif ($tipeSuratKode === 'SE') {
            $nomor_surat = str_pad($sequence_number, 3, '0', STR_PAD_LEFT) . '/SE-DIR/' . date('Y');
        } else {
            $nomor_surat = 'DRAFT-' . date('YmdHis') . '-' . $userId;
        }

        $sql = "INSERT INTO office (
                    kepentingan, tipe_surat_id, jenis_surat_id, tempat_dibuat, klasifikasi_arsip_id, subjenis_surat_id,
                    nomor_surat, sequence_number, tanggal_surat, dari_tipe, dari_pribadi, dibuat_oleh, kepada, perihal, naskah, paraf,
                    lampiran_path, lampiran_orig_name, lampiran_mime, lampiran_size, file_surat, created_at, updated_at
                ) VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?, NOW(), NOW())";
        $st = $conn->prepare($sql);
        if (!$st) {
            $errors[] = 'Gagal menyiapkan query.';
        } else {
            // Set from POST values
            $dari_tipe = in_array($dari_tipe ?? '', ['internal', 'eksternal', 'pribadi', 'sendiri'], true) ? $dari_tipe : 'internal';
            $dari_pribadi_db = null;
            if ($dari_tipe === 'pribadi') {
                $dari_pribadi_db = $dari_pribadi > 0 ? $dari_pribadi : null;
            } elseif ($dari_tipe === 'sendiri') {
                $dari_pribadi_db = $userId;
            }
            // Created by should always be the logged-in user, not the signer
            $dibuat_oleh_final = $userId;
            // Note: $sequence_number already set from getNextSequenceNumber() above
            $jenis_surat_id_db = ($jenis_surat_id > 0) ? $jenis_surat_id : null;
            $klasifikasi_arsip_id_db = ($klasifikasi_arsip_id > 0) ? $klasifikasi_arsip_id : null;
            $subjenis_surat_id_db = ($subjenis_surat_id > 0) ? $subjenis_surat_id : null;
            // types: kepentingan(s), tipe_surat_id(i), tempat_dibuat(s), nomor_surat(s), tanggal_surat(s), dari_tipe(s), dibuat_oleh(i), kepada(s), perihal(s), naskah(s), file_surat(s)
            $lampiran_path_json = $lampiran_orig ? json_encode([$lampiran_path], JSON_UNESCAPED_SLASHES) : json_encode([], JSON_UNESCAPED_SLASHES);
            $lampiran_size_int = $lampiran_size ?? 0;
            $st->bind_param(
                'siisississiisssssssis',
                $kepentingan,
                $tipe_surat_id,
                $jenis_surat_id_db,
                $tempat_dibuat,
                $klasifikasi_arsip_id_db,
                $subjenis_surat_id_db,
                $nomor_surat,
                $sequence_number,
                $tanggal_surat,
                $dari_tipe,
                $dari_pribadi_db,
                $dibuat_oleh_final,
                $kepada,
                $perihal,
                $naskah_html,
                $paraf,
                $lampiran_path_json,
                $lampiran_orig,
                $lampiran_mime,
                $lampiran_size_int,
                $fileSuratPath
            );
            if (!$st->execute()) {
                $errors[] = 'Gagal menyimpan data: ' . $st->error;
            } else {
                $insertedId = $conn->insert_id;

                // Auto-initialize approval chain if penandatangan is set
                if ($dari_pribadi_db !== null && $dari_pribadi_db > 0) {
                    initializeApprovalChainForLetter($conn, $insertedId, $dibuat_oleh_final, $dari_pribadi_db);
                }

                // Note: Sequence already persisted by getNextSequenceNumber() with reserve=true
                // Redirect to record list after successful creation
                header('Location: record-letter.php?success=1');
                exit();
            }
            $st->close();
        }
    }
}

?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Buat Surat (User)</title>
    <link href="/jasa_marga/img/baru/logojp.png" rel="icon">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css">
    <style>
        .greeting {
            text-align: center;
            animation: fadeIn 2s ease-in-out;
        }

        .icon {
            margin-top: 40px;
            font-size: 50px;
            margin-bottom: 0px;
            animation: bounce 1.5s infinite;
        }

        @keyframes bounce {

            0%,
            100% {
                transform: translateY(0);
            }

            50% {
                transform: translateY(-10px);
            }
        }

        .sidebar {
            background-color: #0c50e1;
            color: white;
            min-height: 100vh;
            padding: 15px;
            width: 250px;
            flex-shrink: 0;
            overflow-y: auto;
        }

        .sidebar a {
            color: white;
            text-decoration: none;
            display: block;
            padding: 10px;
            margin: 5px 0;
            border-radius: 5px;
            transition: all 0.3s ease;
        }

        .sidebar a:hover,
        .sidebar a.active {
            background-color: #ffc107;
            color: #0047ab;
            transform: translateX(5px);
        }

        .btn-success {
            background-color: #ffc107;
            border-color: #ffc107;
            color: #0047ab;
            font-weight: bold;
        }

        .btn-success:hover {
            background-color: #e0a800;
            border-color: #e0a800;
            color: #0047ab;
        }

        .table thead {
            background-color: #0047ab;
            color: #ffc107;
            text-transform: uppercase;
            font-weight: bold;
        }

        .table thead th {
            background-color: #0c50e1;
            color: #ffc107;
        }

        .sidebar-animated {
            background-color: #0c50e1;
            padding: 20px 10px;
            font-family: Arial, sans-serif;
            height: 100vh;
            width: 250px;
            transition: all 0.3s ease;
            max-height: 100vh;
            overflow-y: auto;
            overflow-x: hidden;
            padding-bottom: 20px;
            margin-bottom: 0;
            scrollbar-width: thin;
        }

        .sidebar-animated::-webkit-scrollbar {
            width: 8px;
        }

        .sidebar-animated::-webkit-scrollbar-thumb {
            background-color: #e1b12c;
            border-radius: 10px;
        }

        .sidebar-animated::-webkit-scrollbar-thumb:hover {
            background-color: #ffc107;
        }

        .sidebar-animated a {
            color: #FFFFFFFF;
            text-decoration: none;
            font-size: 18px;
            display: flex;
            align-items: center;
            padding: 10px;
            border-radius: 5px;
            transition: background-color 0.3s ease, transform 0.3s ease;
        }

        .sidebar-animated a:hover {
            background-color: #ffc107;
            color: #fff;
            transform: translateX(5px);
        }

        .sidebar-animated a i {
            margin-right: 10px;
        }

        .sidebar-animated .submenu-toggle {
            cursor: pointer;
            transition: color 0.3s;
        }

        .sidebar-animated .submenu-toggle:hover .arrow-icon {
            transform: rotate(90deg);
            color: #0c50e1;
        }

        .sidebar-animated .submenu {
            padding-left: 20px;
            overflow: hidden;
            transition: max-height 0.4s ease;
        }

        .sidebar-animated .collapse.show {
            max-height: 300px;
            transition: max-height 0.4s ease-in;
        }

        .arrow-icon {
            margin-left: auto;
            transition: transform 0.3s;
        }

        .submenu a {
            font-size: 0.9rem;
            color: #FFFFFFFF;
            padding: 8px 10px;
            transition: color 0.3s ease, transform 0.3s ease;
        }

        .submenu a:hover {
            color: #001eff;
            transform: translateX(5px);
        }

        .sidebar-animated a.active {
            background-color: #ffc107;
            color: #fff;
            font-weight: bold;
        }

        .sidebar-animated a.active i {
            color: #fff;
        }

        .navbar-custom {
            background-color: #0c50e1;
            transition: background-color 0.3s ease;
            padding: 15px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }

        .navbar-custom .navbar-brand img {
            transition: transform 0.3s ease;
        }

        .navbar-custom .navbar-brand:hover img {
            transform: scale(1.05);
        }

        .toggle-sidebar-btn {
            color: #fff;
            font-size: 1.3rem;
            cursor: pointer;
            margin-left: 10px;
            transition: color 0.3s ease;
        }

        .toggle-sidebar-btn:hover {
            color: #ffc107;
        }

        @keyframes fadeInUp {
            0% {
                opacity: 0;
                transform: translateY(20px);
            }

            100% {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .highlight-card {
            border-radius: 15px;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
            animation: fadeInUp 0.8s ease-in-out;
        }

        .highlight-card:hover {
            transform: translateY(-10px);
            box-shadow: 0 10px 20px rgba(0, 0, 0, 0.2);
        }

        .highlight-card h3 {
            font-size: 2.5rem;
            font-weight: bold;
            margin: 0;
        }

        .highlight-card p {
            font-size: 1.1rem;
            margin-top: 5px;
            opacity: 0.9;
        }

        .highlight-card .card-footer {
            background: rgba(0, 0, 0, 0.1);
            border-top: 1px solid rgba(255, 255, 255, 0.2);
        }

        .highlight-card .card-footer a {
            text-decoration: none;
            font-weight: bold;
            transition: color 0.3s ease;
        }

        .highlight-card .card-footer a:hover {
            color: #ffd700;
        }

        .highlight-card i {
            animation: bounce 1.5s infinite;
        }

        .pagination .page-link {
            color: #007bff;
        }

        .pagination .active .page-link {
            background-color: #007bff;
            border-color: #007bff;
            color: white;
        }

        .search-container-new {
            position: relative;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .search-box-new {
            display: flex;
            align-items: center;
            transition: all 0.3s ease;
        }

        .search-input-new {
            border-radius: 25px;
            padding: 8px 12px;
            border: 1px solid #ced4da;
            background-color: #ffc107;
            color: #001eff;
            transition: width 0.4s ease, box-shadow 0.3s ease;
            width: 184px;
        }

        .search-input-new::placeholder {
            color: #001eff;
            font-style: italic;
            opacity: 0.8;
        }

        .search-input-new:focus {
            width: 250px;
            background-color: #ffffff;
            border-color: #ffc107;
            box-shadow: 0 0 8px rgba(255, 193, 7, 0.6);
            outline: none;
        }

        .search-btn-new {
            background-color: #001eff;
            color: #fff;
            border: none;
            border-radius: 20px;
            padding: 6px 20px;
            font-size: 14px;
            cursor: pointer;
            transition: background-color 0.3s ease, transform 0.3s ease;
            display: none;
        }

        .search-btn-new.show {
            display: inline-block;
        }

        .stepper {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 14px;
            margin: 25px 0 24px;
        }

        .stepper .step {
            display: flex;
            align-items: center;
            gap: 10px;
            position: relative;
        }

        .stepper .step:not(:last-child)::after {
            content: "";
            width: 300px;
            height: 4px;
            background: #0c50e1;
            opacity: .35;
            border-radius: 4px;
            margin-left: 12px;
            margin-right: 4px;
        }

        .stepper .step.completed:not(:last-child)::after {
            background: #ffc107;
            opacity: 1;
        }

        .step-circle {
            width: 36px;
            height: 36px;
            border-radius: 50%;
            display: grid;
            place-items: center;
            font-weight: 700;
            border: 3px solid #0c50e1;
            color: #0c50e1;
            background: #fff;
        }

        .step.active .step-circle {
            background: #ffc107;
            border-color: #ffc107;
            color: #001eff;
            box-shadow: 0 0 0 4px rgba(255, 193, 7, .25);
        }

        .step.completed .step-circle {
            background: #0c50e1;
            border-color: #0c50e1;
            color: #ffc107;
        }

        .step-label {
            font-weight: 600;
            color: #001eff;
        }

        .step-desc {
            font-size: .85rem;
            color: #6c757d;
            margin-top: -2px;
        }

        .card-header {
            background: #0c50e1;
            color: #ffc107;
            border-bottom: 0;
            font-weight: 700;
        }

        .card {
            border: 1px solid rgba(0, 0, 0, .06);
            box-shadow: 0 6px 18px rgba(0, 0, 0, .06);
            border-radius: 16px;
        }

        .btn-print-custom {
            background: #ffc107;
            color: #0c50e1;
            font-weight: 700;
        }

        .btn-print-custom:hover {
            background: #0c50e1;
            color: #ffc107;
            border-color: #0c50e1;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(12, 80, 225, 0.45);
        }

        .btn-save-custom {
            background: #0c50e1;
            color: #ffc107;
            font-weight: 700;
        }

        .btn-save-custom:hover {
            background: #ffc107;
            color: #001eff;
            border-color: #ffc107;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(255, 193, 7, 0.45);
        }

        .btn-print-custom,
        .btn-save-custom {
            transition: all 0.25s ease;
        }

        .form-control,
        .form-select {
            border-radius: 10px;
        }

        .badge-kode {
            font-size: .75rem;
            background: #e9f1ff;
            color: #0c50e1;
            border: 1px solid #0c50e1;
            border-radius: 8px;
            padding: .25rem .5rem;
        }

        .flex-grow-1 {
            flex: 1;
        }

        .submenu-toggle:hover {
            background-color: rgba(255, 193, 7, 0.1);
        }
    </style>
</head>

<body>
    <div class="d-flex">
        <?php include __DIR__ . '/../includes/sidebar-user.php'; ?>
        <!-- Surat Content -->
        <div id="mainContent" class="container-fluid py-3">
            <div class="d-flex align-items-center justify-content-between mb-2">
                <h1 class="h3 m-0">Buat Surat</h1>
                <span class="badge-kode">Tanggal:
                    <?php
                    $dateValue = $_POST['tanggal_surat'] ?? date('Y-m-d');
                    if ($dateValue) {
                        // Convert to Indonesian format "20 November 2025"
                        $timestamp = strtotime($dateValue);
                        if ($timestamp !== false) {
                            $bulanIndonesia = [
                                1 => 'Januari',
                                2 => 'Februari',
                                3 => 'Maret',
                                4 => 'April',
                                5 => 'Mei',
                                6 => 'Juni',
                                7 => 'Juli',
                                8 => 'Agustus',
                                9 => 'September',
                                10 => 'Oktober',
                                11 => 'November',
                                12 => 'Desember'
                            ];
                            $hari = date('j', $timestamp);
                            $bulan = $bulanIndonesia[date('n', $timestamp)];
                            $tahun = date('Y', $timestamp);
                            echo htmlspecialchars("$hari $bulan $tahun", ENT_QUOTES, 'UTF-8');
                        } else {
                            echo htmlspecialchars($dateValue, ENT_QUOTES, 'UTF-8');
                        }
                    } else {
                        $timestamp = time();
                        $bulanIndonesia = [
                            1 => 'Januari',
                            2 => 'Februari',
                            3 => 'Maret',
                            4 => 'April',
                            5 => 'Mei',
                            6 => 'Juni',
                            7 => 'Juli',
                            8 => 'Agustus',
                            9 => 'September',
                            10 => 'Oktober',
                            11 => 'November',
                            12 => 'Desember'
                        ];
                        $hari = date('j', $timestamp);
                        $bulan = $bulanIndonesia[date('n', $timestamp)];
                        $tahun = date('Y', $timestamp);
                        echo htmlspecialchars("$hari $bulan $tahun", ENT_QUOTES, 'UTF-8');
                    }
                    ?></span>
            </div>

            <!-- STEPPER -->
            <nav class="stepper" aria-label="Progress">
                <div class="step active" data-step="1">
                    <div class="step-circle">1</div>
                    <div>
                        <div class="step-label">Formulir Surat</div>
                    </div>
                </div>
                <div class="step" data-step="2">
                    <div class="step-circle">2</div>
                    <div>
                        <div class="step-label">Konten Surat</div>
                    </div>
                </div>
            </nav>

            <form method="post" action="" enctype="multipart/form-data" autocomplete="off" id="formNaskah">
                <!-- ============ STEP 1 ============ -->
                <section class="step-pane" data-step="1">
                    <!-- CARD 1 (IDENTITAS SURAT) -->
                    <div class="card mb-4">
                        <div class="card-header">Identitas Surat</div>
                        <div class="card-body row g-3">
                            <!-- Kepentingan -->
                            <div class="col-12">
                                <label class="form-label d-block">Kepentingan <span class="text-danger">*</span></label>
                                <div class="d-flex gap-4">
                                    <label class="form-check">
                                        <input type="radio" class="form-check-input" name="kepentingan" value="internal"
                                            <?php echo (($_POST['kepentingan'] ?? '') === 'internal') ? 'checked' : ''; ?>>
                                        <span class="form-check-label">Internal</span>
                                    </label>
                                    <label class="form-check">
                                        <input type="radio" class="form-check-input" name="kepentingan"
                                            value="eksternal"
                                            <?php echo (($_POST['kepentingan'] ?? '') === 'eksternal') ? 'checked' : ''; ?>>
                                        <span class="form-check-label">Eksternal</span>
                                    </label>
                                </div>
                            </div>

                            <!-- Tipe Surat -->
                            <div class="col-12">
                                <label for="tipe_surat_id" class="form-label">Tipe Surat <span
                                        class="text-danger">*</span></label>
                                <select id="tipe_surat_id" name="tipe_surat_id" class="form-select" required>
                                    <option value="" disabled selected>-- Pilih Tipe Surat --</option>
                                    <!-- Will be populated via JavaScript -->
                                </select>
                            </div>

                            <!-- Classification Fields (Only for Surat Keluar) -->
                            <div id="classification-fields" class="d-none">
                                <!-- Klasifikasi (First) -->
                                <div class="col-12 mb-3">
                                    <label for="klasifikasi_arsip_new" class="form-label">Klasifikasi <span
                                            class="text-danger">*</span></label>
                                    <select id="klasifikasi_arsip_new" name="klasifikasi_arsip_new" class="form-select">
                                        <option value="" disabled selected>-- Pilih Klasifikasi --</option>
                                    </select>
                                </div>

                                <!-- Jenis Surat (Second) -->
                                <div class="col-12 mb-3">
                                    <label for="jenis_surat_new" class="form-label">Jenis Surat <span
                                            class="text-danger">*</span></label>
                                    <select id="jenis_surat_new" name="jenis_surat_new" class="form-select">
                                        <option value="" disabled selected>-- Pilih Klasifikasi Terlebih Dahulu --
                                        </option>
                                    </select>
                                </div>

                                <!-- Subjenis Surat (Third) -->
                                <div class="col-12 mb-3">
                                    <label for="subjenis_surat_id" class="form-label">Subjenis Surat <span
                                            class="text-danger">*</span></label>
                                    <select id="subjenis_surat_id" name="subjenis_surat_id" class="form-select">
                                        <option value="" disabled selected>-- Pilih Jenis Surat Terlebih Dahulu --
                                        </option>
                                    </select>
                                </div>
                            </div>

                            <!-- Legacy Jenis Surat (Hidden by default) -->
                            <div id="legacy-jenis-field" class="col-12 d-none">
                                <label for="jenis_surat_id" class="form-label">Jenis Surat (Legacy) <span
                                        class="text-danger">*</span></label>
                                <select id="jenis_surat_id" name="jenis_surat_id" class="form-select">
                                    <option value="" disabled
                                        <?php echo empty($_POST['jenis_surat_id']) ? 'selected' : ''; ?>>-- Pilih Jenis
                                        Surat --</option>
                                    <?php foreach ($jenisSurat as $row): ?>
                                        <option value="<?php echo (int)$row['id']; ?>"
                                            data-kode="<?php echo htmlspecialchars($row['kode'], ENT_QUOTES, 'UTF-8'); ?>"
                                            <?php echo ((string)($_POST['jenis_surat_id'] ?? '') === (string)$row['id']) ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($row['nama'], ENT_QUOTES, 'UTF-8'); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>

                        <!-- Legacy Klasifikasi (Hidden by default) -->
                        <div class="row mb-3">
                            <div id="legacy-klasifikasi-field" class="col-12 d-none">
                                <label for="klasifikasi_arsip_id" class="form-label">Klasifikasi (Legacy) <span
                                        class="text-danger">*</span></label>
                                <select id="klasifikasi_arsip_id" name="klasifikasi_arsip_id" class="form-select">
                                    <option value="" disabled
                                        <?php echo empty($_POST['klasifikasi_arsip_id']) ? 'selected' : ''; ?>>-- Pilih
                                        Klasifikasi --</option>
                                    <?php foreach ($klasifikasi as $row): ?>
                                        <option value="<?php echo (int)$row['id']; ?>"
                                            data-kode="<?php echo htmlspecialchars($row['kode'], ENT_QUOTES, 'UTF-8'); ?>"
                                            <?php echo ((string)($_POST['klasifikasi_arsip_id'] ?? '') === (string)$row['id']) ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($row['nama'], ENT_QUOTES, 'UTF-8'); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>

                        <div class="row mb-3">
                            <!-- Tempat -->
                            <div class="col-12">
                                <label for="tempat_dibuat" class="form-label">Tempat Surat Dibuat <span
                                        class="text-danger">*</span></label>
                                <input type="text" id="tempat_dibuat" name="tempat_dibuat" class="form-control" required
                                    value="<?php echo htmlspecialchars($_POST['tempat_dibuat'] ?? '', ENT_QUOTES, 'UTF-8'); ?>">
                            </div>

                            <!-- Pejabat yang Menandatangani (moved from Subyek card) -->
                            <div class="col-12">
                                <label class="form-label d-block">Pejabat yang Menandatangani <span
                                        class="text-danger">*</span></label>
                                <div class="d-flex flex-wrap gap-4">
                                    <label class="form-check">
                                        <input type="radio" class="form-check-input" name="dari_tipe" value="pribadi"
                                            <?php echo (($_POST['dari_tipe'] ?? '') === 'pribadi') ? 'checked' : ''; ?>>
                                        <span class="form-check-label">Pejabat</span>
                                    </label>
                                    <label class="form-check">
                                        <input type="radio" class="form-check-input" name="dari_tipe" value="sendiri"
                                            id="dariTipeSendiri"
                                            <?php echo (($_POST['dari_tipe'] ?? 'sendiri') === 'sendiri') ? 'checked' : ''; ?>>
                                        <span class="form-check-label">Diri Sendiri</span>
                                    </label>
                                </div>
                            </div>

                            <!-- Nama Yang Menandatangani Dropdown (filtered to 16 authorized signers) -->
                            <div class="col-12" id="wrapPribadi" style="display:none;">
                                <label for="dari_pribadi" class="form-label">Nama Yang Menandatangani</label>
                                <select id="dari_pribadi" name="dari_pribadi" class="form-select">
                                    <option value="" disabled selected>-- Pilih Pejabat --</option>
                                    <!-- Will be populated via JavaScript with authorized signers only -->
                                </select>
                            </div>
                            <input type="hidden" id="user_nama" value="<?= htmlspecialchars($name, ENT_QUOTES) ?>">
                            <input type="hidden" id="user_jabatan"
                                value="<?= htmlspecialchars($user['jabatan'] ?? '', ENT_QUOTES) ?>">
                            <input type="hidden" id="user_id" value="<?= (int)$userId ?>">
                            <input type="hidden" id="user_tipe"
                                value="<?= htmlspecialchars($user['tipe'] ?? '', ENT_QUOTES) ?>">

                            <!-- Backdate Checkbox -->
                            <div class="col-12">
                                <div class="form-check">
                                    <input type="checkbox" class="form-check-input" id="backdate_checkbox"
                                        name="is_backdate">
                                    <label class="form-check-label" for="backdate_checkbox">Backdate</label>
                                </div>
                            </div>

                            <!-- Nomor Surat with Previous Number Indicator -->
                            <div class="col-12">
                                <label for="nomor_surat" class="form-label">
                                    Nomor Surat <span class="text-danger">*</span>
                                    <small id="prev_nomor_indicator" class="text-muted ms-2"
                                        style="font-weight: normal;"></small>
                                </label>
                                <input type="text" id="nomor_surat" name="nomor_surat" class="form-control" required
                                    readonly
                                    value="<?php echo htmlspecialchars($_POST['nomor_surat'] ?? '', ENT_QUOTES, 'UTF-8'); ?>">

                                <!-- Backdate Edit Fields (hidden by default) -->
                                <div id="backdate_fields" class="d-none mt-2">
                                    <div class="row g-2">
                                        <div class="col-md-4">
                                            <label for="backdate_sequence" class="form-label small">Nomor Urut</label>
                                            <input type="number" id="backdate_sequence" name="backdate_sequence"
                                                class="form-control form-control-sm" min="1" placeholder="Urutan">
                                        </div>
                                        <div class="col-md-4" id="backdate_year_field" style="display:none;">
                                            <label for="backdate_year" class="form-label small">Tahun</label>
                                            <input type="number" id="backdate_year" name="backdate_year"
                                                class="form-control form-control-sm" min="2020" max="2099"
                                                value="<?= date('Y') ?>" placeholder="Tahun">
                                        </div>
                                    </div>
                                    <small class="text-muted" id="backdate_hint">Masukkan nomor urut untuk
                                        backdate</small>
                                </div>
                            </div>

                            <!-- Tanggal Surat -->
                            <div class="col-12">
                                <label for="tanggal_surat" class="form-label">Tanggal Surat <span
                                        class="text-danger">*</span></label>
                                <input type="date" id="tanggal_surat" name="tanggal_surat" class="form-control" required
                                    value="<?php echo htmlspecialchars($_POST['tanggal_surat'] ?? '', ENT_QUOTES, 'UTF-8'); ?>">
                            </div>

                            <!-- Paraf field removed - now auto-generated from approval chain -->
                        </div>
                    </div>

                    <!-- CARD 2 (SUBYEK) - Only Kepada field remains here -->
                    <div class="card mb-4">
                        <div class="card-header">Subyek</div>
                        <div class="card-body row g-3">
                            <!-- Kepada -->
                            <div class="col-12" id="kepada-field">
                                <label for="kepada" class="form-label">Kepada <span class="text-danger">*</span></label>
                                <input type="text" id="kepada" name="kepada" class="form-control" required
                                    value="<?php echo htmlspecialchars($_POST['kepada'] ?? '', ENT_QUOTES, 'UTF-8'); ?>">
                            </div>
                        </div>
                    </div>

                    <!-- CARD 3 (INFORMASI SURAT) -->
                    <div class="card mb-3">
                        <div class="card-header">Informasi Surat</div>
                        <div class="card-body row g-3">
                            <!-- Perihal -->
                            <div class="col-12">
                                <label for="perihal" class="form-label">Perihal <span
                                        class="text-danger">*</span></label>
                                <input type="text" id="perihal" name="perihal" class="form-control" required
                                    value="<?php echo htmlspecialchars($_POST['perihal'] ?? '', ENT_QUOTES, 'UTF-8'); ?>">
                            </div>

                            <!-- Lampiran -->
                            <div class="col-12">
                                <label for="lampiran" class="form-label">Lampiran</label>
                                <input type="file" id="lampiran" name="lampiran" class="form-control"
                                    accept=".pdf,.jpg,.jpeg,.png">

                                <input type="hidden" id="lampiran_url" name="lampiran_url" value="">
                                <input type="hidden" id="lampiran_mime" name="lampiran_mime" value="">
                            </div>
                        </div>
                    </div>

                    <!-- TOMBOL -->
                    <div class="d-flex justify-content-end gap-2">
                        <button type="button" class="btn btn-save-custom" id="toStep2">Lanjutkan</button>
                    </div>
                </section>

                <!-- ============ STEP 2 ============ -->
                <section class="step-pane d-none" data-step="2">
                    <div class="card mb-3">
                        <div class="card-header d-flex align-items-center justify-content-between">
                            <span style="font-size: 18px;">Isi Surat</span>
                            <!-- <div class="d-flex gap-2">
                                <button type="button" class="btn btn-outline-warning color-warning btn-md" id="printDoc">
                                Preview Hasil Surat
                                </button>
                            </div> -->
                        </div>
                        <div class="card-body p-0">
                            <textarea id="editorNaskah" name="naskah_html"></textarea>
                        </div>
                    </div>

                    <input type="hidden" name="__final_submit" value="1" />

                    <div class="d-flex justify-content-between align-items-center flex-wrap gap-2 mt-3">
                        <button type="button" class="btn btn-danger" id="backTo1">Kembali</button>

                        <div class="d-flex gap-2">
                            <button type="button" id="btnUploadPdf" class="btn btn-print-custom">
                                Cetak & Upload PDF
                            </button>
                            <button type="submit" id="btnSave" class="btn btn-save-custom">
                                Simpan
                            </button>
                        </div>
                    </div>

                    <input type="hidden" name="file_surat" id="fileSuratPath" value="">

                </section>
            </form>
        </div>
    </div>
    </div>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/tinymce@6/tinymce.min.js"></script>
    <script>
        // Ensure listener exists early and will enable the button immediately if html2pdf is already present
        document.addEventListener('html2pdf:loaded', function() {
            try {
                window._html2pdf_loaded = true;
                const b = document.getElementById('btnUploadPdf');
                if (b) b.disabled = false;
            } catch (e) {
                console.error(e);
            }
        });

        // Dynamically load html2pdf: try local first, then CDN fallback. Dispatch event on success.
        (function() {
            function loadScript(src, onload, onerror) {
                var s = document.createElement('script');
                s.src = src;
                s.async = false;
                s.onload = onload;
                s.onerror = onerror;
                document.head.appendChild(s);
            }

            loadScript('/jasa_marga/node_modules/html2pdf.js/dist/html2pdf.bundle.min.js', function() {
                document.dispatchEvent(new Event('html2pdf:loaded'));
            }, function() {
                console.warn('local html2pdf not found, loading CDN');
                loadScript('https://cdnjs.cloudflare.com/ajax/libs/html2pdf.js/0.10.1/html2pdf.bundle.min.js',
                    function() {
                        document.dispatchEvent(new Event('html2pdf:loaded'));
                    },
                    function() {
                        console.error('Failed to load html2pdf from both local and CDN');
                    });
            });
        })();
    </script>
    <script>
        // Form element initialization - NOW ALL ELEMENTS ARE LOADED
        (function() {
            const tipeSelect = document.getElementById('tipe_surat_id');
            const classificationFields = document.getElementById('classification-fields');
            const jenisSuratSelect = document.getElementById('jenis_surat_new');
            const klasifikasiSelect = document.getElementById('klasifikasi_arsip_new');
            const subjenisSelect = document.getElementById('subjenis_surat_id');
            const nomorSuratInput = document.getElementById('nomor_surat');

            // Data from PHP backend
            const tipeSuratList = <?= json_encode($tipeSuratList) ?>;
            const jenisSuratList = <?= json_encode($jenisSurat) ?>;
            const klasifikasiList = <?= json_encode($klasifikasi) ?>;
            const subjenisLists = <?= json_encode($subjenisList) ?>;

            // Populate tipe_surat_id select with options
            function populateTipeSurat() {
                if (!tipeSelect || tipeSuratList.length === 0) return;

                const selectedValue = tipeSelect.value;
                const existingOptions = tipeSelect.querySelectorAll('option:not([disabled])');

                // Only populate if options don't exist yet (avoid duplication)
                if (existingOptions.length <= 1) {
                    tipeSuratList.forEach(tipe => {
                        const option = document.createElement('option');
                        option.value = tipe.id;
                        option.textContent = tipe.nama;
                        option.dataset.kode = tipe.kode;
                        tipeSelect.appendChild(option);
                    });
                }

                if (selectedValue) {
                    tipeSelect.value = selectedValue;
                }
            }

            // Populate klasifikasi options
            function populateKlasifikasi() {
                if (klasifikasiSelect.options.length === 1) {
                    klasifikasiList.forEach(klasif => {
                        const option = document.createElement('option');
                        option.value = klasif.id;
                        option.textContent = `${klasif.kode} - ${klasif.nama}`;
                        option.dataset.kode = klasif.kode;
                        klasifikasiSelect.appendChild(option);
                    });
                }
            }

            // Populate jenis surat filtered by klasifikasi
            function populateJenisByKlasifikasi() {
                const klasifikasiId = parseInt(klasifikasiSelect.value) || 0;

                // Clear jenis dropdown
                jenisSuratSelect.innerHTML = '<option value="" disabled selected>-- Pilih Jenis Surat --</option>';
                // Clear subjenis dropdown
                subjenisSelect.innerHTML =
                    '<option value="" disabled selected>-- Pilih Subjenis Surat Terlebih Dahulu --</option>';

                if (klasifikasiId <= 0) return;

                // Filter jenis by selected klasifikasi
                const filteredJenis = jenisSuratList.filter(jenis =>
                    parseInt(jenis.id_klasifikasi) === klasifikasiId
                );

                if (filteredJenis.length === 0) {
                    jenisSuratSelect.innerHTML =
                        '<option value="" disabled selected>-- Tidak ada Jenis Surat --</option>';
                    return;
                }

                filteredJenis.forEach(jenis => {
                    const option = document.createElement('option');
                    option.value = jenis.id;
                    option.textContent = `${jenis.kode} - ${jenis.nama}`;
                    option.dataset.kode = jenis.kode;
                    jenisSuratSelect.appendChild(option);
                });
            }

            // Populate subjenis surat filtered by jenis
            function populateSubjenisByJenis() {
                const jenisId = parseInt(jenisSuratSelect.value) || 0;

                // Clear subjenis dropdown
                subjenisSelect.innerHTML = '<option value="" disabled selected>-- Pilih Subjenis Surat --</option>';

                if (jenisId <= 0) return;

                // Filter subjenis by selected jenis
                const filteredSubjenis = subjenisLists.filter(subjenis =>
                    parseInt(subjenis.id_jenis) === jenisId
                );

                if (filteredSubjenis.length === 0) {
                    subjenisSelect.innerHTML =
                        '<option value="" disabled selected>-- Tidak ada Subjenis Surat --</option>';
                    return;
                }

                filteredSubjenis.forEach(subjenis => {
                    const option = document.createElement('option');
                    option.value = subjenis.id;
                    option.textContent = `${subjenis.kode} - ${subjenis.nama}`;
                    option.dataset.kode = subjenis.kode;
                    subjenisSelect.appendChild(option);
                });
            }

            // Toggle SK-specific fields
            function toggleClassificationFields() {
                if (!tipeSelect) return;
                const selectedOption = tipeSelect.options[tipeSelect.selectedIndex];
                const kode = selectedOption?.dataset.kode;

                if (kode === 'SK') {
                    classificationFields.classList.remove('d-none');
                    // Populate klasifikasi first
                    populateKlasifikasi();
                    // Initialize jenis and subjenis as empty
                    jenisSuratSelect.innerHTML =
                        '<option value="" disabled selected>-- Pilih Klasifikasi Terlebih Dahulu --</option>';
                    subjenisSelect.innerHTML =
                        '<option value="" disabled selected>-- Pilih Jenis Surat Terlebih Dahulu --</option>';
                } else {
                    classificationFields.classList.add('d-none');
                }

                generateNomorSurat();
            }

            // Generate nomor surat via API (uses sequence from letter_sequences table)
            async function generateNomorSurat() {
                if (!nomorSuratInput) return;

                const tipeSuratId = tipeSelect?.value;
                if (!tipeSuratId) {
                    nomorSuratInput.value = '';
                    return;
                }

                try {
                    const params = new URLSearchParams({
                        action: 'generate_nomor_surat',
                        tipe_surat_id: tipeSuratId
                    });

                    // Add classification data for Surat Keluar
                    const selectedTipeOption = tipeSelect.selectedOptions[0];
                    if (selectedTipeOption && selectedTipeOption.dataset.kode === 'SK') {
                        const klasifikasiOption = klasifikasiSelect?.selectedOptions[0];
                        const jenisOption = jenisSuratSelect?.selectedOptions[0];
                        const subjenisId = subjenisSelect?.value;

                        if (klasifikasiOption && jenisOption && subjenisId) {
                            params.append('klasifikasi_kode', klasifikasiOption.dataset.kode);
                            params.append('jenis_kode', jenisOption.dataset.kode);
                            params.append('subjenis_id', subjenisId);

                            // Get jabatan from dropdown "Pejabat yang Menandatangani"
                            let jabatan = '';
                            const dariTipeChecked = document.querySelector('input[name="dari_tipe"]:checked');

                            if (dariTipeChecked && dariTipeChecked.value === 'pribadi' && dariPribadiSelect &&
                                dariPribadiSelect.value) {
                                const selectedOption = dariPribadiSelect.selectedOptions[0];
                                jabatan = selectedOption?.dataset?.jabatan || '';
                            } else if (dariTipeChecked && dariTipeChecked.value === 'sendiri') {
                                jabatan = document.getElementById('user_jabatan')?.value || '';
                                if (!jabatan && dariPribadiSelect && dariPribadiSelect.value) {
                                    const selectedOption = dariPribadiSelect.selectedOptions[0];
                                    jabatan = selectedOption?.dataset?.jabatan || '';
                                }
                            }

                            params.append('jabatan', jabatan || '');
                        } else {
                            // Not all fields selected yet
                            nomorSuratInput.value = '';
                            return;
                        }
                    }

                    const response = await fetch(`api_classification.php?${params}`);
                    const data = await response.json();

                    if (data.success) {
                        nomorSuratInput.value = data.data.nomor_surat;
                    } else {
                        console.warn('Error generating nomor surat:', data.message);
                        nomorSuratInput.value = '';
                    }
                } catch (error) {
                    console.error('Error generating nomor surat:', error);
                    nomorSuratInput.value = '';
                }
            }

            // Event listeners
            if (tipeSelect) {
                tipeSelect.addEventListener('change', toggleClassificationFields);
                klasifikasiSelect?.addEventListener('change', async () => {
                    populateJenisByKlasifikasi();
                    await generateNomorSurat();
                });
                jenisSuratSelect?.addEventListener('change', async () => {
                    populateSubjenisByJenis();
                    await generateNomorSurat();
                });
                subjenisSelect?.addEventListener('change', async () => await generateNomorSurat());
            }

            // Authorized signers and backdate functionality
            const dariPribadiSelect = document.getElementById('dari_pribadi');
            const backdateCheckbox = document.getElementById('backdate_checkbox');
            const backdateFields = document.getElementById('backdate_fields');
            const backdateSequence = document.getElementById('backdate_sequence');
            const backdateYear = document.getElementById('backdate_year');
            const backdateYearField = document.getElementById('backdate_year_field');
            const backdateHint = document.getElementById('backdate_hint');
            const prevNomorIndicator = document.getElementById('prev_nomor_indicator');
            const dariTipeSendiri = document.getElementById('dariTipeSendiri');
            const wrapPribadi = document.getElementById('wrapPribadi');

            let authorizedSigners = [];
            let currentUserAuthorized = false;
            const currentUserName = document.getElementById('user_nama')?.value || '';
            const currentUserId = document.getElementById('user_id')?.value || '';

            // Load authorized signers
            async function loadAuthorizedSigners() {
                try {
                    const response = await fetch(
                        `api_classification.php?action=get_authorized_signers&current_user_id=${encodeURIComponent(currentUserId)}`
                    );
                    const data = await response.json();

                    if (data.success && dariPribadiSelect) {
                        authorizedSigners = data.data;
                        currentUserAuthorized = !!data.current_user_authorized;
                        dariPribadiSelect.innerHTML =
                            '<option value="" disabled selected>-- Pilih Pejabat --</option>';

                        data.data.forEach(signer => {
                            const option = document.createElement('option');
                            option.value = signer.user_id || signer.id;
                            option.textContent = signer.nama + (signer.jabatan ? ` - ${signer.jabatan}` :
                                '');
                            option.dataset.jabatan = signer.jabatan || '';
                            option.dataset.nama = signer.nama;
                            dariPribadiSelect.appendChild(option);
                        });

                        checkDiriSendiriEligibility();
                    }
                } catch (error) {
                    console.error('Error loading authorized signers:', error);
                }
            }

            // Check if current user is in authorized signers list or is admin
            function checkDiriSendiriEligibility() {
                if (!dariTipeSendiri || !currentUserName) return;

                const userTipe = document.getElementById('user_tipe')?.value || '';
                if (userTipe === 'admin') return; // admins can choose Diri Sendiri

                // If API reported current user is authorized, keep Diri Sendiri enabled
                if (!currentUserAuthorized) {
                    // Disable "Diri Sendiri" option for non-authorized users
                    dariTipeSendiri.disabled = true;
                    dariTipeSendiri.parentElement.classList.add('text-muted');
                    dariTipeSendiri.parentElement.title = 'Anda tidak terdaftar sebagai penandatangan resmi';
                    // Auto-select Pejabat option
                    const pejabatRadio = document.querySelector('input[name="dari_tipe"][value="pribadi"]');
                    if (pejabatRadio) {
                        pejabatRadio.checked = true;
                        handleDariTipeChange();
                    }
                }
            }

            // Handle dari_tipe radio change
            function handleDariTipeChange() {
                const dariTipeRadio = document.querySelector('input[name="dari_tipe"]:checked');
                if (!dariTipeRadio) return;

                if (dariTipeRadio.value === 'pribadi') {
                    wrapPribadi.style.display = 'block';
                } else {
                    wrapPribadi.style.display = 'none';
                }
                generateNomorSurat();
            }

            // Handle backdate checkbox toggle
            function handleBackdateToggle() {
                if (backdateCheckbox.checked) {
                    backdateFields.classList.remove('d-none');
                    nomorSuratInput.removeAttribute('readonly');
                    updateBackdateFieldsVisibility();
                } else {
                    backdateFields.classList.add('d-none');
                    nomorSuratInput.setAttribute('readonly', 'readonly');
                    generateNomorSurat();
                }
            }

            // Show/hide year field based on tipe surat
            function updateBackdateFieldsVisibility() {
                const selectedTipeOption = tipeSelect?.options[tipeSelect.selectedIndex];
                const tipeKode = selectedTipeOption?.dataset.kode;

                if (tipeKode === 'SK') {
                    backdateYearField.style.display = 'none';
                    backdateHint.textContent = 'Masukkan nomor urut untuk backdate';
                } else {
                    backdateYearField.style.display = 'block';
                    backdateHint.textContent = 'Masukkan nomor urut dan tahun untuk backdate';
                }
            }

            // Handle backdate input changes
            function handleBackdateInput() {
                if (!backdateCheckbox.checked) return;

                const selectedTipeOption = tipeSelect?.options[tipeSelect.selectedIndex];
                const tipeKode = selectedTipeOption?.dataset.kode;
                const sequence = backdateSequence.value;
                const year = backdateYear.value || new Date().getFullYear();

                if (!sequence) return;

                if (tipeKode === 'SK') {
                    const currentNomor = nomorSuratInput.value;
                    if (currentNomor) {
                        const parts = currentNomor.split('.');
                        parts[parts.length - 1] = sequence;
                        nomorSuratInput.value = parts.join('.');
                    }
                } else if (tipeKode === 'KPTS') {
                    const paddedSeq = String(sequence).padStart(3, '0');
                    nomorSuratInput.value = `${paddedSeq}/KPTS-JPT/${year}`;
                } else if (tipeKode === 'SE') {
                    const paddedSeq = String(sequence).padStart(3, '0');
                    nomorSuratInput.value = `${paddedSeq}/SE-DIR/${year}`;
                }
            }

            // Update previous letter number indicator
            async function updatePrevNomorIndicator() {
                if (!tipeSelect?.value || !prevNomorIndicator) return;

                try {
                    const response = await fetch(
                        `api_classification.php?action=get_last_letter_number&tipe_surat_id=${tipeSelect.value}`
                    );
                    const data = await response.json();

                    if (data.success && data.data) {
                        if (data.data.last_nomor_surat) {
                            prevNomorIndicator.textContent =
                                `(Nomor Surat Sebelumnya: ${data.data.last_nomor_surat})`;
                        } else {
                            prevNomorIndicator.textContent = '(Belum ada surat sebelumnya)';
                        }
                    }
                } catch (error) {
                    console.error('Error loading last letter number:', error);
                    prevNomorIndicator.textContent = '';
                }
            }

            // Event listeners for new functionality
            document.querySelectorAll('input[name="dari_tipe"]').forEach(radio => {
                radio.addEventListener('change', handleDariTipeChange);
            });

            if (dariPribadiSelect) {
                dariPribadiSelect.addEventListener('change', async () => await generateNomorSurat());
            }

            if (backdateCheckbox) {
                backdateCheckbox.addEventListener('change', handleBackdateToggle);
            }
            if (backdateSequence) {
                backdateSequence.addEventListener('input', handleBackdateInput);
            }
            if (backdateYear) {
                backdateYear.addEventListener('input', handleBackdateInput);
            }

            if (tipeSelect) {
                tipeSelect.addEventListener('change', () => {
                    updateBackdateFieldsVisibility();
                    updatePrevNomorIndicator();
                });
            }

            // Initialize on page load
            populateTipeSurat();
            toggleClassificationFields();
            loadAuthorizedSigners();
            handleDariTipeChange();
        })();

        // Utility functions
        const get = (id) => document.getElementById(id);
        const getVal = (id) => (get(id)?.value || '').trim();
        const safe = (str) => (typeof str === 'string' ? str.replace(/</g, '&lt;').replace(/>/g, '&gt;') : str);

        function toIDDate(dateStr) {
            if (!dateStr) return null;
            const date = new Date(dateStr);
            const days = ['Minggu', 'Senin', 'Selasa', 'Rabu', 'Kamis', 'Jumat', 'Sabtu'];
            const months = ['Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni', 'Juli', 'Agustus', 'September',
                'Oktober', 'November', 'Desember'
            ];
            const dayName = days[date.getDay()];
            const dateNum = date.getDate();
            const monthName = months[date.getMonth()];
            const year = date.getFullYear();
            return `${dayName}, ${dateNum} ${monthName} ${year}`;
        }

        function todayID() {
            return toIDDate(new Date().toISOString().split('T')[0]);
        }

        // Get template URL based on letter type
        function getTemplateUrl() {
            const tipeSelect = document.getElementById('tipe_surat_id');
            if (!tipeSelect) return 'template-letter.php';

            const selectedOption = tipeSelect.options[tipeSelect.selectedIndex];
            const kode = selectedOption?.dataset.kode || 'SK';

            if (kode === 'SE') {
                return 'template-letter.php?type=edaran';
            } else if (kode === 'KPTS') {
                return 'template-letter.php?type=keputusan';
            }
            return 'template-letter.php';
        }

        // Collect form values for template placeholder replacement
        function collectFormValues() {
            const tipeSelect = document.getElementById('tipe_surat_id');
            const selectedOption = tipeSelect.options[tipeSelect.selectedIndex];
            const tipeKode = selectedOption?.dataset.kode || '';

            let documentTitle = 'Surat Keluar';
            if (tipeKode === 'KPTS') {
                documentTitle = 'KEPUTUSAN DIREKSI PT JASAMARGA PANDAAN TOL';
            } else if (tipeKode === 'SE') {
                documentTitle = 'SURAT EDARAN';
            }

            const tglInput = getVal('tanggal_surat');
            const tglID = toIDDate(tglInput) || todayID();

            // Get current user info (from page PHP variable)
            const userNama = <?= json_encode($name) ?>;
            const userJabatan = <?= json_encode($user['jabatan'] ?? '') ?>;

            return {
                DOCUMENT_TITLE: safe(documentTitle),
                NOMOR_SURAT: safe(getVal('nomor_surat')),
                KOTA: safe(getVal('tempat_dibuat')),
                TANGGAL_SURAT: safe(tglID),
                NAMA_PEJABAT: safe(userNama),
                JABATAN_PEJABAT: safe(userJabatan),
                NAMA_PERUSAHAAN: safe('PT Jasamarga Pandaan Tol'),
                NAMA_MAHASISWA: safe(getVal('kepada') || ''),
                PARAF: safe(getVal('paraf') || ''),
                PATH_TANDA_TANGAN: safe(getVal('path_ttd') || ''),
                QR_CODE: safe(getVal('qr_code') || '')
            };
        }

        // Fetch template HTML
        async function fetchTemplate() {
            try {
                const res = await fetch(getTemplateUrl(), {
                    cache: 'no-store'
                });
                const txt = await res.text();
                // Extract body content only
                const bodyMatch = txt.match(/<body[^>]*>([\s\S]*?)<\/body>/i);
                const bodyContent = bodyMatch ? bodyMatch[1] : txt;
                return bodyContent.trim();
            } catch (err) {
                console.error('fetchTemplate error:', err);
                return '<p>Error loading template</p>';
            }
        }

        /**
         * Extract editable field values from TinyMCE content and sync back to form inputs.
         * This ensures changes made directly in the editor are reflected in the database.
         */
        /**
         * Get the surat type code from tipe_surat_id select
         * Maps select options back to codes (SK, KPTS, SE)
         */
        function getTipeSuratKode() {
            const tipeSuratSelect = document.getElementById('tipe_surat_id');
            if (!tipeSuratSelect || !tipeSuratSelect.value) {
                return '';
            }

            // Get the selected option's text and extract code
            const selectedText = tipeSuratSelect.selectedOptions[0]?.text || '';
            console.log(`   Selected tipe_surat text: "${selectedText}"`);

            // Map common texts to codes
            if (selectedText.includes('Surat Keluar') || selectedText.includes('Keluar')) {
                return 'SK';
            } else if (selectedText.includes('Keputusan') || selectedText.includes('KPTS')) {
                return 'KPTS';
            } else if (selectedText.includes('Edaran') || selectedText.includes('SE')) {
                return 'SE';
            }

            // Try to extract code from text (e.g., "SK - Surat Keluar" -> "SK")
            const match = selectedText.match(/^([A-Z]+)\s*-/);
            if (match) {
                return match[1];
            }

            return selectedText.substring(0, 4).toUpperCase(); // Fallback
        }

        function extractValuesFromEditor() {
            if (!window.tinymce || !tinymce.get('editorNaskah')) {
                console.warn('⚠️ TinyMCE not ready for extraction');
                return;
            }

            const editor = tinymce.get('editorNaskah');
            const content = editor.getContent();

            // Create a temporary DOM to parse the content
            const tempDiv = document.createElement('div');
            tempDiv.innerHTML = content;

            // Get current surat type for nomor_surat validation
            const tipeSuratKode = getTipeSuratKode();
            console.log(`   📋 Using surat type: ${tipeSuratKode}`);

            // Map of data-field values to form input IDs
            const fieldMapping = {
                'TANGGAL_SURAT': 'tanggal_surat',
                'KOTA': 'tempat_dibuat',
                'KEPADA': 'kepada',
                'PERIHAL': 'perihal',
                'NOMOR_SURAT': 'nomor_surat'
            };

            let extractedCount = 0;

            console.log('🔍 DEBUG extractValuesFromEditor():');
            console.log('   Editor content length:', content.length);

            // Find all elements with data-field attribute
            const editableFields = tempDiv.querySelectorAll('[data-field]');
            console.log('   Found', editableFields.length, 'editable fields with data-field attribute');

            editableFields.forEach(el => {
                const fieldName = el.getAttribute('data-field');
                const inputId = fieldMapping[fieldName];

                console.log(`   Processing field: ${fieldName} → ${inputId}`);

                if (inputId) {
                    const formInput = document.getElementById(inputId);
                    if (formInput) {
                        let value = el.textContent.trim();
                        console.log(`     Extracted text: "${value}"`);

                        // Skip if value is still a placeholder
                        if (value.startsWith('[[') && value.endsWith(']]')) {
                            console.log(`     ⏭️ Skipping placeholder`);
                            return;
                        }

                        // Special handling for date field - convert Indonesian date to YYYY-MM-DD
                        if (fieldName === 'TANGGAL_SURAT') {
                            const convertedDate = convertIndonesianDateToISO(value);
                            if (convertedDate) {
                                value = convertedDate;
                                console.log(`     📅 Converted to ISO: "${convertedDate}"`);
                            } else {
                                console.warn(`     ⚠️ Could not convert date: "${value}"`);
                                return; // Don't update if we can't convert
                            }
                        }

                        // Special handling for nomor_surat - validate based on surat type
                        if (fieldName === 'NOMOR_SURAT') {
                            const validatedValue = validateNomorSurat(value, formInput.value, tipeSuratKode);
                            if (validatedValue === null && value !== formInput.value) {
                                // Invalid edit - skip this update
                                console.warn(`     ❌ Invalid nomor_surat edit - NOT updating`);
                                return;
                            }
                            if (validatedValue === null) {
                                // No change or invalid
                                console.log(`     ℹ️ No valid changes to nomor_surat`);
                                return;
                            }
                            value = validatedValue;
                        }

                        // Only update if value is different
                        if (formInput.value !== value) {
                            console.log(`     ✅ UPDATING: "${formInput.value}" → "${value}"`);
                            formInput.value = value;
                            extractedCount++;
                        } else {
                            console.log(`     ℹ️ Value unchanged`);
                        }
                    } else {
                        console.log(`     ❌ Form input #${inputId} not found!`);
                    }
                }
            });

            console.log(`   ✅ Extraction complete: ${extractedCount} field(s) updated`);
            console.log('   Form values after extraction:');
            for (const [field, inputId] of Object.entries(fieldMapping)) {
                const inp = document.getElementById(inputId);
                if (inp) {
                    console.log(`     #${inputId} = "${inp.value}"`);
                }
            }
            return extractedCount;
        }

        /**
         * Convert Indonesian date format to ISO format (YYYY-MM-DD)
         * Handles formats like: "19 Desember 2025", "19-12-2025", "19/12/2025"
         */
        function convertIndonesianDateToISO(dateStr) {
            if (!dateStr) return null;

            // Already in ISO format?
            if (/^\d{4}-\d{2}-\d{2}$/.test(dateStr)) {
                return dateStr;
            }

            // Indonesian month names
            const months = {
                'januari': '01',
                'februari': '02',
                'maret': '03',
                'april': '04',
                'mei': '05',
                'juni': '06',
                'juli': '07',
                'agustus': '08',
                'september': '09',
                'oktober': '10',
                'november': '11',
                'desember': '12'
            };

            // Indonesian day names (to remove if present)
            const dayNames = ['senin', 'selasa', 'rabu', 'kamis', 'jumat', 'sabtu', 'minggu'];

            // Try format: "Minggu, 7 Desember 2025" (with day name)
            const match3 = dateStr.match(/^(\w+),?\s+(\d{1,2})\s+(\w+)\s+(\d{4})$/i);
            if (match3) {
                const dayName = match3[1].toLowerCase();
                const day = match3[2].padStart(2, '0');
                const monthName = match3[3].toLowerCase();
                const year = match3[4];
                const month = months[monthName];
                if (month && dayNames.includes(dayName)) {
                    console.log(`   📅 Parsed format with day name: "${dateStr}" → "${year}-${month}-${day}"`);
                    return `${year}-${month}-${day}`;
                }
            }

            // Try format: "19 Desember 2025"
            const match = dateStr.match(/^(\d{1,2})\s+(\w+)\s+(\d{4})$/i);
            if (match) {
                const day = match[1].padStart(2, '0');
                const monthName = match[2].toLowerCase();
                const year = match[3];
                const month = months[monthName];
                if (month) {
                    return `${year}-${month}-${day}`;
                }
            }

            // Try format: "19-12-2025" or "19/12/2025" (DD-MM-YYYY)
            const match2 = dateStr.match(/^(\d{1,2})[-\/](\d{1,2})[-\/](\d{4})$/);
            if (match2) {
                const day = match2[1].padStart(2, '0');
                const month = match2[2].padStart(2, '0');
                const year = match2[3];
                return `${year}-${month}-${day}`;
            }

            return null;
        }

        /**
         * Validate and extract editable parts of nomor_surat based on surat type
         * Returns the edited number if valid, null if invalid
         * 
         * Patterns:
         * - KPTS: "001/KPTS-JPT/2025" → editable parts: sequence (001) and year (2025)
         * - SE: "001/SE-DIR/2025" → editable parts: sequence (001) and year (2025)
         * - SK: "DKU.JPT.HK.09.03.8" → editable part: only the last number (8)
         */
        function validateNomorSurat(editedValue, originalValue, tipeSuratKode) {
            if (!editedValue || editedValue === originalValue) {
                return null; // No change, skip update
            }

            console.log(
                `   🔢 Validating nomor_surat for tipeSuratKode="${tipeSuratKode}": "${originalValue}" → "${editedValue}"`
            );

            if (tipeSuratKode === 'KPTS') {
                // Format: 001/KPTS-JPT/2025
                // Only sequence (001) and year (2025) should change
                // Middle part KPTS-JPT MUST NOT change
                const regex = /^(\d+)\/KPTS-JPT\/(\d{4})$/i;
                const editMatch = editedValue.match(regex);
                const origMatch = originalValue.match(regex);

                if (!editMatch) {
                    console.warn(
                        `     ❌ Invalid KPTS format in edited value. Expected: ###/KPTS-JPT/YYYY, got: "${editedValue}"`
                    );
                    return null;
                }
                if (!origMatch) {
                    console.warn(
                        `     ❌ Invalid KPTS format in original value. Expected: ###/KPTS-JPT/YYYY, got: "${originalValue}"`
                    );
                    return null;
                }

                const editSeq = editMatch[1];
                const editYear = editMatch[2];
                const origSeq = origMatch[1];
                const origYear = origMatch[2];

                // The middle part must remain unchanged
                if (editMatch[0].toUpperCase() === origMatch[0].toUpperCase()) {
                    console.log(`     ℹ️ No actual change in nomor_surat`);
                    return null;
                }

                console.log(`     ✅ Valid KPTS edit: sequence ${origSeq}→${editSeq}, year ${origYear}→${editYear}`);
                return editedValue;

            } else if (tipeSuratKode === 'SE') {
                // Format: 001/SE-DIR/2025
                // Only sequence (001) and year (2025) should change
                // Middle part SE-DIR MUST NOT change
                const regex = /^(\d+)\/SE-DIR\/(\d{4})$/i;
                const editMatch = editedValue.match(regex);
                const origMatch = originalValue.match(regex);

                if (!editMatch) {
                    console.warn(
                        `     ❌ Invalid SE format in edited value. Expected: ###/SE-DIR/YYYY, got: "${editedValue}"`);
                    return null;
                }
                if (!origMatch) {
                    console.warn(
                        `     ❌ Invalid SE format in original value. Expected: ###/SE-DIR/YYYY, got: "${originalValue}"`
                    );
                    return null;
                }

                const editSeq = editMatch[1];
                const editYear = editMatch[2];
                const origSeq = origMatch[1];
                const origYear = origMatch[2];

                // The middle part must remain unchanged
                if (editMatch[0].toUpperCase() === origMatch[0].toUpperCase()) {
                    console.log(`     ℹ️ No actual change in nomor_surat`);
                    return null;
                }

                console.log(`     ✅ Valid SE edit: sequence ${origSeq}→${editSeq}, year ${origYear}→${editYear}`);
                return editedValue;

            } else if (tipeSuratKode === 'SK') {
                // Format: DKU.JPT.HK.09.03.8
                // Only the last number (urutan) should change
                const parts = editedValue.split('.');
                const origParts = originalValue.split('.');

                if (parts.length !== 6 || origParts.length !== 6) {
                    console.warn(`     ❌ Invalid SK format. Expected 6 parts separated by dots, got: "${editedValue}"`);
                    return null;
                }

                // Check if all parts except last are the same (case-insensitive for prefix)
                const prefixMatch = parts.slice(0, 5).every((p, i) => p.toUpperCase() === origParts[i].toUpperCase());

                if (!prefixMatch) {
                    console.warn(
                        `     ❌ Invalid SK edit. Cannot change the prefix (DKU.JPT.HK.09.03). Tried to change from: "${originalValue}" to: "${editedValue}"`
                    );
                    return null;
                }

                if (isNaN(parts[5]) || isNaN(origParts[5])) {
                    console.warn(`     ❌ Invalid SK format. Last part must be a number. Got: "${parts[5]}"`);
                    return null;
                }

                if (parts[5] === origParts[5]) {
                    console.log(`     ℹ️ No actual change in urutan number`);
                    return null;
                }

                console.log(`     ✅ Valid SK edit: urutan ${origParts[5]}→${parts[5]}`);
                return editedValue;

            } else {
                console.warn(
                    `     ⚠️ Unknown surat type: "${tipeSuratKode}". Cannot validate nomor_surat. Accepting as-is.`);
                // For unknown types, don't allow changes to nomor_surat
                return null;
            }

            return null;
        }

        // Replace placeholders in template
        function fillPlaceholders(html, data) {
            let out = html;
            for (const [key, val] of Object.entries(data)) {
                if (val === null) continue;
                const re = new RegExp(`\\[\\[${key}\\]\\]`, 'g');
                out = out.replace(re, val);
            }
            // Replace remaining image placeholders with empty src to prevent TinyMCE errors
            out = out.replace(/src="\[\[PATH_TANDA_TANGAN\]\]"/g, 'src=""');
            out = out.replace(/src="\[\[PATH_STEMPEL\]\]"/g, 'src=""');
            out = out.replace(/src="\[\[QR_CODE\]\]"/g, 'src=""');
            return out;
        }

        // Wait for TinyMCE to load
        function waitForTinyMCE() {
            return new Promise((resolve) => {
                if (typeof tinymce !== 'undefined') {
                    resolve();
                } else {
                    const checkInterval = setInterval(() => {
                        if (typeof tinymce !== 'undefined') {
                            clearInterval(checkInterval);
                            resolve();
                        }
                    }, 100);
                    // Timeout after 10 seconds
                    setTimeout(() => {
                        clearInterval(checkInterval);
                        console.error('TinyMCE failed to load within timeout');
                        resolve();
                    }, 10000);
                }
            });
        }

        let editorReady = false;
        let editorInstance = null;

        async function initTinyMCE() {
            await waitForTinyMCE();
            if (typeof tinymce === 'undefined') {
                console.error('TinyMCE is not available');
                return;
            }
            if (editorReady && tinymce.get('editorNaskah')) return Promise.resolve();
            return new Promise((resolve) => {
                tinymce.init({
                    selector: '#editorNaskah',
                    height: 780,
                    menubar: true,
                    branding: false,
                    convert_urls: false,
                    remove_script_host: false,
                    entity_encoding: 'raw',
                    automatic_uploads: true,
                    images_reuse_filename: false,
                    paste_data_images: false,
                    file_picker_types: 'image',
                    images_upload_url: '/jasa_marga/api/upload/image.php',
                    images_upload_credentials: true,
                    images_upload_handler: (blobInfo, progress) => {
                        const fd = new FormData();
                        fd.append('image', blobInfo.blob(), blobInfo.filename());

                        return fetch('/jasa_marga/api/upload/image.php', {
                                method: 'POST',
                                credentials: 'same-origin',
                                body: fd
                            })
                            .then(r => {
                                if (!r.ok) throw new Error('HTTP ' + r.status);
                                return r.json();
                            })
                            .then(json => {
                                if (json && json.location) {
                                    return json.location;
                                }
                                throw new Error(json?.error || 'Upload gagal');
                            });
                    },
                    protect: [/\[\[[\s\S]*?\]\]/g],
                    extended_valid_elements: '*[*]',
                    valid_children: '+body[style],+div[div|section|header|footer|img]',
                    verify_html: false,
                    plugins: 'lists table link image code preview autoresize',
                    toolbar: 'undo redo | styles | bold italic underline | alignleft aligncenter alignright alignjustify | bullist numlist outdent indent | table link image | removeformat | preview | code',
                    init_instance_callback: function(editor) {
                        editorReady = true;
                        editorInstance = editor;
                        // Attempt to apply template style for current selection
                        applyTemplateStyleToEditor(editor).catch(e => console.error(
                            'applyTemplateStyleToEditor:', e));
                        // Resolve the promise when instance initialized
                        try {
                            resolve();
                        } catch (e) {
                            /* ignore */
                        }
                    },

                    content_style: `
                        * { box-sizing: border-box; }
                        html, body { margin:0; padding:0; }
                        body {
                            font-family: Arial, Helvetica, sans-serif;
                            color: #111;
                            background: #fff;
                        }
                        @page{
                            size: A4;
                            margin: 12mm 14mm 28mm 12mm;
                        }
                        :root{
                            --page-w:210mm;
                            --page-h:297mm;
                            --pad-top:12mm;
                            --pad-right:14mm;
                            --pad-bottom:42mm;
                            --pad-left:12mm;
                        }
                        .page{
                            width: var(--page-w);
                            min-height: var(--page-h);
                            padding: var(--pad-top) var(--pad-right) var(--pad-bottom) var(--pad-left);
                            margin: 0 auto;
                            position: relative;
                            background: #fff;
                        }
                        .html2pdf__page-break{
                            height:0; border:0; page-break-after:always;
                        }
                        .header {
                            display:flex; align-items:flex-start; justify-content:space-between;
                            gap: 16px; margin-bottom: 10mm;
                        }
                        .brand { display:flex; align-items:center; gap:12px; }
                        .brand img.logo { height:70px; width:auto; object-fit:contain; }
                        /* Header-left / Header-right placeholder to match template-surat-edaran */
                        .header-left { flex: 1; }
                        .header-left img.logo { height: 90px; width: 360px; object-fit: contain; display: block; }
                        .header-right { flex: 1; text-align: right; font-size:13px; color:#333; line-height:1.4; }
                        .brand .title { line-height:1.1; font-weight:700; letter-spacing:.3px; }
                        .brand .title .main { font-size:22px; color:#0c50e1; }
                        .brand .title .sub { font-size:18px; color:#0c50e1; text-transform:uppercase; }
                        .header .topnote { text-align:right; font-size:13px; color:#333; max-width:300px; margin-top: 20px; }
                        .doc-title {
                            text-align:center; font-weight:600; font-size:18px;
                        }
                        .doc-number {
                            text-align:center; font-size:18px; font-weight:600;
                        }
                        .doc-subject {
                            text-align:center; font-size:18px; font-weight:600;
                        }
                        .section-title {
                            font-size:16px; font-weight:700; margin-bottom:4mm;
                        }
                        .numbered-list {
                            padding-left: 4px;
                            list-style: none;
                            margin: 0;
                        }

                        .numbered-list li {
                            margin-bottom: 5px;
                            margin-top: 0px;
                            position: relative;
                        }
                        .divider { height:1px; background:#e6e9ef; margin: 2mm 0 8mm; }
                        .section { margin-bottom: 6mm; font-size: 14px !important; }
                        .lead { margin-bottom: 4mm; }
                        table.kv { border-collapse: collapse; width:100%; font-size: 14px; }
                        table.kv td { padding: 2px 0; vertical-align: top; }
                        table.kv td.col1 { width:120px; } table.kv td.col2 { width:10px; } table.kv td.col3 { width:auto; }
                        span.highlight { font-weight: 700 !important; }
                        .sign-block {
                            margin-top: 14mm;
                            width: 100%;
                            display:flex; justify-content:flex-end;
                            position: relative;
                        }
                        .sign {
                            width: 72mm;
                            text-align:left; font-size:14px;
                            position: relative; padding-top: 0;
                        }
                        .sign .place-date { margin-bottom: 15mm; }
                        .sign .stamp { position:absolute; right: -8mm; top: 18mm; opacity:.85; width: 34mm; height:auto; }
                        .sign .signature { height: 22mm; width:auto; display:block; margin: -6mm 0 0 0; }
                        .sign .signature-info { display: block; margin-top: 20mm; }
                        .sign .name { font-weight:700; margin: 0; }
                        .sign .title { margin-top: 1mm; }
                        .sign-block .paraf {
                            position: absolute;
                            left: 0;
                            bottom: 0;
                            font-size: 14px;
                            color: #333;
                            font-weight: 500;
                        }
                        .footer {
                            position: absolute;
                            right: 14mm;
                            bottom: 12mm;
                            left: auto;
                            width: auto;
                            text-align: right;
                            font-size: 10px;
                            color: #333;
                            line-height: 1.4;
                        }
                        .page .footer{ display:none !important; }
                        .page:first-of-type .footer{ display:block !important; }
                        .sign-block, .footer, .doc-title, .doc-number, table.kv {
                        page-break-inside: avoid;
                        break-inside: avoid;
                        }
                        .page:not(:last-child){ page-break-after: always; }
                        
                        /* Two-column section for Surat Keputusan */
                        .section-title-2 {
                            font-weight: bold;
                            font-size: 16px;
                            margin-bottom: 12px;
                            text-align: center;
                        }
                        .two-column-section {
                            display: flex;
                        }
                        .two-column-section .column {
                            flex: 1;
                        }
                        .two-column-section .column-2 {
                            flex: 8;
                        }
                        .two-column-section .column-2 p {
                            margin: 0;
                        }
                        .page-break {
                            page-break-before: always;
                            break-before: page;
                        }
                        .signature-section {
                            margin-top: 20mm;
                        }
                        .signature-info-table {
                            margin-left: auto;
                            margin-right: 0;
                            width: auto;
                            font-size: 14px;
                        }
                        .signature-info-table td {
                            padding: 2px 0;
                            vertical-align: top;
                        }
                        .signature-info-table .label {
                            font-weight: 600;
                        }
                        .signature-info-table .colon {
                            font-weight: 600;
                        }
                        .signature-table {
                            margin-left: auto;
                            margin-right: 32px;
                            text-align: left;
                            font-size: 14px;
                            margin-bottom: 3mm;
                        }
                        .signature-table td {
                            font-weight: 600;
                            padding-right: 10px;
                        }
                        .signature-section-container {
                            position: relative;
                            text-align: right;
                        }
                        .signature-paraf {
                            position: absolute;
                            left: 0;
                            top: 48mm;
                            font-size: 16px;
                            color: #333;
                        }
                        .signature-qr-container {
                            margin-right: 80px;
                        }
                        .signature-img {
                            height: 22mm;
                            width: auto;
                            display: block;
                            margin: -4mm 0 0 auto;
                        }
                        .signature-name {
                            font-weight: 600;
                            margin-top: 5mm;
                            margin-right: 92px;
                        }
                        .signature-sub-name {
                            margin-right: 32px;
                            font-weight: 700;
                        }
                        .sign-block-keputusan {
                            display: flex;
                            justify-content: center;
                            position: relative;
                        }
                        .sign-paraf-keputusan {
                            position: absolute;
                            left: 0;
                            top: 60mm;
                            font-size: 12px;
                            color: #333;
                        }
                        .lampiran-box{
                            border:1px solid #e0e0e0;
                            padding:0;
                            margin: 0 auto;
                        }
                        .lampiran-slot{
                            width: calc(var(--page-w) - var(--pad-left) - var(--pad-right));
                            height: calc(var(--page-h) - var(--pad-top) - var(--pad-bottom) - 26mm);
                            margin: 0 auto;
                            display:flex; align-items:center; justify-content:center;
                            overflow:hidden;
                            page-break-inside: avoid;
                            break-inside: avoid;
                        }
                        .lampiran-slot > img.lampiran-img{
                            max-width: 100%;
                            max-height: 100%;
                            width: auto; height: auto;
                            object-fit: contain;
                            display: block; margin: 0;
                        }
                        .lampiran-slot > iframe.lampiran-frame{
                            width: 100%; height: 100%; border: 0;
                        }
                        img.signature[src=""], img.stamp[src=""] { display:none !important; }
                        table, th, td { border: none !important; outline: none !important; }
                        .mce-item-table, .mce-item-table td, .mce-item-table th { border: none !important; outline: none !important; }
                        @media print {
                            @page{ size:A4; margin:12mm 20mm 28mm; }
                            .page{ -webkit-print-color-adjust: exact; print-color-adjust: exact; }
                        }
                    `
                });
            });
        }

        // Apply template CSS into the TinyMCE iframe for matching templates (Surat Edaran)
        async function applyTemplateStyleToEditor(editorInstance) {
            if (!editorInstance) editorInstance = tinymce.get('editorNaskah');
            if (!editorInstance) return;

            const tplUrl = getTemplateUrl();
            const isEdaran = tplUrl && tplUrl.toLowerCase().includes('edaran');
            try {
                const doc = editorInstance.getDoc();
                if (!doc) return;
                // Ensure relative URLs inside templates resolve correctly by adding a base href
                let baseEl = doc.getElementById('editorTemplateBase');
                if (!baseEl) {
                    baseEl = doc.createElement('base');
                    baseEl.id = 'editorTemplateBase';
                    baseEl.href = '/jasa_marga/';
                    doc.head.appendChild(baseEl);
                } else {
                    baseEl.href = '/jasa_marga/';
                }
                let styleEl = doc.getElementById('editorTemplateStyle');

                if (!isEdaran) {
                    if (styleEl) styleEl.parentNode.removeChild(styleEl);
                    return;
                }

                const cssText = await fetchTemplateStyles();
                if (!styleEl) {
                    styleEl = doc.createElement('style');
                    styleEl.id = 'editorTemplateStyle';
                    doc.head.appendChild(styleEl);
                }
                // Ensure the exact template style is used for the editor
                styleEl.textContent = cssText +
                    '\n.page{ width:210mm !important; min-height:297mm !important; padding:12mm 14mm 42mm 12mm !important; margin:0 auto !important; background:#fff !important; }';
            } catch (e) {
                console.error('applyTemplateStyleToEditor error', e);
            }
        }

        // Fetch template CSS styles
        async function fetchTemplateStyles() {
            try {
                const res = await fetch(getTemplateUrl(), {
                    cache: 'no-store'
                });
                const html = await res.text();
                // Extract CSS from <style> tags in template
                const styleMatch = html.match(/<style[^>]*>([\s\S]*?)<\/style>/);
                return styleMatch ? styleMatch[1] : '';
            } catch (err) {
                console.error('fetchTemplateStyles error:', err);
                return '';
            }
        }

        // Step Navigation
        (function() {
            const to2 = document.getElementById('toStep2');
            const back1 = document.getElementById('backTo1');
            const panes = document.querySelectorAll('.step-pane');
            const steps = document.querySelectorAll('.stepper .step');

            function setStep(n) {
                panes.forEach(p => p.classList.toggle('d-none', p.dataset.step !== String(n)));
                steps.forEach(s => {
                    const cur = Number(s.dataset.step);
                    s.classList.toggle('active', cur === n);
                    s.classList.toggle('completed', cur < n);
                });
                window.scrollTo({
                    top: 0,
                    behavior: 'smooth'
                });

                // Initialize TinyMCE and populate with template when moving to step 2
                if (n === 2) {
                    const editorTextarea = document.getElementById('editorNaskah');
                    if (editorTextarea) {
                        editorTextarea.style.display = '';
                    }

                    setTimeout(async () => {
                        await initTinyMCE();

                        // Fetch template and populate editor
                        try {
                            const template = await fetchTemplate();
                            const formValues = collectFormValues();
                            const filledContent = fillPlaceholders(template, formValues);

                            if (editorInstance) {
                                editorInstance.setContent(filledContent);
                                await applyTemplateStyleToEditor(editorInstance);
                                editorInstance.focus();
                            } else {
                                const editor = tinymce.get('editorNaskah');
                                if (editor) {
                                    editor.setContent(filledContent);
                                    await applyTemplateStyleToEditor(editor);
                                    editor.focus();
                                }
                            }
                        } catch (err) {
                            console.error('Error populating template:', err);
                        }
                    }, 200);
                }
            }

            if (back1) {
                back1.addEventListener('click', () => setStep(1));
            }

            if (to2) {
                to2.addEventListener('click', async () => {
                    // Basic validation
                    const tipeSuratId = document.getElementById('tipe_surat_id').value;
                    const tempatDibuat = document.getElementById('tempat_dibuat').value.trim();
                    const tanggalSurat = document.getElementById('tanggal_surat').value;
                    const perihal = document.getElementById('perihal').value.trim();

                    if (!tipeSuratId) {
                        alert('Tipe Surat wajib dipilih');
                        return;
                    }
                    if (!tempatDibuat) {
                        alert('Tempat Surat Dibuat wajib diisi');
                        return;
                    }
                    if (!tanggalSurat) {
                        alert('Tanggal Surat wajib diisi');
                        return;
                    }
                    if (!perihal) {
                        alert('Perihal wajib diisi');
                        return;
                    }

                    setStep(2);
                });
            }
        })();

        // Custom form validation
        document.getElementById('formNaskah').addEventListener('submit', function(e) {
            const tipeSelect = document.getElementById('tipe_surat_id');
            const selectedTipeOption = tipeSelect.options[tipeSelect.selectedIndex];
            const tipeKode = selectedTipeOption?.dataset.kode || '';

            // If SK, validate classification fields
            if (tipeKode === 'SK') {
                const jenisSuratSelect = document.getElementById('jenis_surat_new');
                const klasifikasiSelect = document.getElementById('klasifikasi_arsip_new');
                const subjenisSelect = document.getElementById('subjenis_surat_id');

                if (!jenisSuratSelect.value) {
                    e.preventDefault();
                    alert('Jenis Surat wajib dipilih untuk SK');
                    jenisSuratSelect.focus();
                    return false;
                }
                if (!klasifikasiSelect.value) {
                    e.preventDefault();
                    alert('Klasifikasi wajib dipilih untuk SK');
                    klasifikasiSelect.focus();
                    return false;
                }
                if (!subjenisSelect.value) {
                    e.preventDefault();
                    alert('Subjenis Surat wajib dipilih untuk SK');
                    subjenisSelect.focus();
                    return false;
                }
            }
        });

        // Upload button handler
        window.addEventListener('load', function() {
            const uploadBtn = document.getElementById('btnUploadPdf');
            const saveBtn = document.getElementById('btnSave');
            const fileInput = document.getElementById('fileSuratPath');

            if (!uploadBtn) {
                console.error('❌ Tombol Cetak & Upload PDF tidak ditemukan setelah load!');
                return;
            }
            if (!saveBtn || !fileInput) {
                console.error('❌ Elemen btnSave atau fileSuratPath tidak ditemukan!');
                return;
            }

            // Enable upload only if html2pdf already loaded, otherwise wait for event
            uploadBtn.disabled = !window._html2pdf_loaded;
            document.addEventListener('html2pdf:loaded', function() {
                try {
                    uploadBtn.disabled = false;
                } catch (e) {}
            });

            saveBtn.addEventListener('click', async function(e) {
                // Prevent default form submission - we'll submit manually after sync
                e.preventDefault();

                // Extract values from TinyMCE editor back to form fields
                // This ensures changes made directly in the editor are saved to the database
                extractValuesFromEditor();

                // Sync TinyMCE content to textarea before form submission
                if (window.tinymce && tinymce.get('editorNaskah')) {
                    const editor = tinymce.get('editorNaskah');
                    const editorContent = editor.getContent() || '';

                    // Fetch CSS from template
                    let cssToUse = '';
                    try {
                        cssToUse = await fetchTemplateStyles();
                        console.log('✅ Fetched CSS for save');
                        console.log('📊 CREATE - Raw CSS length:', cssToUse.length);
                        console.log('📊 CREATE - Template URL:', getTemplateUrl());
                    } catch (err) {
                        console.error('Failed to fetch CSS:', err);
                    }

                    // Clean CSS for mPDF compatibility
                    const cleanedCss = cssToUse
                        .replace(/:root\s*\{[\s\S]*?\}/g, '')
                        .replace(/@media[\s\S]*?\{[\s\S]*?\}/g, '')
                        .replace(/@page\s*\{[\s\S]*?\}/g, '')
                        .replace(/var\(--page-w\)/g, '210mm')
                        .replace(/var\(--page-h\)/g, '297mm')
                        .replace(/var\(--pad-top\)/g, '12mm')
                        .replace(/var\(--pad-right\)/g, '14mm')
                        .replace(/var\(--pad-bottom\)/g, '42mm')
                        .replace(/var\(--pad-left\)/g, '12mm')
                        .replace(/var\(--content-h\)/g, '243mm')
                        .replace(/var\(--content-w\)/g, '182mm')
                        .replace(/var\(--slot-gap\)/g, '6mm')
                        .replace(/var\(--slot-h\)/g, '237mm')
                        .replace(/calc\s*\(\s*297mm\s*-\s*12mm\s*-\s*42mm\s*\)/gi, '243mm')
                        .replace(
                            /calc\s*\(\s*210mm\s*-\s*(?:var\(--pad-left\)|12mm)\s*-\s*(?:var\(--pad-right\)|14mm)\s*\)/gi,
                            '182mm')
                        .replace(
                            /calc\s*\(\s*(?:var\(--content-h\)|243mm)\s*-\s*(?:var\(--slot-gap\)|6mm)\s*\)/gi,
                            '237mm')
                        .replace(/calc\s*\([^)]*\)/gi, 'auto')
                        .replace(/var\s*\([^)]*\)/gi, 'auto')
                        .replace(
                            /\.page\s*:\s*not\(\s*:last-child\s*\)\s*\{\s*page-break-after\s*:\s*always;?\s*\}/gi,
                            '')
                        .replace(/\.page\s*\{[^}]*\}/gi, m => m.replace(
                            /page-break-(after|before)\s*:\s*always;?/gi, ''));

                    // Combine CSS with content
                    const finalNaskah = cleanedCss.trim() ?
                        '<style>' + cleanedCss + '</style>' + editorContent :
                        editorContent;

                    document.getElementById('editorNaskah').value = finalNaskah;
                    console.log('✅ TinyMCE content WITH CSS synced to textarea before save');
                    console.log('📄 CREATE - Final naskah length:', finalNaskah.length, 'chars');
                    console.log('📊 CREATE - Cleaned CSS length:', cleanedCss.length, 'chars');
                    console.log('📊 CREATE - CSS contains padding:', cleanedCss.includes('padding') ?
                        'YES' : 'NO');
                }

                // Check if PDF has been uploaded
                if (!fileInput.value.trim()) {
                    alert('⚠️ Harap klik "Cetak & Upload PDF" terlebih dahulu sebelum menyimpan!');
                    return; // Don't submit form
                }

                // Submit the form
                document.getElementById('formNaskah').submit();
            });

            uploadBtn.addEventListener('click', async function() {
                try {
                    if (!window.html2pdf) {
                        alert(
                            'html2pdf.js belum tersedia. Sistem sedang mencoba memuat library dari CDN. Silakan coba lagi sebentar.'
                        );
                        return;
                    }
                    if (!window.tinymce || !tinymce.get('editorNaskah')) {
                        alert('Editor TinyMCE belum siap.');
                        return;
                    }

                    let content = tinymce.get('editorNaskah').getContent() || '';
                    if (!content.trim()) {
                        alert('Konten surat masih kosong!');
                        return;
                    }

                    if (/\[\[[A-Z0-9_]+\]\]/i.test(content)) {
                        const vals = collectFormValues();
                        content = fillPlaceholders(content, vals);
                    }

                    content = content
                        .replace(
                            /<div[^>]*class=["'][^"']*html2pdf__page-break[^"']*["'][^>]*>\s*<\/div>/gi,
                            '')
                        .replace(
                            /<div[^>]*class=["'][^"']*(page-appendix|lampiran-page)[^"']*["'][^>]*>[\s\S]*?<\/div>/gi,
                            '');

                    const styleText = await fetchTemplateStyles();
                    console.log('✅ CREATE PDF: Using fresh template CSS');

                    // Clean CSS for mPDF compatibility (same as edit-letter.php)
                    const cleanedStyle = styleText
                        .replace(/:root\s*\{[\s\S]*?\}/g, '')
                        .replace(/@media[\s\S]*?\{[\s\S]*?\}/g, '')
                        .replace(/@page\s*\{[\s\S]*?\}/g, '')
                        .replace(/var\(--page-w\)/g, '210mm')
                        .replace(/var\(--page-h\)/g, '297mm')
                        .replace(/var\(--pad-top\)/g, '12mm')
                        .replace(/var\(--pad-right\)/g, '14mm')
                        .replace(/var\(--pad-bottom\)/g, '42mm')
                        .replace(/var\(--pad-left\)/g, '12mm')
                        .replace(/var\(--content-h\)/g, '243mm')
                        .replace(/var\(--content-w\)/g, '182mm')
                        .replace(/var\(--slot-gap\)/g, '6mm')
                        .replace(/var\(--slot-h\)/g, '237mm')
                        .replace(/calc\s*\(\s*297mm\s*-\s*12mm\s*-\s*42mm\s*\)/gi, '243mm')
                        .replace(
                            /calc\s*\(\s*210mm\s*-\s*(?:var\(--pad-left\)|12mm)\s*-\s*(?:var\(--pad-right\)|14mm)\s*\)/gi,
                            '182mm')
                        .replace(
                            /calc\s*\(\s*(?:var\(--content-h\)|243mm)\s*-\s*(?:var\(--slot-gap\)|6mm)\s*\)/gi,
                            '237mm')
                        .replace(/calc\s*\([^)]*\)/gi, 'auto')
                        .replace(/var\s*\([^)]*\)/gi, 'auto')
                        .replace(
                            /\.page\s*:\s*not\(\s*:last-child\s*\)\s*\{\s*page-break-after\s*:\s*always;?\s*\}/gi,
                            '')
                        .replace(/\.page\s*\{[^}]*\}/gi, m => m.replace(
                            /page-break-(after|before)\s*:\s*always;?/gi, ''));

                    const appendixCss = `
                    .page{
                        width: 210mm !important;
                        min-height: 297mm !important;
                        padding: 12mm 14mm 42mm 12mm !important;
                        margin: 0 auto !important;
                        position: relative !important;
                        background: #fff !important;
                    }
                    .sign-block {
                        margin-top: 8mm !important;
                        width: 100% !important;
                        display: flex !important;
                        justify-content: flex-end !important;
                        page-break-inside: avoid !important;
                        position: relative !important;
                    }
                    .sign {
                        width: 72mm !important;
                        text-align: left !important;
                        font-size: 14px !important;
                        position: relative !important;
                    }
                    .sign .place-date {
                        margin-bottom: 15mm !important;
                    }
                    .sign .signature-info {
                        display: block !important;
                        margin-top: 20mm !important;
                    }
                    .sign .name {
                        font-weight: 700 !important;
                        margin: 0 !important;
                    }
                    .sign .sub-name {
                        font-weight: 700 !important;
                    }
                    .sign .title {
                        margin-top: 1mm !important;
                    }
                    .sign-block .paraf {
                        position: absolute !important;
                        left: 0 !important;
                        bottom: 0 !important;
                        font-size: 14px !important;
                        color: #333 !important;
                        text-align: left !important;
                        font-weight: 500 !important;
                        white-space: nowrap !important;
                    }
                    .company-title {
                        text-align: right;
                        font-weight: 600;
                        font-size: 14px;
                        margin-bottom: 24px;
                    }
                    .footer {
                        position: absolute !important;
                        right: 14mm !important;
                        bottom: 8mm !important;
                        text-align: right !important;
                        font-size: 10px !important;
                    }
                    .page-appendix{
                        break-before: page !important;
                        page-break-before: always !important;
                    }
                    .lampiran-box{ 
                        display:flex; 
                        align-items:center; 
                        justify-content:center; 
                        border:0 !important; 
                    }
                    .lampiran-img,
                    .lampiran-frame{
                        width:100% !important; 
                        max-width:100% !important;
                        object-fit: contain; 
                        border:0 !important;
                    }
                    .html2pdf__page-break{ 
                        break-before: page !important; 
                        page-break-before: always !important; 
                    }
                    `;

                    const styles = [
                        cleanedStyle,
                        appendixCss,
                    ].join('\n');

                    (function appendLampiran() {
                        const inp = document.getElementById('lampiran');
                        if (!inp || !inp.files || !inp.files[0]) return;

                        const f = inp.files[0];
                        const type = (f.type || '').toLowerCase();
                        const url = URL.createObjectURL(f);

                        const lampiranInner = type.startsWith('image/') ?
                            `<img src="${url}" class="lampiran-img" alt="Lampiran">` :
                            (type === 'application/pdf' ?
                                `<iframe src="${url}" class="lampiran-frame" title="Lampiran PDF"></iframe>` :
                                `<p style="font-size:12px;text-align:center;">(Lampiran ${type || 'unknown'} tidak bisa dipratinjau)</p>`
                            );

                        content = content
                            .replace(
                                /<div[^>]*class=["'][^"']*html2pdf__page-break[^"']*["'][^>]*>\s*<\/div>/gi,
                                '')
                            .replace(
                                /<div[^>]*class=["'][^"']*(page-appendix|lampiran-page)[^"']*["'][^>]*>[\s\S]*?<\/div>/gi,
                                '');

                        content += `
                        <div class="html2pdf__page-break"></div>
                        <div class="page page-appendix">
                        <div class="lampiran-box">
                            <div class="lampiran-slot">${lampiranInner}</div>
                        </div>
                        </div>`;
                    })();

                    const finalHtml = [
                        '<!DOCTYPE html>',
                        '<html lang="id">',
                        '<head>',
                        '<meta charset="utf-8">',
                        '<meta name="viewport" content="width=device-width, initial-scale=1">',
                        '<style>',
                        styles,
                        '</style>',
                        '</head>',
                        '<body>',
                        content,
                        '</body>',
                        '</html>'
                    ].join('');

                    let safeHtml = finalHtml
                        .replace(/\sdata-mce-[a-z-]+="[^"]*"/gi, '');

                    const opt = {
                        margin: [0, 0, 0, 0],
                        filename: 'surat_temp.pdf',
                        image: {
                            type: 'png',
                            quality: 0.98
                        },
                        html2canvas: {
                            scale: 2,
                            useCORS: true,
                            allowTaint: true
                        },
                        jsPDF: {
                            unit: 'mm',
                            format: 'a4',
                            orientation: 'portrait',
                            compress: false
                        },
                        pagebreak: {
                            mode: ['css']
                        }
                    };

                    console.log('🔄 Membuat PDF dengan html2pdf.js…');
                    const worker = html2pdf().set(opt).from(safeHtml).toPdf();
                    const pdf = await worker.get('pdf');

                    let total = pdf.internal.getNumberOfPages();
                    while (total > 1) {
                        const ops = pdf.internal.pages[total];
                        const opCount = Array.isArray(ops) ? ops.filter(Boolean).length : 0;

                        if (opCount <= 3) {
                            pdf.deletePage(total);
                            total--;
                        } else {
                            break;
                        }
                    }

                    const blobPdf = pdf.output('blob');

                    const fd = new FormData();
                    fd.append('pdf_file', blobPdf, `surat_${Date.now()}.pdf`);

                    console.log('⬆️ Upload ke server…');
                    const res = await fetch('/jasa_marga/api/upload/pdf.php', {
                        method: 'POST',
                        body: fd
                    });
                    const data = await res.json();

                    if (data && data.ok && data.path) {
                        fileInput.value = data.path;
                        alert('✅ PDF berhasil diupload ke server!');
                    } else {
                        console.error('Upload response:', data);
                        alert('❌ Gagal upload PDF: ' + (data?.error || 'Tidak diketahui.'));
                    }

                } catch (err) {
                    console.error('❌ Error saat generate/upload PDF:', err);
                    alert('Terjadi kesalahan saat membuat atau mengupload PDF!');
                }
            });
        });
    </script>
</body>

</html>