<?php

/**
 * API endpoint for managing letter catatan (notes)
 * Only supervisors/approvers with higher positions can add notes
 */

session_start();
header('Content-Type: application/json');

include(__DIR__ . '/../includes/koneksi.php');
include(__DIR__ . '/../includes/functions.php');

// Check authentication
if (!isset($_SESSION['username'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Unauthorized']);
    exit();
}

$username = $_SESSION['username'];
$stmt = $conn->prepare("SELECT id, tipe, nama FROM users WHERE username = ?");
$stmt->bind_param("s", $username);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->fetch_assoc();
$stmt->close();

if (!$user) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'User not found']);
    exit();
}

$userId = (int)$user['id'];
$action = $_POST['action'] ?? '';

// Get subordinates and approver status
$subordinateIds = getSubordinatesForUser($conn, $userId);
$isSupervisor = !empty($subordinateIds);
$isApprover = isApprover($conn, $userId);

if ($action === 'save') {
    $officeId = isset($_POST['office_id']) ? (int)$_POST['office_id'] : 0;
    $catatan = $_POST['catatan'] ?? '';

    if ($officeId <= 0) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Invalid office_id']);
        exit();
    }

    // Get the letter to check creator
    $stmt = $conn->prepare("SELECT dibuat_oleh FROM office WHERE id = ?");
    $stmt->bind_param("i", $officeId);
    $stmt->execute();
    $letterResult = $stmt->get_result();
    $letter = $letterResult->fetch_assoc();
    $stmt->close();

    if (!$letter) {
        http_response_code(404);
        echo json_encode(['success' => false, 'error' => 'Letter not found']);
        exit();
    }

    $creatorId = (int)$letter['dibuat_oleh'];

    // Check permissions: can only add note if user is supervisor/approver of letter creator
    if ($creatorId === $userId || (!$isSupervisor && !$isApprover) || ($isSupervisor && !in_array($creatorId, $subordinateIds))) {
        http_response_code(403);
        echo json_encode(['success' => false, 'error' => 'You do not have permission to add notes for this letter']);
        exit();
    }

    // Sanitize catatan
    $catatan = trim($catatan);
    $catatan = $conn->real_escape_string($catatan);

    // Try to insert or update
    $checkStmt = $conn->prepare("SELECT id FROM letter_catatan WHERE office_id = ? AND user_id = ?");
    $checkStmt->bind_param("ii", $officeId, $userId);
    $checkStmt->execute();
    $checkResult = $checkStmt->get_result();
    $exists = $checkResult->num_rows > 0;
    $checkStmt->close();

    if ($exists) {
        // Update existing catatan
        $updateStmt = $conn->prepare("UPDATE letter_catatan SET catatan = ?, updated_at = NOW() WHERE office_id = ? AND user_id = ?");
        $updateStmt->bind_param("sii", $catatan, $officeId, $userId);
        if ($updateStmt->execute()) {
            echo json_encode(['success' => true, 'message' => 'Catatan berhasil diperbarui']);
        } else {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => 'Failed to update catatan']);
        }
        $updateStmt->close();
    } else {
        // Insert new catatan
        $insertStmt = $conn->prepare("INSERT INTO letter_catatan (office_id, user_id, catatan) VALUES (?, ?, ?)");
        $insertStmt->bind_param("iis", $officeId, $userId, $catatan);
        if ($insertStmt->execute()) {
            echo json_encode(['success' => true, 'message' => 'Catatan berhasil disimpan']);
        } else {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => 'Failed to save catatan']);
        }
        $insertStmt->close();
    }
} elseif ($action === 'get') {
    $officeId = isset($_GET['office_id']) ? (int)$_GET['office_id'] : 0;

    if ($officeId <= 0) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Invalid office_id']);
        exit();
    }

    // Get catatan for current user
    $stmt = $conn->prepare("SELECT catatan, created_at, updated_at FROM letter_catatan WHERE office_id = ? AND user_id = ?");
    $stmt->bind_param("ii", $officeId, $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    $catatan = $result->fetch_assoc();
    $stmt->close();

    echo json_encode([
        'success' => true,
        'catatan' => $catatan['catatan'] ?? '',
        'created_at' => $catatan['created_at'] ?? null,
        'updated_at' => $catatan['updated_at'] ?? null
    ]);
} else {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Invalid action']);
}
