<?php
require_once 'koneksi.php';

/**
 * SMART Classification Parser with Auto-Fix for Typos and Inconsistencies
 * Handles: missing leading zeros, typos (OP→OT), flexible matching
 */
class SmartClassificationParser
{
    private $pdo;

    public function __construct($pdo)
    {
        $this->pdo = $pdo;
    }

    public function importFromFile($filename)
    {
        if (!file_exists($filename)) {
            throw new Exception("File not found: $filename");
        }

        $content = file_get_contents($filename);
        if (!$content) {
            throw new Exception("Cannot read file: $filename");
        }

        echo "=== SMART CLASSIFICATION IMPORT ===\n\n";

        // STEP 1: Import Klasifikasi
        echo "STEP 1: Importing Klasifikasi...\n";
        $klasifikasiCount = $this->importKlasifikasi($content);
        echo "✓ Imported $klasifikasiCount klasifikasi records\n\n";

        // STEP 2: Import Jenis Surat  
        echo "STEP 2: Importing Jenis Surat...\n";
        $jenisCount = $this->importJenisSurat($content);
        echo "✓ Imported $jenisCount jenis surat records\n\n";

        // STEP 3: Import Subjenis Surat with SMART matching
        echo "STEP 3: Importing Subjenis Surat (with smart matching)...\n";
        $subjenisCount = $this->importSubjenisSuratSmart($content);
        echo "✓ Imported $subjenisCount subjenis surat records\n\n";

        echo "=== IMPORT COMPLETED ===\n";
        echo "Total: $klasifikasiCount klasifikasi + $jenisCount jenis + $subjenisCount subjenis\n";
    }

    private function importKlasifikasi($content)
    {
        $klasifikasiData = [];

        preg_match('/KLASIFIKASI\s*:(.*?)JENIS SURAT\s*:/s', $content, $matches);
        if (!$matches) return 0;

        $klasifikasiText = $matches[1];
        $lines = explode("\n", $klasifikasiText);

        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line)) continue;

            if (preg_match('/^\d+\.\s*([A-Z]{2})\s+(.+)$/', $line, $match)) {
                $kode = trim($match[1]);
                $nama = trim($match[2]);

                $klasifikasiData[] = [
                    'kode' => $kode,
                    'nama' => $nama,
                    'deskripsi' => "Klasifikasi $nama"
                ];
            }
        }

        $stmt = $this->pdo->prepare("
            INSERT IGNORE INTO klasifikasi_arsip (kode, nama, deskripsi, is_active, created_at, updated_at) 
            VALUES (?, ?, ?, 1, NOW(), NOW())
        ");

        $inserted = 0;
        foreach ($klasifikasiData as $data) {
            if ($stmt->execute([$data['kode'], $data['nama'], $data['deskripsi']])) {
                $inserted++;
            }
        }

        return $inserted;
    }

    private function importJenisSurat($content)
    {
        $jenisData = [];

        preg_match('/JENIS SURAT\s*:(.*?)SUBJENIS SURAT\s*:/s', $content, $matches);
        if (!$matches) return 0;

        $jenisText = $matches[1];

        preg_match_all('/\*(\d+)\.\s*Klasifikasi\s+([A-Z]{2,3})(.*?)(?=\*\d+\.\s*Klasifikasi|\z)/s', $jenisText, $klasifikasiBlocks, PREG_SET_ORDER);

        echo "DEBUG: Found " . count($klasifikasiBlocks) . " klasifikasi blocks\n";

        foreach ($klasifikasiBlocks as $block) {
            $klasifikasiKode = trim($block[2]);
            $jenisContent = $block[3];

            $stmt = $this->pdo->prepare("SELECT id FROM klasifikasi_arsip WHERE kode = ?");
            $stmt->execute([$klasifikasiKode]);
            $klasifikasiRecord = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$klasifikasiRecord) {
                echo "DEBUG: Klasifikasi $klasifikasiKode NOT FOUND\n";
                continue;
            }

            $klasifikasiId = $klasifikasiRecord['id'];

            preg_match_all('/^-\s*(\d+)\s+(.+?)$/m', $jenisContent, $jenisMatches, PREG_SET_ORDER);

            echo "DEBUG: $klasifikasiKode has " . count($jenisMatches) . " jenis\n";

            foreach ($jenisMatches as $jenisMatch) {
                $kodeJenis = str_pad(trim($jenisMatch[1]), 2, '0', STR_PAD_LEFT);
                $namaJenis = trim($jenisMatch[2]);

                $jenisData[] = [
                    'kode' => $kodeJenis,
                    'nama' => $namaJenis,
                    'deskripsi' => "Jenis Surat: $namaJenis",
                    'id_klasifikasi' => $klasifikasiId
                ];
            }
        }

        $stmt = $this->pdo->prepare("
            INSERT IGNORE INTO jenis_surat (kode, nama, deskripsi, id_klasifikasi, is_active, created_at, updated_at) 
            VALUES (?, ?, ?, ?, 1, NOW(), NOW())
        ");

        $inserted = 0;
        $failed = 0;
        foreach ($jenisData as $data) {
            try {
                $stmt->execute([$data['kode'], $data['nama'], $data['deskripsi'], $data['id_klasifikasi']]);
                if ($stmt->rowCount() > 0) {
                    $inserted++;
                } else {
                    $failed++;
                    echo "DEBUG: Failed to insert (duplicate?) - Kode:{$data['kode']}, Nama:{$data['nama']}, KlasID:{$data['id_klasifikasi']}\n";
                }
            } catch (Exception $e) {
                $failed++;
                echo "DEBUG: Error inserting {$data['kode']}: " . $e->getMessage() . "\n";
            }
        }

        echo "DEBUG: Inserted $inserted, Failed $failed\n";

        return $inserted;
    }

    /**
     * SMART subjenis import with flexible matching
     */
    private function importSubjenisSuratSmart($content)
    {
        $subjenisData = [];

        preg_match('/SUBJENIS SURAT\s*:(.*)/s', $content, $matches);
        if (!$matches) return 0;

        $subjenisText = $matches[1];

        // Split by klasifikasi blocks
        preg_match_all('/\*(\d+)\.\s*Klasifikasi\s+([A-Z]{2,3})(.*?)(?=\*\d+\.\s*Klasifikasi|\z)/s', $subjenisText, $klasifikasiBlocks, PREG_SET_ORDER);

        $totalImported = 0;
        $totalSkipped = 0;

        foreach ($klasifikasiBlocks as $block) {
            $klasifikasiKode = trim($block[2]);
            $subjenisContent = $block[3];

            // Get klasifikasi ID
            $stmt = $this->pdo->prepare("SELECT id FROM klasifikasi_arsip WHERE kode = ?");
            $stmt->execute([$klasifikasiKode]);
            $klasifikasiRecord = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$klasifikasiRecord) continue;
            $klasifikasiId = $klasifikasiRecord['id'];

            // Split by jenis blocks (*XX YY*)
            preg_match_all('/\*([A-Z]{2,3})\s+(\d+)\*(.*?)(?=\*[A-Z]{2,3}\s+\d+\*|\*\d+\.\s*Klasifikasi|\z)/s', $subjenisContent, $jenisBlocks, PREG_SET_ORDER);

            foreach ($jenisBlocks as $jenisBlock) {
                $jenisKodeRaw = $jenisBlock[2];
                $jenisKlasifikasiRaw = $jenisBlock[1];
                $subjenisContentBlock = $jenisBlock[3];

                // SMART MATCHING: Try multiple strategies to find jenis
                $jenisId = $this->findJenisIdSmart($jenisKlasifikasiRaw, $jenisKodeRaw, $klasifikasiId, $klasifikasiKode);

                if (!$jenisId) {
                    echo "  ⚠ Skipped: $jenisKlasifikasiRaw-$jenisKodeRaw (jenis not found)\n";
                    $totalSkipped++;
                    continue;
                }

                // Parse subjenis items
                preg_match_all('/^-\s*(\d+)\s*([a-z]\.)?\s*(.+?)$/m', $subjenisContentBlock, $subjenisMatches, PREG_SET_ORDER);

                foreach ($subjenisMatches as $subjenisMatch) {
                    $kodeSubjenis = str_pad(trim($subjenisMatch[1]), 2, '0', STR_PAD_LEFT);
                    $namaSubjenis = trim($subjenisMatch[3]);

                    // Clean up name
                    $namaSubjenis = preg_replace('/\s+$/', '', $namaSubjenis);

                    $subjenisData[] = [
                        'kode' => $kodeSubjenis,
                        'nama' => $namaSubjenis,
                        'deskripsi' => "Subjenis Surat: $namaSubjenis",
                        'id_jenis' => $jenisId,
                        'id_klasifikasi' => $klasifikasiId
                    ];

                    $totalImported++;
                }
            }
        }

        // Insert to database
        $stmt = $this->pdo->prepare("
            INSERT IGNORE INTO subjenis_surat (kode, nama, deskripsi, id_jenis, id_klasifikasi, created_at) 
            VALUES (?, ?, ?, ?, ?, NOW())
        ");

        $inserted = 0;
        foreach ($subjenisData as $data) {
            if ($stmt->execute([
                $data['kode'],
                $data['nama'],
                $data['deskripsi'],
                $data['id_jenis'],
                $data['id_klasifikasi']
            ])) {
                $inserted++;
            }
        }

        echo "\nSummary: $inserted imported, $totalSkipped skipped\n";

        return $inserted;
    }

    /**
     * SMART MATCHING: Try multiple strategies to find jenis ID
     */
    private function findJenisIdSmart($jenisKlasifikasiRaw, $jenisKodeRaw, $klasifikasiId, $klasifikasiKodeExpected)
    {
        // Strategy 1: Direct match with klasifikasi and kode (with leading zero)
        $kodeWithZero = str_pad($jenisKodeRaw, 2, '0', STR_PAD_LEFT);
        $stmt = $this->pdo->prepare("SELECT id FROM jenis_surat WHERE kode = ? AND id_klasifikasi = ?");
        $stmt->execute([$kodeWithZero, $klasifikasiId]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($result) return $result['id'];

        // Strategy 2: Try without leading zero
        $kodeWithoutZero = ltrim($jenisKodeRaw, '0');
        if ($kodeWithoutZero !== $kodeWithZero) {
            $stmt->execute([$kodeWithoutZero, $klasifikasiId]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($result) return $result['id'];
        }

        // Strategy 3: Handle typo - if klasifikasi in subjenis doesn't match expected
        if ($jenisKlasifikasiRaw !== $klasifikasiKodeExpected) {
            // Common typos
            $typoMap = [
                'OP' => 'OT',
                'OT' => 'OP'
            ];

            $correctedKode = $typoMap[$jenisKlasifikasiRaw] ?? null;
            if ($correctedKode) {
                // Get correct klasifikasi ID
                $stmtKlas = $this->pdo->prepare("SELECT id FROM klasifikasi_arsip WHERE kode = ?");
                $stmtKlas->execute([$correctedKode]);
                $klasRecord = $stmtKlas->fetch(PDO::FETCH_ASSOC);

                if ($klasRecord) {
                    $stmt->execute([$kodeWithZero, $klasRecord['id']]);
                    $result = $stmt->fetch(PDO::FETCH_ASSOC);
                    if ($result) {
                        echo "  ✓ Auto-fixed typo: $jenisKlasifikasiRaw → $correctedKode\n";
                        return $result['id'];
                    }
                }
            }
        }

        // Strategy 4: Find by position if kode doesn't match
        // Get all jenis for this klasifikasi ordered by kode
        $stmtAll = $this->pdo->prepare("SELECT id, kode FROM jenis_surat WHERE id_klasifikasi = ? ORDER BY CAST(kode AS UNSIGNED)");
        $stmtAll->execute([$klasifikasiId]);
        $allJenis = $stmtAll->fetchAll(PDO::FETCH_ASSOC);

        // Try to match by numeric value
        $numericValue = intval($jenisKodeRaw);
        foreach ($allJenis as $idx => $jenis) {
            $jenisNumeric = intval($jenis['kode']);
            if ($jenisNumeric === $numericValue) {
                return $jenis['id'];
            }
        }

        return null;
    }
}

// Execute smart import
try {
    $host = 'localhost';
    $dbname = 'jasa_marga';
    $username = 'root';
    $password = '';

    $dsn = "mysql:host=$host;dbname=$dbname;charset=utf8mb4";
    $pdo = new PDO($dsn, $username, $password, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
    ]);

    echo "Starting smart import with auto-fix for typos and inconsistencies...\n\n";

    $parser = new SmartClassificationParser($pdo);
    $parser->importFromFile('KLASIFIKASI JENIS SUBJENIS_SURAT ULUM.txt');
} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    echo $e->getTraceAsString() . "\n";
}
