<?php
require_once 'koneksi.php';

/**
 * IMPROVED Classification Data Parser
 * Fixed regex patterns to properly capture all jenis and subjenis
 */
class ImprovedClassificationParser
{
    private $pdo;

    public function __construct($pdo)
    {
        $this->pdo = $pdo;
    }

    /**
     * Main import function with sequential processing
     */
    public function importFromFile($filename)
    {
        if (!file_exists($filename)) {
            throw new Exception("File not found: $filename");
        }

        $content = file_get_contents($filename);
        if (!$content) {
            throw new Exception("Cannot read file: $filename");
        }

        echo "=== IMPROVED CLASSIFICATION IMPORT ===\n\n";

        // STEP 1: Import Klasifikasi
        echo "STEP 1: Importing Klasifikasi...\n";
        $klasifikasiCount = $this->importKlasifikasi($content);
        echo "✓ Imported $klasifikasiCount klasifikasi records\n\n";

        // STEP 2: Import Jenis Surat  
        echo "STEP 2: Importing Jenis Surat...\n";
        $jenisCount = $this->importJenisSurat($content);
        echo "✓ Imported $jenisCount jenis surat records\n\n";

        // STEP 3: Import Subjenis Surat
        echo "STEP 3: Importing Subjenis Surat...\n";
        $subjenisCount = $this->importSubjenisSurat($content);
        echo "✓ Imported $subjenisCount subjenis surat records\n\n";

        echo "=== IMPORT COMPLETED SUCCESSFULLY ===\n";
        echo "Total: $klasifikasiCount klasifikasi + $jenisCount jenis + $subjenisCount subjenis\n";
    }

    /**
     * Import klasifikasi only
     */
    private function importKlasifikasi($content)
    {
        $klasifikasiData = [];

        // Extract KLASIFIKASI section
        preg_match('/KLASIFIKASI\s*:(.*?)JENIS SURAT\s*:/s', $content, $matches);
        if (!$matches) return 0;

        $klasifikasiText = $matches[1];
        $lines = explode("\n", $klasifikasiText);

        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line)) continue;

            // Match pattern: "1. PR PERANCANGAN"
            if (preg_match('/^\d+\.\s*([A-Z]{2})\s+(.+)$/', $line, $match)) {
                $kode = trim($match[1]);
                $nama = trim($match[2]);

                $klasifikasiData[] = [
                    'kode' => $kode,
                    'nama' => $nama,
                    'deskripsi' => "Klasifikasi $nama"
                ];
            }
        }

        // Insert to database
        $stmt = $this->pdo->prepare("
            INSERT IGNORE INTO klasifikasi_arsip (kode, nama, deskripsi, is_active, created_at, updated_at) 
            VALUES (?, ?, ?, 1, NOW(), NOW())
        ");

        $inserted = 0;
        foreach ($klasifikasiData as $data) {
            if ($stmt->execute([$data['kode'], $data['nama'], $data['deskripsi']])) {
                $inserted++;
            }
        }

        return $inserted;
    }

    /**
     * Import jenis surat with IMPROVED regex
     */
    private function importJenisSurat($content)
    {
        $jenisData = [];

        // Extract JENIS SURAT section
        preg_match('/JENIS SURAT\s*:(.*?)SUBJENIS SURAT\s*:/s', $content, $matches);
        if (!$matches) {
            echo "ERROR: Cannot find JENIS SURAT section\n";
            return 0;
        }

        $jenisText = $matches[1];

        // Split by klasifikasi blocks: *1. Klasifikasi XX
        preg_match_all('/\*(\d+)\.\s*Klasifikasi\s+([A-Z]{2,3})(.*?)(?=\*\d+\.\s*Klasifikasi|\z)/s', $jenisText, $klasifikasiBlocks, PREG_SET_ORDER);

        echo "DEBUG: Found " . count($klasifikasiBlocks) . " klasifikasi blocks in JENIS section\n";

        foreach ($klasifikasiBlocks as $block) {
            $klasifikasiKode = trim($block[2]);
            $jenisContent = $block[3];

            // Find klasifikasi ID from database
            $stmt = $this->pdo->prepare("SELECT id FROM klasifikasi_arsip WHERE kode = ?");
            $stmt->execute([$klasifikasiKode]);
            $klasifikasiRecord = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$klasifikasiRecord) {
                echo "Warning: Klasifikasi $klasifikasiKode not found in database\n";
                continue;
            }

            $klasifikasiId = $klasifikasiRecord['id'];

            // IMPROVED: Parse jenis items - match lines starting with "- XX "
            // This pattern captures everything after the number until end of line
            preg_match_all('/^-\s*(\d+)\s+(.+?)$/m', $jenisContent, $jenisMatches, PREG_SET_ORDER);

            echo "DEBUG: Found " . count($jenisMatches) . " jenis for klasifikasi $klasifikasiKode\n";

            foreach ($jenisMatches as $jenisMatch) {
                $kodeJenis = str_pad(trim($jenisMatch[1]), 2, '0', STR_PAD_LEFT);
                $namaJenis = trim($jenisMatch[2]);

                // Remove trailing spaces and clean up the name
                $namaJenis = preg_replace('/\s+$/', '', $namaJenis);

                $jenisData[] = [
                    'kode' => $kodeJenis,
                    'nama' => $namaJenis,
                    'deskripsi' => "Jenis Surat: $namaJenis",
                    'id_klasifikasi' => $klasifikasiId,
                    'klasifikasi_kode' => $klasifikasiKode
                ];

                echo "  - $klasifikasiKode-$kodeJenis: $namaJenis\n";
            }
        }

        // Insert to database
        $stmt = $this->pdo->prepare("
            INSERT IGNORE INTO jenis_surat (kode, nama, deskripsi, id_klasifikasi, is_active, created_at, updated_at) 
            VALUES (?, ?, ?, ?, 1, NOW(), NOW())
        ");

        $inserted = 0;
        foreach ($jenisData as $data) {
            if ($stmt->execute([$data['kode'], $data['nama'], $data['deskripsi'], $data['id_klasifikasi']])) {
                $inserted++;
            }
        }

        return $inserted;
    }

    /**
     * Import subjenis surat with IMPROVED regex
     */
    private function importSubjenisSurat($content)
    {
        $subjenisData = [];

        // Extract SUBJENIS SURAT section
        preg_match('/SUBJENIS SURAT\s*:(.*)/s', $content, $matches);
        if (!$matches) {
            echo "ERROR: Cannot find SUBJENIS SURAT section\n";
            return 0;
        }

        $subjenisText = $matches[1];

        // Split by klasifikasi blocks
        preg_match_all('/\*(\d+)\.\s*Klasifikasi\s+([A-Z]{2,3})(.*?)(?=\*\d+\.\s*Klasifikasi|\z)/s', $subjenisText, $klasifikasiBlocks, PREG_SET_ORDER);

        echo "DEBUG: Found " . count($klasifikasiBlocks) . " klasifikasi blocks in SUBJENIS section\n";

        foreach ($klasifikasiBlocks as $block) {
            $klasifikasiKode = trim($block[2]);
            $subjenisContent = $block[3];

            // Find klasifikasi ID
            $stmt = $this->pdo->prepare("SELECT id FROM klasifikasi_arsip WHERE kode = ?");
            $stmt->execute([$klasifikasiKode]);
            $klasifikasiRecord = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$klasifikasiRecord) continue;
            $klasifikasiId = $klasifikasiRecord['id'];

            // Split by jenis blocks (*PR 01*)
            preg_match_all('/\*([A-Z]{2,3})\s+(\d+)\*(.*?)(?=\*[A-Z]{2,3}\s+\d+\*|\*\d+\.\s*Klasifikasi|\z)/s', $subjenisContent, $jenisBlocks, PREG_SET_ORDER);

            echo "DEBUG: Found " . count($jenisBlocks) . " jenis blocks for klasifikasi $klasifikasiKode\n";

            foreach ($jenisBlocks as $jenisBlock) {
                $jenisKode = str_pad($jenisBlock[2], 2, '0', STR_PAD_LEFT);
                $subjenisContentBlock = $jenisBlock[3];

                // Find jenis ID from database
                $stmt = $this->pdo->prepare("SELECT id FROM jenis_surat WHERE kode = ? AND id_klasifikasi = ?");
                $stmt->execute([$jenisKode, $klasifikasiId]);
                $jenisRecord = $stmt->fetch(PDO::FETCH_ASSOC);

                if (!$jenisRecord) {
                    echo "Warning: Jenis $jenisKode not found for klasifikasi $klasifikasiKode\n";
                    continue;
                }

                $jenisId = $jenisRecord['id'];

                // IMPROVED: Parse subjenis items - match lines starting with "- XX "
                preg_match_all('/^-\s*(\d+)\s*([a-z]\.)?\s*(.+?)$/m', $subjenisContentBlock, $subjenisMatches, PREG_SET_ORDER);

                echo "  DEBUG: Found " . count($subjenisMatches) . " subjenis for jenis $klasifikasiKode-$jenisKode\n";

                foreach ($subjenisMatches as $subjenisMatch) {
                    $kodeSubjenis = str_pad(trim($subjenisMatch[1]), 2, '0', STR_PAD_LEFT);
                    $namaSubjenis = trim($subjenisMatch[3]);

                    // Clean up the name
                    $namaSubjenis = preg_replace('/\s+$/', '', $namaSubjenis);

                    $subjenisData[] = [
                        'kode' => $kodeSubjenis,
                        'nama' => $namaSubjenis,
                        'deskripsi' => "Subjenis Surat: $namaSubjenis",
                        'id_jenis' => $jenisId,
                        'id_klasifikasi' => $klasifikasiId
                    ];

                    echo "    - $klasifikasiKode-$jenisKode-$kodeSubjenis: $namaSubjenis\n";
                }
            }
        }

        // Insert to database
        $stmt = $this->pdo->prepare("
            INSERT IGNORE INTO subjenis_surat (kode, nama, deskripsi, id_jenis, id_klasifikasi, created_at) 
            VALUES (?, ?, ?, ?, ?, NOW())
        ");

        $inserted = 0;
        foreach ($subjenisData as $data) {
            if ($stmt->execute([
                $data['kode'],
                $data['nama'],
                $data['deskripsi'],
                $data['id_jenis'],
                $data['id_klasifikasi']
            ])) {
                $inserted++;
            }
        }

        return $inserted;
    }
}

// Execute improved import
try {
    // Database connection
    $host = 'localhost';
    $dbname = 'jasa_marga';
    $username = 'root';
    $password = '';

    $dsn = "mysql:host=$host;dbname=$dbname;charset=utf8mb4";
    $pdo = new PDO($dsn, $username, $password, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
    ]);

    $parser = new ImprovedClassificationParser($pdo);
    $parser->importFromFile('KLASIFIKASI JENIS SUBJENIS_SURAT ULUM.txt');
} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    echo $e->getTraceAsString() . "\n";
}
