<?php
require_once 'koneksi.php';

/**
 * Classification Data Parser for ULUM System
 * Handles hierarchical import: Klasifikasi -> Jenis -> Subjenis
 */
class ClassificationDataParser
{
    private $pdo;
    private $klasifikasiData = [];
    private $jenisData = [];
    private $subjenisData = [];

    public function __construct($pdo)
    {
        $this->pdo = $pdo;
    }

    /**
     * Main import function
     */
    public function importFromFile($filename)
    {
        if (!file_exists($filename)) {
            throw new Exception("File not found: $filename");
        }

        $content = file_get_contents($filename);
        if (!$content) {
            throw new Exception("Cannot read file: $filename");
        }

        echo "Parsing classification file...\n";

        // Parse each section
        $this->parseKlasifikasi($content);
        $this->parseJenisSurat($content);
        $this->parseSubjenisSurat($content);

        // Display parse results
        echo "Parse Results:\n";
        echo "- Klasifikasi: " . count($this->klasifikasiData) . " items\n";
        echo "- Jenis Surat: " . count($this->jenisData) . " items\n";
        echo "- Subjenis Surat: " . count($this->subjenisData) . " items\n\n";

        // Show sample data for debugging
        echo "Data Summary:\n";
        print_r([
            'klasifikasi_count' => count($this->klasifikasiData),
            'jenis_count' => count($this->jenisData),
            'subjenis_count' => count($this->subjenisData),
            'klasifikasi_sample' => array_slice($this->klasifikasiData, 0, 3),
            'jenis_sample' => array_slice($this->jenisData, 0, 3),
            'subjenis_sample' => array_slice($this->subjenisData, 0, 3)
        ]);

        // Insert to database
        echo "\nInserting data to database...\n";
        $this->insertKlasifikasi();
        $this->insertJenisSurat();
        $this->insertSubjenisSurat();

        echo "Insert Results:\n";
        echo "- Klasifikasi inserted: " . count($this->klasifikasiData) . " items\n";
        echo "- Jenis Surat inserted: " . count($this->jenisData) . " items\n";
        echo "- Subjenis Surat inserted: " . count($this->subjenisData) . " items\n\n";

        echo "Data import completed successfully!\n";
    }

    /**
     * Parse KLASIFIKASI section
     */
    private function parseKlasifikasi($content)
    {
        // Extract section between "KLASIFIKASI :" and "JENIS SURAT :"
        preg_match('/KLASIFIKASI\s*:(.*?)JENIS SURAT\s*:/s', $content, $matches);
        if (!$matches) {
            echo "Warning: KLASIFIKASI section not found\n";
            return;
        }

        $klasifikasiText = $matches[1];
        $lines = explode("\n", $klasifikasiText);

        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line)) continue;

            // Match pattern: "1. PR PERANCANGAN"
            if (preg_match('/^\d+\.\s*([A-Z]{2})\s+(.+)$/', $line, $match)) {
                $kode = trim($match[1]);
                $nama = trim($match[2]);

                $this->klasifikasiData[] = [
                    'kode' => $kode,
                    'nama' => $nama,
                    'deskripsi' => "Klasifikasi $nama"
                ];
            }
        }
    }

    /**
     * Parse JENIS SURAT section
     */
    private function parseJenisSurat($content)
    {
        // Extract JENIS SURAT section
        preg_match('/JENIS SURAT\s*:(.*?)SUBJENIS SURAT\s*:/s', $content, $matches);
        if (!$matches) {
            echo "Warning: JENIS SURAT section not found\n";
            return;
        }

        $jenisText = $matches[1];
        echo "DEBUG: Found JENIS SURAT section, length: " . strlen($jenisText) . "\n";

        // Split by klasifikasi blocks
        preg_match_all('/\*(\d+)\.\s*Klasifikasi\s+([A-Z]{2,3})(.*?)(?=\*\d+\.\s*Klasifikasi|\z)/s', $jenisText, $klasifikasiBlocks, PREG_SET_ORDER);

        echo "DEBUG: Found " . count($klasifikasiBlocks) . " klasifikasi blocks\n";

        foreach ($klasifikasiBlocks as $i => $block) {
            $klasifikasiKode = trim($block[2]);
            $jenisContent = $block[3];

            echo "DEBUG: Processing klasifikasi $klasifikasiKode\n";

            // Find matching klasifikasi ID
            $klasifikasiId = $this->getKlasifikasiIdByKode($klasifikasiKode);
            if (!$klasifikasiId) {
                echo "Warning: Klasifikasi $klasifikasiKode not found in database\n";
                continue;
            }

            // Parse jenis items within this klasifikasi
            preg_match_all('/^-\s*(\d+)\s+(.+?)(?=series:|uraian:|^-|\*|$)/ms', $jenisContent, $jenisMatches, PREG_SET_ORDER);

            echo "DEBUG: Found " . count($jenisMatches) . " jenis for klasifikasi $klasifikasiKode\n";

            foreach ($jenisMatches as $jenisMatch) {
                $kodeJenis = str_pad(trim($jenisMatch[1]), 2, '0', STR_PAD_LEFT);
                $namaJenis = trim($jenisMatch[2]);

                $this->jenisData[] = [
                    'kode' => $kodeJenis,
                    'nama' => $namaJenis,
                    'deskripsi' => "Jenis Surat: $namaJenis",
                    'id_klasifikasi' => $klasifikasiId
                ];
            }
        }
    }

    /**
     * Parse SUBJENIS SURAT section  
     */
    private function parseSubjenisSurat($content)
    {
        // Extract SUBJENIS SURAT section (from "SUBJENIS SURAT :" to end)
        preg_match('/SUBJENIS SURAT\s*:(.*)/s', $content, $matches);
        if (!$matches) {
            echo "Warning: SUBJENIS SURAT section not found\n";
            return;
        }

        $subjenisText = $matches[1];
        echo "DEBUG: Found SUBJENIS SURAT section, length: " . strlen($subjenisText) . "\n";

        // Split by klasifikasi blocks (*1. Klasifikasi PR)
        preg_match_all('/\*(\d+)\.\s*Klasifikasi\s+([A-Z]{2,3})(.*?)(?=\*\d+\.\s*Klasifikasi|\z)/s', $subjenisText, $klasifikasiBlocks, PREG_SET_ORDER);

        echo "DEBUG: Found " . count($klasifikasiBlocks) . " klasifikasi blocks in subjenis\n";

        foreach ($klasifikasiBlocks as $block) {
            $klasifikasiKode = trim($block[2]);
            $subjenisContent = $block[3];

            echo "DEBUG: Processing subjenis for klasifikasi $klasifikasiKode\n";

            // Find klasifikasi ID
            $klasifikasiId = $this->getKlasifikasiIdByKode($klasifikasiKode);
            if (!$klasifikasiId) continue;

            // Split by jenis blocks (*PR 01*)
            preg_match_all('/\*([A-Z]{2,3})\s+(\d+)\*(.*?)(?=\*[A-Z]{2,3}\s+\d+\*|\*\d+\.\s*Klasifikasi|\z)/s', $subjenisContent, $jenisBlocks, PREG_SET_ORDER);

            echo "DEBUG: Found " . count($jenisBlocks) . " jenis blocks for klasifikasi $klasifikasiKode\n";

            foreach ($jenisBlocks as $jenisBlock) {
                $jenisKode = str_pad($jenisBlock[2], 2, '0', STR_PAD_LEFT);
                $subjenisContent = $jenisBlock[3];

                // Find jenis ID 
                $jenisId = $this->getJenisIdByKodeAndKlasifikasi($jenisKode, $klasifikasiId);
                if (!$jenisId) {
                    echo "Warning: Jenis $jenisKode not found for klasifikasi $klasifikasiKode\n";
                    continue;
                }

                // Parse subjenis items (- 01 a. Name or - 01 Name)
                preg_match_all('/^-\s*(\d+)\s*([a-z]\.)?\s*(.+?)(?=series:|uraian:|^-|$)/ms', $subjenisContent, $subjenisMatches, PREG_SET_ORDER);

                echo "DEBUG: Found " . count($subjenisMatches) . " subjenis for jenis $jenisKode\n";

                foreach ($subjenisMatches as $subjenisMatch) {
                    $kodeSubjenis = str_pad(trim($subjenisMatch[1]), 2, '0', STR_PAD_LEFT);
                    $namaSubjenis = trim($subjenisMatch[3]);

                    $this->subjenisData[] = [
                        'kode' => $kodeSubjenis,
                        'nama' => $namaSubjenis,
                        'deskripsi' => "Subjenis Surat: $namaSubjenis",
                        'id_jenis' => $jenisId,
                        'id_klasifikasi' => $klasifikasiId
                    ];
                }
            }
        }
    }

    /**
     * Get klasifikasi ID by kode
     */
    private function getKlasifikasiIdByKode($kode)
    {
        $stmt = $this->pdo->prepare("SELECT id FROM klasifikasi_arsip WHERE kode = ?");
        $stmt->execute([$kode]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result ? $result['id'] : null;
    }

    /**
     * Get jenis ID by kode and klasifikasi
     */
    private function getJenisIdByKodeAndKlasifikasi($kode, $klasifikasiId)
    {
        $stmt = $this->pdo->prepare("SELECT id FROM jenis_surat WHERE kode = ? AND id_klasifikasi = ?");
        $stmt->execute([$kode, $klasifikasiId]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result ? $result['id'] : null;
    }

    /**
     * Insert klasifikasi data
     */
    private function insertKlasifikasi()
    {
        $stmt = $this->pdo->prepare("
            INSERT IGNORE INTO klasifikasi_arsip (kode, nama, deskripsi, is_active, created_at, updated_at) 
            VALUES (?, ?, ?, 1, NOW(), NOW())
        ");

        foreach ($this->klasifikasiData as $data) {
            $stmt->execute([$data['kode'], $data['nama'], $data['deskripsi']]);
        }
    }

    /**
     * Insert jenis surat data
     */
    private function insertJenisSurat()
    {
        $stmt = $this->pdo->prepare("
            INSERT IGNORE INTO jenis_surat (kode, nama, deskripsi, id_klasifikasi, is_active, created_at, updated_at) 
            VALUES (?, ?, ?, ?, 1, NOW(), NOW())
        ");

        foreach ($this->jenisData as $data) {
            $stmt->execute([$data['kode'], $data['nama'], $data['deskripsi'], $data['id_klasifikasi']]);
        }
    }

    /**
     * Insert subjenis surat data
     */
    private function insertSubjenisSurat()
    {
        $stmt = $this->pdo->prepare("
            INSERT IGNORE INTO subjenis_surat (kode, nama, deskripsi, id_jenis, id_klasifikasi, created_at) 
            VALUES (?, ?, ?, ?, ?, NOW())
        ");

        foreach ($this->subjenisData as $data) {
            $stmt->execute([
                $data['kode'],
                $data['nama'],
                $data['deskripsi'],
                $data['id_jenis'],
                $data['id_klasifikasi']
            ]);
        }
    }
}

// Execute import
try {
    // Database connection
    $host = 'localhost';
    $dbname = 'jasa_marga';
    $username = 'root';
    $password = '';

    $dsn = "mysql:host=$host;dbname=$dbname;charset=utf8mb4";
    $pdo = new PDO($dsn, $username, $password, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
    ]);

    $parser = new ClassificationDataParser($pdo);
    $parser->importFromFile('KLASIFIKASI JENIS SUBJENIS_SURAT ULUM.txt');
} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
}
