<?php

/**
 * Master Data Parser and Inserter for Hierarchical Classification System
 * 
 * This script parses KLASIFIKASI JENIS SUBJENIS_SURAT ULUM.txt and inserts
 * structured data into the new hierarchical classification tables
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once 'koneksi.php';

class ClassificationDataParser
{

    private $conn;
    private $klasifikasiData = [];
    private $jenisData = [];
    private $subjenisData = [];

    public function __construct($dbConnection)
    {
        $this->conn = $dbConnection;
    }

    /**
     * Parse the classification file and extract structured data
     */
    public function parseClassificationFile($filePath)
    {
        if (!file_exists($filePath)) {
            throw new Exception("Classification file not found: $filePath");
        }

        $content = file_get_contents($filePath);
        if (!$content) {
            throw new Exception("Could not read classification file");
        }

        $this->parseKlasifikasi($content);
        $this->parseJenisSurat($content);
        $this->parseSubjenisSurat($content);

        return [
            'klasifikasi' => count($this->klasifikasiData),
            'jenis' => count($this->jenisData),
            'subjenis' => count($this->subjenisData)
        ];
    }

    /**
     * Parse KLASIFIKASI section
     */
    private function parseKlasifikasi($content)
    {
        // Extract KLASIFIKASI section
        preg_match('/KLASIFIKASI\s*:(.*?)JENIS SURAT\s*:/s', $content, $matches);
        if (!$matches) return;

        $klasifikasiText = $matches[1];
        $lines = explode("\n", $klasifikasiText);

        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line)) continue;

            // Match pattern: "1. PR PERANCANGAN"
            if (preg_match('/^\d+\.\s*([A-Z]{2})\s+(.+)$/', $line, $match)) {
                $kode = trim($match[1]);
                $nama = trim($match[2]);

                $this->klasifikasiData[] = [
                    'kode' => $kode,
                    'nama' => $nama,
                    'deskripsi' => "Klasifikasi $nama"
                ];
            }
        }
    }

    /**
     * Parse JENIS SURAT section
     */
    private function parseJenisSurat($content)
    {
        // Extract JENIS SURAT section
        preg_match('/JENIS SURAT\s*:(.*?)SUBJENIS SURAT\s*:/s', $content, $matches);
        if (!$matches) return;

        $jenisText = $matches[1];

        // Split by klasifikasi blocks
        preg_match_all('/\*(\d+)\.\s*Klasifikasi\s+([A-Z]{2,3})(.*?)(?=\*\d+\.\s*Klasifikasi|\z)/s', $jenisText, $klasifikasiBlocks, PREG_SET_ORDER);

        foreach ($klasifikasiBlocks as $block) {
            $klasifikasiKode = trim($block[2]);
            $jenisContent = $block[3];

            // Find matching klasifikasi ID
            $klasifikasiId = $this->getKlasifikasiIdByKode($klasifikasiKode);
            if (!$klasifikasiId) continue;

            // Parse jenis items within this klasifikasi
            preg_match_all('/^-\s*(\d+)\s+(.+?)(?=series:|uraian:|^-|\*|$)/ms', $jenisContent, $jenisMatches, PREG_SET_ORDER);

            foreach ($jenisMatches as $jenisMatch) {
                $kodeJenis = str_pad(trim($jenisMatch[1]), 2, '0', STR_PAD_LEFT);
                $namaJenis = trim($jenisMatch[2]);

                // Extract description/series info if available
                $deskripsi = '';
                $startPos = strpos($jenisContent, $jenisMatch[0]);
                $endPos = strpos($jenisContent, '- ' . ($jenisMatch[1] + 1), $startPos);
                if ($endPos === false) {
                    $endPos = strpos($jenisContent, '*', $startPos + 1);
                }

                if ($endPos !== false) {
                    $jenisBlock = substr($jenisContent, $startPos, $endPos - $startPos);
                } else {
                    $jenisBlock = substr($jenisContent, $startPos);
                }

                // Extract uraian
                if (preg_match('/uraian:\s*(.+?)(?=series:|uraian:|^-|\*|$)/s', $jenisBlock, $uraianMatch)) {
                    $deskripsi = trim($uraianMatch[1]);
                }

                $this->jenisData[] = [
                    'kode' => $kodeJenis,
                    'nama' => $namaJenis,
                    'deskripsi' => $deskripsi,
                    'id_klasifikasi' => $klasifikasiId,
                    'klasifikasi_kode' => $klasifikasiKode
                ];
            }
        }
    }

    /**
     * Parse SUBJENIS SURAT section
     */
    private function parseSubjenisSurat($content)
    {
        // Extract SUBJENIS SURAT section
        preg_match('/SUBJENIS SURAT\s*:(.*?)$/s', $content, $matches);
        if (!$matches) return;

        $subjenisText = $matches[1];

        // Split by klasifikasi blocks
        preg_match_all('/\*(\d+)\.\s*Klasifikasi\s+([A-Z]{2,3})(.*?)(?=\*\d+\.\s*Klasifikasi|\z)/s', $subjenisText, $klasifikasiBlocks, PREG_SET_ORDER);

        foreach ($klasifikasiBlocks as $block) {
            $klasifikasiKode = trim($block[2]);
            $subjenisContent = $block[3];

            // Find matching klasifikasi ID
            $klasifikasiId = $this->getKlasifikasiIdByKode($klasifikasiKode);
            if (!$klasifikasiId) continue;

            // Parse subjenis blocks like "*PR 01*"
            preg_match_all('/\*([A-Z]{2,3})\s+(\d+)\*(.*?)(?=\*[A-Z]{2,3}\s+\d+\*|\*\d+\.\s*Klasifikasi|\z)/s', $subjenisContent, $jenisBlocks, PREG_SET_ORDER);

            foreach ($jenisBlocks as $jenisBlock) {
                $jenisKode = str_pad(trim($jenisBlock[2]), 2, '0', STR_PAD_LEFT);
                $jenisContent = $jenisBlock[3];

                // Find matching jenis ID
                $jenisId = $this->getJenisIdByKode($jenisKode, $klasifikasiId);
                if (!$jenisId) continue;

                // Parse subjenis items
                preg_match_all('/^-\s*(\d+)\s+([a-z]\.\s*)?(.+?)(?=uraian:|series:|^-|\*|$)/ms', $jenisContent, $subjenisMatches, PREG_SET_ORDER);

                foreach ($subjenisMatches as $subjenisMatch) {
                    $kodeSubjenis = str_pad(trim($subjenisMatch[1]), 2, '0', STR_PAD_LEFT);
                    $namaSubjenis = trim($subjenisMatch[3]);

                    // Extract description
                    $deskripsi = '';
                    $startPos = strpos($jenisContent, $subjenisMatch[0]);
                    $nextMatch = strpos($jenisContent, '- ' . ($subjenisMatch[1] + 1), $startPos);
                    if ($nextMatch === false) {
                        $nextMatch = strpos($jenisContent, '*', $startPos + 1);
                    }

                    if ($nextMatch !== false) {
                        $subjenisBlock = substr($jenisContent, $startPos, $nextMatch - $startPos);
                    } else {
                        $subjenisBlock = substr($jenisContent, $startPos);
                    }

                    if (preg_match('/uraian:\s*(.+?)(?=series:|uraian:|^-|\*|$)/s', $subjenisBlock, $uraianMatch)) {
                        $deskripsi = trim($uraianMatch[1]);
                    }

                    $this->subjenisData[] = [
                        'kode' => $kodeSubjenis,
                        'nama' => $namaSubjenis,
                        'deskripsi' => $deskripsi,
                        'id_jenis' => $jenisId,
                        'id_klasifikasi' => $klasifikasiId
                    ];
                }
            }
        }
    }

    /**
     * Insert all parsed data into database
     */
    public function insertData()
    {
        $this->conn->begin_transaction();

        try {
            $results = [];

            // Insert klasifikasi data
            $results['klasifikasi'] = $this->insertKlasifikasiData();

            // Insert jenis surat data
            $results['jenis'] = $this->insertJenisData();

            // Insert subjenis surat data
            $results['subjenis'] = $this->insertSubjenisData();

            $this->conn->commit();
            return $results;
        } catch (Exception $e) {
            $this->conn->rollback();
            throw $e;
        }
    }

    private function insertKlasifikasiData()
    {
        $stmt = $this->conn->prepare("
            INSERT INTO klasifikasi_arsip (nama, kode, deskripsi) 
            VALUES (?, ?, ?)
            ON DUPLICATE KEY UPDATE 
            nama = VALUES(nama), 
            deskripsi = VALUES(deskripsi)
        ");

        $count = 0;
        foreach ($this->klasifikasiData as $item) {
            $stmt->bind_param('sss', $item['nama'], $item['kode'], $item['deskripsi']);
            if ($stmt->execute()) $count++;
        }

        return $count;
    }

    private function insertJenisData()
    {
        $stmt = $this->conn->prepare("
            INSERT INTO jenis_surat (nama, kode, deskripsi, id_klasifikasi) 
            VALUES (?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE 
            nama = VALUES(nama), 
            deskripsi = VALUES(deskripsi)
        ");

        $count = 0;
        foreach ($this->jenisData as $item) {
            $stmt->bind_param('sssi', $item['nama'], $item['kode'], $item['deskripsi'], $item['id_klasifikasi']);
            if ($stmt->execute()) $count++;
        }

        return $count;
    }

    private function insertSubjenisData()
    {
        $stmt = $this->conn->prepare("
            INSERT INTO subjenis_surat (nama, kode, deskripsi, id_jenis, id_klasifikasi) 
            VALUES (?, ?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE 
            nama = VALUES(nama), 
            deskripsi = VALUES(deskripsi)
        ");

        $count = 0;
        foreach ($this->subjenisData as $item) {
            $stmt->bind_param('sssii', $item['nama'], $item['kode'], $item['deskripsi'], $item['id_jenis'], $item['id_klasifikasi']);
            if ($stmt->execute()) $count++;
        }

        return $count;
    }

    private function getKlasifikasiIdByKode($kode)
    {
        $stmt = $this->conn->prepare("SELECT id FROM klasifikasi_arsip WHERE kode = ?");
        $stmt->bind_param('s', $kode);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        return $row ? $row['id'] : null;
    }

    private function getJenisIdByKode($kode, $klasifikasiId)
    {
        $stmt = $this->conn->prepare("SELECT id FROM jenis_surat WHERE kode = ? AND id_klasifikasi = ?");
        $stmt->bind_param('si', $kode, $klasifikasiId);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        return $row ? $row['id'] : null;
    }

    /**
     * Get parsed data for debugging
     */
    public function getDataSummary()
    {
        return [
            'klasifikasi_count' => count($this->klasifikasiData),
            'jenis_count' => count($this->jenisData),
            'subjenis_count' => count($this->subjenisData),
            'klasifikasi_sample' => array_slice($this->klasifikasiData, 0, 3),
            'jenis_sample' => array_slice($this->jenisData, 0, 3),
            'subjenis_sample' => array_slice($this->subjenisData, 0, 3)
        ];
    }
}

// Usage Example
if (php_sapi_name() === 'cli') {
    // Command line execution
    $filePath = __DIR__ . '/KLASIFIKASI JENIS SUBJENIS_SURAT ULUM.txt';

    try {
        $parser = new ClassificationDataParser($conn);

        echo "Parsing classification file...\n";
        $parseResults = $parser->parseClassificationFile($filePath);

        echo "Parse Results:\n";
        echo "- Klasifikasi: {$parseResults['klasifikasi']} items\n";
        echo "- Jenis Surat: {$parseResults['jenis']} items\n";
        echo "- Subjenis Surat: {$parseResults['subjenis']} items\n\n";

        // Show summary
        $summary = $parser->getDataSummary();
        echo "Data Summary:\n";
        print_r($summary);

        echo "\nInserting data to database...\n";
        $insertResults = $parser->insertData();

        echo "Insert Results:\n";
        echo "- Klasifikasi inserted: {$insertResults['klasifikasi']} items\n";
        echo "- Jenis Surat inserted: {$insertResults['jenis']} items\n";
        echo "- Subjenis Surat inserted: {$insertResults['subjenis']} items\n";

        echo "\nData import completed successfully!\n";
    } catch (Exception $e) {
        echo "Error: " . $e->getMessage() . "\n";
        exit(1);
    }
}
