<?php
// report_letter_order_mismatches.php
// Usage: php report_letter_order_mismatches.php
// Checks the 16 pre-configured users vs `users.nama` exact matches in DB.

require_once 'koneksi.php';

$targets = [
    'Netty Renova',
    'Indarani',
    'Andri Supriadi',
    'Muhammad Reza Pahlevi Guntur',
    'Bambang Hermawan',
    'Fakirno',
    'Ragil Suparmadi',
    'Auzi Trisiyas Sulkha',
    'Muhammad Tachyuddin',
    'Kukuh Prasojo',
    'Rendy Hendra Prasetya',
    'Tedy Agus Harianto',
    'Budhi Satria Permana',
    'M.Nafi Uddin',
    'Harbetta Yurisa Tanydag',
    'Abdul Rokhman'
];

function findExactMatch($conn, $name)
{
    $sql = "SELECT id, username, nama FROM users WHERE nama = ? LIMIT 1";
    $stmt = $conn->prepare($sql);
    if (!$stmt) return false;
    $stmt->bind_param('s', $name);
    $stmt->execute();
    $res = $stmt->get_result();
    $row = $res->fetch_assoc();
    $stmt->close();
    return $row ?: false;
}

function findLikeMatches($conn, $name)
{
    // create flexible LIKE pattern: space-separated words joined with %
    $parts = preg_split('/\s+/', trim($name));
    $pattern = '%' . implode('%', array_filter($parts)) . '%';
    $sql = "SELECT id, username, nama FROM users WHERE nama LIKE ? LIMIT 8";
    $stmt = $conn->prepare($sql);
    if (!$stmt) return [];
    $stmt->bind_param('s', $pattern);
    $stmt->execute();
    $res = $stmt->get_result();
    $rows = $res->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
    return $rows;
}

function findClosestByLevenshtein($conn, $name)
{
    // Pull some potential candidates (LIMIT) and compute levenshtein in PHP
    $sql = "SELECT id, username, nama FROM users LIMIT 200"; // limit for speed
    $res = $conn->query($sql);
    if (!$res) return [];
    $rows = $res->fetch_all(MYSQLI_ASSOC);
    $distances = [];
    foreach ($rows as $row) {
        $dist = levenshtein(mb_strtolower($name, 'UTF-8'), mb_strtolower($row['nama'], 'UTF-8'));
        $distances[] = ['dist' => $dist, 'row' => $row];
    }
    usort($distances, function ($a, $b) {
        return $a['dist'] <=> $b['dist'];
    });
    $best = array_slice($distances, 0, 6);
    $result = [];
    foreach ($best as $b) $result[] = ['dist' => $b['dist'], 'id' => $b['row']['id'], 'nama' => $b['row']['nama'], 'username' => $b['row']['username']];
    return $result;
}

function findLetterOrder($conn, $userId)
{
    $sql = "SELECT lo.id, lo.user_id, lo.supervisor_id, u.nama as user_name, s.nama as supervisor_name
            FROM letter_order lo
            LEFT JOIN users u ON lo.user_id = u.id
            LEFT JOIN users s ON lo.supervisor_id = s.id
            WHERE lo.user_id = ? LIMIT 1";
    $stmt = $conn->prepare($sql);
    if (!$stmt) return false;
    $stmt->bind_param('i', $userId);
    $stmt->execute();
    $res = $stmt->get_result();
    $row = $res->fetch_assoc();
    $stmt->close();
    return $row ?: false;
}

echo "Report - Letter Order Name Checks\n";
echo str_repeat('=', 40) . "\n";

$found = 0;
$notFound = 0;
foreach ($targets as $name) {
    echo "\nChecking: \"$name\"\n";
    $exact = findExactMatch($conn, $name);
    if ($exact) {
        $found++;
        echo "  Exact match found: ID={$exact['id']}, username={$exact['username']}, nama={$exact['nama']}\n";
        // Check if letter_order mapping exists
        $lo = findLetterOrder($conn, (int)$exact['id']);
        if ($lo) {
            echo "    [letter_order exists] id={$lo['id']}, supervisor={$lo['supervisor_name']}, supervisor_id={$lo['supervisor_id']}\n";
        } else {
            echo "    [letter_order missing] - no mapping found yet.\n";
        }
        continue;
    }

    $notFound++;
    echo "  EXACT MATCH NOT FOUND for: $name\n";
    // Show LIKE matches
    $likes = findLikeMatches($conn, $name);
    if (!empty($likes)) {
        echo "    LIKE matches:\n";
        foreach ($likes as $l) echo "      ID={$l['id']}, username={$l['username']}, nama={$l['nama']}\n";
    } else {
        echo "    No LIKE matches found.\n";
    }
    // Show closest via levenshtein
    $closest = findClosestByLevenshtein($conn, $name);
    if (!empty($closest)) {
        echo "    Closest candidates (levenshtein):\n";
        foreach ($closest as $c) echo "      dist={$c['dist']}, ID={$c['id']}, nama={$c['nama']}, username={$c['username']}\n";
    }
}

echo "\nSummary: $found exact matches, $notFound not found.\n";
