<?php
session_start();

// Log errors to file instead of output
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/error_log.txt');

// Ensure JSON response even on fatal errors
ob_start();
register_shutdown_function(function () {
    $error = error_get_last();
    if ($error !== null && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR, E_USER_ERROR])) {
        ob_clean();
        header('Content-Type: application/json');
        echo json_encode(['success' => false, 'message' => 'Server error: ' . $error['message']]);
    }
});

header('Content-Type: application/json');

// Include dependencies first
require_once __DIR__ . '/../vendor/autoload.php';

use chillerlan\QRCode\QRCode;
use chillerlan\QRCode\QROptions;

try {
    include(__DIR__ . '/../includes/koneksi.php');

    if (!isset($conn) || $conn->connect_error) {
        throw new Exception('Database connection failed');
    }
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => 'Initialization error: ' . $e->getMessage()]);
    exit();
}

// Log request info for debugging
dev_log("VALIDATE LETTER REQUEST - Method: " . $_SERVER['REQUEST_METHOD'] . ", User-Agent: " . ($_SERVER['HTTP_USER_AGENT'] ?? 'Unknown') . ", POST data: " . json_encode($_POST));

// Check if user is admin or approver
if (!isset($_SESSION['username'])) {
    dev_log("VALIDATE LETTER ERROR - No session username");
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit();
}

$username = $_SESSION['username'];
$query = "SELECT id, tipe FROM users WHERE username = ?";
$stmt = $conn->prepare($query);
$stmt->bind_param("s", $username);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->fetch_assoc();

if (!$user) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit();
}

// Get letter ID
if (!isset($_POST['letter_id']) || empty($_POST['letter_id'])) {
    echo json_encode(['success' => false, 'message' => 'Letter ID is required']);
    exit();
}

$letterId = (int)$_POST['letter_id'];

// Handle signature upload/drawing
$signaturePath = null;
if (isset($_FILES['signature']) && $_FILES['signature']['error'] === UPLOAD_ERR_OK) {
    $signatureDir = 'letter/signatures';
    if (!is_dir($signatureDir)) {
        mkdir($signatureDir, 0755, true);
    }

    $uploadedFile = $_FILES['signature'];
    $fileExtension = strtolower(pathinfo($uploadedFile['name'], PATHINFO_EXTENSION));

    // Validate file type
    $allowedTypes = ['png', 'jpg', 'jpeg', 'gif'];
    if (!in_array($fileExtension, $allowedTypes)) {
        echo json_encode(['success' => false, 'message' => 'Invalid signature file type. Only PNG, JPG, JPEG, GIF allowed.']);
        exit();
    }

    // Validate file size (5MB max)
    if ($uploadedFile['size'] > 5 * 1024 * 1024) {
        echo json_encode(['success' => false, 'message' => 'Signature file too large. Maximum 5MB allowed.']);
        exit();
    }

    // Generate unique filename
    $signatureFileName = 'signature_' . $letterId . '_' . time() . '.png';
    $signaturePath = $signatureDir . '/' . $signatureFileName;

    // Move uploaded file
    if (!move_uploaded_file($uploadedFile['tmp_name'], $signaturePath)) {
        echo json_encode(['success' => false, 'message' => 'Failed to save signature file']);
        exit();
    }
}

$currentUserId = (int)$user['id'];

// Check if letter exists
$checkQuery = "SELECT id, nomor_surat, is_validated, file_surat, dibuat_oleh FROM office WHERE id = ?";
$checkStmt = $conn->prepare($checkQuery);
$checkStmt->bind_param("i", $letterId);
$checkStmt->execute();
$letterResult = $checkStmt->get_result();
$letter = $letterResult->fetch_assoc();

if (!$letter) {
    echo json_encode(['success' => false, 'message' => 'Letter not found']);
    exit();
}

if ($letter['is_validated'] == 1) {
    echo json_encode(['success' => false, 'message' => 'Letter already validated']);
    exit();
}

// If not admin, verify current user is either the creator or an approver in approval_chain for the creator
if ($user['tipe'] !== 'admin') {
    $creatorId = (int)$letter['dibuat_oleh'];

    // Check if user is the creator
    $isCreator = ($currentUserId === $creatorId);

    // Check if user is an approver
    $checkStmt = $conn->prepare("SELECT COUNT(*) AS c FROM approval_chain WHERE approver_id = ? AND user_id = ?");
    $checkStmt->bind_param('ii', $currentUserId, $creatorId);
    $checkStmt->execute();
    $isApprover = ($checkStmt->get_result()->fetch_assoc()['c'] ?? 0) > 0;
    $checkStmt->close();

    if (!$isCreator && !$isApprover) {
        echo json_encode(['success' => false, 'message' => 'Unauthorized: not approver for this letter']);
        exit();
    }
}

// Generate unique token for QR code
$token = bin2hex(random_bytes(16));

// Create QR code directory if not exists
$qrDir = 'letter/qrcodes';
if (!is_dir($qrDir)) {
    mkdir($qrDir, 0755, true);
}

$qrCodePath = $qrDir . '/qr_' . $letterId . '_' . time() . '.png';
// Buat URL dinamis berdasarkan domain yang dipakai (tidak hardcode port)
$protocol = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? "https" : "http";
$host = $_SERVER['HTTP_HOST']; // Gunakan host dan port yang sedang dipakai
$basePath = dirname($_SERVER['PHP_SELF']);
$verifyUrl = $protocol . "://" . $host . $basePath . "/verify-letter.php?token=" . $token;
// $verifyUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . 
//              "://" . $_SERVER['HTTP_HOST'] . 
//              dirname($_SERVER['PHP_SELF']) . 
//              "/verify-letter.php?token=" . $token;

try {
    dev_log("🔍 QR CODE GENERATION START - Letter ID: " . $letterId);
    dev_log("🔍 QR Code path will be: " . $qrCodePath);
    dev_log("🔍 Token generated: " . $token);

    // Configure QR code options for PNG output
    // Using ECC_M for balance between error correction and data capacity
    // Auto version selection to handle any URL length
    $options = new QROptions([
        'version'          => QRCode::VERSION_AUTO,
        'outputType'       => QRCode::OUTPUT_IMAGE_PNG,
        'outputBase64'     => false,
        'eccLevel'         => QRCode::ECC_M,
        'scale'            => 15,
        'imageTransparent' => false,
        'imageBase64'      => false,
    ]);

    dev_log("🔍 Verify URL: " . $verifyUrl);

    // Generate QR code and save as PNG
    $qrcode = new QRCode($options);
    dev_log("🔍 QRCode object created successfully");

    $qrImageData = $qrcode->render($verifyUrl);
    dev_log("🔍 QR code rendered, image data size: " . strlen($qrImageData) . " bytes");

    // The render() returns base64 or raw image data depending on settings
    // For chillerlan 5.x, we need to save directly
    $bytesWritten = file_put_contents($qrCodePath, $qrImageData);
    dev_log("🔍 QR code saved to: " . $qrCodePath . " (" . $bytesWritten . " bytes written)");

    if ($bytesWritten === false) {
        throw new Exception("Failed to write QR code to disk at " . $qrCodePath);
    }

    // Verify the file was created
    if (!file_exists($qrCodePath)) {
        throw new Exception("QR code file does not exist after saving: " . $qrCodePath);
    }
    dev_log("🔍 QR code file verified to exist");

    // Add logo as watermark to QR code
    $logoPath = 'img/logojpt-new/Logo QR New.png';
    dev_log("🔍 Logo path: " . $logoPath . " (exists: " . (file_exists($logoPath) ? "YES" : "NO") . ")");

    if (file_exists($logoPath)) {
        // Load QR code image
        $qrImage = @imagecreatefrompng($qrCodePath);
        if ($qrImage === false) {
            throw new Exception('Failed to load QR code image');
        }

        // Enable alpha blending and save alpha for QR image
        imagealphablending($qrImage, true);
        imagesavealpha($qrImage, true);

        $qrWidth = imagesx($qrImage);
        $qrHeight = imagesy($qrImage);

        // Load logo with proper transparency support
        $logo = @imagecreatefrompng($logoPath);
        if ($logo === false) {
            imagedestroy($qrImage);
            throw new Exception('Failed to load logo image');
        }

        // Preserve PNG transparency for logo
        imagealphablending($logo, true);
        imagesavealpha($logo, true);

        $logoWidth = imagesx($logo);
        $logoHeight = imagesy($logo);

        // Calculate logo size - Adjust the multiplier to change size
        // 0.2 = 20% (small), 0.3 = 30% (medium), 0.4 = 40% (large)
        $logoSizeRatio = 0.35; // ← Change this value to adjust logo size
        $logoQrWidth = $qrWidth * $logoSizeRatio;
        $logoQrHeight = $logoQrWidth; // Keep square aspect ratio

        // Calculate position to center logo
        $logoX = ($qrWidth - $logoQrWidth) / 2;
        $logoY = ($qrHeight - $logoQrHeight) / 2;

        // Create a temporary image for the logo with transparency
        $logoResized = imagecreatetruecolor($logoQrWidth, $logoQrHeight);

        // Enable alpha blending and save transparency
        imagealphablending($logoResized, false);
        imagesavealpha($logoResized, true);

        // Fill with transparent background
        $transparent = imagecolorallocatealpha($logoResized, 0, 0, 0, 127);
        imagefill($logoResized, 0, 0, $transparent);

        // Enable alpha blending for copying
        imagealphablending($logoResized, true);

        // Resize logo onto transparent background
        imagecopyresampled(
            $logoResized,
            $logo,
            0,
            0,
            0,
            0,
            $logoQrWidth,
            $logoQrHeight,
            $logoWidth,
            $logoHeight
        );

        // Create a function to preserve transparency during merge
        function imagecopymerge_alpha($dst_im, $src_im, $dst_x, $dst_y, $src_x, $src_y, $src_w, $src_h, $pct)
        {
            // Get image width and height and percentage
            $pct /= 100;
            $w = imagesx($src_im);
            $h = imagesy($src_im);

            // Turn alpha blending off
            imagealphablending($src_im, false);

            // Find the most opaque pixel in the image (the one with the smallest alpha value)
            $minalpha = 127;
            for ($x = 0; $x < $w; $x++) {
                for ($y = 0; $y < $h; $y++) {
                    $alpha = (imagecolorat($src_im, $x, $y) >> 24) & 0xFF;
                    if ($alpha < $minalpha) {
                        $minalpha = $alpha;
                    }
                }
            }

            // Loop through image pixels and modify alpha for each
            for ($x = 0; $x < $w; $x++) {
                for ($y = 0; $y < $h; $y++) {
                    // Get current alpha value (represents the OPPOSITE of opacity)
                    $colorxy = imagecolorat($src_im, $x, $y);
                    $alpha = ($colorxy >> 24) & 0xFF;

                    // Calculate new alpha
                    if ($minalpha !== 127) {
                        $alpha = 127 + 127 * $pct * ($alpha - 127) / (127 - $minalpha);
                    } else {
                        $alpha = $alpha * $pct;
                    }

                    // Get the color index with new alpha
                    $alphacolorxy = imagecolorallocatealpha($src_im, ($colorxy >> 16) & 0xFF, ($colorxy >> 8) & 0xFF, $colorxy & 0xFF, $alpha);

                    // Set pixel with the new color + alpha
                    if (!imagesetpixel($src_im, $x, $y, $alphacolorxy)) {
                        return false;
                    }
                }
            }

            // Copy image
            imagecopy($dst_im, $src_im, $dst_x, $dst_y, $src_x, $src_y, $src_w, $src_h);
        }

        // Use alpha-preserving merge instead of regular imagecopymerge
        // This preserves PNG transparency and prevents black background
        imagecopymerge_alpha(
            $qrImage,
            $logoResized,
            $logoX,
            $logoY,
            0,
            0,
            $logoQrWidth,
            $logoQrHeight,
            85 // 70% opacity - keeps transparency intact
        );

        // Save the final QR code with watermark
        imagepng($qrImage, $qrCodePath);

        // Clean up
        imagedestroy($qrImage);
        imagedestroy($logo);
        imagedestroy($logoResized);
    }

    // Now embed the QR code into the existing PDF letter
    if (!empty($letter['file_surat']) && file_exists($letter['file_surat'])) {
        require_once('vendor/setasign/fpdf/fpdf.php');
        require_once('vendor/setasign/fpdi/src/autoload.php');

        $pdf = new \setasign\Fpdi\Fpdi();
        $pageCount = $pdf->setSourceFile($letter['file_surat']);

        // Import all pages
        // Default QR placement settings (kept unchanged for backward compatibility)
        $qrSize = 38; // mm size
        $signatureBlockLeft = 118; // mm from left edge
        $signatureBlockWidth = 72; // mm
        $xPos = $signatureBlockLeft + ($signatureBlockWidth - $qrSize) / 2; // Center in signature block

        // Determine which page to place QR code on
        $qrPageNo = ($pageCount >= 2) ? 2 : 1; // Page 2 if available, otherwise page 1
        $qrYPos = 87; // Position near top of the page where QR will be placed

        // Check for special placement rules (e.g., 'Surat Edaran Direksi' / 'SE-DIR')
        // The number format for SE-DIR is usually like "001/SE-DIR/2025", so we detect the substring '/SE-DIR'.
        $nomorSurat = strtoupper(trim($letter['nomor_surat'] ?? ''));
        if (strpos($nomorSurat, '/SE-DIR') !== false) {
            // Override placement for Surat Edaran Direksi (SE-DIR)
            // NOTE: We're not changing defaults; this is an override for SE-DIR only.
            // Adjust these values as needed for your template and sign area. These are initial values to help
            // position the QR code where a 'doc preview/signature' layout for SE-DIR might need it.
            $qrSize = 36; // smaller QR size for SE-DIR
            $signatureBlockLeft = 128; // move more to the right
            $signatureBlockWidth = 48; // narrower block
            $qrYPos = 178; // move slightly up on the page
            // Recompute x position after overrides
            $xPos = $signatureBlockLeft + ($signatureBlockWidth - $qrSize) / 2;
            // For SE-DIR, we often put QR on page 1, but keep behavior flexible
            $qrPageNo = 1;
            dev_log("🔍 QR placement OVERRIDE: SE-DIR detected - qrSize={$qrSize}, xPos={$xPos}, qrYPos={$qrYPos}, qrPageNo={$qrPageNo}");
        } elseif (strpos($nomorSurat, '/SK') !== false) {
            // Override placement for Surat Keputusan (SK)
            $qrSize = 36;
            $signatureBlockLeft = 118;
            $signatureBlockWidth = 72;
            $qrYPos = 270; // Lower position for SK
            // Recompute x position after overrides
            $xPos = $signatureBlockLeft + ($signatureBlockWidth - $qrSize) / 2;
            $qrPageNo = 1; // Place on first page
            dev_log("🔍 QR placement OVERRIDE: SK detected - qrSize={$qrSize}, xPos={$xPos}, qrYPos={$qrYPos}, qrPageNo={$qrPageNo}");
        } elseif (strpos($nomorSurat, '/KPTS') !== false) {
            // Override placement for Keputusan (KPTS)
            $qrSize = 36;
            $signatureBlockLeft = 118;
            $signatureBlockWidth = 72;
            $qrYPos = 265; // Slightly lower position for KPTS
            // Recompute x position after overrides
            $xPos = $signatureBlockLeft + ($signatureBlockWidth - $qrSize) / 2;
            $qrPageNo = 1; // Place on first page
            dev_log("🔍 QR placement OVERRIDE: KPTS detected - qrSize={$qrSize}, xPos={$xPos}, qrYPos={$qrYPos}, qrPageNo={$qrPageNo}");
        }

        for ($pageNo = 1; $pageNo <= $pageCount; $pageNo++) {
            $templateId = $pdf->importPage($pageNo);
            $size = $pdf->getTemplateSize($templateId);

            $pdf->AddPage($size['orientation'], [$size['width'], $size['height']]);
            $pdf->useTemplate($templateId);

            // Add QR code to the designated page
            if ($pageNo == $qrPageNo) {
                dev_log("🔍 QR CODE PLACEMENT ON PDF:");
                dev_log("🔍  - QR Code Size: " . $qrSize . "mm x " . $qrSize . "mm");
                dev_log("🔍  - Signature Block Left: " . $signatureBlockLeft . "mm");
                dev_log("🔍  - Signature Block Width: " . $signatureBlockWidth . "mm");
                dev_log("🔍  - Calculated X Position: " . $xPos . "mm from left");
                dev_log("🔍  - Y Position on page: " . $qrYPos . "mm from page top");
                dev_log("🔍  - Page Number: " . $pageNo);
                dev_log("🔍  - QR Code image path: " . $qrCodePath);
                dev_log("🔍  - Page size: " . $size['width'] . "mm x " . $size['height'] . "mm");

                $pdf->Image($qrCodePath, $xPos, $qrYPos, $qrSize, $qrSize, 'PNG');
                dev_log("✅ QR code image placed on page " . $pageNo . " at X=" . $xPos . "mm, Y=" . $qrYPos . "mm");
            }
        }

        // Save the updated PDF (overwrite the original)
        $pdf->Output('F', $letter['file_surat']);
    }

    // Update database
    dev_log("🔍 DATABASE UPDATE START - Letter ID: " . $letterId);
    dev_log("🔍 QR Code path to store: " . $qrCodePath);
    dev_log("🔍 Signature path to store: " . ($signaturePath ?: "None"));
    dev_log("🔍 Token to store: " . $token);

    $updateQuery = "UPDATE office SET is_validated = 1, qr_code = ?, validation_token = ?, signature_path = ?, validated_at = NOW() WHERE id = ?";
    $updateStmt = $conn->prepare($updateQuery);

    if (!$updateStmt) {
        dev_log("❌ Prepare failed: " . $conn->error);
        throw new Exception('Failed to prepare database statement: ' . $conn->error);
    }

    $updateStmt->bind_param("sssi", $qrCodePath, $token, $signaturePath, $letterId);
    dev_log("🔍 Parameters bound successfully");

    if ($updateStmt->execute()) {
        dev_log("✅ DATABASE UPDATE SUCCESS - Letter ID: " . $letterId);
        dev_log("✅ Affected rows: " . $updateStmt->affected_rows);
        echo json_encode([
            'success' => true,
            'message' => 'Letter validated successfully and QR code added to PDF',
            'qr_code_path' => $qrCodePath,
            'signature_path' => $signaturePath,
            'verify_url' => $verifyUrl
        ]);
    } else {
        dev_log("❌ DATABASE UPDATE FAILED - Error: " . $updateStmt->error);
        // Delete QR code and signature files if database update fails
        if (file_exists($qrCodePath)) {
            unlink($qrCodePath);
            dev_log("🗑️ Deleted QR code file: " . $qrCodePath);
        }
        if ($signaturePath && file_exists($signaturePath)) {
            unlink($signaturePath);
            dev_log("🗑️ Deleted signature file: " . $signaturePath);
        }
        echo json_encode(['success' => false, 'message' => 'Failed to update database: ' . $updateStmt->error]);
    }
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => 'Error generating QR code: ' . $e->getMessage()]);
}

$conn->close();
