<?php
session_start();
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

include(__DIR__ . '/../includes/koneksi.php');
require_once(__DIR__ . '/../api/classification.php');

/**
 * Helper function to initialize approval chain when letter is created
 * Updated to use only signer (0/1 approval), not creator + signer
 */
function initializeApprovalChainForLetter($conn, $officeId, $creatorUserId, $signerId)
{
    // Get the signer info from authorized_signers
    $stmt = $conn->prepare("
        SELECT id, user_id, nama
        FROM authorized_signers 
        WHERE user_id = ? AND is_active = 1
        LIMIT 1
    ");
    $stmt->bind_param("i", $signerId);
    $stmt->execute();
    $signer = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if (!$signer) {
        // Try finding by authorized_signers.id
        $stmt = $conn->prepare("
            SELECT id, user_id, nama
            FROM authorized_signers 
            WHERE id = ? AND is_active = 1
            LIMIT 1
        ");
        $stmt->bind_param("i", $signerId);
        $stmt->execute();
        $signer = $stmt->get_result()->fetch_assoc();
        $stmt->close();
    }

    if (!$signer) {
        return false; // No signer found
    }

    error_log("DEBUG initializeApprovalChainForLetter - Office ID: $officeId, Signer ID: " . $signer['user_id'] . ", Signer Name: " . $signer['nama']);

    // Build approval chain with ONLY the selected penandatangan (not creator)
    // This gives us 0/1 approval instead of 0/2
    $approvalChain = [
        [
            'user_id' => (int)$signer['user_id'],
            'nama' => $signer['nama']
        ]
    ];

    error_log("DEBUG Approval chain created with " . count($approvalChain) . " step(s)");

    // Delete existing chain entries for this letter
    $stmt = $conn->prepare("DELETE FROM urutan_paraf WHERE office_id = ?");
    $stmt->bind_param("i", $officeId);
    $stmt->execute();
    error_log("DEBUG DELETE from urutan_paraf: office_id=$officeId, affected_rows=" . $stmt->affected_rows);
    $stmt->close();

    // Insert approval chain into urutan_paraf (just the signer)
    $urutan = 1;
    foreach ($approvalChain as $member) {
        $stmt = $conn->prepare("INSERT INTO urutan_paraf (office_id, user_id, urutan) VALUES (?, ?, ?)");
        $stmt->bind_param("iii", $officeId, $member['user_id'], $urutan);
        if ($stmt->execute()) {
            error_log("DEBUG Inserted urutan_paraf: office_id=$officeId, user_id=" . $member['user_id'] . ", urutan=$urutan");
        } else {
            error_log("DEBUG FAILED to insert urutan_paraf: " . $stmt->error);
        }
        $stmt->close();
        $urutan++;
    }

    // Verify what was inserted
    $verifyStmt = $conn->prepare("SELECT COUNT(*) as total FROM urutan_paraf WHERE office_id = ?");
    $verifyStmt->bind_param("i", $officeId);
    $verifyStmt->execute();
    $verifyResult = $verifyStmt->get_result()->fetch_assoc();
    $verifyStmt->close();
    error_log("DEBUG After insert, urutan_paraf count for office_id=$officeId: " . $verifyResult['total']);

    // Update office with penandatangan_id
    $signerUserId = (int)$signer['user_id'];
    $stmt = $conn->prepare("UPDATE office SET penandatangan_id = ? WHERE id = ?");
    $stmt->bind_param("ii", $signerUserId, $officeId);
    $stmt->execute();
    $stmt->close();

    return true;
}

if (!isset($_SESSION['username'])) {
    header('Location: ../login.php');
    exit();
}

// Ambil user & cek admin
$username = $_SESSION['username'];
$qUser = $conn->prepare("SELECT id, tipe, nama, jabatan FROM users WHERE username = ?");
$qUser->bind_param("s", $username);
$qUser->execute();
$res  = $qUser->get_result();
$user = $res ? $res->fetch_assoc() : null;
$res?->free();
$qUser->close();

if (!$user || strtolower(trim($user['tipe'])) !== 'admin') {
    header('Location: ../login.php');
    exit();
}

$idLogin      = $user['id'] ?? 0;
$namaLogin    = $user['nama'] ?: $username;
$jabatanLogin = $user['jabatan'] ?? '';

date_default_timezone_set("Asia/Jakarta");
$today   = date('Y-m-d');
$nowTime = date('H:i');

// Helper function to get kode jabatan from JSON mapping with alias support
function getKodeJabatanFromJSON($jabatan)
{
    $jsonFile = __DIR__ . '/../data/kode_jabatan.json';

    if (!file_exists($jsonFile)) {
        return null;
    }

    $jsonContent = file_get_contents($jsonFile);
    $data = json_decode($jsonContent, true);

    if (!$data || !isset($data['kodeJabatan'])) {
        return null;
    }

    // Normalize jabatan for comparison (case-insensitive, trim spaces, remove extra spaces)
    $jabatanNormalized = strtolower(trim(preg_replace('/\s+/', ' ', $jabatan)));

    foreach ($data['kodeJabatan'] as $item) {
        // Check main jabatan
        $itemJabatanNormalized = strtolower(trim(preg_replace('/\s+/', ' ', $item['jabatan'])));

        if ($itemJabatanNormalized === $jabatanNormalized) {
            return $item['kode'];
        }

        // Check aliases if available
        if (isset($item['aliases']) && is_array($item['aliases'])) {
            foreach ($item['aliases'] as $alias) {
                $aliasNormalized = strtolower(trim(preg_replace('/\s+/', ' ', $alias)));

                if ($aliasNormalized === $jabatanNormalized) {
                    return $item['kode'];
                }
            }
        }
    }

    return null; // Jabatan not found in mapping
}

// Hitung agenda aktif (opsional)
$jumlahAgendaAktif = 0;
if ($s = $conn->prepare("SELECT COUNT(*) FROM agenda WHERE tanggal > ? OR (tanggal = ? AND waktu >= ?)")) {
    $s->bind_param("sss", $today, $today, $nowTime);
    $s->execute();
    $s->bind_result($jumlah);
    if ($s->fetch()) $jumlahAgendaAktif = (int)$jumlah;
    $s->close();
}

// ================================ Store Surat to Database ================================
$jenisSurat = [];
$klasifikasi = [];
$usersList = [];
if ($r = $conn->query("SELECT id, kode, nama FROM jenis_surat WHERE is_active=1 ORDER BY nama")) {
    $jenisSurat = $r->fetch_all(MYSQLI_ASSOC);
    $r->free();
}
if ($r = $conn->query("SELECT id, kode, nama FROM klasifikasi_arsip WHERE is_active=1 ORDER BY nama")) {
    $klasifikasi = $r->fetch_all(MYSQLI_ASSOC);
    $r->free();
}
if ($r = $conn->query("SELECT id, nama, jabatan FROM users WHERE tipe IN ('user', 'admin') ORDER BY nama")) {
    $usersList = $r->fetch_all(MYSQLI_ASSOC);
    $r->free();
}

// load kode jabatan mapping for client-side preview
$kodeJabatanMap = [];
if (file_exists(__DIR__ . '/../data/kode_jabatan.json')) {
    $kb = json_decode(file_get_contents(__DIR__ . '/../data/kode_jabatan.json'), true);
    if (isset($kb['kodeJabatan'])) $kodeJabatanMap = $kb['kodeJabatan'];
}

// Note: get_next_office_id(), format_nomor_surat(), extractSequenceFromNomorSurat() removed
// Now using getNextSequenceNumber() from api/classification.php for proper sequence handling

$errors = [];
$success = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['__final_submit']) && $_POST['__final_submit'] === '1') {
    $kepentingan       = $_POST['kepentingan'] ?? '';
    $tipe_surat_id     = (int)($_POST['tipe_surat_id'] ?? 0);
    $tempat_dibuat     = trim($_POST['tempat_dibuat'] ?? '');
    $tanggal_surat     = $_POST['tanggal_surat'] ?? '';
    $dari_tipe         = $_POST['dari_tipe'] ?? '';
    $dari_pribadi      = trim($_POST['dari_pribadi'] ?? '');
    $kepada            = trim($_POST['kepada'] ?? '');
    $perihal           = trim($_POST['perihal'] ?? '');
    $fileSuratPath     = trim($_POST['file_surat'] ?? '');
    $nomor_surat       = trim($_POST['nomor_surat'] ?? '');
    $paraf             = trim($_POST['paraf'] ?? '');

    // Get tipe surat kode to determine which classification system to use
    $tipeSuratKode = '';
    if ($tipe_surat_id > 0) {
        $stmtTipe = $conn->prepare("SELECT kode FROM tipe_surat WHERE id = ?");
        $stmtTipe->bind_param('i', $tipe_surat_id);
        $stmtTipe->execute();
        $resTipe = $stmtTipe->get_result();
        if ($rowTipe = $resTipe->fetch_assoc()) {
            $tipeSuratKode = $rowTipe['kode'];
        }
        $stmtTipe->close();
    }

    // Determine which fields to use based on tipe surat
    $jenis_surat_id = 0;
    $klasifikasi_arsip_id = 0;
    $subjenis_surat_id = 0;

    // STRICT: Only SK (Surat Keluar) uses classification system
    if ($tipeSuratKode === 'SK') {
        // New classification system for Surat Keluar (SK)
        $klasifikasi_arsip_id = (int)($_POST['klasifikasi_arsip_new'] ?? 0);
        $jenis_surat_id = (int)($_POST['jenis_surat_new'] ?? 0);
        $subjenis_surat_id = (int)($_POST['subjenis_surat_id'] ?? 0);
    }
    // KPTS and SE: NO classification - all fields remain 0 (will be NULL in DB)
    // This prevents accidental data leak when switching from SK to KPTS/SE

    $enumKepentingan = ['internal', 'eksternal'];
    $enumDari        = ['internal', 'eksternal', 'pribadi', 'sendiri'];

    // Validation
    if (!in_array($kepentingan, $enumKepentingan, true)) $errors[] = 'Kepentingan tidak valid.';
    if ($tipe_surat_id <= 0)                             $errors[] = 'Tipe surat wajib diisi.';
    if ($tempat_dibuat === '')                           $errors[] = 'Tempat surat dibuat wajib diisi.';
    if ($tanggal_surat === '')                           $errors[] = 'Tanggal surat wajib diisi.';
    if ($nomor_surat === '')                             $errors[] = 'Nomor surat belum digenerate.';
    if (!in_array($dari_tipe, $enumDari, true))          $errors[] = 'Dari (tipe) tidak valid.';
    if ($dari_tipe === 'pribadi' && $dari_pribadi === '') $errors[] = 'Nama atas nama pribadi tidak boleh kosong.';
    if ($dari_tipe === 'sendiri' && $dari_pribadi === '') $dari_pribadi = $namaLogin;
    // Only require 'Kepada' for Surat Keluar (SK), not for Surat Keputusan (KPTS) or Surat Edaran (SE)
    if ($tipeSuratKode === 'SK' && $kepada === '')       $errors[] = 'Kepada wajib diisi.';
    if ($perihal === '')                                 $errors[] = 'Perihal wajib diisi.';
    if ($fileSuratPath === '')                           $errors[] = 'File surat belum diupload.';

    $lampiran_path = $lampiran_orig = $lampiran_mime = null;
    $lampiran_size = null;

    if (!$errors && isset($_FILES['lampiran']) && $_FILES['lampiran']['error'] !== UPLOAD_ERR_NO_FILE) {
        if ($_FILES['lampiran']['error'] !== UPLOAD_ERR_OK) {
            $errors[] = 'Gagal mengunggah lampiran.';
        } else {
            $tmp  = $_FILES['lampiran']['tmp_name'];
            $mime = @mime_content_type($tmp) ?: ($_FILES['lampiran']['type'] ?? '');
            $size = (int)$_FILES['lampiran']['size'];
            $allowed = ['application/pdf', 'image/jpeg', 'image/png'];
            if (!in_array($mime, $allowed, true)) {
                $errors[] = 'Tipe lampiran tidak diizinkan. Gunakan PDF/JPG/PNG.';
            } elseif ($size > 25 * 1024 * 1024) {
                $errors[] = 'Lampiran melebihi 25 MB.';
            } else {
                $dir = __DIR__ . '/letter/uploads/lampiran/';
                if (!is_dir($dir)) mkdir($dir, 0775, true);
                $orig = $_FILES['lampiran']['name'];
                $ext  = strtolower(pathinfo($orig, PATHINFO_EXTENSION));
                $safe = preg_replace('/[^a-zA-Z0-9_\-\.]/', '_', pathinfo($orig, PATHINFO_FILENAME));
                $fname = $safe . '_' . time() . '.' . $ext;
                $dest  = $dir . $fname;
                if (!move_uploaded_file($tmp, $dest)) {
                    $errors[] = 'Tidak dapat menyimpan lampiran.';
                } else {
                    $lampiran_path = 'letter/uploads/lampiran/' . $fname;
                    $lampiran_orig = $orig;
                    $lampiran_mime = $mime;
                    $lampiran_size = $size;
                }
            }
        }
    }

    if (!$errors) {
        $conn->begin_transaction();
        try {
            // Ensure dari_tipe_db is always a valid ENUM value for the database
            switch ($dari_tipe) {
                case 'pribadi':
                case 'sendiri':
                case 'internal':
                    $dari_tipe_db = 'internal';
                    break;
                case 'eksternal':
                    $dari_tipe_db = 'eksternal';
                    break;
                default:
                    // Fallback to internal if invalid value
                    $dari_tipe_db = 'internal';
                    dev_log("WARNING: Invalid dari_tipe value '$dari_tipe', defaulting to 'internal'");
                    break;
            }

            // Debug logging
            dev_log("DEBUG INSERT - dari_tipe (original): " . $dari_tipe);
            dev_log("DEBUG INSERT - dari_tipe_db (final): " . $dari_tipe_db);
            dev_log("DEBUG INSERT - dari_pribadi (from POST): " . $dari_pribadi);
            dev_log("DEBUG INSERT - idLogin: " . ($idLogin ?? 'NOT SET'));
            dev_log("DEBUG INSERT - kepada: " . $kepada);

            // Handle dari_pribadi based on type
            if ($dari_tipe === 'pribadi') {
                $dari_pribadi_db = (int)$dari_pribadi;
            } elseif ($dari_tipe === 'sendiri') {
                // For "Diri Sendiri", always use logged-in user's ID
                $dari_pribadi_db = $idLogin;
            } else {
                $dari_pribadi_db = null;
            }

            dev_log("DEBUG INSERT - dari_pribadi_db (final): " . ($dari_pribadi_db ?? 'NULL'));

            $lampiran_path_json = $lampiran_path
                ? json_encode([$lampiran_path], JSON_UNESCAPED_SLASHES)
                : json_encode([], JSON_UNESCAPED_SLASHES);

            // Get proper sequence number from letter_sequences table (reserve=true to persist)
            $sequence_number = getNextSequenceNumber($conn, $tipe_surat_id, true);

            // Regenerate nomor_surat server-side with the correct reserved sequence number
            if ($tipeSuratKode === 'SK') {
                // Get klasifikasi kode
                $stmtK = $conn->prepare("SELECT kode FROM klasifikasi_arsip WHERE id = ?");
                $stmtK->bind_param('i', $klasifikasi_arsip_id);
                $stmtK->execute();
                $resK = $stmtK->get_result();
                $klasRow = $resK->fetch_assoc();
                $klasifikasiKode = $klasRow['kode'] ?? '';
                $stmtK->close();

                // Get jenis kode
                $stmtJ = $conn->prepare("SELECT kode FROM jenis_surat WHERE id = ?");
                $stmtJ->bind_param('i', $jenis_surat_id);
                $stmtJ->execute();
                $resJ = $stmtJ->get_result();
                $jenisRow = $resJ->fetch_assoc();
                $jenisKode = $jenisRow['kode'] ?? '';
                $stmtJ->close();

                // Get subjenis kode
                $stmtS = $conn->prepare("SELECT kode FROM subjenis_surat WHERE id = ?");
                $stmtS->bind_param('i', $subjenis_surat_id);
                $stmtS->execute();
                $resS = $stmtS->get_result();
                $subRow = $resS->fetch_assoc();
                $subjenisKode = $subRow['kode'] ?? '';
                $stmtS->close();

                // Get jabatan based on dari_tipe
                $jabatan = '';
                if ($dari_tipe === 'pribadi' && !empty($dari_pribadi_db)) {
                    $stmtJab = $conn->prepare("SELECT jabatan FROM users WHERE id = ?");
                    $stmtJab->bind_param('i', $dari_pribadi_db);
                    $stmtJab->execute();
                    $resJab = $stmtJab->get_result();
                    if ($rowJab = $resJab->fetch_assoc()) {
                        $jabatan = $rowJab['jabatan'] ?? '';
                    }
                    $stmtJab->close();
                } elseif ($dari_tipe === 'sendiri') {
                    $jabatan = $jabatanLogin;
                }

                // Get kode jabatan from JSON mapping
                $kodeJabatan = getKodeJabatanFromJSON($jabatan);

                // Format: KODE.KLASIFIKASI.JENIS.SUBJENIS.SEQ or KLASIFIKASI.JENIS.SUBJENIS.SEQ
                $formattedSeq = str_pad($sequence_number, 2, '0', STR_PAD_LEFT);
                if ($kodeJabatan) {
                    $nomor_surat = "{$kodeJabatan}.{$klasifikasiKode}.{$jenisKode}.{$subjenisKode}.{$formattedSeq}";
                } else {
                    $nomor_surat = "{$klasifikasiKode}.{$jenisKode}.{$subjenisKode}.{$formattedSeq}";
                }
            } elseif ($tipeSuratKode === 'KPTS') {
                $nomor_surat = str_pad($sequence_number, 3, '0', STR_PAD_LEFT) . '/KPTS-JPT/' . date('Y');
            } elseif ($tipeSuratKode === 'SE') {
                $nomor_surat = str_pad($sequence_number, 3, '0', STR_PAD_LEFT) . '/SE-DIR/' . date('Y');
            }
            // Note: $nomor_surat now contains the correct sequence number

            // Check if sequence_number column exists in database
            $checkColumn = $conn->prepare("SHOW COLUMNS FROM office LIKE 'sequence_number'");
            $checkColumn->execute();
            $sequenceColumnExists = $checkColumn->get_result()->num_rows > 0;

            // Set jenis_surat_id, klasifikasi_arsip_id, and subjenis_surat_id to NULL if not applicable (for KPTS/SE)
            $jenis_surat_id_db = ($jenis_surat_id > 0) ? $jenis_surat_id : null;
            $klasifikasi_arsip_id_db = ($klasifikasi_arsip_id > 0) ? $klasifikasi_arsip_id : null;
            $subjenis_surat_id_db = ($subjenis_surat_id > 0) ? $subjenis_surat_id : null;

            // Debug: Log all bind_param values
            dev_log("DEBUG BIND PARAMS:");
            dev_log("1. kepentingan: " . $kepentingan);
            dev_log("2. tipe_surat_id: " . $tipe_surat_id);
            dev_log("3. jenis_surat_id_db: " . ($jenis_surat_id_db ?? 'NULL'));
            dev_log("4. tempat_dibuat: " . $tempat_dibuat);
            dev_log("5. klasifikasi_arsip_id_db: " . ($klasifikasi_arsip_id_db ?? 'NULL'));
            dev_log("6. subjenis_surat_id_db: " . ($subjenis_surat_id_db ?? 'NULL'));
            dev_log("7. nomor_surat: " . $nomor_surat);
            dev_log("8. sequence_number: " . $sequence_number);
            dev_log("9. tanggal_surat: " . $tanggal_surat);
            dev_log("10. dari_tipe_db: '" . $dari_tipe_db . "' (length: " . strlen($dari_tipe_db) . ")");
            dev_log("10. dari_pribadi_db: " . ($dari_pribadi_db ?? 'NULL'));
            dev_log("11. kepada: " . $kepada);
            dev_log("12. perihal: " . $perihal);
            dev_log("13. lampiran_path_json: " . $lampiran_path_json);
            dev_log("14. lampiran_orig: " . ($lampiran_orig ?? 'NULL'));
            dev_log("15. lampiran_mime: " . ($lampiran_mime ?? 'NULL'));
            dev_log("16. lampiran_size: " . ($lampiran_size ?? 'NULL'));
            dev_log("17. fileSuratPath: " . $fileSuratPath);

            // Extra validation before database insert
            if (!in_array($dari_tipe_db, ['internal', 'eksternal'], true)) {
                throw new Exception("Invalid dari_tipe_db value: '$dari_tipe_db'. Must be 'internal' or 'eksternal'");
            }

            $sql = "INSERT INTO office(
                    kepentingan, tipe_surat_id, jenis_surat_id, tempat_dibuat, klasifikasi_arsip_id, subjenis_surat_id,
                    nomor_surat, sequence_number, tanggal_surat, dari_tipe, dari_pribadi, dibuat_oleh,
                    kepada, perihal, paraf,
                    lampiran_path, lampiran_orig_name, lampiran_mime, lampiran_size, file_surat,
                    created_at, updated_at
                ) VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?, NOW(), NOW())";

            $st = $conn->prepare($sql);
            $lampiran_size_int = $lampiran_size !== null ? (int)$lampiran_size : 0;

            // Set dibuat_oleh to the currently logged-in admin (do not use signer as creator)
            $dibuat_oleh_final = $idLogin; // admin who is creating the letter

            $st->bind_param(
                'siisississiissssssis',
                $kepentingan,
                $tipe_surat_id,
                $jenis_surat_id_db,
                $tempat_dibuat,
                $klasifikasi_arsip_id_db,
                $subjenis_surat_id_db,
                $nomor_surat,
                $sequence_number,
                $tanggal_surat,
                $dari_tipe_db,
                $dari_pribadi_db,
                $dibuat_oleh_final,
                $kepada,
                $perihal,
                $paraf,
                $lampiran_path_json,
                $lampiran_orig,
                $lampiran_mime,
                $lampiran_size_int,
                $fileSuratPath
            );

            if (!$st->execute()) throw new Exception($st->error);

            // Get the inserted ID
            $insertedId = $conn->insert_id;

            // Auto-initialize approval chain if penandatangan is set
            if ($dari_pribadi_db !== null && $dari_pribadi_db > 0) {
                initializeApprovalChainForLetter($conn, $insertedId, $dibuat_oleh_final, $dari_pribadi_db);
            }

            // Note: Sequence already persisted by getNextSequenceNumber() with reserve=true
            // Note: nomor_surat already regenerated with correct sequence before INSERT

            $st->close();
            $conn->commit();

            header('Location: record-letter.php?success=1');
            exit();
        } catch (Exception $ex) {
            $conn->rollback();
            if (isset($finalPathAbs) && is_file($finalPathAbs)) @unlink($finalPathAbs);
            if (isset($lampiran_path)) {
                $absLampiran = __DIR__ . '/' . $lampiran_path;
                if (is_file($absLampiran)) @unlink($absLampiran);
            }
            dev_log('SURAT_SAVE_FAIL: ' . $ex->getMessage());
            $errors[] = "Gagal menyimpan: " . $ex->getMessage();
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <!-- Favicon -->
    <link href="/jasa_marga/img/baru/logojp.png" rel="icon">

    <title>Buat Surat</title>

    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">

    <style>
        .greeting {
            text-align: center;
            color: <?php echo $color;
                    ?>;
            animation: fadeIn 2s ease-in-out;
        }

        .icon {
            margin-top: 40px;
            font-size: 50px;
            margin-bottom: 0px;
            animation: bounce 1.5s infinite;
        }

        @keyframes bounce {

            0%,
            100% {
                transform: translateY(0);
            }

            50% {
                transform: translateY(-10px);
            }
        }

        .sidebar {
            background-color: #0c50e1;
            color: white;
            min-height: 100vh;
            padding: 15px;
        }

        .sidebar a {
            color: white;
            text-decoration: none;
            display: block;
            padding: 10px;
            margin: 5px 0;
            border-radius: 5px;
        }

        .sidebar a:hover,
        .sidebar a.active {
            background-color: #ffc107;
            color: #0047ab;
        }

        .btn-success {
            background-color: #ffc107;
            /* Kuning mencolok */
            border-color: #ffc107;
            color: #0047ab;
            font-weight: bold;
        }

        /* Warna untuk header tabel */
        .table thead {
            background-color: #0047ab;
            /* Biru Jasamarga */
            color: #0c50e1;
            text-transform: uppercase;
            font-weight: bold;
            color: #ffc107;
        }

        .table thead th {
            background-color: #0c50e1;
            /* Biru Jasamarga */
            color: #ffc107;
        }

        .sidebar-animated {
            background-color: #001eff;
            ;
            padding: 20px 10px;
            font-family: Arial, sans-serif;
            height: 100vh;
            width: 250px;
            transition: all 0.3s ease;
            max-height: 100vh;
            /* Limits sidebar height to viewport */
            overflow-y: auto;
            /* Enables vertical scroll */
            overflow-x: hidden;
            /* Hides horizontal scroll */
            padding-bottom: 20px;
            /* Adds extra padding to avoid cut-off */
            margin-bottom: 0;
            /* Ensures no unwanted margin at the bottom */
            scrollbar-width: thin;
        }

        /* Optional: Styling untuk scrollbar pada Chrome, Edge, dan Safari */
        .sidebar-animated::-webkit-scrollbar {
            width: 8px;
            /* Mengatur lebar scrollbar */
        }

        .sidebar-animated::-webkit-scrollbar-thumb {
            background-color: #e1b12c;
            /* Warna scrollbar */
            border-radius: 10px;
            /* Membuat sudut melengkung */
        }

        .sidebar-animated::-webkit-scrollbar-thumb:hover {
            background-color: #ffc107;
            /* Warna scrollbar saat di-hover */
        }

        .sidebar-animated a {
            color: #FFFFFFFF;
            text-decoration: none;
            font-size: 18px;
            display: flex;
            align-items: center;
            padding: 10px;
            border-radius: 5px;
            transition: background-color 0.3s ease, transform 0.3s ease;
        }

        .sidebar-animated a:hover {
            background-color: #ffc107;
            color: #fff;
            transform: translateX(5px);
        }

        .sidebar-animated a i {
            margin-right: 10px;
        }

        .sidebar-animated .submenu-toggle {
            cursor: pointer;
            transition: color 0.3s;
        }

        .sidebar-animated .submenu-toggle:hover .arrow-icon {
            transform: rotate(90deg);
            color: #001eff;
        }

        .sidebar-animated .submenu {
            padding-left: 1px;
            overflow: hidden;
            transition: max-height 0.4s ease;
        }

        .sidebar-animated .collapse.show {
            max-height: 300px;
            /* adjust as needed */
            transition: max-height 0.4s ease-in;
        }

        .arrow-icon {
            margin-left: auto;
            transition: transform 0.3s;
        }

        /* Hover and animation effects for submenu */
        .submenu a {
            font-size: 16px;
            color: #FFFFFFFF;
            transition: color 0.3s ease, transform 0.3s ease;
        }

        .submenu a:hover {
            color: #001eff;
            transform: translateX(5px);
        }

        /* Sidebar active link style */
        .sidebar-animated a.active {
            background-color: #ffc107;
            /* Warna latar belakang khusus untuk menu aktif */
            color: #fff;
            /* Warna teks untuk menu aktif */
            font-weight: bold;
        }

        .sidebar-animated a.active i {
            color: #fff;
            /* Warna ikon untuk menu aktif */
        }

        /* Navbar custom styling */
        .navbar-custom {
            background-color: #001eff;
            /* Same as sidebar background color */
            transition: background-color 0.3s ease;
            padding: 15px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            /* Adds a subtle shadow */
        }

        /* Logo and Sidebar Toggle */
        .navbar-custom .navbar-brand img {
            transition: transform 0.3s ease;
        }

        .navbar-custom .navbar-brand:hover img {
            transform: scale(1.05);
        }

        /* Toggle Sidebar button */
        .toggle-sidebar-btn {
            color: #fff;
            font-size: 1.3rem;
            cursor: pointer;
            margin-left: 10px;
            transition: color 0.3s ease;
        }

        .toggle-sidebar-btn:hover {
            color: #ffc107;
        }

        /* Animasi untuk kartu */
        @keyframes fadeInUp {
            0% {
                opacity: 0;
                transform: translateY(20px);
            }

            100% {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .highlight-card {
            border-radius: 15px;
            /* Membuat sudut melengkung */
            transition: transform 0.3s ease, box-shadow 0.3s ease;
            animation: fadeInUp 0.8s ease-in-out;
        }

        /* Hover efek */
        .highlight-card:hover {
            transform: translateY(-10px);
            box-shadow: 0 10px 20px rgba(0, 0, 0, 0.2);
        }

        /* Gaya untuk teks */
        .highlight-card h3 {
            font-size: 2.5rem;
            font-weight: bold;
            margin: 0;
        }

        .highlight-card p {
            font-size: 1.1rem;
            margin-top: 5px;
            opacity: 0.9;
        }

        /* Footer styling */
        .highlight-card .card-footer {
            background: rgba(0, 0, 0, 0.1);
            border-top: 1px solid rgba(255, 255, 255, 0.2);
        }

        .highlight-card .card-footer a {
            text-decoration: none;
            font-weight: bold;
            transition: color 0.3s ease;
        }

        .highlight-card .card-footer a:hover {
            color: #ffd700;
            /* Warna emas saat hover */
        }

        /* Animasi untuk ikon */
        .highlight-card i {
            animation: bounce 1.5s infinite;
        }

        /* Bounce animasi untuk ikon */
        @keyframes bounce {

            0%,
            20%,
            50%,
            80%,
            100% {
                transform: translateY(0);
            }

            40% {
                transform: translateY(-10px);
            }

            60% {
                transform: translateY(-5px);
            }
        }

        .pagination .page-link {
            color: #007bff;
        }

        .pagination .active .page-link {
            background-color: #007bff;
            border-color: #007bff;
            color: white;
        }


        .search-container-new {
            position: relative;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .search-box-new {
            display: flex;
            align-items: center;
            transition: all 0.3s ease;
        }

        .search-input-new {
            border-radius: 25px;
            padding: 8px 12px;
            border: 1px solid #ced4da;
            background-color: #ffc107;
            color: #001eff;
            transition: width 0.4s ease, box-shadow 0.3s ease;
            width: 184px;
        }

        .search-input-new::placeholder {
            color: #001eff;
            font-style: italic;
            opacity: 0.8;
        }

        .search-input-new:focus {
            width: 250px;
            background-color: #ffffff;
            border-color: #ffc107;
            box-shadow: 0 0 8px rgba(255, 193, 7, 0.6);
            /* Glow kuning */
            outline: none;
        }

        .search-btn-new {
            background-color: #001eff;
            color: #fff;
            border: none;
            border-radius: 20px;
            padding: 6px 20px;
            font-size: 14px;
            cursor: pointer;
            transition: background-color 0.3s ease, transform 0.3s ease;
            display: none;
        }

        .search-btn-new.show {
            display: inline-block;
        }

        /* ---------- CSS Tambahan ---------- */
        .stepper {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 14px;
            margin: 25px 0 24px;
        }

        .stepper .step {
            display: flex;
            align-items: center;
            gap: 10px;
            position: relative;
        }

        .stepper .step:not(:last-child)::after {
            content: "";
            width: 300px;
            height: 4px;
            background: #0c50e1;
            opacity: .35;
            border-radius: 4px;
            margin-left: 12px;
            margin-right: 4px;
        }

        .stepper .step.completed:not(:last-child)::after {
            background: #ffc107;
            opacity: 1;
        }

        .step-circle {
            width: 36px;
            height: 36px;
            border-radius: 50%;
            display: grid;
            place-items: center;
            font-weight: 700;
            border: 3px solid #0c50e1;
            color: #0c50e1;
            background: #fff;
        }

        .step.active .step-circle {
            background: #ffc107;
            border-color: #ffc107;
            color: #001eff;
            box-shadow: 0 0 0 4px rgba(255, 193, 7, .25);
        }

        .step.completed .step-circle {
            background: #0c50e1;
            border-color: #0c50e1;
            color: #ffc107;
        }

        .step-label {
            font-weight: 600;
            color: #001eff;
        }

        .step-desc {
            font-size: .85rem;
            color: #6c757d;
            margin-top: -2px;
        }

        .card-header {
            background: #0c50e1;
            color: #ffc107;
            border-bottom: 0;
            font-weight: 700;
        }

        .card {
            border: 1px solid rgba(0, 0, 0, .06);
            box-shadow: 0 6px 18px rgba(0, 0, 0, .06);
            border-radius: 16px;
        }

        .btn-print-custom {
            background: #ffc107;
            color: #0c50e1;
            font-weight: 700;
        }

        .btn-print-custom:hover {
            background: #0c50e1;
            color: #ffc107;
            border-color: #0c50e1;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(12, 80, 225, 0.45);
        }

        .btn-save-custom {
            background: #0c50e1;
            color: #ffc107;
            font-weight: 700;
        }

        .btn-save-custom:hover {
            background: #ffc107;
            color: #001eff;
            border-color: #ffc107;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(255, 193, 7, 0.45);
        }

        .btn-print-custom,
        .btn-save-custom {
            transition: all 0.25s ease;
        }

        .form-control,
        .form-select {
            border-radius: 10px;
        }

        .badge-kode {
            font-size: .75rem;
            background: #e9f1ff;
            color: #0c50e1;
            border: 1px solid #0c50e1;
            border-radius: 8px;
            padding: .25rem .5rem;
        }
    </style>
</head>

<body>
    <?php if (!empty($errors)) {
        echo '<pre style="color:#b00;background:#fee;padding:8px;border:1px solid #f99;">'
            . htmlspecialchars(print_r($errors, true), ENT_QUOTES, 'UTF-8') . '</pre>';
    } ?>

    <?php if (isset($_GET['updated'])): ?>
        <script>
            Swal.fire({
                icon: 'success',
                title: 'Berhasil!',
                text: 'Surat berhasil diperbarui.',
                showConfirmButton: true
            });
        </script>

    <?php elseif (isset($_GET['deleted'])): ?>
        <script>
            Swal.fire({
                icon: 'success',
                title: 'Berhasil!',
                text: 'Surat berhasil dihapus.',
                showConfirmButton: true
            });
        </script>
    <?php endif; ?>

    <div class="d-flex">

        <!-- Sidebar -->
        <div class="sidebar">
            <a href="#"><img src="/jasa_marga/img/logojpt-new/Logo Header New.png" alt="#" style="width: 200px"></a>
            <a href="dashboard.php"><i class="fas fa-bars"></i> Dashboard</a>
            <a href="gaji.php"><i class="fas fa-money-bill-wave"></i> Gaji Pegawai</a>
            <a href="surat.php"><i class="fas fa-file-alt"></i> Surat Lainnya</a>

            <a href="#letterSubmenu" class="submenu-toggle" data-bs-toggle="collapse" data-bs-target="#letterSubmenu">
                <i class="fas fa-envelope-open-text"></i> Surat
                <i class="fas fa-caret-down arrow-icon ml-1" id="locArrow"></i>
            </a>
            <div id="letterSubmenu" class="collapse submenu">
                <a href="create-letter.php" class="active"><i class="fas fa-pen-to-square"></i> Buat Surat</a>
                <a href="record-letter.php"><i class="fas fa-folder-open"></i> Record Surat</a>
                <a href="manajemen-surat.php"><i class="fas fa-cog"></i> Manajemen Data Surat</a>
                <a href="letter-order.php" class="<?= (basename($_SERVER['PHP_SELF']) === 'letter-order.php') ? 'active' : ''; ?>"><i class="fas fa-list-ol"></i> Urutan Surat</a>
            </div>

            <a href="#todolistSubmenu" class="submenu-toggle" data-bs-toggle="collapse"
                data-bs-target="#todolistSubmenu">
                <i class="fas fa-tasks"></i> To Do List Pekerjaan
                <i class="fas fa-caret-down arrow-icon ml-1" id="locArrow"></i>
            </a>
            <div id="todolistSubmenu" class="collapse submenu">
                <a href="todolist.php"><i class="fas fa-database"></i> Manajemen Data</a>
                <a href="record-todolist.php"><i class="fas fa-history"></i> Record To Do List</a>
            </div>

            <a href="#agendaSubmenu" class="submenu-toggle" data-bs-toggle="collapse" data-bs-target="#agendaSubmenu">
                <i class="fas fa-calendar-check"></i> Agenda Kerja
                <?php if ($jumlahAgendaAktif > 0): ?>
                    <span class="badge bg-danger ms-2"><?= $jumlahAgendaAktif ?></span>
                <?php endif; ?>
                <i class="fas fa-caret-down arrow-icon ml-1" id="agendaArrow"></i>
            </a>

            <div id="agendaSubmenu" class="collapse submenu">
                <a href="list-agenda.php">
                    <i class="fas fa-clock"></i> List
                    <?php if ($jumlahAgendaAktif > 0): ?>
                        <span class="badge bg-danger ms-1"><?= $jumlahAgendaAktif ?></span>
                    <?php endif; ?>
                </a>
                <a href="record-agenda.php">
                    <i class="fas fa-clipboard-list"></i> Record
                </a>
            </div>
            <a href="sub-unit-kerja.php"><i class="fas fa-sitemap"></i> Manajemen Unit Kerja</a>
            <a href="#systemSubmenu" class="submenu-toggle" data-bs-toggle="collapse" data-bs-target="#systemSubmenu">
                <i class="fas fa-newspaper"></i> Interface
                <i class="fas fa-caret-down arrow-icon ml-1" id="systemArrow"></i>
            </a>
            <div id="systemSubmenu" class="collapse submenu">
                <a href="kontak.php"><i class="fas fa-paper-plane"></i> Contact</a>
                <a href="footer.php"><i class="fas fa-shoe-prints"></i> Footer </a>
                <a href="header.php"><i class="fas fa-heading"></i> Header </a>
                <a href="home.php"><i class="fas fa-house-user"></i> Home </a>
                <a href="nav.php"><i class="fas fa-landmark"></i> Navbar </a>
                <a href="prof.php"><i class="fas fa-user-circle"></i> Profil </a>
                <a href="trav.php" class=""><i class="fas fa-font"></i> Travoy </a>
                <a href="jaringan.php" class=""><i class="fas fa-car"></i> Jaringan Tol</a>
                <a href="chatbot1.php" class=""><i class="fas fa-comments"></i> Chatbot </a>
                <a href="gratifi.php"><i class="fas fa-gift"></i> Gratifikasi</a>
            </div>
            <a href="#locSubmenu" class="submenu-toggle" data-bs-toggle="collapse" data-bs-target="#locSubmenu">
                <i class="fas fa-images" class="active"></i> Tampilan Gambar
                <i class="fas fa-caret-down arrow-icon ml-1" id="locArrow"></i>
            </a>
            <div id="locSubmenu" class="collapse submenu">
                <a href="berita.php"><i class="fas fa-newspaper"></i> Berita </a>
                <a href="sosial.php"><i class="fas fa-file"></i> Tanggung jawab </a>
                <a href="direk.php"><i class="fas fa-user-tie"></i> Direksi </a>
                <a href="komis.php"><i class="fas fa-users"></i> Komisaris </a>
                <a href="organisasi.php"><i class="fas fa-layer-group"></i> Organisasi </a>
                <a href="gerbang.php"><i class="fas fa-road"></i> Gerbang Tol </a>
            </div>
            <a href="/jasa_marga/auth/logout.php"><i class="fas fa-sign-out-alt"></i> Logout</a>
        </div>

        <!-- Surat Content -->
        <div id="mainContent" class="container-fluid py-3">
            <div class="d-flex align-items-center justify-content-between mb-2">
                <h1 class="h3 m-0">Buat Surat</h1>
                <span class="badge-kode">Tanggal:
                    <?php
                    $dateValue = $_POST['tanggal_surat'] ?? date('Y-m-d');
                    if ($dateValue) {
                        // Convert to Indonesian format "20 November 2025"
                        $timestamp = strtotime($dateValue);
                        if ($timestamp !== false) {
                            $bulanIndonesia = [
                                1 => 'Januari',
                                2 => 'Februari',
                                3 => 'Maret',
                                4 => 'April',
                                5 => 'Mei',
                                6 => 'Juni',
                                7 => 'Juli',
                                8 => 'Agustus',
                                9 => 'September',
                                10 => 'Oktober',
                                11 => 'November',
                                12 => 'Desember'
                            ];
                            $hari = date('j', $timestamp);
                            $bulan = $bulanIndonesia[date('n', $timestamp)];
                            $tahun = date('Y', $timestamp);
                            echo htmlspecialchars("$hari $bulan $tahun", ENT_QUOTES, 'UTF-8');
                        } else {
                            echo htmlspecialchars($dateValue, ENT_QUOTES, 'UTF-8');
                        }
                    } else {
                        $timestamp = time();
                        $bulanIndonesia = [
                            1 => 'Januari',
                            2 => 'Februari',
                            3 => 'Maret',
                            4 => 'April',
                            5 => 'Mei',
                            6 => 'Juni',
                            7 => 'Juli',
                            8 => 'Agustus',
                            9 => 'September',
                            10 => 'Oktober',
                            11 => 'November',
                            12 => 'Desember'
                        ];
                        $hari = date('j', $timestamp);
                        $bulan = $bulanIndonesia[date('n', $timestamp)];
                        $tahun = date('Y', $timestamp);
                        echo htmlspecialchars("$hari $bulan $tahun", ENT_QUOTES, 'UTF-8');
                    }
                    ?></span>
            </div>

            <!-- STEPPER -->
            <nav class="stepper" aria-label="Progress">
                <div class="step active" data-step="1">
                    <div class="step-circle">1</div>
                    <div>
                        <div class="step-label">Formulir Surat</div>
                    </div>
                </div>
                <div class="step" data-step="2">
                    <div class="step-circle">2</div>
                    <div>
                        <div class="step-label">Konten Surat</div>
                    </div>
                </div>
            </nav>

            <form method="post" action="" enctype="multipart/form-data" autocomplete="off" id="formNaskah">
                <!-- ============ STEP 1 ============ -->
                <section class="step-pane" data-step="1">
                    <!-- CARD 1 (IDENTITAS SURAT) -->
                    <div class="card mb-4">
                        <div class="card-header">Identitas Surat</div>
                        <div class="card-body row g-3">
                            <!-- Kepentingan -->
                            <div class="col-12">
                                <label class="form-label d-block">Kepentingan <span class="text-danger">*</span></label>
                                <div class="d-flex gap-4">
                                    <label class="form-check">
                                        <input type="radio" class="form-check-input" name="kepentingan" value="internal"
                                            <?php echo (($_POST['kepentingan'] ?? '') === 'internal') ? 'checked' : ''; ?>>
                                        <span class="form-check-label">Internal</span>
                                    </label>
                                    <label class="form-check">
                                        <input type="radio" class="form-check-input" name="kepentingan"
                                            value="eksternal"
                                            <?php echo (($_POST['kepentingan'] ?? '') === 'eksternal') ? 'checked' : ''; ?>>
                                        <span class="form-check-label">Eksternal</span>
                                    </label>
                                </div>
                            </div>

                            <!-- Tipe Surat -->
                            <div class="col-12">
                                <label for="tipe_surat_id" class="form-label">Tipe Surat <span
                                        class="text-danger">*</span></label>
                                <select id="tipe_surat_id" name="tipe_surat_id" class="form-select" required>
                                    <option value="" disabled selected>-- Pilih Tipe Surat --</option>
                                    <!-- Will be populated via JavaScript -->
                                </select>
                            </div>

                            <!-- Classification Fields (Only for Surat Keluar) -->
                            <div id="classification-fields" class="d-none">
                                <!-- Klasifikasi -->
                                <div class="col-12 mb-3">
                                    <label for="klasifikasi_arsip_new" class="form-label">Klasifikasi <span
                                            class="text-danger">*</span></label>
                                    <select id="klasifikasi_arsip_new" name="klasifikasi_arsip_new" class="form-select">
                                        <option value="" disabled selected>-- Pilih Klasifikasi --</option>
                                    </select>
                                </div>

                                <!-- Jenis Surat -->
                                <div class="col-12 mb-3">
                                    <label for="jenis_surat_new" class="form-label">Jenis Surat <span
                                            class="text-danger">*</span></label>
                                    <select id="jenis_surat_new" name="jenis_surat_new" class="form-select">
                                        <option value="" disabled selected>-- Pilih Klasifikasi Terlebih Dahulu --
                                        </option>
                                    </select>
                                </div>

                                <!-- Subjenis Surat -->
                                <div class="col-12 mb-3">
                                    <label for="subjenis_surat_id" class="form-label">Subjenis Surat <span
                                            class="text-danger">*</span></label>
                                    <select id="subjenis_surat_id" name="subjenis_surat_id" class="form-select">
                                        <option value="" disabled selected>-- Pilih Jenis Surat Terlebih Dahulu --
                                        </option>
                                    </select>
                                </div>
                            </div>

                            <!-- Legacy Jenis Surat (Hidden by default) -->
                            <div id="legacy-jenis-field" class="col-12 d-none">
                                <label for="jenis_surat_id" class="form-label">Jenis Surat (Legacy) <span
                                        class="text-danger">*</span></label>
                                <select id="jenis_surat_id" name="jenis_surat_id" class="form-select" required>
                                    <option value="" disabled
                                        <?php echo empty($_POST['jenis_surat_id']) ? 'selected' : ''; ?>>-- Pilih Jenis
                                        Surat --</option>
                                    <?php foreach ($jenisSurat as $row): ?>
                                        <option value="<?php echo (int)$row['id']; ?>"
                                            data-kode="<?php echo htmlspecialchars($row['kode'], ENT_QUOTES, 'UTF-8'); ?>"
                                            <?php echo ((string)($_POST['jenis_surat_id'] ?? '') === (string)$row['id']) ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($row['nama'], ENT_QUOTES, 'UTF-8'); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>

                        <!-- Legacy Klasifikasi (Hidden by default) -->
                        <div class="row mb-3">
                            <div id="legacy-klasifikasi-field" class="col-12 d-none">
                                <label for="klasifikasi_arsip_id" class="form-label">Klasifikasi (Legacy) <span
                                        class="text-danger">*</span></label>
                                <select id="klasifikasi_arsip_id" name="klasifikasi_arsip_id" class="form-select">
                                    <option value="" disabled
                                        <?php echo empty($_POST['klasifikasi_arsip_id']) ? 'selected' : ''; ?>>-- Pilih
                                        Klasifikasi --</option>
                                    <?php foreach ($klasifikasi as $row): ?>
                                        <option value="<?php echo (int)$row['id']; ?>"
                                            data-kode="<?php echo htmlspecialchars($row['kode'], ENT_QUOTES, 'UTF-8'); ?>"
                                            <?php echo ((string)($_POST['klasifikasi_arsip_id'] ?? '') === (string)$row['id']) ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($row['nama'], ENT_QUOTES, 'UTF-8'); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>

                        <div class="row mb-3">
                            <!-- Tempat -->
                            <div class="col-12">
                                <label for="tempat_dibuat" class="form-label">Tempat Surat Dibuat <span
                                        class="text-danger">*</span></label>
                                <input type="text" id="tempat_dibuat" name="tempat_dibuat" class="form-control" required
                                    value="<?php echo htmlspecialchars($_POST['tempat_dibuat'] ?? '', ENT_QUOTES, 'UTF-8'); ?>">
                            </div>

                            <!-- Pejabat yang Menandatangani (moved from Subyek card) -->
                            <div class="col-12">
                                <label class="form-label d-block">Pejabat yang Menandatangani <span
                                        class="text-danger">*</span></label>
                                <div class="d-flex flex-wrap gap-4">
                                    <label class="form-check">
                                        <input type="radio" class="form-check-input" name="dari_tipe" value="pribadi"
                                            <?php echo (($_POST['dari_tipe'] ?? '') === 'pribadi') ? 'checked' : ''; ?>>
                                        <span class="form-check-label">Pejabat</span>
                                    </label>
                                    <label class="form-check">
                                        <input type="radio" class="form-check-input" name="dari_tipe" value="sendiri" id="dariTipeSendiri"
                                            <?php echo (($_POST['dari_tipe'] ?? '') === 'sendiri') ? 'checked' : ''; ?>>
                                        <span class="form-check-label">Diri Sendiri</span>
                                    </label>
                                </div>
                            </div>

                            <!-- Nama Yang Menandatangani Dropdown (filtered to 16 authorized signers) -->
                            <div class="col-12" id="wrapPribadi" style="display:none;">
                                <label for="dari_pribadi" class="form-label">Nama Yang Menandatangani</label>
                                <select id="dari_pribadi" name="dari_pribadi" class="form-select">
                                    <option value="" disabled selected>-- Pilih Pejabat --</option>
                                    <!-- Will be populated via JavaScript with authorized signers only -->
                                </select>
                            </div>
                            <input type="hidden" id="user_nama" value="<?= htmlspecialchars($namaLogin, ENT_QUOTES) ?>">
                            <input type="hidden" id="user_jabatan" value="<?= htmlspecialchars($jabatanLogin, ENT_QUOTES) ?>">
                            <input type="hidden" id="user_id" value="<?= (int)$idLogin ?>">
                            <input type="hidden" id="user_tipe" value="<?= htmlspecialchars($user['tipe'] ?? '', ENT_QUOTES) ?>">

                            <!-- Backdate Checkbox -->
                            <div class="col-12">
                                <div class="form-check">
                                    <input type="checkbox" class="form-check-input" id="backdate_checkbox" name="is_backdate">
                                    <label class="form-check-label" for="backdate_checkbox">Backdate</label>
                                </div>
                            </div>

                            <!-- Nomor Surat with Previous Number Indicator -->
                            <div class="col-12">
                                <label for="nomor_surat" class="form-label">
                                    Nomor Surat <span class="text-danger">*</span>
                                    <small id="prev_nomor_indicator" class="text-muted ms-2" style="font-weight: normal;"></small>
                                </label>
                                <input type="text" id="nomor_surat" name="nomor_surat" class="form-control" required
                                    readonly
                                    value="<?php echo htmlspecialchars($_POST['nomor_surat'] ?? '', ENT_QUOTES, 'UTF-8'); ?>">

                                <!-- Backdate Edit Fields (hidden by default) -->
                                <div id="backdate_fields" class="d-none mt-2">
                                    <div class="row g-2">
                                        <div class="col-md-4">
                                            <label for="backdate_sequence" class="form-label small">Nomor Urut</label>
                                            <input type="number" id="backdate_sequence" name="backdate_sequence"
                                                class="form-control form-control-sm" min="1" placeholder="Urutan">
                                        </div>
                                        <div class="col-md-4" id="backdate_year_field" style="display:none;">
                                            <label for="backdate_year" class="form-label small">Tahun</label>
                                            <input type="number" id="backdate_year" name="backdate_year"
                                                class="form-control form-control-sm" min="2020" max="2099"
                                                value="<?= date('Y') ?>" placeholder="Tahun">
                                        </div>
                                    </div>
                                    <small class="text-muted" id="backdate_hint">Masukkan nomor urut untuk backdate</small>
                                </div>
                            </div>

                            <!-- Tanggal Surat -->
                            <div class="col-12">
                                <label for="tanggal_surat" class="form-label">Tanggal Surat <span
                                        class="text-danger">*</span></label>
                                <input type="date" id="tanggal_surat" name="tanggal_surat" class="form-control" required
                                    value="<?php echo htmlspecialchars($_POST['tanggal_surat'] ?? '', ENT_QUOTES, 'UTF-8'); ?>">
                            </div>

                            <!-- Paraf field removed - now auto-generated from approval chain -->
                        </div>
                    </div>

                    <!-- CARD 2 (SUBYEK) - Only Kepada field remains here -->
                    <div class="card mb-4">
                        <div class="card-header">Subyek</div>
                        <div class="card-body row g-3">
                            <!-- Kepada -->
                            <div class="col-12" id="kepada-field">
                                <label for="kepada" class="form-label">Kepada <span class="text-danger">*</span></label>
                                <input type="text" id="kepada" name="kepada" class="form-control" required
                                    value="<?php echo htmlspecialchars($_POST['kepada'] ?? '', ENT_QUOTES, 'UTF-8'); ?>">
                            </div>
                        </div>
                    </div>

                    <!-- CARD 3 (INFORMASI SURAT) -->
                    <div class="card mb-3">
                        <div class="card-header">Informasi Surat</div>
                        <div class="card-body row g-3">
                            <!-- Perihal -->
                            <div class="col-12">
                                <label for="perihal" class="form-label">Perihal <span
                                        class="text-danger">*</span></label>
                                <input type="text" id="perihal" name="perihal" class="form-control" required
                                    value="<?php echo htmlspecialchars($_POST['perihal'] ?? '', ENT_QUOTES, 'UTF-8'); ?>">
                            </div>

                            <!-- Lampiran -->
                            <div class="col-12">
                                <label for="lampiran" class="form-label">Lampiran</label>
                                <input type="file" id="lampiran" name="lampiran" class="form-control"
                                    accept=".pdf,.jpg,.jpeg,.png">

                                <input type="hidden" id="lampiran_url" name="lampiran_url" value="">
                                <input type="hidden" id="lampiran_mime" name="lampiran_mime" value="">
                            </div>
                        </div>
                    </div>

                    <!-- TOMBOL -->
                    <div class="d-flex justify-content-end gap-2">
                        <button type="button" class="btn btn-save-custom" id="toStep2">Lanjutkan</button>
                    </div>
                </section>

                <!-- ============ STEP 2 ============ -->
                <section class="step-pane d-none" data-step="2">
                    <div class="card mb-3">
                        <div class="card-header d-flex align-items-center justify-content-between">
                            <span style="font-size: 18px;">Isi Surat</span>
                            <!-- <div class="d-flex gap-2">
                                <button type="button" class="btn btn-outline-warning color-warning btn-md" id="printDoc">
                                Preview Hasil Surat
                                </button>
                            </div> -->
                        </div>
                        <div class="card-body p-0">
                            <textarea id="editorNaskah" name="naskah_html"></textarea>
                        </div>
                    </div>

                    <input type="hidden" name="__final_submit" value="1" />

                    <div class="d-flex justify-content-between align-items-center flex-wrap gap-2 mt-3">
                        <button type="button" class="btn btn-danger" id="backTo1">Kembali</button>

                        <div class="d-flex gap-2">
                            <button type="button" id="btnUploadPdf" class="btn btn-print-custom">
                                Cetak & Upload PDF
                            </button>
                            <button type="submit" id="btnSave" class="btn btn-save-custom">
                                Simpan
                            </button>
                        </div>
                    </div>

                    <input type="hidden" name="file_surat" id="fileSuratPath" value="">

                </section>
            </form>
        </div>

    </div>

    <!-- JS -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

    <!-- JS Surat -->
    <script>
        (function() {
            const jenisSelect = document.getElementById('jenis_surat_id');
            const klasSelect = document.getElementById('klasifikasi_arsip_id');
            const nomorInput = document.getElementById('nomor_surat');

            async function updateNomor() {
                if (!jenisSelect || !klasSelect || !nomorInput) return;
                const jenisId = parseInt(jenisSelect.value || '0', 10);
                const klasId = parseInt(klasSelect.value || '0', 10);
                if (!jenisId || !klasId) return;

                try {
                    const url =
                        `/jasa_marga/api/letter-autofill.php?jenis_id=${encodeURIComponent(jenisId)}&klas_id=${encodeURIComponent(klasId)}`;
                    const res = await fetch(url, {
                        credentials: 'same-origin'
                    });

                    const ctype = res.headers.get('content-type') || '';
                    if (!ctype.includes('application/json')) {
                        console.warn('Autofill nomor: respons bukan JSON. Status:', res.status);
                        return;
                    }

                    const data = await res.json();
                    if (data && data.ok && data.nomor) {
                        // Try to compute kodeJabatan prefix for SK if possible
                        let finalNomor = data.nomor;
                        try {
                            const kodeJabatanList = <?= json_encode($kodeJabatanMap) ?>;
                            const usersData = <?= json_encode($usersList) ?>;
                            const currentUserId = <?= json_encode($idLogin) ?>;
                            const tipe = document.getElementById('tipe_surat_id');
                            const jenisId = parseInt(jenisSelect.value || '0', 10);
                            const klasId = parseInt(klasSelect.value || '0', 10);
                            let createdBy = currentUserId;
                            const dariChecked = (document.querySelector('input[name="dari_tipe"]:checked') || {}).value || 'sendiri';
                            if (dariChecked === 'pribadi') {
                                const dariSelect = document.getElementById('dari_pribadi');
                                const v = parseInt(dariSelect.value || 0);
                                if (v > 0) createdBy = v;
                            }
                            // Only for SK try to add kodeJabatan
                            if (tipe && tipe.options[tipe.selectedIndex] && tipe.options[tipe.selectedIndex].dataset.kode === 'SK') {
                                const usr = usersData.find(u => parseInt(u.id) === parseInt(createdBy)) || null;
                                if (usr && usr.jabatan) {
                                    const jab = usr.jabatan.toString().trim().toLowerCase();
                                    let kode = null;
                                    for (const it of kodeJabatanList) {
                                        const main = (it.jabatan || '').toString().trim().toLowerCase();
                                        if (main === jab) {
                                            kode = it.kode;
                                            break;
                                        }
                                        if (it.aliases && Array.isArray(it.aliases)) {
                                            for (const a of it.aliases) {
                                                if ((a || '').toString().trim().toLowerCase() === jab) {
                                                    kode = it.kode;
                                                    break;
                                                }
                                            }
                                            if (kode) break;
                                        }
                                    }
                                    if (kode) finalNomor = kode + '.' + data.nomor;
                                }
                            }
                        } catch (e) {
                            console.warn('kodeJabatan preview failed', e);
                        }
                        nomorInput.value = finalNomor;
                    } else {
                        console.warn('Autofill nomor: data tidak ok', data);
                    }
                } catch (e) {
                    console.warn('Gagal auto nomor_surat', e);
                }
            }

            if (jenisSelect) jenisSelect.addEventListener('change', updateNomor);
            if (klasSelect) klasSelect.addEventListener('change', updateNomor);
        })();

        // Dynamic Classification System JavaScript
        (function() {
            const tipeSuratSelect = document.getElementById('tipe_surat_id');
            const classificationFields = document.getElementById('classification-fields');
            const legacyJenisField = document.getElementById('legacy-jenis-field');
            const legacyKlasifikasiField = document.getElementById('legacy-klasifikasi-field');

            const klasifikasiSelect = document.getElementById('klasifikasi_arsip_new');
            const jenisSelect = document.getElementById('jenis_surat_new');
            const subjenisSelect = document.getElementById('subjenis_surat_id');
            const nomorInput = document.getElementById('nomor_surat');
            const dariPribadiSelect = document.getElementById('dari_pribadi');

            // Backdate elements
            const backdateCheckbox = document.getElementById('backdate_checkbox');
            const backdateFields = document.getElementById('backdate_fields');
            const backdateSequence = document.getElementById('backdate_sequence');
            const backdateYear = document.getElementById('backdate_year');
            const backdateYearField = document.getElementById('backdate_year_field');
            const backdateHint = document.getElementById('backdate_hint');
            const prevNomorIndicator = document.getElementById('prev_nomor_indicator');
            const dariTipeSendiri = document.getElementById('dariTipeSendiri');

            // Authorized signers data (will be loaded from API)
            let authorizedSigners = [];
            const currentUserName = document.getElementById('user_nama')?.value || '';
            const currentUserId = document.getElementById('user_id')?.value || '';

            // Initialize form
            initializeForm();

            // Event listeners
            if (tipeSuratSelect) {
                tipeSuratSelect.addEventListener('change', handleTipeSuratChange);
            }
            if (klasifikasiSelect) {
                klasifikasiSelect.addEventListener('change', handleKlasifikasiChange);
            }
            if (jenisSelect) {
                jenisSelect.addEventListener('change', handleJenisChange);
            }
            if (subjenisSelect) {
                subjenisSelect.addEventListener('change', handleSubjenisChange);
            }
            if (dariPribadiSelect) {
                dariPribadiSelect.addEventListener('change', handleDariPribadiChange);
            }

            // Backdate checkbox listener
            if (backdateCheckbox) {
                backdateCheckbox.addEventListener('change', handleBackdateToggle);
            }
            if (backdateSequence) {
                backdateSequence.addEventListener('input', handleBackdateInput);
            }
            if (backdateYear) {
                backdateYear.addEventListener('input', handleBackdateInput);
            }

            async function handleDariPribadiChange() {
                // Generate nomor surat when pejabat changes
                await generateNomorSurat();
            }

            async function initializeForm() {
                await loadAuthorizedSigners();
                await loadTipeSurat();
                await loadKlasifikasi();
                checkDiriSendiriEligibility();
            }

            // Load authorized signers (16 registered names)
            async function loadAuthorizedSigners() {
                try {
                    const response = await fetch('api_classification.php?action=get_authorized_signers');
                    const data = await response.json();

                    if (data.success && dariPribadiSelect) {
                        authorizedSigners = data.data;
                        dariPribadiSelect.innerHTML = '<option value="" disabled selected>-- Pilih Pejabat --</option>';

                        data.data.forEach(signer => {
                            const option = document.createElement('option');
                            option.value = signer.user_id || signer.id;
                            option.textContent = signer.nama + (signer.jabatan ? ` - ${signer.jabatan}` : '');
                            option.dataset.jabatan = signer.jabatan || '';
                            option.dataset.nama = signer.nama;
                            dariPribadiSelect.appendChild(option);
                        });
                    }
                } catch (error) {
                    console.error('Error loading authorized signers:', error);
                }
            }

            // Check if current user is in authorized signers list or is admin
            function checkDiriSendiriEligibility() {
                if (!dariTipeSendiri || !currentUserName) return;

                const userTipe = document.getElementById('user_tipe')?.value || '';
                // Allow admin to always select 'Diri Sendiri'
                if (userTipe === 'admin') return;

                const isAuthorized = authorizedSigners.some(signer =>
                    signer.nama.toLowerCase().trim() === currentUserName.toLowerCase().trim()
                );

                if (!isAuthorized) {
                    // Disable "Diri Sendiri" option for non-authorized users
                    dariTipeSendiri.disabled = true;
                    dariTipeSendiri.parentElement.classList.add('text-muted');
                    dariTipeSendiri.parentElement.title = 'Anda tidak terdaftar sebagai penandatangan resmi';
                }
            }

            // Handle backdate checkbox toggle
            function handleBackdateToggle() {
                if (backdateCheckbox.checked) {
                    backdateFields.classList.remove('d-none');
                    nomorInput.removeAttribute('readonly');
                    updateBackdateFieldsVisibility();
                } else {
                    backdateFields.classList.add('d-none');
                    nomorInput.setAttribute('readonly', 'readonly');
                    // Re-generate normal nomor surat
                    generateNomorSurat();
                }
            }

            // Show/hide year field based on tipe surat
            function updateBackdateFieldsVisibility() {
                const selectedOption = tipeSuratSelect?.selectedOptions[0];
                const tipeSuratKode = selectedOption?.dataset.kode;

                if (tipeSuratKode === 'SK') {
                    // Surat Keluar - only edit sequence number
                    backdateYearField.style.display = 'none';
                    backdateHint.textContent = 'Masukkan nomor urut untuk backdate';
                } else {
                    // KPTS/SE - edit sequence and year
                    backdateYearField.style.display = 'block';
                    backdateHint.textContent = 'Masukkan nomor urut dan tahun untuk backdate';
                }
            }

            // Handle backdate input changes
            function handleBackdateInput() {
                if (!backdateCheckbox.checked) return;

                const selectedOption = tipeSuratSelect?.selectedOptions[0];
                const tipeSuratKode = selectedOption?.dataset.kode;
                const sequence = backdateSequence.value;
                const year = backdateYear.value || new Date().getFullYear();

                if (!sequence) return;

                // Build nomor surat based on type
                if (tipeSuratKode === 'SK') {
                    // For SK, we need to rebuild the full nomor
                    // Get current nomor format and replace just the sequence
                    const currentNomor = nomorInput.value;
                    if (currentNomor) {
                        const parts = currentNomor.split('.');
                        parts[parts.length - 1] = sequence;
                        nomorInput.value = parts.join('.');
                    }
                } else if (tipeSuratKode === 'KPTS') {
                    const paddedSeq = String(sequence).padStart(3, '0');
                    nomorInput.value = `${paddedSeq}/KPTS-JPT/${year}`;
                } else if (tipeSuratKode === 'SE') {
                    const paddedSeq = String(sequence).padStart(3, '0');
                    nomorInput.value = `${paddedSeq}/SE-DIR/${year}`;
                }
            }

            // Update previous letter number indicator
            async function updatePrevNomorIndicator() {
                if (!tipeSuratSelect?.value || !prevNomorIndicator) return;

                try {
                    const response = await fetch(`api_classification.php?action=get_last_letter_number&tipe_surat_id=${tipeSuratSelect.value}`);
                    const data = await response.json();

                    if (data.success && data.data) {
                        if (data.data.last_nomor_surat) {
                            prevNomorIndicator.textContent = `(Nomor Surat Sebelumnya: ${data.data.last_nomor_surat})`;
                        } else {
                            prevNomorIndicator.textContent = '(Belum ada surat sebelumnya)';
                        }
                    }
                } catch (error) {
                    console.error('Error loading last letter number:', error);
                    prevNomorIndicator.textContent = '';
                }
            }

            async function loadTipeSurat() {
                try {
                    const response = await fetch('api_classification.php?action=get_tipe_surat');
                    const data = await response.json();

                    if (data.success && tipeSuratSelect) {
                        tipeSuratSelect.innerHTML =
                            '<option value="" disabled selected>-- Pilih Tipe Surat --</option>';
                        data.data.forEach(item => {
                            const option = document.createElement('option');
                            option.value = item.id;
                            option.textContent = item.nama;
                            option.dataset.kode = item.kode;
                            tipeSuratSelect.appendChild(option);
                        });
                    }
                } catch (error) {
                    console.error('Error loading tipe surat:', error);
                }
            }

            async function loadKlasifikasi() {
                try {
                    const response = await fetch('api_classification.php?action=get_klasifikasi');
                    const data = await response.json();

                    if (data.success && klasifikasiSelect) {
                        klasifikasiSelect.innerHTML =
                            '<option value="" disabled selected>-- Pilih Klasifikasi --</option>';
                        data.data.forEach(item => {
                            const option = document.createElement('option');
                            option.value = item.id;
                            option.textContent = `${item.kode} - ${item.nama}`;
                            option.dataset.kode = item.kode;
                            klasifikasiSelect.appendChild(option);
                        });
                    }
                } catch (error) {
                    console.error('Error loading klasifikasi:', error);
                }
            }

            async function handleTipeSuratChange() {
                const selectedOption = tipeSuratSelect.selectedOptions[0];
                if (!selectedOption) return;

                const tipeSuratKode = selectedOption.dataset.kode;
                const kepadaField = document.getElementById('kepada-field');
                const kepadaInput = document.getElementById('kepada');

                // Show/hide fields based on tipe surat
                if (tipeSuratKode === 'SK') { // Surat Keluar
                    classificationFields.classList.remove('d-none');
                    legacyJenisField.classList.add('d-none');
                    legacyKlasifikasiField.classList.add('d-none');
                    if (kepadaField) kepadaField.classList.remove('d-none');
                    if (kepadaInput) kepadaInput.required = true;

                    // Make new fields required
                    setFieldsRequired(['klasifikasi_arsip_new', 'jenis_surat_new', 'subjenis_surat_id'], true);
                    setFieldsRequired(['jenis_surat_id', 'klasifikasi_arsip_id'], false);
                } else { // Keputusan Direksi atau Edaran Direksi
                    classificationFields.classList.add('d-none');
                    legacyJenisField.classList.add('d-none');
                    legacyKlasifikasiField.classList.add('d-none');
                    if (kepadaField) kepadaField.classList.add('d-none');
                    if (kepadaInput) kepadaInput.required = false;

                    // Remove required from all classification fields
                    setFieldsRequired(['klasifikasi_arsip_new', 'jenis_surat_new', 'subjenis_surat_id'], false);
                    setFieldsRequired(['jenis_surat_id', 'klasifikasi_arsip_id'], false);
                }

                // Update backdate fields visibility and previous number indicator
                updateBackdateFieldsVisibility();
                await updatePrevNomorIndicator();

                // Generate nomor surat
                await generateNomorSurat();
            }

            async function handleKlasifikasiChange() {
                const klasifikasiId = klasifikasiSelect.value;
                if (!klasifikasiId) return;

                // Clear dependent fields
                jenisSelect.innerHTML = '<option value="" disabled selected>Loading...</option>';
                subjenisSelect.innerHTML =
                    '<option value="" disabled selected>-- Pilih Jenis Surat Terlebih Dahulu --</option>';

                try {
                    const response = await fetch(
                        `api_classification.php?action=get_jenis_by_klasifikasi&klasifikasi_id=${klasifikasiId}`
                    );
                    const data = await response.json();

                    if (data.success) {
                        jenisSelect.innerHTML =
                            '<option value="" disabled selected>-- Pilih Jenis Surat --</option>';
                        data.data.forEach(item => {
                            const option = document.createElement('option');
                            option.value = item.id;
                            option.textContent = `${item.kode} - ${item.nama}`;
                            option.dataset.kode = item.kode;
                            jenisSelect.appendChild(option);
                        });
                    } else {
                        jenisSelect.innerHTML =
                            '<option value="" disabled selected>-- Tidak ada jenis surat --</option>';
                    }
                } catch (error) {
                    console.error('Error loading jenis surat:', error);
                    jenisSelect.innerHTML = '<option value="" disabled selected>-- Error loading data --</option>';
                }
            }

            async function handleJenisChange() {
                const jenisId = jenisSelect.value;
                if (!jenisId) return;

                // Clear dependent fields
                subjenisSelect.innerHTML = '<option value="" disabled selected>Loading...</option>';

                try {
                    const response = await fetch(
                        `api_classification.php?action=get_subjenis_by_jenis&jenis_id=${jenisId}`);
                    const data = await response.json();

                    if (data.success) {
                        subjenisSelect.innerHTML =
                            '<option value="" disabled selected>-- Pilih Subjenis Surat --</option>';
                        data.data.forEach(item => {
                            const option = document.createElement('option');
                            option.value = item.id;
                            option.textContent = `${item.kode} - ${item.nama}`;
                            option.dataset.kode = item.kode;
                            subjenisSelect.appendChild(option);
                        });
                    } else {
                        subjenisSelect.innerHTML =
                            '<option value="" disabled selected>-- Tidak ada subjenis surat --</option>';
                    }
                } catch (error) {
                    console.error('Error loading subjenis surat:', error);
                    subjenisSelect.innerHTML =
                        '<option value="" disabled selected>-- Error loading data --</option>';
                }

                // Generate nomor surat
                await generateNomorSurat();
            }

            async function handleSubjenisChange() {
                // Generate nomor surat when subjenis changes
                await generateNomorSurat();
            }

            async function generateNomorSurat() {
                if (!nomorInput) return;

                const tipeSuratId = tipeSuratSelect.value;
                if (!tipeSuratId) {
                    nomorInput.value = '';
                    return;
                }

                try {
                    const params = new URLSearchParams({
                        action: 'generate_nomor_surat',
                        tipe_surat_id: tipeSuratId
                    });

                    // Add classification data for Surat Keluar
                    const selectedTipeOption = tipeSuratSelect.selectedOptions[0];
                    if (selectedTipeOption && selectedTipeOption.dataset.kode === 'SK') {
                        const klasifikasiOption = klasifikasiSelect.selectedOptions[0];
                        const jenisOption = jenisSelect.selectedOptions[0];
                        const subjenisId = subjenisSelect.value;

                        if (klasifikasiOption && jenisOption && subjenisId) {
                            params.append('klasifikasi_kode', klasifikasiOption.dataset.kode);
                            params.append('jenis_kode', jenisOption.dataset.kode);
                            params.append('subjenis_id', subjenisId);

                            // Get jabatan from dropdown "Nama Yang Mentandatangani"
                            let jabatan = '';
                            const dariTipeChecked = document.querySelector('input[name="dari_tipe"]:checked');

                            if (dariTipeChecked && dariTipeChecked.value === 'pribadi' && dariPribadiSelect &&
                                dariPribadiSelect.value) {
                                // Get jabatan from selected pejabat
                                const selectedOption = dariPribadiSelect.selectedOptions[0];
                                jabatan = selectedOption?.dataset?.jabatan || '';
                            } else if (dariTipeChecked && dariTipeChecked.value === 'sendiri') {
                                // Get jabatan from logged-in user (hidden field)
                                jabatan = document.getElementById('user_jabatan')?.value || '';

                                // If jabatan still empty, try to get from selected option in dropdown
                                if (!jabatan && dariPribadiSelect && dariPribadiSelect.value) {
                                    const selectedOption = dariPribadiSelect.selectedOptions[0];
                                    jabatan = selectedOption?.dataset?.jabatan || '';
                                }
                            }

                            // Jabatan is optional - if empty, nomor will be generated without kode jabatan prefix
                            // This allows users without jabatan mapping to still create letters
                            if (!jabatan) {
                                console.info('Jabatan kosong - nomor surat akan dibuat tanpa kode jabatan prefix');
                            }

                            params.append('jabatan', jabatan || '');
                        } else {
                            // Not all fields selected yet
                            nomorInput.value = '';
                            return;
                        }
                    }

                    const response = await fetch(`api_classification.php?${params}`);
                    const data = await response.json();

                    if (data.success) {
                        nomorInput.value = data.data.nomor_surat;
                    } else {
                        console.warn('Error generating nomor surat:', data.message);
                        nomorInput.value = '';
                    }
                } catch (error) {
                    console.error('Error generating nomor surat:', error);
                    nomorInput.value = '';
                }
            }

            function setFieldsRequired(fieldIds, required) {
                fieldIds.forEach(fieldId => {
                    const field = document.getElementById(fieldId);
                    if (field) {
                        if (required) {
                            field.setAttribute('required', '');
                        } else {
                            field.removeAttribute('required');
                        }
                    }
                });
            }
        })();

        (function() {
            const get = (id) => document.getElementById(id);
            const fileInput = document.getElementById('lampiran');
            const hidUrl = document.getElementById('lampiran_url');
            const hidMime = document.getElementById('lampiran_mime');

            let lastBlobUrl = null;

            fileInput?.addEventListener('change', (e) => {
                const f = e.target.files && e.target.files[0] ? e.target.files[0] : null;

                if (lastBlobUrl) {
                    URL.revokeObjectURL(lastBlobUrl);
                    lastBlobUrl = null;
                }

                if (!f) {
                    hidUrl.value = '';
                    hidMime.value = '';
                    return;
                }

                const blobUrl = URL.createObjectURL(f);
                lastBlobUrl = blobUrl;

                hidUrl.value = blobUrl;
                hidMime.value = f.type || '';
            });
        })();
    </script>

    <!-- TinyMCE Self-Hosted Community Version (No API Key Required) -->
    <script src="https://cdn.jsdelivr.net/npm/tinymce@6.8.3/tinymce.min.js" referrerpolicy="origin"></script>
    <script>
        (function() {
            const pribadiWrap = document.getElementById('wrapPribadi');
            const pribadiSelect = document.getElementById('dari_pribadi');
            const userNama = document.getElementById('user_nama').value;
            const userJabatan = document.getElementById('user_jabatan').value;

            // Function to select current user in dropdown by matching nama
            function selectCurrentUser() {
                const options = pribadiSelect.options;
                for (let i = 0; i < options.length; i++) {
                    const optionText = options[i].textContent.trim();
                    // Check if option text starts with current user's name
                    if (optionText.startsWith(userNama)) {
                        pribadiSelect.selectedIndex = i;
                        return;
                    }
                }
            }

            document.querySelectorAll('input[name="dari_tipe"]').forEach(r => {
                r.addEventListener('change', async () => {
                    if (r.value === 'pribadi' && r.checked) {
                        pribadiWrap.style.display = 'block';
                        pribadiSelect.disabled = false;
                        pribadiSelect.value = '';
                    } else if (r.value === 'sendiri' && r.checked) {
                        pribadiWrap.style.display = 'block';
                        pribadiSelect.disabled = true;
                        selectCurrentUser();
                        // Trigger nomor surat generation after selecting current user
                        if (typeof generateNomorSurat === 'function') {
                            await generateNomorSurat();
                        }
                    } else {
                        pribadiWrap.style.display = 'none';
                        pribadiSelect.disabled = false;
                    }
                });

                // Initialize on page load
                if (r.checked) {
                    if (r.value === 'pribadi') {
                        pribadiWrap.style.display = 'block';
                        pribadiSelect.disabled = false;
                    } else if (r.value === 'sendiri') {
                        pribadiWrap.style.display = 'block';
                        pribadiSelect.disabled = true;
                        selectCurrentUser();
                    }
                }
            });

            const to2 = document.getElementById('toStep2');
            const back1 = document.getElementById('backTo1');
            const panes = document.querySelectorAll('.step-pane');
            const steps = document.querySelectorAll('.stepper .step');
            const form = document.getElementById('formNaskah');

            function setStep(n) {
                panes.forEach(p => p.classList.toggle('d-none', p.dataset.step !== String(n)));
                steps.forEach(s => {
                    const cur = Number(s.dataset.step);
                    s.classList.toggle('active', cur === n);
                    s.classList.toggle('completed', cur < n);
                });
                window.scrollTo({
                    top: 0,
                    behavior: 'smooth'
                });
            }
            back1.addEventListener('click', () => setStep(1));

            const get = id => document.getElementById(id);
            const getVal = id => (get(id)?.value || '').trim();
            const safe = (v) => String(v ?? '');

            function toIDDate(str) {
                if (!str) return '';
                const m = /^(\d{4})-(\d{2})-(\d{2})$/.exec(str);
                if (!m) return str;

                const months = [
                    'Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni',
                    'Juli', 'Agustus', 'September', 'Oktober', 'November', 'Desember'
                ];

                const day = parseInt(m[3]);
                const monthIndex = parseInt(m[2]) - 1;
                const year = m[1];

                return `${day} ${months[monthIndex]} ${year}`;
            }

            function todayID() {
                const d = new Date();
                const months = [
                    'Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni',
                    'Juli', 'Agustus', 'September', 'Oktober', 'November', 'Desember'
                ];

                const day = d.getDate();
                const month = months[d.getMonth()];
                const year = d.getFullYear();
                return `${day} ${month} ${year}`;
            }

            function collectFormValues() {
                const jenisOpt = get('jenis_surat_id')?.selectedOptions?.[0];
                const klasOpt = get('klasifikasi_arsip_id')?.selectedOptions?.[0];
                const dariTipe = (document.querySelector('input[name="dari_tipe"]:checked')?.value || '');

                const namaLogin = getVal('user_nama');
                const jabatanUser = getVal('user_jabatan');

                let namaPejabat = getVal('nama_pejabat') || '';
                let jabatanPejabat = getVal('jabatan_pejabat') || '';

                if (dariTipe === 'pribadi') {
                    const selectedOption = get('dari_pribadi')?.selectedOptions?.[0];
                    // Get the text content (nama) from the option, not the value (id)
                    if (selectedOption) {
                        const optionText = selectedOption.textContent.trim();
                        // Extract nama (before the dash) if format is "Nama - Jabatan"
                        namaPejabat = optionText.split('-')[0].trim();
                    } else {
                        namaPejabat = namaLogin;
                    }
                    jabatanPejabat = selectedOption?.dataset?.jabatan || jabatanUser || '';
                } else if (dariTipe === 'sendiri') {
                    namaPejabat = namaLogin;
                    jabatanPejabat = jabatanUser || '';
                }

                const tglInput = getVal('tanggal_surat');
                const tglID = toIDDate(tglInput) || todayID();

                // Determine document title based on tipe_surat
                const tipeSuratSelect = document.getElementById('tipe_surat_id');
                let documentTitle = 'Surat Keluar'; // default
                if (tipeSuratSelect) {
                    const selectedOption = tipeSuratSelect.options[tipeSuratSelect.selectedIndex];
                    if (selectedOption) {
                        const tipeSuratText = selectedOption.textContent.trim();
                        if (tipeSuratText.includes('Keputusan')) {
                            documentTitle = 'KEPUTUSAN DIREKSI PT JASAMARGA PANDAAN TOL';
                        } else if (tipeSuratText.includes('Edaran')) {
                            documentTitle = 'SURAT EDARAN';
                        } else {
                            // For regular Surat Keluar, always use "Surat Keluar"
                            documentTitle = 'Surat Keluar';
                        }
                    }
                }

                return {
                    JENIS_SURAT: safe(jenisOpt ? jenisOpt.textContent.trim() : ''),
                    DOCUMENT_TITLE: safe(documentTitle),
                    NOMOR_SURAT: safe(getVal('nomor_surat')),
                    KOTA: safe(getVal('tempat_dibuat')),
                    TANGGAL_SURAT: safe(tglID),

                    NAMA_PEJABAT: safe(namaPejabat),
                    NIK_PEJABAT: safe(getVal('nik_pejabat') || ''),
                    JABATAN_PEJABAT: safe(jabatanPejabat),
                    DEPARTEMEN: safe(getVal('departemen') || ''),
                    NAMA_PERUSAHAAN: safe(getVal('nama_perusahaan') || 'PT Jasamarga Pandaan Tol'),

                    NAMA_MAHASISWA: safe(getVal('kepada') || ''),
                    NIM: safe(getVal('nim') || ''),
                    JURUSAN: safe(getVal('jurusan') || ''),
                    PROGRAM_STUDI: safe(getVal('program_studi') || ''),
                    UNIVERSITAS: safe(getVal('universitas') || ''),

                    TANGGAL_MULAI: getVal('tanggal_mulai') || null,
                    TANGGAL_SELESAI: getVal('tanggal_selesai') || null,

                    PATH_TANDA_TANGAN: safe(getVal('path_ttd') || ''),
                    QR_CODE: safe(getVal('qr_code') || ''),
                    // PATH_STEMPEL:      safe(getVal('path_stempel') || ''),

                    PARAF: safe(getVal('paraf') || ''),

                    DARI_TIPE: dariTipe
                };
            }

            // Apply template CSS into the TinyMCE iframe for matching templates (Surat Edaran)
            async function applyTemplateStyleToEditor(editorInstance) {
                if (!editorInstance) editorInstance = tinymce.get('editorNaskah');
                if (!editorInstance) return;

                const tplUrl = getTemplateUrl();
                const isEdaran = tplUrl && tplUrl.toLowerCase().includes('edaran');
                try {
                    const doc = editorInstance.getDoc();
                    if (!doc) return;
                    // Ensure relative URLs inside templates resolve correctly by adding a base href
                    let baseEl = doc.getElementById('editorTemplateBase');
                    if (!baseEl) {
                        baseEl = doc.createElement('base');
                        baseEl.id = 'editorTemplateBase';
                        baseEl.href = '/jasa_marga/';
                        doc.head.appendChild(baseEl);
                    } else {
                        baseEl.href = '/jasa_marga/';
                    }

                    let styleEl = doc.getElementById('editorTemplateStyle');
                    if (!isEdaran) {
                        if (styleEl) styleEl.parentNode.removeChild(styleEl);
                        return;
                    }

                    const cssText = await fetchTemplateStyles();
                    if (!styleEl) {
                        styleEl = doc.createElement('style');
                        styleEl.id = 'editorTemplateStyle';
                        doc.head.appendChild(styleEl);
                    }
                    // Ensure the exact template style is used for the editor
                    styleEl.textContent = cssText + '\n.page{ width:210mm !important; min-height:297mm !important; padding:12mm 14mm 42mm 12mm !important; margin:0 auto !important; background:#fff !important; }';
                } catch (e) {
                    console.error('applyTemplateStyleToEditor error', e);
                }
            }

            async function fetchTemplate() {
                const res = await fetch(getTemplateUrl(), {
                    cache: 'no-store'
                });
                const txt = await res.text();
                const bodyOnly = txt.replace(/^[\s\S]*?<body[^>]*>/i, '').replace(/<\/body>[\s\S]*$/i, '').trim();
                return bodyOnly;
            }

            function fillPlaceholders(html, data) {
                let out = html;

                for (const [key, val] of Object.entries(data)) {
                    if (val === null) continue;
                    const re = new RegExp(`\\[\\[${key}\\]\\]`, 'g');
                    out = out.replace(re, val);
                }

                if (!data.PATH_TANDA_TANGAN) {
                    out = out.replace(/<img[^>]*class=["']signature["'][^>]*>/gi, '');
                }
                if (!data.PATH_STEMPEL) {
                    out = out.replace(/<img[^>]*class=["']stamp["'][^>]*>/gi, '');
                }

                // Hide empty paraf div
                if (!data.PARAF || data.PARAF.trim() === '') {
                    out = out.replace(/<div[^>]*class=["']paraf["'][^>]*>.*?<\/div>/gi, '');
                }

                return out;
            }

            function waitForTinyMCE() {
                return new Promise((resolve) => {
                    if (typeof tinymce !== 'undefined') {
                        resolve();
                    } else {
                        const checkInterval = setInterval(() => {
                            if (typeof tinymce !== 'undefined') {
                                clearInterval(checkInterval);
                                resolve();
                            }
                        }, 100);
                        // Timeout after 10 seconds
                        setTimeout(() => {
                            clearInterval(checkInterval);
                            console.error('TinyMCE failed to load within timeout');
                            resolve();
                        }, 10000);
                    }
                });
            }

            async function initTinyMCE() {
                await waitForTinyMCE();
                if (typeof tinymce === 'undefined') {
                    console.error('TinyMCE is not available');
                    return;
                }
                if (editorReady && tinymce.get('editorNaskah')) return Promise.resolve();
                return new Promise((resolve) => {
                    tinymce.init({
                        selector: '#editorNaskah',
                        height: 780,
                        menubar: true,
                        branding: false,
                        convert_urls: false,
                        remove_script_host: false,
                        entity_encoding: 'raw',
                        automatic_uploads: true,
                        images_reuse_filename: false,
                        paste_data_images: false,
                        file_picker_types: 'image',
                        images_upload_url: '/jasa_marga/api/upload/image.php',
                        images_upload_credentials: true,
                        images_upload_handler: (blobInfo, progress) => {
                            const fd = new FormData();
                            fd.append('image', blobInfo.blob(), blobInfo.filename());

                            return fetch('/jasa_marga/api/upload/image.php', {
                                    method: 'POST',
                                    credentials: 'same-origin',
                                    body: fd
                                })
                                .then(r => {
                                    if (!r.ok) throw new Error('HTTP ' + r.status);
                                    return r.json();
                                })
                                .then(json => {
                                    if (json && json.location) {
                                        return json.location;
                                    }
                                    throw new Error(json?.error || 'Upload gagal');
                                });
                        },
                        protect: [/\[\[[\s\S]*?\]\]/g],
                        extended_valid_elements: '*[*]',
                        valid_children: '+body[style],+div[div|section|header|footer|img]',
                        verify_html: false,
                        plugins: 'lists table link image code preview autoresize',
                        toolbar: 'undo redo | styles | bold italic underline | alignleft aligncenter alignright alignjustify | bullist numlist outdent indent | table link image | removeformat | preview | code',
                        init_instance_callback: function(editor) {
                            editorReady = true;
                            // Attempt to apply template style for current selection
                            applyTemplateStyleToEditor(editor).catch(e => console.error('applyTemplateStyleToEditor:', e));
                            // Resolve the promise when instance initialized
                            try {
                                resolve();
                            } catch (e) {
                                /* ignore */
                            }
                        },

                        content_style: `
                        * { box-sizing: border-box; }
                        html, body { margin:0; padding:0; }
                        body {
                            font-family: Arial, Helvetica, sans-serif;
                            color: #111;
                            background: #fff;
                        }
                        @page{
                            size: A4;
                            margin: 12mm 14mm 28mm 12mm;
                        }
                        :root{
                            --page-w:210mm;
                            --page-h:297mm;
                            --pad-top:12mm;
                            --pad-right:14mm;
                            --pad-bottom:42mm;
                            --pad-left:12mm;
                        }
                        .page{
                            width: var(--page-w);
                            min-height: var(--page-h);
                            padding: var(--pad-top) var(--pad-right) var(--pad-bottom) var(--pad-left);
                            margin: 0 auto;
                            position: relative;
                            background: #fff;
                        }
                        .html2pdf__page-break{
                            height:0; border:0; page-break-after:always;
                        }
                        .header {
                            display:flex; align-items:flex-start; justify-content:space-between;
                            gap: 16px; margin-bottom: 10mm;
                        }
                        .brand { display:flex; align-items:center; gap:12px; }
                        .brand img.logo { height:70px; width:auto; object-fit:contain; }
                        /* Header-left / Header-right placeholder to match template-surat-edaran */
                        .header-left { flex: 1; }
                        .header-left img.logo { height: 90px; width: 360px; object-fit: contain; display: block; }
                        .header-right { flex: 1; text-align: right; font-size:13px; color:#333; line-height:1.4; }
                        .brand .title { line-height:1.1; font-weight:700; letter-spacing:.3px; }
                        .brand .title .main { font-size:22px; color:#0c50e1; }
                        .brand .title .sub { font-size:18px; color:#0c50e1; text-transform:uppercase; }
                        .header .topnote { text-align:right; font-size:13px; color:#333; max-width:300px; margin-top: 20px; }
                        .doc-title {
                            text-align:center; font-weight:600; font-size:18px;
                        }
                        .doc-number {
                            text-align:center; font-size:18px; font-weight:600;
                        }
                        .doc-subject {
                            text-align:center; font-size:18px; font-weight:600;
                        }
                        .section-title {
                            font-size:16px; font-weight:700; margin-bottom:4mm;
                        }
                        .numbered-list {
                            padding-left: 4px;
                            list-style: none;
                            margin: 0;
                        }

                        .numbered-list li {
                            margin-bottom: 5px;
                            margin-top: 0px;
                            position: relative;
                        }
                        .divider { height:1px; background:#e6e9ef; margin: 2mm 0 8mm; }
                        .section { margin-bottom: 6mm; font-size: 14px !important; }
                        .lead { margin-bottom: 4mm; }
                        table.kv { border-collapse: collapse; width:100%; font-size: 14px; }
                        table.kv td { padding: 2px 0; vertical-align: top; }
                        table.kv td.col1 { width:120px; } table.kv td.col2 { width:10px; } table.kv td.col3 { width:auto; }
                        span.highlight { font-weight: 700 !important; }
                        .sign-block {
                            margin-top: 14mm;
                            width: 100%;
                            display:flex; justify-content:flex-end;
                            position: relative;
                        }
                        .sign {
                            width: 72mm;
                            text-align:left; font-size:14px;
                            position: relative; padding-top: 0;
                        }
                        .sign .place-date { margin-bottom: 15mm; }
                        .sign .stamp { position:absolute; right: -8mm; top: 18mm; opacity:.85; width: 34mm; height:auto; }
                        .sign .signature { height: 22mm; width:auto; display:block; margin: -6mm 0 0 0; }
                        .sign .signature-info { display: block; margin-top: 20mm; }
                        .sign .name { font-weight:700; margin: 0; }
                        .sign .title { margin-top: 1mm; }
                        .sign-block .paraf {
                            position: absolute;
                            left: 0;
                            bottom: 0;
                            font-size: 14px;
                            color: #333;
                            font-weight: 500;
                        }
                        .footer {
                            position: absolute;
                            right: 14mm;
                            bottom: 12mm;
                            left: auto;
                            width: auto;
                            text-align: right;
                            font-size: 10px;
                            color: #333;
                            line-height: 1.4;
                        }
                        .page .footer{ display:none !important; }
                        .page:first-of-type .footer{ display:block !important; }
                        .sign-block, .footer, .doc-title, .doc-number, table.kv {
                        page-break-inside: avoid;
                        break-inside: avoid;
                        }
                        .page:not(:last-child){ page-break-after: always; }
                        
                        /* Two-column section for Surat Keputusan */
                        .section-title-2 {
                            font-weight: bold;
                            font-size: 16px;
                            margin-bottom: 12px;
                            text-align: center;
                        }
                        .two-column-section {
                            display: flex;
                        }
                        .two-column-section .column {
                            flex: 1;
                        }
                        .two-column-section .column-2 {
                            flex: 8;
                        }
                        .two-column-section .column-2 p {
                            margin: 0;
                        }
                        .page-break {
                            page-break-before: always;
                            break-before: page;
                        }
                        .signature-section {
                            margin-top: 20mm;
                        }
                        .signature-info-table {
                            margin-left: auto;
                            margin-right: 0;
                            width: auto;
                            font-size: 14px;
                        }
                        .signature-info-table td {
                            padding: 2px 0;
                            vertical-align: top;
                        }
                        .signature-info-table .label {
                            font-weight: 600;
                        }
                        .signature-info-table .colon {
                            font-weight: 600;
                        }
                        .signature-table {
                            margin-left: auto;
                            margin-right: 32px;
                            text-align: left;
                            font-size: 14px;
                            margin-bottom: 3mm;
                        }
                        .signature-table td {
                            font-weight: 600;
                            padding-right: 10px;
                        }
                        .signature-section-container {
                            position: relative;
                            text-align: right;
                        }
                        .signature-paraf {
                            position: absolute;
                            left: 0;
                            top: 48mm;
                            font-size: 16px;
                            color: #333;
                        }
                        .signature-qr-container {
                            margin-right: 80px;
                        }
                        .signature-img {
                            height: 22mm;
                            width: auto;
                            display: block;
                            margin: -4mm 0 0 auto;
                        }
                        .signature-name {
                            font-weight: 600;
                            margin-top: 5mm;
                            margin-right: 92px;
                        }
                        .signature-sub-name {
                            margin-right: 32px;
                            font-weight: 700;
                        }
                        .sign-block-keputusan {
                            display: flex;
                            justify-content: center;
                            position: relative;
                        }
                        .sign-paraf-keputusan {
                            position: absolute;
                            left: 0;
                            top: 60mm;
                            font-size: 12px;
                            color: #333;
                        }
                        .lampiran-box{
                            border:1px solid #e0e0e0;
                            padding:0;
                            margin: 0 auto;
                        }
                        .lampiran-slot{
                            width: calc(var(--page-w) - var(--pad-left) - var(--pad-right));
                            height: calc(var(--page-h) - var(--pad-top) - var(--pad-bottom) - 26mm);
                            margin: 0 auto;
                            display:flex; align-items:center; justify-content:center;
                            overflow:hidden;
                            page-break-inside: avoid;
                            break-inside: avoid;
                        }
                        .lampiran-slot > img.lampiran-img{
                            max-width: 100%;
                            max-height: 100%;
                            width: auto; height: auto;
                            object-fit: contain;
                            display: block; margin: 0;
                        }
                        .lampiran-slot > iframe.lampiran-frame{
                            width: 100%; height: 100%; border: 0;
                        }
                        img.signature[src=""], img.stamp[src=""] { display:none !important; }
                        table, th, td { border: none !important; outline: none !important; }
                        .mce-item-table, .mce-item-table td, .mce-item-table th { border: none !important; outline: none !important; }
                        @media print {
                            @page{ size:A4; margin:12mm 20mm 28mm; }
                            .page{ -webkit-print-color-adjust: exact; print-color-adjust: exact; }
                        }
                    `,
                        setup(ed) {
                            ed.on('init', () => {
                                editorReady = true;
                                resolve();
                            });
                        }
                    });
                });
            }

            async function buildEditorFromTemplate() {
                if (!form.checkValidity()) {
                    form.reportValidity();
                    return;
                }

                await initTinyMCE();

                // Check if TinyMCE loaded successfully
                if (typeof tinymce === 'undefined') {
                    alert('Editor gagal dimuat. Silakan refresh halaman dan coba lagi.');
                    return;
                }

                const tpl = await fetchTemplate();
                const vals = collectFormValues();
                const html = fillPlaceholders(tpl, vals);

                const editor = tinymce.get('editorNaskah');
                if (editor) {
                    editor.setContent(html);
                    await applyTemplateStyleToEditor(editor);
                    setStep(2);
                } else {
                    alert('Editor belum siap. Silakan tunggu sebentar dan coba lagi.');
                }
            }

            to2.addEventListener('click', buildEditorFromTemplate);

            const syncBtn = document.getElementById('syncFromForm');
            if (syncBtn) {
                syncBtn.addEventListener('click', async () => {
                    const tpl = await fetchTemplate();
                    const vals = collectFormValues();
                    const html = fillPlaceholders(tpl, vals);
                    if (editorReady) {
                        const editor = tinymce.get('editorNaskah');
                        editor.setContent(html);
                        await applyTemplateStyleToEditor(editor);
                    }
                });
            }

            const printBtn = document.getElementById('printDoc');
            if (printBtn) {
                printBtn.addEventListener('click', () => {
                    if (editorReady) {
                        const ed = tinymce.get('editorNaskah');
                        ed.execCommand('mcePrint');
                    }
                });
            }
        })();
    </script>
    <!-- Global variable to track editor readiness - accessible across all script blocks -->
    <script>
        let editorReady = false;
    </script>
    <script>
        (function() {
            const form = document.getElementById('formNaskah');
            form.addEventListener('submit', function(e) {
                if (window.tinymce) tinymce.triggerSave();
                const html = (document.getElementById('editorNaskah').value || '');
                const m = html.match(/<img[^>]+src=["']([^"']+)["'][^>]*>/gi) || [];
                const hasBase64 = /src\s*=\s*["']data:image\//i.test(html);
                console.log('BASE64_DETECTED:', hasBase64, 'IMG_TAGS_COUNT:', m.length);
            });
        })();

        // Global function to determine template URL based on tipe_surat
        function getTemplateUrl() {
            const tipeSuratSelect = document.getElementById('tipe_surat_id');
            if (!tipeSuratSelect) return 'template-letter.php';

            const selectedOption = tipeSuratSelect.options[tipeSuratSelect.selectedIndex];
            if (!selectedOption) return 'template-letter.php';

            const tipeSuratText = selectedOption.textContent.trim();

            if (tipeSuratText.includes('Keputusan')) {
                return '/jasa_marga/surat/templates/surat-keputusan.php';
            } else if (tipeSuratText.includes('Edaran')) {
                return '/jasa_marga/surat/templates/surat-edaran.php';
            } else {
                return '/jasa_marga/surat/templates/surat-keluar.php'; // Default for Surat Keluar
            }
        }
    </script>

    <!-- Generate PDF → Upload (load local first, then CDN) -->
    <script>
        (function() {
            function loadScript(src, onload, onerror) {
                var s = document.createElement('script');
                s.src = src;
                s.async = false;
                s.onload = onload;
                s.onerror = onerror;
                document.head.appendChild(s);
            }
            loadScript('/jasa_marga/node_modules/html2pdf.js/dist/html2pdf.bundle.min.js', function() {
                document.dispatchEvent(new Event('html2pdf:loaded'));
            }, function() {
                loadScript('https://cdnjs.cloudflare.com/ajax/libs/html2pdf.js/0.10.1/html2pdf.bundle.min.js', function() {
                    document.dispatchEvent(new Event('html2pdf:loaded'));
                }, function() {
                    console.error('Failed to load html2pdf');
                });
            });
        })();
    </script>
    <script>
        async function fetchTemplateStyles() {
            const res = await fetch(getTemplateUrl(), {
                cache: 'no-store'
            });
            const html = await res.text();
            const m = html.match(/<style[^>]*>([\s\S]*?)<\/style>/i);
            return m ? m[1] : '';
        }

        function collectStep1Values() {
            const get = id => (document.getElementById(id)?.value || '').trim();

            const jenisOpt = document.getElementById('jenis_surat_id')?.selectedOptions?.[0];
            const klasOpt = document.getElementById('klasifikasi_arsip_id')?.selectedOptions?.[0];

            const dariTipe = (document.querySelector('input[name="dari_tipe"]:checked')?.value || '');
            const namaLogin = (document.getElementById('user_nama')?.value || '').trim();
            const jabatanLogin = (document.getElementById('user_jabatan')?.value || '').trim();

            let namaPejabat = '';
            let jabatanPejabat = '';

            if (dariTipe === 'pribadi') {
                const selectedOption = document.getElementById('dari_pribadi')?.selectedOptions?.[0];
                // Get the text content (nama) from the option, not the value (id)
                if (selectedOption) {
                    const optionText = selectedOption.textContent.trim();
                    // Extract nama (before the dash) if format is "Nama - Jabatan"
                    namaPejabat = optionText.split('-')[0].trim();
                } else {
                    namaPejabat = namaLogin;
                }
                jabatanPejabat = selectedOption?.dataset?.jabatan || jabatanLogin || '';
            } else if (dariTipe === 'sendiri') {
                namaPejabat = namaLogin;
                jabatanPejabat = jabatanLogin;
            } else {
                namaPejabat = get('nama_pejabat') || '';
                jabatanPejabat = get('jabatan_pejabat') || '';
            }

            const tglRaw = get('tanggal_surat');
            const tglID = (() => {
                if (!tglRaw) return '';
                const m = /^(\d{4})-(\d{2})-(\d{2})$/.exec(tglRaw);
                return m ? `${m[3]}-${m[2]}-${m[1]}` : tglRaw;
            })();

            const fallbackPerusahaan = 'PT Jasamarga Pandaan Tol';

            return {
                JENIS_SURAT: (jenisOpt ? jenisOpt.textContent.trim() : ''),
                NOMOR_SURAT: get('nomor_surat'),
                KOTA: get('tempat_dibuat'),
                TANGGAL_SURAT: tglID,

                NAMA_PEJABAT: namaPejabat,
                NIK_PEJABAT: get('nik_pejabat'),
                JABATAN_PEJABAT: jabatanPejabat,
                DEPARTEMEN: get('departemen'),
                NAMA_PERUSAHAAN: get('nama_perusahaan') || fallbackPerusahaan,

                NAMA_MAHASISWA: get('kepada'),
                NIM: get('nim'),
                JURUSAN: get('jurusan'),
                PROGRAM_STUDI: get('program_studi'),
                UNIVERSITAS: get('universitas'),

                TANGGAL_MULAI: get('tanggal_mulai') || '',
                TANGGAL_SELESAI: get('tanggal_selesai') || '',

                PATH_TANDA_TANGAN: (get('path_ttd') || ''),
                QR_CODE: (get('qr_code') || '')
            };
        }

        function replacePlaceholders(html, data) {
            let out = html;
            console.log('🔍 replacePlaceholders called with data:', data);
            console.log('🔍 QR_CODE value:', data.QR_CODE);
            console.log('🔍 PATH_TANDA_TANGAN value:', data.PATH_TANDA_TANGAN);

            for (const [k, v] of Object.entries(data)) {
                const re = new RegExp(`\\[\\[${k}\\]\\]`, 'g');
                const matches = html.match(re);
                if (matches) {
                    console.log(`🔍 Found ${matches.length} occurrences of [[${k}]]`);
                }
                out = out.replace(re, String(v ?? ''));
            }
            // Remove any remaining unreplaced placeholders
            const unreplaced = out.match(/\[\[[A-Z0-9_]+\]\]/g);
            if (unreplaced) {
                console.log('🔍 Unreplaced placeholders:', unreplaced);
            }
            out = out.replace(/\[\[[A-Z0-9_]+\]\]/g, '');

            if (!data.PATH_TANDA_TANGAN) {
                out = out.replace(/<img[^>]*class=["']signature["'][^>]*>/gi, '');
            }

            console.log('🔍 Final HTML length:', out.length);
            return out;
        }

        window.addEventListener('load', function() {
            console.log('✅ Halaman sudah dimuat, script siap.');

            const uploadBtn = document.getElementById('btnUploadPdf');
            const saveBtn = document.getElementById('btnSave');
            const fileInput = document.getElementById('fileSuratPath');
            const tipeSuratSelect = document.getElementById('tipe_surat_id');

            if (!uploadBtn) {
                console.error('❌ Tombol Cetak & Upload PDF tidak ditemukan setelah load!');
                return;
            }
            if (!saveBtn || !fileInput) {
                console.error('❌ Elemen btnSave atau fileSuratPath tidak ditemukan!');
                return;
            }

            // Disable upload until html2pdf available
            uploadBtn.disabled = true;
            document.addEventListener('html2pdf:loaded', function() {
                try {
                    uploadBtn.disabled = false;
                } catch (e) {}
            });

            saveBtn.addEventListener('click', function(e) {
                if (!fileInput.value.trim()) {
                    e.preventDefault();
                    alert('⚠️ Harap klik "Cetak & Upload PDF" terlebih dahulu sebelum menyimpan!');
                }
            });

            if (tipeSuratSelect) {
                tipeSuratSelect.addEventListener('change', async () => {
                    if (editorReady && window.tinymce) {
                        await applyTemplateStyleToEditor(tinymce.get('editorNaskah'));
                    }
                });
            }

            // Load html2pdf (try local first, then CDN) and dispatch event on ready
            (function() {
                function loadScript(src, onload, onerror) {
                    var s = document.createElement('script');
                    s.src = src;
                    s.async = false;
                    s.onload = onload;
                    s.onerror = onerror;
                    document.head.appendChild(s);
                }
                loadScript('/jasa_marga/node_modules/html2pdf.js/dist/html2pdf.bundle.min.js', function() {
                    document.dispatchEvent(new Event('html2pdf:loaded'));
                }, function() {
                    loadScript('https://cdnjs.cloudflare.com/ajax/libs/html2pdf.js/0.10.1/html2pdf.bundle.min.js', function() {
                        document.dispatchEvent(new Event('html2pdf:loaded'));
                    }, function() {
                        console.error('Failed to load html2pdf');
                    });
                });
            })();

            uploadBtn.addEventListener('click', async function() {
                try {
                    if (!window.html2pdf) {
                        alert('html2pdf.js belum tersedia. Sistem sedang mencoba memuat library dari CDN. Silakan coba lagi sebentar.');
                        return;
                    }
                    if (!window.tinymce || !tinymce.get('editorNaskah')) {
                        alert('Editor TinyMCE belum siap.');
                        return;
                    }

                    let content = tinymce.get('editorNaskah').getContent() || '';
                    if (!content.trim()) {
                        alert('Konten surat masih kosong!');
                        return;
                    }

                    if (/\[\[[A-Z0-9_]+\]\]/i.test(content)) {
                        const vals = collectStep1Values();
                        content = replacePlaceholders(content, vals);
                    }

                    content = content
                        .replace(
                            /<div[^>]*class=["'][^"']*html2pdf__page-break[^"']*["'][^>]*>\s*<\/div>/gi,
                            '')
                        .replace(
                            /<div[^>]*class=["'][^"']*(page-appendix|lampiran-page)[^"']*["'][^>]*>[\s\S]*?<\/div>/gi,
                            '');

                    const styleText = await fetchTemplateStyles();

                    // Keep the original styles mostly intact for proper layout
                    const cleanedStyle = styleText;

                    const appendixCss = `
                    /* Ensure proper single-page layout - match template exactly */
                    .page{
                        width: 210mm !important;
                        min-height: 297mm !important;
                        padding: 10mm 14mm 25mm 12mm !important;
                        margin: 0 auto !important;
                        position: relative !important;
                        background: #fff !important;
                    }
                    
                    /* Sign block styling */
                    .sign-block {
                        margin-top: 8mm !important;
                        width: 100% !important;
                        display: flex !important;
                        justify-content: flex-end !important;
                        page-break-inside: avoid !important;
                        position: relative !important;
                    }
                    
                    .sign {
                        width: 72mm !important;
                        text-align: left !important;
                        font-size: 14px !important;
                        position: relative !important;
                    }
                    
                    .sign .place-date {
                        margin-bottom: 15mm !important;
                    }
                    
                    .sign .signature-info {
                        display: block !important;
                        margin-top: 20mm !important;
                    }
                    
                    .sign-block .paraf {
                        position: absolute !important;
                        left: 0 !important;
                        bottom: 0 !important;
                        font-size: 14px !important;
                        color: #333 !important;
                        text-align: left !important;
                        font-weight: 500 !important;
                        white-space: nowrap !important;
                    }
                    
                    .company-title {
                        text-align: right;
                        font-weight: 600;
                        font-size: 14px;
                        margin-bottom: 24px;
                    }

                    .sign .name {
                        font-weight: 700 !important;
                        margin: 0 !important;
                    }
                    
                    .sign .sub-name {
                        font-weight: 700 !important;
                    }
                    
                    .sign .title {
                        margin-top: 1mm !important;
                    }
                    
                    /* Footer at bottom of page */
                    .footer {
                        position: absolute !important;
                        right: 14mm !important;
                        bottom: 8mm !important;
                        text-align: right !important;
                        font-size: 10px !important;
                    }
                    
                    /* Lampiran/Appendix page */
                    .page-appendix{
                        break-before: page !important;
                        page-break-before: always !important;
                    }
                    
                    .lampiran-box{ 
                        display:flex; 
                        align-items:center; 
                        justify-content:center; 
                        border:0 !important; 
                    }
                    
                    .lampiran-img,
                    .lampiran-frame{
                        width:100% !important; 
                        max-width:100% !important;
                        object-fit: contain; 
                        border:0 !important;
                    }

                    .html2pdf__page-break{ 
                        break-before: page !important; 
                        page-break-before: always !important; 
                    }
                `;

                    const styles = [
                        cleanedStyle,
                        appendixCss,
                    ].join('\n');

                    (function appendLampiran() {
                        const inp = document.getElementById('lampiran');
                        if (!inp || !inp.files || !inp.files[0]) return;

                        const f = inp.files[0];
                        const type = (f.type || '').toLowerCase();
                        const url = URL.createObjectURL(f);

                        const lampiranInner = type.startsWith('image/') ?
                            `<img src="${url}" class="lampiran-img" alt="Lampiran">` :
                            (type === 'application/pdf' ?
                                `<iframe src="${url}" class="lampiran-frame" title="Lampiran PDF"></iframe>` :
                                `<p style="font-size:12px;text-align:center;">(Lampiran ${type || 'unknown'} tidak bisa dipratinjau)</p>`
                            );

                        content = content
                            .replace(
                                /<div[^>]*class=["'][^"']*html2pdf__page-break[^"']*["'][^>]*>\s*<\/div>/gi,
                                '')
                            .replace(
                                /<div[^>]*class=["'][^"']*(page-appendix|lampiran-page)[^"']*["'][^>]*>[\s\S]*?<\/div>/gi,
                                '');

                        content += `
                        <div class="html2pdf__page-break"></div>
                        <div class="page page-appendix">
                        <div class="lampiran-box">
                            <div class="lampiran-slot">${lampiranInner}</div>
                        </div>
                        </div>`;
                    })();

                    const finalHtml = [
                        '<!DOCTYPE html>',
                        '<html lang="id">',
                        '<head>',
                        '<meta charset="utf-8">',
                        '<meta name="viewport" content="width=device-width, initial-scale=1">',
                        '<style>',
                        styles,
                        '</style>',
                        '</head>',
                        '<body>',
                        content,
                        '</body>',
                        '</html>'
                    ].join('');

                    let safeHtml = finalHtml
                        .replace(/\sdata-mce-[a-z-]+="[^"]*"/gi, '');

                    const opt = {
                        margin: [0, 0, 0, 0],
                        filename: 'surat_temp.pdf',
                        image: {
                            type: 'png',
                            quality: 0.98
                        },
                        html2canvas: {
                            scale: 2,
                            useCORS: true,
                            allowTaint: true
                        },
                        jsPDF: {
                            unit: 'mm',
                            format: 'a4',
                            orientation: 'portrait',
                            compress: false
                        },
                        pagebreak: {
                            mode: ['css']
                        }
                    };

                    console.log('🔄 Membuat PDF…');
                    const worker = html2pdf().set(opt).from(safeHtml).toPdf();
                    const pdf = await worker.get('pdf');

                    let total = pdf.internal.getNumberOfPages();
                    while (total > 1) {
                        const ops = pdf.internal.pages[total];
                        const opCount = Array.isArray(ops) ? ops.filter(Boolean).length : 0;

                        if (opCount <= 3) {
                            pdf.deletePage(total);
                            total--;
                        } else {
                            break;
                        }
                    }

                    const blobPdf = pdf.output('blob');

                    const fd = new FormData();
                    fd.append('pdf_file', blobPdf, `surat_${Date.now()}.pdf`);

                    console.log('⬆️ Upload ke server…');
                    const res = await fetch('/jasa_marga/api/upload/pdf.php', {
                        method: 'POST',
                        body: fd
                    });
                    const data = await res.json();

                    if (data && data.ok && data.path) {
                        fileInput.value = data.path;
                        alert('✅ PDF berhasil diupload ke server!');
                    } else {
                        console.error('Upload response:', data);
                        alert('❌ Gagal upload PDF: ' + (data?.error || 'Tidak diketahui.'));
                    }

                } catch (err) {
                    console.error('❌ Error saat generate/upload PDF:', err);
                    alert('Terjadi kesalahan saat membuat atau mengupload PDF!');
                }
            });
        });
    </script>

    <!-- Preview Hasil Surat JS -->
    <!-- <script>
        (function(){
        const btn = document.getElementById('printDoc');

        const PRINT_CSS = `
            @page { size: A4; margin: 12mm 20mm 28mm; }

            :root{
                --page-w:210mm;
                --page-h:297mm;
                --pad-top:12mm;
                --pad-bottom:28mm;
            }

            html, body { background:#fff; }
            body { -webkit-print-color-adjust: exact; print-color-adjust: exact; }
            *{ box-sizing:border-box; }
            .page{
                width: var(--page-w);
                min-height: var(--page-h);
                padding: var(--pad-top) 20mm var(--pad-bottom);
                margin: 0 auto;
                background:#fff;
                position: relative;
                overflow: hidden;
                page-break-after: always;
                break-inside: avoid;
            }
            .page:last-child{ page-break-after: auto; }
            .header { display:flex; align-items:flex-start; justify-content:space-between; gap:16px; margin-bottom:10mm; }
            .brand { display:flex; align-items:center; gap:12px; }
            .brand img.logo { height:70px; width:auto; object-fit:contain; }
            .brand .title { line-height:1.1; font-weight:700; letter-spacing:.3px; }
            .brand .title .main { font-size:22px; color:#0c50e1; }
            .brand .title .sub  { font-size:18px; color:#0c50e1; text-transform:uppercase; }
            .topnote{ text-align:right; font-size:13px; color:#333; max-width:300px; margin-top:20px; }
            .doc-title { text-align:center; margin:2mm 0 .8mm; font-weight:900; font-size:20px; text-transform: uppercase; }
            .doc-number{ text-align:center; font-size:18px; margin-bottom:10mm; font-weight:700; }
            .divider { height:1px; background:#e6e9ef; margin:2mm 0 8mm; }
            .section { margin-bottom:6mm; font-size:15px; }
            .lead { margin-bottom:4mm; }
            table.kv { border-collapse:collapse; width:100%; font-size:15px; }
            table.kv td { padding:2px 0; vertical-align:top; }
            table.kv td.col1 { width:120px; } table.kv td.col2 { width:10px; } table.kv td.col3 { width:auto; }
            .sign-block{ margin-top:14mm; width:100%; display:flex; justify-content:flex-end; }
            .sign { width:72mm; text-align:left; font-size:15px; position:relative; padding-top:0; }
            .sign .place-date { margin-bottom:14mm; }
            .sign .stamp{ position:absolute; right:-8mm; top:18mm; opacity:.85; width:34mm; height:auto; }
            .sign .signature{ height:30mm; width:auto; display:block; margin:-6mm 0 0 0; }
            .sign .name{ font-weight:700; margin-top:2mm; }
            .sign .title{ margin-top:1mm; }
            .footer{
                position: absolute;
                left:20mm; right:20mm; bottom:12mm;
                font-size:10px; color:#333;
            }
            .footer .address{ text-align:right; line-height:1.4; }
            .page:not(:first-child) .footer{ display:none !important; }
            .lampiran-frame { width:100%; border:0; 
                height: calc(var(--page-h) - var(--pad-top) - var(--pad-bottom) - 26mm);
            }
            .lampiran-img { max-width:100%; height:auto; display:block; margin:0 auto;
                max-height: calc(var(--page-h) - var(--pad-top) - var(--pad-bottom) - 26mm);
            }
            table, th, td { border: none !important; outline: none !important; }
            .mce-item-table, .mce-item-table td, .mce-item-table th { border: none !important; outline: none !important; }
            img.signature[src=""], img.stamp[src=""] { display:none !important; }
        `;

        function openPrintPreview(htmlBody){
            const w = window.open('', '_blank');
            if(!w){ alert('Popup diblokir. Izinkan pop-up untuk melihat preview.'); return; }
            w.document.open();
            w.document.write(`<!DOCTYPE html>
            <html lang="id">
            <head>
            <meta charset="utf-8">
            <title>Preview Surat</title>
            <meta name="viewport" content="width=device-width, initial-scale=1">
            <style>${PRINT_CSS}</style>
            </head>
            <body>${htmlBody}</body>
            </html>`);

            // Ambil lampiran dari input file
            const inp = document.getElementById('lampiran');
            if (inp && inp.files && inp.files[0]) {
            const f = inp.files[0];
            const mime = (f.type || '').toLowerCase();
            const url = URL.createObjectURL(f);

            let lampiranBlock = `
                <div style="page-break-before:always"></div>
                <div class="page page-appendix">
                <h3 style="text-align:center;margin:0 0 6mm;">Lampiran</h3>
                <div style="font-size:14px;margin-bottom:4mm;">Nomor: ${
                    (document.getElementById('nomor_surat')?.value || '')
                }</div>
            `;

            if (mime.startsWith('image/')) {
                lampiranBlock += `
                <div style="border:1px solid #e0e0e0; padding:6mm; text-align:center;">
                    <img src="${url}" class="lampiran-img" alt="Lampiran">
                </div>`;
            } else if (mime === 'application/pdf') {
                lampiranBlock += `
                <div style="border:1px solid #e0e0e0; padding:0;">
                    <iframe src="${url}" class="lampiran-frame"></iframe>
                </div>`;
            } else {
                lampiranBlock += `<p><b>Lampiran (${mime || 'unknown'}) tidak dapat dipratinjau.</b></p>`;
            }
            lampiranBlock += `</div>`;

            // selipkan ke body preview
            w.document.body.insertAdjacentHTML('beforeend', lampiranBlock);
            }

            w.document.close();
            w.focus();
            setTimeout(()=>{ w.print(); }, 600);
        }

        btn?.addEventListener('click', function(){
            let html = '';
            if (window.tinymce && tinymce.get('editorNaskah')) {
            html = tinymce.get('editorNaskah').getContent({ format: 'html' });
            } else {
            const el = document.getElementById('editorNaskah');
            html = el ? (el.value || el.innerHTML || '') : '';
            }
            if(!html.trim()){
            alert('Konten surat masih kosong. Silakan isi/Generate terlebih dahulu.');
            return;
            }
            html = html
            .replace(/\sdata-mce-[a-z-]+="[^"]*"/g, '')
            .replace(/\sclass="([^"]*)"/g, (m, cls) => {
                const cleaned = cls.split(/\s+/).filter(c => !/^mce-/.test(c)).join(' ');
                return cleaned ? ` class="${cleaned}"` : '';
            })
            .replace(/\scontenteditable="[^"]*"/g, '')
            .replace(/\srole="application"/g, '');
            openPrintPreview(html);
        });
        })();
    </script> -->

    <script>
        // Event listener untuk memperlihatkan tombol ketika input diklik
        const searchInputNew = document.getElementById('searchInputNew');
        const searchButtonNew = document.getElementById('searchButtonNew');

        if (searchInputNew && searchButtonNew) {
            searchInputNew.addEventListener('focus', function() {
                searchButtonNew.classList.add('show');
                searchInputNew.placeholder = '';
            });

            searchInputNew.addEventListener('blur', function() {
                if (!searchInputNew.value.trim()) {
                    searchButtonNew.classList.remove('show');
                    searchInputNew.placeholder = 'Tap untuk mencari...';
                }
            });
        }

        document.querySelectorAll('.btn-delete').forEach(button => {
            button.addEventListener('click', function() {
                document.getElementById('deleteUserId').value = this.dataset.id;
                new bootstrap.Modal(document.getElementById('deleteUserModal')).show();
            });
        });

        document.querySelectorAll('.btn-edit').forEach(button => {
            button.addEventListener('click', function() {
                document.getElementById('editUserId').value = this.dataset.id;
                document.getElementById('editUsername').value = this.dataset.username;
                document.getElementById('editNama').value = this.dataset.nama;
                document.getElementById('editEmail').value = this.dataset.email;
                document.getElementById('editTipe').value = this.dataset.tipe;
                document.getElementById('editJabatan').value = this.dataset.jabatan;
                document.getElementById('editGender').value = this.dataset.gender;
                document.getElementById('editSubUnit').value = this.dataset.sub_unit_id;

                // Reset dan centang akses
                document.getElementById('aksesBerita').checked = false;
                document.getElementById('aksesSosial').checked = false;
                document.getElementById('aksesGaji').checked = false;
                document.getElementById('aksesSurat').checked = false;
                document.getElementById('aksesAgenda').checked = false;

                try {
                    const akses = JSON.parse(this.dataset.access);
                    if (akses.berita === 'on') document.getElementById('aksesBerita').checked = true;
                    if (akses.sosial === 'on') document.getElementById('aksesSosial').checked = true;
                    if (akses.gaji === 'on') document.getElementById('aksesGaji').checked = true;
                    if (akses.surat === 'on') document.getElementById('aksesSurat').checked = true;
                    if (akses.agenda === 'on') document.getElementById('aksesAgenda').checked = true;
                    if (akses.todolist === 'on') document.getElementById('aksesTodolist').checked = true;
                } catch (e) {
                    console.error("Gagal parse access_modul");
                }

                new bootstrap.Modal(document.getElementById('editUserModal')).show();
            });
        });

        document.addEventListener('DOMContentLoaded', function() {
            var editButtons = document.querySelectorAll('.btn-edit');
            var modal = new bootstrap.Modal(document.getElementById('editUserModal'));

            editButtons.forEach(function(button) {
                button.addEventListener('click', function() {
                    var userId = this.getAttribute('data-id');
                    var username = this.getAttribute('data-username');
                    document.getElementById('editUserId').value = userId;
                    document.getElementById('editUsername').value = username;
                    modal.show();
                });
            });
        });
    </script>

</body>

</html>