-- ============================================
-- QR CODE VALIDATION SYSTEM - DATABASE UPDATE
-- PT Jasamarga Pandaan Tol
-- Version: 1.0.0
-- Date: November 19, 2025
-- ============================================

-- Backup recommendation: 
-- Run this before executing: mysqldump -u root -p jasa_marga > backup_before_qr_update.sql

USE jasa_marga;

-- Check if columns already exist to prevent errors
SET @exist_validated := (
    SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
    WHERE TABLE_SCHEMA = 'jasa_marga' 
    AND TABLE_NAME = 'office' 
    AND COLUMN_NAME = 'is_validated'
);

-- Add columns only if they don't exist
SET @sql_add_columns = IF(
    @exist_validated = 0,
    'ALTER TABLE `office` 
     ADD COLUMN `is_validated` TINYINT(1) NOT NULL DEFAULT 0 COMMENT ''Validation status: 0=not validated, 1=validated'',
     ADD COLUMN `qr_code` VARCHAR(255) NULL COMMENT ''Path to QR code image file'',
     ADD COLUMN `validation_token` VARCHAR(255) NULL COMMENT ''Unique token for validation verification'',
     ADD COLUMN `validated_at` TIMESTAMP NULL COMMENT ''Timestamp when letter was validated'';',
    'SELECT ''Columns already exist, skipping...'' AS message;'
);

PREPARE stmt FROM @sql_add_columns;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add indexes for better query performance
SET @exist_index := (
    SELECT COUNT(*) FROM INFORMATION_SCHEMA.STATISTICS 
    WHERE TABLE_SCHEMA = 'jasa_marga' 
    AND TABLE_NAME = 'office' 
    AND INDEX_NAME = 'idx_validation_token'
);

SET @sql_add_indexes = IF(
    @exist_index = 0,
    'ALTER TABLE `office` 
     ADD INDEX `idx_validation_token` (`validation_token`),
     ADD INDEX `idx_is_validated` (`is_validated`);',
    'SELECT ''Indexes already exist, skipping...'' AS message;'
);

PREPARE stmt FROM @sql_add_indexes;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Display current table structure
DESCRIBE `office`;

-- Show summary
SELECT 
    CONCAT('✓ Database update completed successfully!') AS status,
    COUNT(*) AS total_letters,
    SUM(CASE WHEN is_validated = 1 THEN 1 ELSE 0 END) AS validated_letters,
    SUM(CASE WHEN is_validated = 0 THEN 1 ELSE 0 END) AS pending_validation
FROM `office`;

-- ============================================
-- INSTALLATION COMPLETE
-- ============================================
-- Next Steps:
-- 1. Refresh record-letter.php page
-- 2. Click "Validasi" button on any letter
-- 3. Test QR code scanning
-- 4. Verify public access at verify-letter.php
-- ============================================
