-- Migration: Independent Sequence Numbers per Tipe Surat
-- Date: 2025-12-15
-- Description: Changes sequence numbering from global to independent per letter type

-- =====================================================
-- STEP 1: Backup existing letter_sequences table (if needed)
-- =====================================================
-- CREATE TABLE letter_sequences_backup AS SELECT * FROM letter_sequences;

-- =====================================================
-- STEP 2: Drop and recreate letter_sequences with proper structure
-- =====================================================

DROP TABLE IF EXISTS letter_sequences;

CREATE TABLE `letter_sequences` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `tipe_surat_id` int(11) NOT NULL COMMENT 'FK to tipe_surat.id (1=SK, 2=KPTS, 3=SE)',
  `tahun` int(4) NOT NULL,
  `last_number` int(11) NOT NULL DEFAULT 0,
  `created_at` timestamp NULL DEFAULT current_timestamp(),
  `updated_at` timestamp NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
  PRIMARY KEY (`id`),
  UNIQUE KEY `unique_type_year` (`tipe_surat_id`, `tahun`),
  KEY `idx_tipe_surat` (`tipe_surat_id`),
  KEY `idx_tahun` (`tahun`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- =====================================================
-- STEP 3: Initialize sequence records for 2025
-- Each tipe_surat starts fresh at 0
-- =====================================================

INSERT INTO `letter_sequences` (`tipe_surat_id`, `tahun`, `last_number`) VALUES
(1, 2025, 0),  -- Surat Keluar (SK)
(2, 2025, 0),  -- Surat Keputusan Direksi (KPTS)
(3, 2025, 0);  -- Surat Edaran Direksi (SE)

-- =====================================================
-- STEP 4: Create authorized_signers table
-- 16 registered names who can sign letters
-- =====================================================

DROP TABLE IF EXISTS authorized_signers;

CREATE TABLE `authorized_signers` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `user_id` int(11) DEFAULT NULL COMMENT 'FK to users.id if exists',
  `nama` varchar(100) NOT NULL,
  `jabatan` varchar(150) DEFAULT NULL,
  `approval_chain` varchar(500) DEFAULT NULL COMMENT 'Format: Name1→Name2→Name3',
  `is_active` tinyint(1) NOT NULL DEFAULT 1,
  `created_at` timestamp NULL DEFAULT current_timestamp(),
  `updated_at` timestamp NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
  PRIMARY KEY (`id`),
  KEY `idx_nama` (`nama`),
  KEY `idx_user_id` (`user_id`),
  KEY `idx_active` (`is_active`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- =====================================================
-- STEP 5: Insert 16 authorized signers
-- =====================================================

INSERT INTO `authorized_signers` (`nama`, `jabatan`, `approval_chain`) VALUES
('Netty Renova', NULL, 'Netty Renova'),
('Indarani', NULL, 'Indarani→Netty Renova'),
('Andri Supriadi', NULL, 'Andri Supriadi→Indarani→Netty Renova'),
('Muhammad Reza Pahlevi Guntur', NULL, 'Muhammad Reza Pahlevi Guntur→Netty Renova'),
('Bambang Hermawan', NULL, 'Bambang Hermawan→Andri Supriadi→Indarani→Netty Renova'),
('Fakirno', NULL, 'Fakirno→Muhammad Reza Pahlevi Guntur→Netty Renova'),
('Ragil Suparmadi', NULL, 'Ragil Suparmadi→Muhammad Reza Pahlevi Guntur→Netty Renova'),
('Auzi Trisiyas Sulkha', NULL, 'Auzi Trisiyas Sulkha→Bambang Hermawan→Andri Supriadi→Indarani→Netty Renova'),
('Muhammad Tachyuddin', NULL, 'Muhammad Tachyuddin→Andri Supriadi→Indarani→Netty Renova'),
('Kukuh Prasojo', NULL, 'Kukuh Prasojo→Ragil Suparmadi→Muhammad Reza Pahlevi Guntur→Netty Renova'),
('Rendy Hendra Prasetya', NULL, 'Rendy Hendra Prasetya→Bambang Hermawan→Andri Supriadi→Indarani→Netty Renova'),
('Tedy Agus Harianto', NULL, 'Tedy Agus Harianto→Andri Supriadi→Indarani→Netty Renova'),
('Budhi Satria Permana', NULL, 'Budhi Satria Permana→Bambang Hermawan→Andri Supriadi→Indarani→Netty Renova'),
('M.Nafi Uddin', NULL, 'M.Nafi Uddin→Fakirno→Muhammad Reza Pahlevi Guntur→Netty Renova'),
('Harbetta Yurisa Tanydag', NULL, 'Harbetta Yurisa Tanydag→Ragil Suparmadi→Muhammad Reza Pahlevi Guntur→Netty Renova'),
('Abdul Rokhman', NULL, 'Abdul Rokhman→Andri Supriadi→Indarani→Netty Renova');

-- =====================================================
-- STEP 6: Update user_id references (run after checking users table)
-- This will link authorized_signers to existing users if their names match
-- =====================================================

UPDATE authorized_signers AS a
INNER JOIN users AS u ON LOWER(TRIM(a.nama)) = LOWER(TRIM(u.nama))
SET a.user_id = u.id, a.jabatan = u.jabatan;

-- =====================================================
-- Verification queries (run to check)
-- =====================================================

-- Check letter_sequences
SELECT * FROM letter_sequences;

-- Check authorized_signers with user links
SELECT a.*, u.username 
FROM authorized_signers a 
LEFT JOIN users u ON a.user_id = u.id;

-- Check tipe_surat reference
SELECT * FROM tipe_surat;
