-- Migration: Fix Letter Sequences for Independent Numbering per Tipe Surat
-- Date: 2025-12-15
-- Purpose: 
--   1. Fix letter_sequences table to use tipe_surat_id (int) instead of tipe_surat_kode (varchar)
--   2. Remove year constraint - global sequence per tipe surat
--   3. Remove UNIQUE constraint on office.sequence_number to allow independent sequences
--   4. Fresh start: Initialize all sequences at 0

-- ============================================
-- STEP 1: Drop and recreate letter_sequences
-- ============================================

DROP TABLE IF EXISTS letter_sequences;

CREATE TABLE letter_sequences (
    id INT(11) NOT NULL AUTO_INCREMENT,
    tipe_surat_id INT(11) NOT NULL,
    last_number INT(11) NOT NULL DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (id),
    UNIQUE KEY unique_tipe_surat (tipe_surat_id),
    CONSTRAINT fk_letter_seq_tipe_surat FOREIGN KEY (tipe_surat_id) REFERENCES tipe_surat(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- STEP 2: Initialize sequences for each tipe surat (fresh start at 0)
-- ============================================

-- Insert sequence records for all 3 tipe surat
INSERT INTO letter_sequences (tipe_surat_id, last_number) VALUES
    (1, 0),  -- Surat Keluar (SK)
    (2, 0),  -- Surat Keputusan Direksi (KPTS)
    (3, 0);  -- Surat Edaran Direksi (SE)

-- ============================================
-- STEP 3: Fix office table - remove global unique constraint on sequence_number
-- ============================================

-- First check if the constraint exists and drop it
SET @constraint_exists = (
    SELECT COUNT(*) FROM information_schema.TABLE_CONSTRAINTS 
    WHERE CONSTRAINT_SCHEMA = DATABASE() 
    AND TABLE_NAME = 'office' 
    AND CONSTRAINT_NAME = 'unique_sequence_number'
);

-- Drop the global unique constraint if exists (allows same sequence number across different tipe_surat)
ALTER TABLE office DROP INDEX IF EXISTS unique_sequence_number;

-- Add composite unique index: sequence must be unique within each tipe_surat
-- First drop if exists to avoid errors
ALTER TABLE office DROP INDEX IF EXISTS unique_seq_per_type;

-- Create composite unique (tipe_surat_id + sequence_number)
-- This allows: SK seq 1, KPTS seq 1, SE seq 1 to coexist
ALTER TABLE office ADD UNIQUE INDEX unique_seq_per_type (tipe_surat_id, sequence_number);

-- ============================================
-- STEP 4: Verify the setup
-- ============================================

SELECT 'letter_sequences table created:' AS status;
SELECT * FROM letter_sequences;

SELECT 'tipe_surat reference table:' AS status;
SELECT id, kode, nama FROM tipe_surat ORDER BY id;

SELECT 'office table indexes:' AS status;
SHOW INDEX FROM office WHERE Key_name LIKE '%seq%';
