-- =====================================================
-- ULTRA SAFE DATABASE MIGRATION: DUAL LETTER NUMBERING SYSTEM
-- =====================================================
-- This script handles foreign key dependencies properly

-- Disable foreign key checks temporarily
SET FOREIGN_KEY_CHECKS = 0;

-- =====================================================
-- STEP 1: CREATE TABLES WITHOUT FOREIGN KEYS
-- =====================================================

-- Create tipe_surat table
CREATE TABLE IF NOT EXISTS `tipe_surat` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `nama` varchar(100) NOT NULL,
  `kode` varchar(10) NOT NULL,
  `created_at` timestamp NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  UNIQUE KEY `uk_tipe_surat_kode` (`kode`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Insert tipe surat data
INSERT IGNORE INTO `tipe_surat` (`nama`, `kode`) VALUES 
('Surat Keluar', 'SK'),
('Surat Keputusan Direksi', 'KPTS'),
('Surat Edaran Direksi', 'SE');

-- Create subjenis_surat table WITHOUT foreign keys first
CREATE TABLE IF NOT EXISTS `subjenis_surat` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `kode` varchar(10) NOT NULL,
  `nama` varchar(255) NOT NULL,
  `deskripsi` text DEFAULT NULL,
  `id_jenis` int(11) NOT NULL,
  `id_klasifikasi` int(11) NOT NULL,
  `created_at` timestamp NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  UNIQUE KEY `uk_subjenis_kode_jenis_klasifikasi` (`kode`, `id_jenis`, `id_klasifikasi`),
  KEY `idx_subjenis_jenis` (`id_jenis`),
  KEY `idx_subjenis_klasifikasi` (`id_klasifikasi`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- =====================================================
-- STEP 2: ADD COLUMNS TO EXISTING TABLES
-- =====================================================

-- Add deskripsi to klasifikasi_arsip if not exists
SET @column_exists = (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
                     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'klasifikasi_arsip' AND COLUMN_NAME = 'deskripsi');

SET @sql = IF(@column_exists = 0, 
             'ALTER TABLE `klasifikasi_arsip` ADD COLUMN `deskripsi` text DEFAULT NULL', 
             'SELECT "deskripsi already exists in klasifikasi_arsip" as message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add deskripsi to jenis_surat if not exists
SET @column_exists = (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
                     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'jenis_surat' AND COLUMN_NAME = 'deskripsi');

SET @sql = IF(@column_exists = 0, 
             'ALTER TABLE `jenis_surat` ADD COLUMN `deskripsi` text DEFAULT NULL', 
             'SELECT "deskripsi already exists in jenis_surat" as message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add id_klasifikasi to jenis_surat if not exists
SET @column_exists = (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
                     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'jenis_surat' AND COLUMN_NAME = 'id_klasifikasi');

SET @sql = IF(@column_exists = 0, 
             'ALTER TABLE `jenis_surat` ADD COLUMN `id_klasifikasi` int(11) DEFAULT NULL', 
             'SELECT "id_klasifikasi already exists in jenis_surat" as message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- =====================================================
-- STEP 3: ADD OFFICE TABLE COLUMNS
-- =====================================================

-- Add tipe_surat_id to office
SET @column_exists = (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
                     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND COLUMN_NAME = 'tipe_surat_id');

SET @sql = IF(@column_exists = 0, 
             'ALTER TABLE `office` ADD COLUMN `tipe_surat_id` int(11) DEFAULT NULL', 
             'SELECT "tipe_surat_id already exists in office" as message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add klasifikasi_arsip_kode to office
SET @column_exists = (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
                     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND COLUMN_NAME = 'klasifikasi_arsip_kode');

SET @sql = IF(@column_exists = 0, 
             'ALTER TABLE `office` ADD COLUMN `klasifikasi_arsip_kode` varchar(10) DEFAULT NULL', 
             'SELECT "klasifikasi_arsip_kode already exists in office" as message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add jenis_surat_kode to office
SET @column_exists = (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
                     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND COLUMN_NAME = 'jenis_surat_kode');

SET @sql = IF(@column_exists = 0, 
             'ALTER TABLE `office` ADD COLUMN `jenis_surat_kode` varchar(10) DEFAULT NULL', 
             'SELECT "jenis_surat_kode already exists in office" as message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add subjenis_surat_id to office
SET @column_exists = (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
                     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND COLUMN_NAME = 'subjenis_surat_id');

SET @sql = IF(@column_exists = 0, 
             'ALTER TABLE `office` ADD COLUMN `subjenis_surat_id` int(11) DEFAULT NULL', 
             'SELECT "subjenis_surat_id already exists in office" as message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add new numbering columns
SET @column_exists = (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
                     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND COLUMN_NAME = 'nomor_surat_surat_keluar');

SET @sql = IF(@column_exists = 0, 
             'ALTER TABLE `office` ADD COLUMN `nomor_surat_surat_keluar` varchar(100) DEFAULT NULL', 
             'SELECT "nomor_surat_surat_keluar already exists in office" as message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SET @column_exists = (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
                     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND COLUMN_NAME = 'nomor_surat_surat_keputusan_direksi');

SET @sql = IF(@column_exists = 0, 
             'ALTER TABLE `office` ADD COLUMN `nomor_surat_surat_keputusan_direksi` varchar(100) DEFAULT NULL', 
             'SELECT "nomor_surat_surat_keputusan_direksi already exists in office" as message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SET @column_exists = (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
                     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND COLUMN_NAME = 'nomor_surat_surat_edaran_direksi');

SET @sql = IF(@column_exists = 0, 
             'ALTER TABLE `office` ADD COLUMN `nomor_surat_surat_edaran_direksi` varchar(100) DEFAULT NULL', 
             'SELECT "nomor_surat_surat_edaran_direksi already exists in office" as message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add migration tracking columns
SET @column_exists = (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
                     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND COLUMN_NAME = 'is_migrated_to_new_system');

SET @sql = IF(@column_exists = 0, 
             'ALTER TABLE `office` ADD COLUMN `is_migrated_to_new_system` tinyint(1) DEFAULT 0', 
             'SELECT "is_migrated_to_new_system already exists in office" as message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SET @column_exists = (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
                     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND COLUMN_NAME = 'migration_notes');

SET @sql = IF(@column_exists = 0, 
             'ALTER TABLE `office` ADD COLUMN `migration_notes` text DEFAULT NULL', 
             'SELECT "migration_notes already exists in office" as message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- =====================================================
-- STEP 4: ADD INDEXES SAFELY
-- =====================================================

-- Add unique key to klasifikasi_arsip.kode if not exists
SET @index_exists = (SELECT COUNT(*) FROM INFORMATION_SCHEMA.STATISTICS 
                    WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'klasifikasi_arsip' AND INDEX_NAME = 'uk_klasifikasi_kode');

SET @sql = IF(@index_exists = 0, 
             'ALTER TABLE `klasifikasi_arsip` ADD UNIQUE KEY `uk_klasifikasi_kode` (`kode`)', 
             'SELECT "uk_klasifikasi_kode already exists" as message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add unique key to jenis_surat if not exists  
SET @index_exists = (SELECT COUNT(*) FROM INFORMATION_SCHEMA.STATISTICS 
                    WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'jenis_surat' AND INDEX_NAME = 'uk_jenis_kode_klasifikasi');

SET @sql = IF(@index_exists = 0, 
             'ALTER TABLE `jenis_surat` ADD UNIQUE KEY `uk_jenis_kode_klasifikasi` (`kode`, `id_klasifikasi`)', 
             'SELECT "uk_jenis_kode_klasifikasi already exists" as message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add index to jenis_surat.id_klasifikasi if not exists
SET @index_exists = (SELECT COUNT(*) FROM INFORMATION_SCHEMA.STATISTICS 
                    WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'jenis_surat' AND INDEX_NAME = 'idx_jenis_klasifikasi');

SET @sql = IF(@index_exists = 0, 
             'ALTER TABLE `jenis_surat` ADD KEY `idx_jenis_klasifikasi` (`id_klasifikasi`)', 
             'SELECT "idx_jenis_klasifikasi already exists" as message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add index to office.tipe_surat_id if not exists
SET @index_exists = (SELECT COUNT(*) FROM INFORMATION_SCHEMA.STATISTICS 
                    WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND INDEX_NAME = 'idx_office_tipe_surat');

SET @sql = IF(@index_exists = 0, 
             'ALTER TABLE `office` ADD KEY `idx_office_tipe_surat` (`tipe_surat_id`)', 
             'SELECT "idx_office_tipe_surat already exists" as message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- =====================================================
-- STEP 5: ADD FOREIGN KEYS SAFELY
-- =====================================================

-- Add foreign key jenis_surat -> klasifikasi_arsip
SET @fk_exists = (SELECT COUNT(*) FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
                 WHERE CONSTRAINT_SCHEMA = DATABASE() AND TABLE_NAME = 'jenis_surat' AND CONSTRAINT_NAME = 'fk_jenis_klasifikasi');

SET @sql = IF(@fk_exists = 0, 
             'ALTER TABLE `jenis_surat` ADD CONSTRAINT `fk_jenis_klasifikasi` FOREIGN KEY (`id_klasifikasi`) REFERENCES `klasifikasi_arsip` (`id`) ON UPDATE CASCADE ON DELETE SET NULL', 
             'SELECT "fk_jenis_klasifikasi already exists" as message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add foreign key subjenis_surat -> jenis_surat  
SET @fk_exists = (SELECT COUNT(*) FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
                 WHERE CONSTRAINT_SCHEMA = DATABASE() AND TABLE_NAME = 'subjenis_surat' AND CONSTRAINT_NAME = 'fk_subjenis_jenis');

SET @sql = IF(@fk_exists = 0, 
             'ALTER TABLE `subjenis_surat` ADD CONSTRAINT `fk_subjenis_jenis` FOREIGN KEY (`id_jenis`) REFERENCES `jenis_surat` (`id`) ON UPDATE CASCADE ON DELETE CASCADE', 
             'SELECT "fk_subjenis_jenis already exists" as message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add foreign key subjenis_surat -> klasifikasi_arsip
SET @fk_exists = (SELECT COUNT(*) FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
                 WHERE CONSTRAINT_SCHEMA = DATABASE() AND TABLE_NAME = 'subjenis_surat' AND CONSTRAINT_NAME = 'fk_subjenis_klasifikasi');

SET @sql = IF(@fk_exists = 0, 
             'ALTER TABLE `subjenis_surat` ADD CONSTRAINT `fk_subjenis_klasifikasi` FOREIGN KEY (`id_klasifikasi`) REFERENCES `klasifikasi_arsip` (`id`) ON UPDATE CASCADE ON DELETE CASCADE', 
             'SELECT "fk_subjenis_klasifikasi already exists" as message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add foreign key office -> tipe_surat
SET @fk_exists = (SELECT COUNT(*) FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
                 WHERE CONSTRAINT_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND CONSTRAINT_NAME = 'fk_office_tipe_surat');

SET @sql = IF(@fk_exists = 0, 
             'ALTER TABLE `office` ADD CONSTRAINT `fk_office_tipe_surat` FOREIGN KEY (`tipe_surat_id`) REFERENCES `tipe_surat` (`id`) ON UPDATE CASCADE ON DELETE SET NULL', 
             'SELECT "fk_office_tipe_surat already exists" as message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add foreign key office -> subjenis_surat
SET @fk_exists = (SELECT COUNT(*) FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
                 WHERE CONSTRAINT_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND CONSTRAINT_NAME = 'fk_office_subjenis');

SET @sql = IF(@fk_exists = 0, 
             'ALTER TABLE `office` ADD CONSTRAINT `fk_office_subjenis` FOREIGN KEY (`subjenis_surat_id`) REFERENCES `subjenis_surat` (`id`) ON UPDATE CASCADE ON DELETE SET NULL', 
             'SELECT "fk_office_subjenis already exists" as message');
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- =====================================================
-- STEP 6: CREATE VIEW SAFELY
-- =====================================================

DROP VIEW IF EXISTS `v_office_dual_numbering`;
CREATE VIEW `v_office_dual_numbering` AS
SELECT 
  o.*,
  ts.nama as tipe_surat_nama,
  ts.kode as tipe_surat_kode,
  ka_new.nama as klasifikasi_nama_new,
  ka_new.kode as klasifikasi_kode_new,
  js_new.nama as jenis_surat_nama_new,
  js_new.kode as jenis_surat_kode_new,
  ss.nama as subjenis_surat_nama,
  ss.kode as subjenis_surat_kode,
  -- Dynamic numbering logic
  CASE 
    WHEN o.tipe_surat_id IS NOT NULL THEN
      CASE ts.kode
        WHEN 'SK' THEN o.nomor_surat_surat_keluar
        WHEN 'KPTS' THEN o.nomor_surat_surat_keputusan_direksi  
        WHEN 'SE' THEN o.nomor_surat_surat_edaran_direksi
        ELSE NULL
      END
    ELSE o.nomor_surat
  END as nomor_surat_display,
  -- Migration status
  CASE 
    WHEN o.is_migrated_to_new_system = 1 THEN 'New System'
    ELSE 'Legacy System'
  END as system_type
FROM office o
LEFT JOIN tipe_surat ts ON o.tipe_surat_id = ts.id
LEFT JOIN klasifikasi_arsip ka_new ON o.klasifikasi_arsip_kode = ka_new.kode
LEFT JOIN jenis_surat js_new ON o.jenis_surat_kode = js_new.kode AND js_new.id_klasifikasi = ka_new.id
LEFT JOIN subjenis_surat ss ON o.subjenis_surat_id = ss.id;

-- Re-enable foreign key checks
SET FOREIGN_KEY_CHECKS = 1;

-- Success message
SELECT 'Ultra Safe Migration completed successfully!' as status;
SELECT 'Tables created, columns added, indexes created, foreign keys established!' as details;

-- Show table status
SELECT 
  COUNT(*) as tipe_surat_count,
  (SELECT COUNT(*) FROM subjenis_surat) as subjenis_count,
  (SELECT COUNT(*) FROM office) as office_count
FROM tipe_surat;