-- =====================================================
-- SAFE DATABASE MIGRATION: DUAL LETTER NUMBERING SYSTEM
-- =====================================================
-- This script safely adds new columns/tables without conflicts

-- Check if tipe_surat table exists, if not create it
CREATE TABLE IF NOT EXISTS `tipe_surat` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `nama` varchar(100) NOT NULL,
  `kode` varchar(10) NOT NULL,
  `created_at` timestamp NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  UNIQUE KEY `uk_tipe_surat_kode` (`kode`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Insert tipe surat data if table is empty
INSERT IGNORE INTO `tipe_surat` (`nama`, `kode`) VALUES 
('Surat Keluar', 'SK'),
('Surat Keputusan Direksi', 'KPTS'),
('Surat Edaran Direksi', 'SE');

-- Create subjenis_surat table WITHOUT foreign keys first
CREATE TABLE IF NOT EXISTS `subjenis_surat` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `kode` varchar(10) NOT NULL,
  `nama` varchar(255) NOT NULL,
  `deskripsi` text DEFAULT NULL,
  `id_jenis` int(11) NOT NULL,
  `id_klasifikasi` int(11) NOT NULL,
  `created_at` timestamp NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  UNIQUE KEY `uk_subjenis_kode_jenis_klasifikasi` (`kode`, `id_jenis`, `id_klasifikasi`),
  KEY `idx_subjenis_jenis` (`id_jenis`),
  KEY `idx_subjenis_klasifikasi` (`id_klasifikasi`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- =====================================================
-- SAFE COLUMN ADDITIONS
-- =====================================================

-- Add deskripsi to klasifikasi_arsip if not exists
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'klasifikasi_arsip' AND COLUMN_NAME = 'deskripsi') = 0,
    'ALTER TABLE `klasifikasi_arsip` ADD COLUMN `deskripsi` text DEFAULT NULL',
    'SELECT "deskripsi column already exists in klasifikasi_arsip"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add deskripsi to jenis_surat if not exists
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'jenis_surat' AND COLUMN_NAME = 'deskripsi') = 0,
    'ALTER TABLE `jenis_surat` ADD COLUMN `deskripsi` text DEFAULT NULL',
    'SELECT "deskripsi column already exists in jenis_surat"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add id_klasifikasi to jenis_surat if not exists
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'jenis_surat' AND COLUMN_NAME = 'id_klasifikasi') = 0,
    'ALTER TABLE `jenis_surat` ADD COLUMN `id_klasifikasi` int(11) DEFAULT NULL',
    'SELECT "id_klasifikasi column already exists in jenis_surat"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add new columns to office table for dual system
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND COLUMN_NAME = 'tipe_surat_id') = 0,
    'ALTER TABLE `office` ADD COLUMN `tipe_surat_id` int(11) DEFAULT NULL',
    'SELECT "tipe_surat_id column already exists in office"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND COLUMN_NAME = 'klasifikasi_arsip_kode') = 0,
    'ALTER TABLE `office` ADD COLUMN `klasifikasi_arsip_kode` varchar(10) DEFAULT NULL',
    'SELECT "klasifikasi_arsip_kode column already exists in office"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND COLUMN_NAME = 'jenis_surat_kode') = 0,
    'ALTER TABLE `office` ADD COLUMN `jenis_surat_kode` varchar(10) DEFAULT NULL',
    'SELECT "jenis_surat_kode column already exists in office"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND COLUMN_NAME = 'subjenis_surat_id') = 0,
    'ALTER TABLE `office` ADD COLUMN `subjenis_surat_id` int(11) DEFAULT NULL',
    'SELECT "subjenis_surat_id column already exists in office"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add new numbering columns
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND COLUMN_NAME = 'nomor_surat_surat_keluar') = 0,
    'ALTER TABLE `office` ADD COLUMN `nomor_surat_surat_keluar` varchar(100) DEFAULT NULL',
    'SELECT "nomor_surat_surat_keluar column already exists in office"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND COLUMN_NAME = 'nomor_surat_surat_keputusan_direksi') = 0,
    'ALTER TABLE `office` ADD COLUMN `nomor_surat_surat_keputusan_direksi` varchar(100) DEFAULT NULL',
    'SELECT "nomor_surat_surat_keputusan_direksi column already exists in office"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND COLUMN_NAME = 'nomor_surat_surat_edaran_direksi') = 0,
    'ALTER TABLE `office` ADD COLUMN `nomor_surat_surat_edaran_direksi` varchar(100) DEFAULT NULL',
    'SELECT "nomor_surat_surat_edaran_direksi column already exists in office"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add migration tracking columns
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND COLUMN_NAME = 'is_migrated_to_new_system') = 0,
    'ALTER TABLE `office` ADD COLUMN `is_migrated_to_new_system` tinyint(1) DEFAULT 0',
    'SELECT "is_migrated_to_new_system column already exists in office"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND COLUMN_NAME = 'migration_notes') = 0,
    'ALTER TABLE `office` ADD COLUMN `migration_notes` text DEFAULT NULL',
    'SELECT "migration_notes column already exists in office"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- =====================================================
-- CREATE INDEXES SAFELY
-- =====================================================

-- Add unique key to klasifikasi_arsip if not exists
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.STATISTICS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'klasifikasi_arsip' AND INDEX_NAME = 'uk_klasifikasi_kode') = 0,
    'ALTER TABLE `klasifikasi_arsip` ADD UNIQUE KEY `uk_klasifikasi_kode` (`kode`)',
    'SELECT "uk_klasifikasi_kode index already exists"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add indexes to jenis_surat if not exists
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.STATISTICS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'jenis_surat' AND INDEX_NAME = 'uk_jenis_kode_klasifikasi') = 0,
    'ALTER TABLE `jenis_surat` ADD UNIQUE KEY `uk_jenis_kode_klasifikasi` (`kode`, `id_klasifikasi`)',
    'SELECT "uk_jenis_kode_klasifikasi index already exists"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.STATISTICS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'jenis_surat' AND INDEX_NAME = 'idx_jenis_klasifikasi') = 0,
    'ALTER TABLE `jenis_surat` ADD KEY `idx_jenis_klasifikasi` (`id_klasifikasi`)',
    'SELECT "idx_jenis_klasifikasi index already exists"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add indexes to office table
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.STATISTICS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND INDEX_NAME = 'idx_office_tipe_surat') = 0,
    'ALTER TABLE `office` ADD KEY `idx_office_tipe_surat` (`tipe_surat_id`)',
    'SELECT "idx_office_tipe_surat index already exists"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- =====================================================
-- CREATE FOREIGN KEYS SAFELY  
-- =====================================================

-- Add foreign keys if they don't exist
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
     WHERE CONSTRAINT_SCHEMA = DATABASE() AND TABLE_NAME = 'subjenis_surat' AND CONSTRAINT_NAME = 'fk_subjenis_jenis') = 0,
    'ALTER TABLE `subjenis_surat` ADD CONSTRAINT `fk_subjenis_jenis` FOREIGN KEY (`id_jenis`) REFERENCES `jenis_surat` (`id`) ON UPDATE CASCADE ON DELETE CASCADE',
    'SELECT "fk_subjenis_jenis constraint already exists"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
     WHERE CONSTRAINT_SCHEMA = DATABASE() AND TABLE_NAME = 'subjenis_surat' AND CONSTRAINT_NAME = 'fk_subjenis_klasifikasi') = 0,
    'ALTER TABLE `subjenis_surat` ADD CONSTRAINT `fk_subjenis_klasifikasi` FOREIGN KEY (`id_klasifikasi`) REFERENCES `klasifikasi_arsip` (`id`) ON UPDATE CASCADE ON DELETE CASCADE',
    'SELECT "fk_subjenis_klasifikasi constraint already exists"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
     WHERE CONSTRAINT_SCHEMA = DATABASE() AND TABLE_NAME = 'jenis_surat' AND CONSTRAINT_NAME = 'fk_jenis_klasifikasi') = 0,
    'ALTER TABLE `jenis_surat` ADD CONSTRAINT `fk_jenis_klasifikasi` FOREIGN KEY (`id_klasifikasi`) REFERENCES `klasifikasi_arsip` (`id`) ON UPDATE CASCADE ON DELETE SET NULL',
    'SELECT "fk_jenis_klasifikasi constraint already exists"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
     WHERE CONSTRAINT_SCHEMA = DATABASE() AND TABLE_NAME = 'office' AND CONSTRAINT_NAME = 'fk_office_tipe_surat') = 0,
    'ALTER TABLE `office` ADD CONSTRAINT `fk_office_tipe_surat` FOREIGN KEY (`tipe_surat_id`) REFERENCES `tipe_surat` (`id`) ON UPDATE CASCADE ON DELETE SET NULL',
    'SELECT "fk_office_tipe_surat constraint already exists"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- =====================================================
-- CREATE VIEW SAFELY
-- =====================================================

DROP VIEW IF EXISTS `v_office_dual_numbering`;
CREATE VIEW `v_office_dual_numbering` AS
SELECT 
  o.*,
  ts.nama as tipe_surat_nama,
  ts.kode as tipe_surat_kode,
  ka_new.nama as klasifikasi_nama_new,
  ka_new.kode as klasifikasi_kode_new,
  js_new.nama as jenis_surat_nama_new,
  js_new.kode as jenis_surat_kode_new,
  ss.nama as subjenis_surat_nama,
  ss.kode as subjenis_surat_kode,
  -- Dynamic numbering logic
  CASE 
    WHEN o.tipe_surat_id IS NOT NULL THEN
      CASE ts.kode
        WHEN 'SK' THEN o.nomor_surat_surat_keluar
        WHEN 'KPTS' THEN o.nomor_surat_surat_keputusan_direksi  
        WHEN 'SE' THEN o.nomor_surat_surat_edaran_direksi
        ELSE NULL
      END
    ELSE o.nomor_surat
  END as nomor_surat_display,
  -- Migration status
  CASE 
    WHEN o.is_migrated_to_new_system = 1 THEN 'New System'
    ELSE 'Legacy System'
  END as system_type
FROM office o
LEFT JOIN tipe_surat ts ON o.tipe_surat_id = ts.id
LEFT JOIN klasifikasi_arsip ka_new ON o.klasifikasi_arsip_kode = ka_new.kode
LEFT JOIN jenis_surat js_new ON o.jenis_surat_kode = js_new.kode AND js_new.id_klasifikasi = ka_new.id
LEFT JOIN subjenis_surat ss ON o.subjenis_surat_id = ss.id;

SELECT 'Migration completed successfully!' as status;