-- =====================================================
-- DATABASE MIGRATION: DUAL LETTER NUMBERING SYSTEM
-- =====================================================
-- This script implements hierarchical classification system
-- while maintaining backward compatibility with existing data

-- Step 1: Create new tables
-- =====================================================

-- 1.1 Create tipe_surat table
CREATE TABLE `tipe_surat` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `nama` varchar(100) NOT NULL,
  `kode` varchar(10) NOT NULL,
  `created_at` timestamp NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  UNIQUE KEY `uk_tipe_surat_kode` (`kode`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Insert tipe surat data
INSERT INTO `tipe_surat` (`nama`, `kode`) VALUES 
('Surat Keluar', 'SK'),
('Surat Keputusan Direksi', 'KPTS'),
('Surat Edaran Direksi', 'SE');

-- 1.2 Create subjenis_surat table
CREATE TABLE `subjenis_surat` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `kode` varchar(10) NOT NULL,
  `nama` varchar(255) NOT NULL,
  `deskripsi` text DEFAULT NULL,
  `id_jenis` int(11) NOT NULL,
  `id_klasifikasi` int(11) NOT NULL,
  `created_at` timestamp NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  UNIQUE KEY `uk_subjenis_kode_jenis_klasifikasi` (`kode`, `id_jenis`, `id_klasifikasi`),
  KEY `idx_subjenis_jenis` (`id_jenis`),
  KEY `idx_subjenis_klasifikasi` (`id_klasifikasi`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- Step 2: Modify existing tables
-- =====================================================

-- 2.1 Add new columns to klasifikasi_arsip table (if not exists)
-- Check and add kode column if it doesn't exist
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'klasifikasi_arsip' AND COLUMN_NAME = 'kode') = 0,
    'ALTER TABLE `klasifikasi_arsip` ADD COLUMN `kode` varchar(10) NOT NULL AFTER `nama`',
    'SELECT "Column kode already exists" as message'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Check and add deskripsi column if it doesn't exist
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'klasifikasi_arsip' AND COLUMN_NAME = 'deskripsi') = 0,
    'ALTER TABLE `klasifikasi_arsip` ADD COLUMN `deskripsi` text DEFAULT NULL AFTER `kode`',
    'SELECT "Column deskripsi already exists" as message'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add unique key if it doesn't exist
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.STATISTICS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'klasifikasi_arsip' AND INDEX_NAME = 'uk_klasifikasi_kode') = 0,
    'ALTER TABLE `klasifikasi_arsip` ADD UNIQUE KEY `uk_klasifikasi_kode` (`kode`)',
    'SELECT "Unique key uk_klasifikasi_kode already exists" as message'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- 2.2 Add new columns to jenis_surat table (if not exists)
-- Check and add kode column if it doesn't exist
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'jenis_surat' AND COLUMN_NAME = 'kode') = 0,
    'ALTER TABLE `jenis_surat` ADD COLUMN `kode` varchar(10) NOT NULL AFTER `nama`',
    'SELECT "Column kode already exists in jenis_surat" as message'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Check and add deskripsi column if it doesn't exist
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'jenis_surat' AND COLUMN_NAME = 'deskripsi') = 0,
    'ALTER TABLE `jenis_surat` ADD COLUMN `deskripsi` text DEFAULT NULL AFTER `kode`',
    'SELECT "Column deskripsi already exists in jenis_surat" as message'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Check and add id_klasifikasi column if it doesn't exist
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'jenis_surat' AND COLUMN_NAME = 'id_klasifikasi') = 0,
    'ALTER TABLE `jenis_surat` ADD COLUMN `id_klasifikasi` int(11) DEFAULT NULL AFTER `deskripsi`',
    'SELECT "Column id_klasifikasi already exists in jenis_surat" as message'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add unique key if it doesn't exist
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.STATISTICS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'jenis_surat' AND INDEX_NAME = 'uk_jenis_kode_klasifikasi') = 0,
    'ALTER TABLE `jenis_surat` ADD UNIQUE KEY `uk_jenis_kode_klasifikasi` (`kode`, `id_klasifikasi`)',
    'SELECT "Unique key uk_jenis_kode_klasifikasi already exists" as message'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Add index if it doesn't exist
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.STATISTICS 
     WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'jenis_surat' AND INDEX_NAME = 'idx_jenis_klasifikasi') = 0,
    'ALTER TABLE `jenis_surat` ADD KEY `idx_jenis_klasifikasi` (`id_klasifikasi`)',
    'SELECT "Index idx_jenis_klasifikasi already exists" as message'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- 2.3 Add new columns to office table (DUAL SYSTEM)
ALTER TABLE `office`
-- New classification columns (code-based instead of ID-based)
ADD COLUMN `tipe_surat_id` int(11) DEFAULT NULL AFTER `kepentingan`,
ADD COLUMN `klasifikasi_arsip_kode` varchar(10) DEFAULT NULL AFTER `klasifikasi_arsip_id`,
ADD COLUMN `jenis_surat_kode` varchar(10) DEFAULT NULL AFTER `jenis_surat_id`,
ADD COLUMN `subjenis_surat_id` int(11) DEFAULT NULL AFTER `jenis_surat_kode`,

-- New numbering columns (separate for each letter type)
ADD COLUMN `nomor_surat_surat_keluar` varchar(100) DEFAULT NULL AFTER `nomor_surat`,
ADD COLUMN `nomor_surat_surat_keputusan_direksi` varchar(100) DEFAULT NULL AFTER `nomor_surat_surat_keluar`,
ADD COLUMN `nomor_surat_surat_edaran_direksi` varchar(100) DEFAULT NULL AFTER `nomor_surat_surat_keputusan_direksi`,

-- Migration tracking
ADD COLUMN `is_migrated_to_new_system` tinyint(1) DEFAULT 0 AFTER `nomor_surat_surat_edaran_direksi`,
ADD COLUMN `migration_notes` text DEFAULT NULL AFTER `is_migrated_to_new_system`,

-- Indexes for performance
ADD KEY `idx_office_tipe_surat` (`tipe_surat_id`),
ADD KEY `idx_office_klasifikasi_kode` (`klasifikasi_arsip_kode`),
ADD KEY `idx_office_jenis_kode` (`jenis_surat_kode`),
ADD KEY `idx_office_subjenis` (`subjenis_surat_id`),
ADD KEY `idx_office_migration` (`is_migrated_to_new_system`);

-- Step 3: Add foreign key constraints
-- =====================================================

-- Add foreign keys for new relationships
ALTER TABLE `subjenis_surat`
ADD CONSTRAINT `fk_subjenis_jenis` FOREIGN KEY (`id_jenis`) REFERENCES `jenis_surat` (`id`) ON UPDATE CASCADE ON DELETE CASCADE,
ADD CONSTRAINT `fk_subjenis_klasifikasi` FOREIGN KEY (`id_klasifikasi`) REFERENCES `klasifikasi_arsip` (`id`) ON UPDATE CASCADE ON DELETE CASCADE;

ALTER TABLE `jenis_surat`
ADD CONSTRAINT `fk_jenis_klasifikasi` FOREIGN KEY (`id_klasifikasi`) REFERENCES `klasifikasi_arsip` (`id`) ON UPDATE CASCADE ON DELETE SET NULL;

ALTER TABLE `office`
ADD CONSTRAINT `fk_office_tipe_surat` FOREIGN KEY (`tipe_surat_id`) REFERENCES `tipe_surat` (`id`) ON UPDATE CASCADE ON DELETE SET NULL,
ADD CONSTRAINT `fk_office_subjenis` FOREIGN KEY (`subjenis_surat_id`) REFERENCES `subjenis_surat` (`id`) ON UPDATE CASCADE ON DELETE SET NULL;

-- Step 4: Create views for backward compatibility
-- =====================================================

-- View to show both old and new numbering formats
CREATE VIEW `v_office_dual_numbering` AS
SELECT 
  o.*,
  ts.nama as tipe_surat_nama,
  ts.kode as tipe_surat_kode,
  ka_new.nama as klasifikasi_nama_new,
  ka_new.kode as klasifikasi_kode_new,
  js_new.nama as jenis_surat_nama_new,
  js_new.kode as jenis_surat_kode_new,
  ss.nama as subjenis_surat_nama,
  ss.kode as subjenis_surat_kode,
  -- Dynamic numbering logic
  CASE 
    WHEN o.tipe_surat_id IS NOT NULL THEN
      CASE ts.kode
        WHEN 'SK' THEN o.nomor_surat_surat_keluar
        WHEN 'KPTS' THEN o.nomor_surat_surat_keputusan_direksi  
        WHEN 'SE' THEN o.nomor_surat_surat_edaran_direksi
        ELSE NULL
      END
    ELSE o.nomor_surat
  END as nomor_surat_display,
  -- Migration status
  CASE 
    WHEN o.is_migrated_to_new_system = 1 THEN 'New System'
    ELSE 'Legacy System'
  END as system_type
FROM office o
LEFT JOIN tipe_surat ts ON o.tipe_surat_id = ts.id
LEFT JOIN klasifikasi_arsip ka_new ON o.klasifikasi_arsip_kode = ka_new.kode
LEFT JOIN jenis_surat js_new ON o.jenis_surat_kode = js_new.kode AND js_new.id_klasifikasi = ka_new.id
LEFT JOIN subjenis_surat ss ON o.subjenis_surat_id = ss.id;

-- Step 5: Data validation queries
-- =====================================================

-- Check existing data that needs migration
SELECT 
  COUNT(*) as total_letters,
  SUM(CASE WHEN is_migrated_to_new_system = 1 THEN 1 ELSE 0 END) as migrated_count,
  SUM(CASE WHEN is_migrated_to_new_system = 0 OR is_migrated_to_new_system IS NULL THEN 1 ELSE 0 END) as legacy_count
FROM office;

-- Check for potential data conflicts
SELECT 
  nomor_surat,
  COUNT(*) as duplicate_count
FROM office 
WHERE nomor_surat IS NOT NULL
GROUP BY nomor_surat
HAVING COUNT(*) > 1;

-- =====================================================
-- ROLLBACK SCRIPT (if needed)
-- =====================================================
/*
-- To rollback these changes, uncomment and run:

-- Drop foreign keys first
ALTER TABLE `office` DROP FOREIGN KEY `fk_office_tipe_surat`;
ALTER TABLE `office` DROP FOREIGN KEY `fk_office_subjenis`;
ALTER TABLE `jenis_surat` DROP FOREIGN KEY `fk_jenis_klasifikasi`;
ALTER TABLE `subjenis_surat` DROP FOREIGN KEY `fk_subjenis_jenis`;
ALTER TABLE `subjenis_surat` DROP FOREIGN KEY `fk_subjenis_klasifikasi`;

-- Drop view
DROP VIEW IF EXISTS `v_office_dual_numbering`;

-- Remove added columns from office table
ALTER TABLE `office` 
DROP COLUMN `tipe_surat_id`,
DROP COLUMN `klasifikasi_arsip_kode`,
DROP COLUMN `jenis_surat_kode`, 
DROP COLUMN `subjenis_surat_id`,
DROP COLUMN `nomor_surat_surat_keluar`,
DROP COLUMN `nomor_surat_surat_keputusan_direksi`,
DROP COLUMN `nomor_surat_surat_edaran_direksi`,
DROP COLUMN `is_migrated_to_new_system`,
DROP COLUMN `migration_notes`;

-- Remove added columns from existing tables
ALTER TABLE `klasifikasi_arsip` DROP COLUMN `kode`, DROP COLUMN `deskripsi`;
ALTER TABLE `jenis_surat` DROP COLUMN `kode`, DROP COLUMN `deskripsi`, DROP COLUMN `id_klasifikasi`;

-- Drop new tables
DROP TABLE IF EXISTS `subjenis_surat`;
DROP TABLE IF EXISTS `tipe_surat`;
*/

-- =====================================================
-- NOTES FOR IMPLEMENTATION
-- =====================================================
/*
1. This script maintains backward compatibility by keeping existing columns
2. New system runs parallel to old system via tipe_surat_id check
3. Migration can be done gradually using is_migrated_to_new_system flag
4. View v_office_dual_numbering handles display logic for both systems
5. Foreign keys ensure data integrity in new system
6. Rollback script provided for safety
*/