-- =====================================================
-- APPROVAL SYSTEM MIGRATION
-- Run this migration to implement the new approval system
-- Date: 2025-12-17
-- =====================================================

SET SQL_MODE = "NO_AUTO_VALUE_ON_ZERO";
START TRANSACTION;

-- =====================================================
-- STEP 1: Create `inisial` table
-- Auto-generate initials from authorized signers
-- =====================================================

DROP TABLE IF EXISTS `inisial`;

CREATE TABLE `inisial` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `user_id` int(11) DEFAULT NULL COMMENT 'FK to users.id',
  `nama` varchar(100) NOT NULL,
  `inisial` varchar(20) NOT NULL COMMENT 'Capital letters from first letter of each word',
  `is_active` tinyint(1) NOT NULL DEFAULT 1,
  `created_at` timestamp NULL DEFAULT current_timestamp(),
  `updated_at` timestamp NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
  PRIMARY KEY (`id`),
  UNIQUE KEY `idx_user_id` (`user_id`),
  KEY `idx_inisial` (`inisial`),
  KEY `idx_active` (`is_active`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- =====================================================
-- STEP 2: Auto-populate `inisial` from authorized_signers
-- Generate initials from first letter of each word (CAPITAL)
-- Example: "Bambang Hermawan" -> "BH"
-- =====================================================

INSERT INTO `inisial` (`user_id`, `nama`, `inisial`, `is_active`)
SELECT 
    a.user_id,
    a.nama,
    -- Generate initials: take first letter of each word, uppercase
    UPPER(
        CONCAT(
            SUBSTRING(SUBSTRING_INDEX(a.nama, ' ', 1), 1, 1),
            CASE WHEN LENGTH(a.nama) - LENGTH(REPLACE(a.nama, ' ', '')) >= 1 
                THEN SUBSTRING(SUBSTRING_INDEX(SUBSTRING_INDEX(a.nama, ' ', 2), ' ', -1), 1, 1) 
                ELSE '' END,
            CASE WHEN LENGTH(a.nama) - LENGTH(REPLACE(a.nama, ' ', '')) >= 2 
                THEN SUBSTRING(SUBSTRING_INDEX(SUBSTRING_INDEX(a.nama, ' ', 3), ' ', -1), 1, 1) 
                ELSE '' END,
            CASE WHEN LENGTH(a.nama) - LENGTH(REPLACE(a.nama, ' ', '')) >= 3 
                THEN SUBSTRING(SUBSTRING_INDEX(SUBSTRING_INDEX(a.nama, ' ', 4), ' ', -1), 1, 1) 
                ELSE '' END
        )
    ) as generated_inisial,
    a.is_active
FROM authorized_signers a
WHERE a.is_active = 1
ON DUPLICATE KEY UPDATE 
    nama = VALUES(nama),
    inisial = VALUES(inisial),
    is_active = VALUES(is_active);

-- =====================================================
-- STEP 3: Create `urutan_paraf` table
-- Track sequential approval chain per letter
-- =====================================================

DROP TABLE IF EXISTS `urutan_paraf`;

CREATE TABLE `urutan_paraf` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `office_id` bigint(20) UNSIGNED NOT NULL COMMENT 'FK to office.id (letter)',
  `user_id` int(11) NOT NULL COMMENT 'FK to users.id (approver)',
  `urutan` int(11) NOT NULL COMMENT 'Position in approval chain (1=first approver)',
  `inisial` varchar(20) DEFAULT NULL COMMENT 'Cached initials at approval time',
  `approved_at` datetime DEFAULT NULL COMMENT 'When approved, NULL if pending',
  `signature_path` varchar(255) DEFAULT NULL COMMENT 'Path to signature image',
  `comment` text DEFAULT NULL COMMENT 'Notes/comments from approver',
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  `updated_at` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
  PRIMARY KEY (`id`),
  UNIQUE KEY `idx_office_user` (`office_id`, `user_id`),
  KEY `idx_office_id` (`office_id`),
  KEY `idx_user_id` (`user_id`),
  KEY `idx_urutan` (`urutan`),
  KEY `idx_approved_at` (`approved_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- =====================================================
-- STEP 4: Create `approval_comments` table
-- Multiple notes per approval (thread-style)
-- =====================================================

DROP TABLE IF EXISTS `approval_comments`;

CREATE TABLE `approval_comments` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `office_id` bigint(20) UNSIGNED NOT NULL COMMENT 'FK to office.id',
  `urutan_paraf_id` int(11) DEFAULT NULL COMMENT 'FK to urutan_paraf.id (optional)',
  `user_id` int(11) NOT NULL COMMENT 'FK to users.id (commenter)',
  `comment` text NOT NULL,
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  `updated_at` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
  PRIMARY KEY (`id`),
  KEY `idx_office_id` (`office_id`),
  KEY `idx_urutan_paraf_id` (`urutan_paraf_id`),
  KEY `idx_user_id` (`user_id`),
  KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;

-- =====================================================
-- STEP 5: Add new columns to `office` table
-- =====================================================

-- Add column to track if approval chain is complete
ALTER TABLE `office` 
ADD COLUMN IF NOT EXISTS `approval_completed` tinyint(1) NOT NULL DEFAULT 0 
COMMENT '1 if all approvals in chain completed' 
AFTER `is_validated`;

-- Add column for penandatangan user_id (signer)
ALTER TABLE `office`
ADD COLUMN IF NOT EXISTS `penandatangan_id` int(11) DEFAULT NULL
COMMENT 'FK to authorized_signers.id - the final signer'
AFTER `dibuat_oleh`;

-- =====================================================
-- STEP 6: Migrate existing validated letters
-- Insert into urutan_paraf with penandatangan as single approver
-- =====================================================

-- For letters that are already validated (is_validated=1),
-- we assume the penandatangan was the final approver
-- This preserves existing data

INSERT INTO `urutan_paraf` (`office_id`, `user_id`, `urutan`, `inisial`, `approved_at`, `signature_path`)
SELECT 
    o.id as office_id,
    COALESCE(a.user_id, 0) as user_id,
    1 as urutan,
    COALESCE(i.inisial, 'N/A') as inisial,
    COALESCE(o.validated_at, NOW()) as approved_at,
    o.signature_path
FROM office o
LEFT JOIN authorized_signers a ON o.kepada = a.nama OR o.perihal LIKE CONCAT('%', a.nama, '%')
LEFT JOIN inisial i ON a.user_id = i.user_id
WHERE o.is_validated = 1
AND NOT EXISTS (
    SELECT 1 FROM urutan_paraf up WHERE up.office_id = o.id
);

-- Update approval_completed for already validated letters
UPDATE office SET approval_completed = 1 WHERE is_validated = 1;

COMMIT;

-- =====================================================
-- VERIFICATION QUERIES (run to check)
-- =====================================================

-- Check inisial table
SELECT * FROM inisial;

-- Check urutan_paraf for migrated data
SELECT up.*, o.nomor_surat, u.nama as approver_name
FROM urutan_paraf up
LEFT JOIN office o ON up.office_id = o.id
LEFT JOIN users u ON up.user_id = u.id
ORDER BY up.office_id, up.urutan;

-- Check office new columns
DESCRIBE office;
