<?php

/**
 * Letter Order Functions
 * Helper functions for letter-order and approval chain features
 */

/**
 * Check if current user is admin
 */
function isAdmin($conn)
{
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    $username = $_SESSION['username'] ?? null;
    if (!$username) return false;
    $stmt = $conn->prepare("SELECT tipe FROM users WHERE username = ? LIMIT 1");
    $stmt->bind_param('s', $username);
    $stmt->execute();
    $res = $stmt->get_result();
    if (!$res) return false;
    $u = $res->fetch_assoc();
    return isset($u['tipe']) && strtolower(trim($u['tipe'])) === 'admin';
}

/**
 * Build supervisor map from letter_order table
 */
function buildSupervisorMap($conn)
{
    $map = [];
    $nameMap = [];
    $sql = "SELECT lo.user_id, lo.supervisor_id, u.nama AS user_name, s.nama AS supervisor_name
            FROM letter_order lo
            LEFT JOIN users u ON lo.user_id = u.id
            LEFT JOIN users s ON lo.supervisor_id = s.id";
    if ($res = $conn->query($sql)) {
        while ($row = $res->fetch_assoc()) {
            $uid = (int)$row['user_id'];
            $sid = empty($row['supervisor_id']) ? null : (int)$row['supervisor_id'];
            $map[$uid] = $sid;
            $nameMap[$uid] = $row['user_name'] ?? '';
            if ($sid) $nameMap[$sid] = $row['supervisor_name'] ?? '';
        }
        $res->free();
    }
    return [$map, $nameMap];
}

/**
 * Get approval chain string from maps
 */
function getApprovalChainFromMaps($startUserId, $nameMap, $map, $maxDepth = 12)
{
    $chain = [];
    $seen = [];
    $current = $startUserId;
    $depth = 0;
    while ($current && $depth < $maxDepth) {
        if (isset($seen[$current])) {
            $chain[] = '(cycle)';
            break;
        }
        $seen[$current] = true;
        $chain[] = $nameMap[$current] ?? ('User:' . $current);
        $next = $map[$current] ?? null;
        if (!$next) break;
        $current = (int)$next;
        $depth++;
    }
    return implode(' → ', $chain);
}

/**
 * Check if adding supervisor creates a cycle
 */
function createsCycle($userId, $candidateSupervisorId, $map)
{
    if ($candidateSupervisorId === null) return false;
    $current = $candidateSupervisorId;
    $seen = [];
    while ($current !== null) {
        if ($current == $userId) return true;
        if (isset($seen[$current])) return true;
        $seen[$current] = true;
        $current = $map[$current] ?? null;
    }
    return false;
}

/**
 * Map jabatan string to kodeJabatan based on kode_jabatan.json
 */
function getKodeFromJabatan($jabatan)
{
    if (empty($jabatan)) return null;
    $jsonFile = __DIR__ . '/../data/kode_jabatan.json';
    if (!file_exists($jsonFile)) return null;
    $data = json_decode(file_get_contents($jsonFile), true);
    if (!isset($data['kodeJabatan']) || !is_array($data['kodeJabatan'])) return null;
    $jabatanNormalized = strtolower(trim(preg_replace('/\s+/', ' ', $jabatan)));
    foreach ($data['kodeJabatan'] as $item) {
        $main = strtolower(trim(preg_replace('/\s+/', ' ', $item['jabatan'] ?? '')));
        if ($main === $jabatanNormalized) return $item['kode'] ?? null;
        if (!empty($item['aliases']) && is_array($item['aliases'])) {
            foreach ($item['aliases'] as $alias) {
                if (strtolower(trim(preg_replace('/\s+/', ' ', $alias))) === $jabatanNormalized) {
                    return $item['kode'] ?? null;
                }
            }
        }
    }
    return null;
}

/**
 * Return an array of subordinate user IDs for a given user
 */
function getSubordinatesForUser($conn, $userId)
{
    $userId = (int)$userId;
    $subordinates = [];

    // 1) load letter_order map and compute descendants
    list($map, $nameMap) = buildSupervisorMap($conn);
    foreach ($map as $candidateId => $supervisorId) {
        $current = (int)$candidateId;
        $seen = [];
        while ($current !== null && isset($map[$current])) {
            if (isset($seen[$current])) break;
            $seen[$current] = true;
            $curSupervisor = $map[$current] ?? null;
            if ($curSupervisor === null) break;
            if ((int)$curSupervisor === $userId) {
                $subordinates[] = (int)$candidateId;
                break;
            }
            $current = (int)$curSupervisor;
        }
    }

    // 2) load users from approval_chain where approver_id = userId
    $stmt = $conn->prepare("SELECT DISTINCT user_id FROM approval_chain WHERE approver_id = ?");
    $stmt->bind_param('i', $userId);
    $stmt->execute();
    $res = $stmt->get_result();
    if ($res) {
        while ($row = $res->fetch_assoc()) {
            $subordinates[] = (int)$row['user_id'];
        }
        $res->free();
    }
    $stmt->close();

    // ensure unique and remove the user itself
    $subordinates = array_values(array_unique($subordinates));
    if (($k = array_search($userId, $subordinates)) !== false) {
        unset($subordinates[$k]);
    }
    return array_values($subordinates);
}

/**
 * Return true if the user is an approver
 */
function isApprover($conn, $userId)
{
    $stmt = $conn->prepare("SELECT 1 FROM approval_chain WHERE approver_id = ? LIMIT 1");
    $stmt->bind_param('i', $userId);
    $stmt->execute();
    $stmt->store_result();
    $is = $stmt->num_rows > 0;
    $stmt->close();
    return $is;
}
