<?php

/**
 * Authentication Helper Functions
 */

/**
 * Start session if not already started
 */
function ensureSession()
{
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
}

/**
 * Check if user is logged in
 */
function isLoggedIn()
{
    ensureSession();
    return isset($_SESSION['username']) && !empty($_SESSION['username']);
}

/**
 * Get current logged in username
 */
function getCurrentUsername()
{
    ensureSession();
    return $_SESSION['username'] ?? null;
}

/**
 * Get current user data from database
 */
function getCurrentUser($conn)
{
    $username = getCurrentUsername();
    if (!$username) return null;

    $stmt = $conn->prepare("SELECT id, username, tipe, nama, access_modul, jabatan FROM users WHERE username = ? LIMIT 1");
    $stmt->bind_param('s', $username);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    $stmt->close();

    return $user;
}

/**
 * Require admin role, redirect to login if not
 */
function requireAdmin($conn, $redirectTo = 'login.php')
{
    ensureSession();
    $user = getCurrentUser($conn);

    if (!$user || $user['tipe'] !== 'admin') {
        header('Location: ' . $redirectTo);
        exit();
    }

    return $user;
}

/**
 * Require user role, redirect to login if not
 */
function requireUser($conn, $redirectTo = 'login.php')
{
    ensureSession();
    $user = getCurrentUser($conn);

    if (!$user || $user['tipe'] !== 'user') {
        header('Location: ' . $redirectTo);
        exit();
    }

    return $user;
}

/**
 * Require any authenticated user
 */
function requireAuth($conn, $redirectTo = 'login.php')
{
    ensureSession();
    $user = getCurrentUser($conn);

    if (!$user) {
        header('Location: ' . $redirectTo);
        exit();
    }

    return $user;
}
