# 4.6 Perancangan Mekanisme Penomoran Surat (Detailed Design)

## 4.6.1 Aturan Penomoran Surat

### Pengenalan Umum

Sistem penomoran surat dalam Jasa Marga dirancang untuk memastikan:
- **Identifikasi Unik**: Setiap surat memiliki nomor yang tidak dapat diduplikasi
- **Traceability**: Memudahkan pelacakan dan referensi dokumen
- **Compliance**: Sesuai dengan standar administrasi dan archival requirements
- **Efisiensi Operasional**: Memungkinkan pencarian dan pengorganisasian dokumen yang cepat

Sistem menggunakan **3 format berbeda** sesuai dengan jenis surat, masing-masing dirancang untuk memenuhi kebutuhan spesifik dari tipe dokumen tersebut.

---

### 1. Format Penomoran Surat Keluar

#### 1.1 Spesifikasi Format

**Format Umum**: `OT.01.03.xxx`

**Struktur Komponen**:

| Komponen | Posisi | Tipe | Deskripsi | Contoh |
|---|---|---|---|---|
| Kode Unit | 1 | String | Identifikasi unit/departemen penerbit | OT (Organisasi dan Transformasi) |
| Kode Jenis | 2 | Numeric | Kode Jenis Surat (01-63) | 01 (Studi dan Survey) |
| Kode Subjenis | 3 | Numeric | Kode Subjenis Surat | 03 (Studi Kelayakan) |
| Nomor Urut | 4 | Numeric | Urutan pembuatan surat | 001, 002, ..., 999 |

#### 1.2 Penjelasan Setiap Komponen

**A. Kode Unit/Departemen (Posisi 1)**

Merepresentasikan unit atau departemen yang mengeluarkan surat:

| Kode | Unit/Departemen | Deskripsi |
|---|---|---|
| OT | Organisasi dan Transformasi | Divisi Organisasi dan Transformasi |
| KU | Keuangan | Divisi Keuangan dan Akuntansi |
| OM | Operasional dan Pemeliharaan | Divisi Operasional |
| PG | Pengadaan | Divisi Pengadaan dan Supply Chain |
| HC | Human Capital | Divisi Sumber Daya Manusia |
| TI | Teknologi Informasi | Divisi IT dan Sistem Informasi |
| HK | Hukum | Divisi Legal dan Compliance |
| (Dapat dikembangkan sesuai struktur organisasi) | - | - |

**Catatan**: Kode unit mengacu pada Klasifikasi Arsip dengan prefix singkat yang disesuaikan.

**B. Kode Jenis Surat (Posisi 2)**

Identifikasi dari tabel `jenis_surat` (ID 1-63):

**Contoh Pemetaan**:
- 01 = Studi dan Survey
- 02 = Rencana Teknik
- 12 = Konsultasi dan Advise
- 22 = Penyusunan Anggaran
- 52 = Employee Capacity Planning
- dst.

**Catatan**: Format numeric 2-digit, dengan zero-padding jika perlu (contoh: 03 untuk ID 3)

**C. Kode Subjenis Surat (Posisi 3)**

Identifikasi dari tabel `subjenis_surat`:

**Contoh Pemetaan untuk Jenis 1 (Studi dan Survey)**:
- 01 = Pra Studi Kelayakan Jalan Tol
- 03 = Studi Kelayakan (Feasibility Study)
- 04 = Studi AMDAL
- 10 = Laporan Hasil Survey

**Validasi**:
- Kode Subjenis harus terdaftar di tabel `subjenis_surat`
- Harus sesuai dengan parent `jenis_surat` yang dipilih
- Format numeric 2-digit

**D. Nomor Urut (Posisi 4)**

**Deskripsi**: Angka urutan yang bertambah satu setiap ada surat keluar baru dari kombinasi unit-jenis-subjenis yang sama.

**Karakteristik**:
- Format numeric: 3-digit, zero-padded (001, 002, ..., 999)
- Reset atau increment tergantung pada strategi penomoran
- **Scope Penghitung**: Per kombinasi (Unit + Jenis Surat + Subjenis Surat)
- Range: 001 - 999

**Strategi Reset**:
- **Opsi A (Recommended)**: Reset per tahun kalender (setiap 1 Januari)
- **Opsi B**: Continue dari tahun sebelumnya (no reset)
- **Opsi C**: Reset per periode tertentu sesuai kebijakan

#### 1.3 Contoh Implementasi Surat Keluar

**Contoh 1: Studi Kelayakan dari Divisi Organisasi**
```
Surat Keluar Pertama:
  Kode Unit: OT (Organisasi dan Transformasi)
  Jenis Surat: 01 (Studi dan Survey)
  Subjenis: 03 (Studi Kelayakan)
  Nomor Urut: 001 (Pertama dari kombinasi OT.01.03)
  
  NOMOR SURAT: OT.01.03.001
```

**Contoh 2: Surat Keluar Berikutnya (Jenis Sama)**
```
Surat Keluar Kedua dari kombinasi yang sama:
  NOMOR SURAT: OT.01.03.002
```

**Contoh 3: Jenis Surat Berbeda dari Unit Sama**
```
Surat Keluar dari jenis surat berbeda:
  Kode Unit: OT (Organisasi dan Transformasi)
  Jenis Surat: 22 (Penyusunan Anggaran)
  Subjenis: 01 (RKAP)
  Nomor Urut: 001 (Reset karena jenis berbeda)
  
  NOMOR SURAT: OT.22.01.001
```

**Contoh 4: Unit Berbeda**
```
Surat Keluar dari unit lain:
  Kode Unit: KU (Keuangan)
  Jenis Surat: 01 (Studi dan Survey)
  Subjenis: 03 (Studi Kelayakan)
  Nomor Urut: 001 (Reset karena unit berbeda)
  
  NOMOR SURAT: KU.01.03.001
```

#### 1.4 Logika Database Penomoran Surat Keluar

**Tabel Terkait**: `office` (Surat Keluar)

**Kolom-Kolom Penting**:
```sql
- id: INT (Primary Key)
- nomor_surat_surat_keluar: VARCHAR(50) - Nomor lengkap surat keluar
- jenis_surat_id: BIGINT - Foreign Key ke jenis_surat
- subjenis_surat_id: INT - Foreign Key ke subjenis_surat
- dari_tipe: VARCHAR(50) - Kode unit yang mengeluarkan
- tanggal_surat: DATE - Tanggal penerbitan
- sequence_number: INT - Urutan global (untuk internal tracking)
- is_validated: TINYINT - Status validasi
- created_at: TIMESTAMP
```

**Algoritma Pemberian Nomor**:

```pseudocode
FUNCTION generate_surat_keluar_number(unit_code, jenis_id, subjenis_id, tahun):
  // 1. Tentukan kombinasi kunci
  composite_key = CONCAT(unit_code, '.', jenis_id, '.', subjenis_id)
  
  // 2. Query terakhir nomor yang ada
  last_sequence = GET_MAX_SEQUENCE(
    FROM office 
    WHERE dari_tipe = unit_code 
      AND jenis_surat_id = jenis_id 
      AND subjenis_surat_id = subjenis_id 
      AND YEAR(tanggal_surat) = tahun
  )
  
  // 3. Increment nomor urut
  next_sequence = IF(last_sequence IS NULL, 1, last_sequence + 1)
  
  // 4. Validasi tidak melebihi batas
  IF next_sequence > 999:
    RAISE ERROR "Nomor urut telah mencapai batas maksimal (999)"
  
  // 5. Format nomor lengkap
  formatted_sequence = ZERO_PADDED(next_sequence, 3)
  full_number = CONCAT(composite_key, '.', formatted_sequence)
  
  // 6. Return hasil
  RETURN full_number
```

---

### 2. Format Penomoran Surat Keputusan Direksi

#### 2.1 Spesifikasi Format

**Format Umum**: `xxx/KPTS-JPT/yyyy`

**Struktur Komponen**:

| Komponen | Posisi | Tipe | Deskripsi | Range/Format |
|---|---|---|---|---|
| Nomor Urut | 1 | Numeric | Urutan pembuatan surat keputusan dalam setahun | 001-999 |
| Identifikasi | 2 | Alphanumeric | Kode tetap identitas dokumen | KPTS-JPT (Keputusan Pimpinan PT Jasamarga) |
| Tahun Terbit | 3 | Numeric | Tahun penerbitan surat keputusan | 4-digit (2024, 2025, dst) |

#### 2.2 Penjelasan Setiap Komponen

**A. Nomor Urut (Posisi 1)**

**Deskripsi**: Angka urutan yang bertambah satu setiap ada surat keputusan direksi baru.

**Karakteristik**:
- Format numeric: 3-digit, zero-padded (001, 002, ..., 999)
- **Scope Penghitung**: Per tahun kalender
- **Reset Policy**: Automatic reset setiap 1 Januari
- Range: 001 - 999

**Logika Penghitung**:
- Surat keputusan pertama di tahun 2025: 001/KPTS-JPT/2025
- Surat keputusan kedua di tahun 2025: 002/KPTS-JPT/2025
- Surat keputusan pertama di tahun 2026: 001/KPTS-JPT/2026

**B. Identifikasi Surat (Posisi 2)**

**Kode Tetap**: `KPTS-JPT`

**Kepanjangan**:
- KPTS = Keputusan Pimpinan (Kepusan Pimpinan)
- JPT = Jasamarga Pandaan Tol (atau dapat diubah sesuai entitas)

**Catatan**: Kode ini tetap tidak berubah untuk semua surat keputusan dan berfungsi sebagai identifikasi jenis dokumen.

**C. Tahun Terbit (Posisi 3)**

**Format**: 4-digit tahun (YYYY)

**Deskripsi**: Tahun penerbitan surat keputusan

**Logika**:
- Menggunakan tahun dari field `tanggal_surat` atau `created_at`
- Harus sesuai dengan tahun kalender saat ini
- Konsisten dengan kebijakan dokumen resmi organisasi

#### 2.3 Contoh Implementasi Surat Keputusan Direksi

**Contoh 1: Keputusan Pertama Tahun 2025**
```
Surat Keputusan Direksi - Penetapan Kebijakan Operasional:
  Nomor Urut: 001
  Identifikasi: KPTS-JPT
  Tahun: 2025
  
  NOMOR SURAT: 001/KPTS-JPT/2025
  
  Deskripsi Dokumen: 
    "Keputusan Direksi PT Jasamarga Pandaan Tol Nomor 001/KPTS-JPT/2025 
     Tentang Penetapan Kebijakan Operasional Tahun 2025"
```

**Contoh 2: Keputusan Kedua Tahun 2025**
```
Surat Keputusan Direksi - Persetujuan Anggaran:
  NOMOR SURAT: 002/KPTS-JPT/2025
```

**Contoh 3: Keputusan Tahun 2026 (Automatic Reset)**
```
Surat Keputusan Direksi - Penetapan Tarif Tol:
  NOMOR SURAT: 001/KPTS-JPT/2026
  
  Catatan: Nomor urut reset ke 001 karena tahun berubah
```

**Contoh 4: Keputusan di Akhir Tahun**
```
Surat Keputusan Direksi - Evaluasi Kinerja Tahun 2025:
  NOMOR SURAT: 152/KPTS-JPT/2025
  
  Catatan: Kemungkinan ada 152 surat keputusan dalam tahun 2025
```

#### 2.4 Logika Database Penomoran Surat Keputusan

**Tabel Terkait**: `office` (Surat Keputusan Direksi) atau `surat_keputusan` (jika terpisah)

**Kolom-Kolom Penting**:
```sql
- id: INT (Primary Key)
- nomor_surat_surat_keputusan_direksi: VARCHAR(50) - Nomor lengkap
- tanggal_surat: DATE - Tanggal penerbitan
- dari_tipe: VARCHAR(50) - 'KEPUTUSAN' atau identifier lain
- tipe_surat_id: INT - Reference ke tipe surat (Keputusan Direksi)
- is_validated: TINYINT
- created_at: TIMESTAMP
```

**Algoritma Pemberian Nomor**:

```pseudocode
FUNCTION generate_surat_keputusan_number(tahun):
  // 1. Query terakhir nomor keputusan di tahun yang sama
  last_sequence = GET_MAX_SEQUENCE(
    FROM office 
    WHERE YEAR(tanggal_surat) = tahun 
      AND tipe_surat = 'KEPUTUSAN_DIREKSI'
  )
  
  // 2. Increment nomor urut
  next_sequence = IF(last_sequence IS NULL, 1, last_sequence + 1)
  
  // 3. Validasi tidak melebihi batas
  IF next_sequence > 999:
    RAISE ERROR "Nomor urut keputusan direksi telah mencapai batas (999)"
  
  // 4. Format nomor lengkap
  formatted_sequence = ZERO_PADDED(next_sequence, 3)
  full_number = CONCAT(formatted_sequence, '/KPTS-JPT/', tahun)
  
  // 5. Return hasil
  RETURN full_number
```

---

### 3. Format Penomoran Surat Edaran Direksi

#### 3.1 Spesifikasi Format

**Format Umum**: `xxx/SE-DIR/yyyy`

**Struktur Komponen**:

| Komponen | Posisi | Tipe | Deskripsi | Range/Format |
|---|---|---|---|---|
| Nomor Urut | 1 | Numeric | Urutan pembuatan surat edaran dalam setahun | 001-999 |
| Identifikasi | 2 | Alphanumeric | Kode tetap identitas dokumen | SE-DIR (Surat Edaran Direksi) |
| Tahun Terbit | 3 | Numeric | Tahun penerbitan surat edaran | 4-digit (2024, 2025, dst) |

#### 3.2 Penjelasan Setiap Komponen

**A. Nomor Urut (Posisi 1)**

**Deskripsi**: Angka urutan yang bertambah satu setiap ada surat edaran direksi baru.

**Karakteristik**:
- Format numeric: 3-digit, zero-padded (001, 002, ..., 999)
- **Scope Penghitung**: Per tahun kalender
- **Reset Policy**: Automatic reset setiap 1 Januari
- Range: 001 - 999

**Logika Penghitung**:
- Surat edaran pertama di tahun 2025: 001/SE-DIR/2025
- Surat edaran kedua di tahun 2025: 002/SE-DIR/2025
- Surat edaran pertama di tahun 2026: 001/SE-DIR/2026

**B. Identifikasi Surat (Posisi 2)**

**Kode Tetap**: `SE-DIR`

**Kepanjangan**:
- SE = Surat Edaran
- DIR = Direksi

**Catatan**: Kode ini tetap tidak berubah untuk semua surat edaran dan berfungsi sebagai identifikasi jenis dokumen yang berbeda dari keputusan.

**Perbedaan SE-DIR vs KPTS-JPT**:
- KPTS = Keputusan (lebih formal, mengikat), biasanya ada perubahan kebijakan permanen
- SE-DIR = Edaran (panduan/instruksi), biasanya bersifat temporal atau prosedural

**C. Tahun Terbit (Posisi 3)**

**Format**: 4-digit tahun (YYYY)

**Deskripsi**: Tahun penerbitan surat edaran

**Logika**: Sama dengan penomoran Surat Keputusan

#### 3.3 Contoh Implementasi Surat Edaran Direksi

**Contoh 1: Edaran Pertama Tahun 2025**
```
Surat Edaran Direksi - Instruksi Penyusunan Rencana Kerja:
  Nomor Urut: 001
  Identifikasi: SE-DIR
  Tahun: 2025
  
  NOMOR SURAT: 001/SE-DIR/2025
  
  Deskripsi Dokumen: 
    "Surat Edaran Direksi Nomor 001/SE-DIR/2025 
     Tentang Instruksi Penyusunan Rencana Kerja dan Anggaran Tahunan"
```

**Contoh 2: Edaran Kebijakan Operasional**
```
Surat Edaran Direksi - Kebijakan Penghematan Energi:
  NOMOR SURAT: 005/SE-DIR/2025
  
  Deskripsi: 
    "Surat Edaran Direksi yang mengatur kebijakan penghematan energi 
     untuk semua unit operasional dalam tahun 2025"
```

**Contoh 3: Edaran Tahun 2026**
```
Surat Edaran Direksi - Panduan Pelaksanaan Program Bina Lingkungan:
  NOMOR SURAT: 001/SE-DIR/2026
  
  Catatan: Reset ke 001 karena tahun berubah
```

#### 3.4 Logika Database Penomoran Surat Edaran

**Tabel Terkait**: `office` (Surat Edaran Direksi) atau `surat_edaran` (jika terpisah)

**Kolom-Kolom Penting**:
```sql
- id: INT (Primary Key)
- nomor_surat_surat_edaran_direksi: VARCHAR(50) - Nomor lengkap
- tanggal_surat: DATE - Tanggal penerbitan
- dari_tipe: VARCHAR(50) - 'EDARAN' atau identifier lain
- tipe_surat_id: INT - Reference ke tipe surat (Edaran Direksi)
- is_validated: TINYINT
- created_at: TIMESTAMP
```

**Algoritma Pemberian Nomor**:

```pseudocode
FUNCTION generate_surat_edaran_number(tahun):
  // 1. Query terakhir nomor edaran di tahun yang sama
  last_sequence = GET_MAX_SEQUENCE(
    FROM office 
    WHERE YEAR(tanggal_surat) = tahun 
      AND tipe_surat = 'EDARAN_DIREKSI'
  )
  
  // 2. Increment nomor urut
  next_sequence = IF(last_sequence IS NULL, 1, last_sequence + 1)
  
  // 3. Validasi tidak melebihi batas
  IF next_sequence > 999:
    RAISE ERROR "Nomor urut surat edaran telah mencapai batas (999)"
  
  // 4. Format nomor lengkap
  formatted_sequence = ZERO_PADDED(next_sequence, 3)
  full_number = CONCAT(formatted_sequence, '/SE-DIR/', tahun)
  
  // 5. Return hasil
  RETURN full_number
```

---

## 4.6.2 Fitur Backdate dan Validasi Nomor Surat

### Definisi dan Tujuan

**Backdate**: Mekanisme yang memungkinkan pengguna untuk mengubah atau menetapkan ulang bagian-bagian tertentu dari nomor surat sesuai dengan aturan yang telah ditentukan.

**Tujuan**:
1. Memungkinkan koreksi nomor surat yang keliru saat pembuatan
2. Mengakomodasi surat yang dibuat pada tanggal berbeda dengan tanggal penerbitan resmi
3. Memberikan fleksibilitas administratif dalam pengelolaan nomor surat
4. Mendukung proses retroaktif dokumentasi

---

### 1. Mekanisme Backdate Surat Keluar

#### 1.1 Komponen yang Dapat Diubah

**Hanya Nomor Urut yang Dapat Dimodifikasi**

Untuk surat keluar dengan format `OT.01.03.xxx`, **hanya komponen xxx (nomor urut) yang dapat diubah** melalui fitur backdate.

| Komponen | Dapat Diubah? | Alasan |
|---|---|---|
| Kode Unit (OT) | ❌ Tidak | Menunjukkan unit penerbit, tidak boleh diubah setelah pembuatan |
| Kode Jenis (01) | ❌ Tidak | Menunjukkan jenis surat, harus konsisten dengan data surat |
| Kode Subjenis (03) | ❌ Tidak | Menunjukkan tipe spesifik surat, tidak boleh diubah |
| Nomor Urut (xxx) | ✅ Ya | Dapat disesuaikan jika ada kesalahan atau keperluan admin |

#### 1.2 Validasi Backdate Surat Keluar

**Validasi I: Duplikasi Nomor**

```pseudocode
FUNCTION validate_backdate_surat_keluar(unit_code, jenis_id, subjenis_id, new_sequence, tahun):
  // 1. Check apakah nomor yang diajukan sudah ada
  existing = GET_SURAT_KELUAR(
    FROM office 
    WHERE dari_tipe = unit_code 
      AND jenis_surat_id = jenis_id 
      AND subjenis_surat_id = subjenis_id 
      AND YEAR(tanggal_surat) = tahun
      AND nomor_urut = new_sequence
      AND id != current_surat_id  // Exclude surat yang sedang diedit
  )
  
  IF existing IS NOT NULL:
    RAISE ERROR "Nomor surat OT." + jenis_id + "." + subjenis_id + "." 
                + new_sequence + " sudah digunakan pada tanggal " 
                + existing.tanggal_surat + ". Silakan gunakan nomor lain."
    RETURN FALSE
  
  RETURN TRUE
```

**Validasi II: Range Nomor Urut**

```pseudocode
FUNCTION validate_sequence_range(new_sequence):
  // 1. Check range valid
  IF new_sequence < 1 OR new_sequence > 999:
    RAISE ERROR "Nomor urut harus berada di antara 001-999"
    RETURN FALSE
  
  RETURN TRUE
```

**Validasi III: Logical Sequence (Optional)**

```pseudocode
FUNCTION validate_logical_sequence(new_sequence, current_sequence, tahun):
  // Optional: Warning jika nomor mundur
  IF new_sequence < current_sequence:
    SHOW WARNING "Anda mengubah nomor urut dari " + current_sequence 
                 + " menjadi " + new_sequence + " (mundur). "
                 + "Pastikan ini adalah tindakan yang disengaja."
  
  RETURN TRUE
```

#### 1.3 Contoh Kasus Backdate Surat Keluar

**Kasus 1: Koreksi Kesalahan Penomoran**

```
Situasi:
  - User membuat surat dengan nomor OT.01.03.010
  - Ternyata nomor 010 sudah digunakan
  - Sistem akan reject: "Nomor OT.01.03.010 sudah digunakan pada [tanggal]"
  
Solusi (Backdate):
  - User dapat mengubah nomor urut dari 010 menjadi 011
  - Hasil: OT.01.03.011 (jika tidak ada duplikasi)
  - Sistem akan memvalidasi bahwa 011 belum digunakan
```

**Kasus 2: Surat Tertanggal Mundur**

```
Situasi:
  - Surat dibuat hari ini (22 Desember 2025) dengan nomor 050/...
  - Tetapi surat ini ternyata harus didatangkan mundur ke 15 Desember 2025
  
Proses:
  - User set tanggal_surat = 15 Desember 2025
  - Sistem akan check: berapa nomor urut yang seharusnya pada 15 Desember?
  - Jika pada 15 Desember sudah ada 035 surat, maka surat ini harus nomor 036
  - User dapat mengubah nomor sesuai dengan validasi sistem
```

**Kasus 3: Pengembalian Dokumen untuk Revisi**

```
Situasi:
  - Surat nomor OT.01.03.020 sudah diterbitkan
  - Dokumen perlu direvisi dan diterbitkan ulang
  
Opsi A (Backdate tidak digunakan):
  - Ambil nomor berikutnya: OT.01.03.021
  
Opsi B (Backdate digunakan):
  - Jika perlu sama dengan nomor asli, gunakan backdate
  - Validasi: Pastikan nomor 020 tidak lagi digunakan di surat lain
```

---

### 2. Mekanisme Backdate Surat Keputusan Direksi

#### 2.1 Komponen yang Dapat Diubah

**Nomor Urut dan Tahun Dapat Dimodifikasi**

Untuk surat keputusan dengan format `xxx/KPTS-JPT/yyyy`, **nomor urut (xxx) dan tahun (yyyy) dapat diubah**, sedangkan identifikasi tetap.

| Komponen | Dapat Diubah? | Alasan |
|---|---|---|
| Nomor Urut (xxx) | ✅ Ya | Dapat disesuaikan jika ada kesalahan penomoran |
| Identifikasi (KPTS-JPT) | ❌ Tidak | Tetap adalah identifikasi keputusan direksi |
| Tahun (yyyy) | ✅ Ya | Dapat disesuaikan dengan tahun penerbitan resmi |

#### 2.2 Validasi Backdate Surat Keputusan

**Validasi I: Duplikasi Nomor per Tahun**

```pseudocode
FUNCTION validate_backdate_surat_keputusan(new_sequence, new_tahun):
  // 1. Check apakah nomor yang diajukan sudah ada di tahun tersebut
  existing = GET_SURAT_KEPUTUSAN(
    FROM office 
    WHERE tipe_surat = 'KEPUTUSAN_DIREKSI'
      AND YEAR(tanggal_surat) = new_tahun
      AND nomor_urut = new_sequence
      AND id != current_surat_id
  )
  
  IF existing IS NOT NULL:
    RAISE ERROR "Nomor keputusan " + new_sequence + "/KPTS-JPT/" + new_tahun 
                + " sudah digunakan. Silakan gunakan nomor lain atau tahun lain."
    RETURN FALSE
  
  RETURN TRUE
```

**Validasi II: Range Nomor dan Tahun**

```pseudocode
FUNCTION validate_sequence_and_tahun_range(new_sequence, new_tahun):
  // 1. Check range nomor urut
  IF new_sequence < 1 OR new_sequence > 999:
    RAISE ERROR "Nomor urut harus berada di antara 001-999"
    RETURN FALSE
  
  // 2. Check tahun valid
  current_year = YEAR(NOW())
  IF new_tahun < 2000 OR new_tahun > current_year + 5:
    RAISE ERROR "Tahun harus berada di antara 2000 dan " + (current_year + 5)
    RETURN FALSE
  
  RETURN TRUE
```

**Validasi III: Konsistensi Tahun dengan Tanggal Surat**

```pseudocode
FUNCTION validate_tahun_consistency(new_tahun, tanggal_surat):
  // Optional: Warn jika tahun tidak sesuai dengan tanggal surat
  surat_tahun = YEAR(tanggal_surat)
  
  IF new_tahun != surat_tahun:
    SHOW WARNING "Tahun yang Anda masukkan (" + new_tahun + ") tidak sesuai " 
                 + "dengan tahun tanggal surat (" + surat_tahun + "). "
                 + "Pastikan ini adalah tindakan yang disengaja."
  
  RETURN TRUE
```

#### 2.3 Contoh Kasus Backdate Surat Keputusan

**Kasus 1: Koreksi Nomor Urut**

```
Situasi:
  - Surat keputusan dibuat dengan nomor 025/KPTS-JPT/2025
  - Ternyata nomor 025 sudah digunakan
  - Sistem akan reject duplikasi
  
Solusi (Backdate):
  - User mengubah nomor urut dari 025 menjadi 026
  - Hasil: 026/KPTS-JPT/2025
  - Sistem memvalidasi bahwa 026 belum digunakan pada tahun 2025
```

**Kasus 2: Keputusan Retroaktif (Backdate Tahun)**

```
Situasi:
  - Surat keputusan dibuat hari ini (22 Desember 2025)
  - Tetapi keputusan ini retroaktif berlaku sejak 2024
  
Proses Backdate:
  - User mengubah tahun dari 2025 menjadi 2024
  - System determine: nomor apa yang seharusnya pada tahun 2024?
  - Jika ada 150 keputusan di tahun 2024, nomor harus 151
  - User dapat set nomor sesuai dengan validasi
  - Hasil: 151/KPTS-JPT/2024
```

**Kasus 3: Keputusan dari Tahun Sebelumnya Baru Diterbitkan**

```
Situasi:
  - Keputusan sebenarnya dibuat 30 Januari 2025
  - Tetapi baru diterbitkan secara resmi 15 Desember 2025
  - Harus menggunakan nomor 2025, bukan 2024
  
Proses:
  - tanggal_surat = 30 Januari 2025 → tahun = 2025
  - nomor_surat = 025/KPTS-JPT/2025 (urutan ke-25 di tahun 2025)
  - Backdate tidak mengubah tahun karena sudah tepat
```

---

### 3. Mekanisme Backdate Surat Edaran Direksi

#### 3.1 Komponen yang Dapat Diubah

**Nomor Urut dan Tahun Dapat Dimodifikasi**

Untuk surat edaran dengan format `xxx/SE-DIR/yyyy`, **nomor urut (xxx) dan tahun (yyyy) dapat diubah**, sedangkan identifikasi tetap.

| Komponen | Dapat Diubah? | Alasan |
|---|---|---|
| Nomor Urut (xxx) | ✅ Ya | Dapat disesuaikan jika ada kesalahan penomoran |
| Identifikasi (SE-DIR) | ❌ Tidak | Tetap adalah identifikasi surat edaran direksi |
| Tahun (yyyy) | ✅ Ya | Dapat disesuaikan dengan tahun penerbitan resmi |

#### 3.2 Validasi Backdate Surat Edaran

**Validasi sama dengan Surat Keputusan**, dengan perbedaan pada tipe surat dan identifikasi:

```pseudocode
FUNCTION validate_backdate_surat_edaran(new_sequence, new_tahun):
  // 1. Check apakah nomor yang diajukan sudah ada di tahun tersebut
  existing = GET_SURAT_EDARAN(
    FROM office 
    WHERE tipe_surat = 'EDARAN_DIREKSI'
      AND YEAR(tanggal_surat) = new_tahun
      AND nomor_urut = new_sequence
      AND id != current_surat_id
  )
  
  IF existing IS NOT NULL:
    RAISE ERROR "Nomor edaran " + new_sequence + "/SE-DIR/" + new_tahun 
                + " sudah digunakan. Silakan gunakan nomor lain atau tahun lain."
    RETURN FALSE
  
  // 2. Validasi range
  IF NOT validate_sequence_and_tahun_range(new_sequence, new_tahun):
    RETURN FALSE
  
  RETURN TRUE
```

#### 3.3 Contoh Kasus Backdate Surat Edaran

**Kasus 1: Edaran Operasional Tahun Baru**

```
Situasi:
  - Edaran tentang kebijakan tahun 2026 dibuat 20 Desember 2025
  - Tetapi harus dinomori dengan tahun 2026
  
Proses:
  - User mengubah tahun dari 2025 menjadi 2026
  - Nomor urut akan dimulai dari 001 (reset tahun)
  - Hasil: 001/SE-DIR/2026
```

**Kasus 2: Edaran Penting yang Tertinggal**

```
Situasi:
  - Ada edaran penting yang lupa dinomori saat penerbitan (bulan Juni 2025)
  - Baru diingat tanggal 22 Desember 2025
  
Proses:
  - Cek: berapa nomor edaran terakhir di tahun 2025 sampai sekarang?
  - Misal sudah ada 48 edaran, maka edaran ini harus 049/SE-DIR/2025
  - User dapat set nomor sesuai dengan backdate validasi
```

---

### 4. Sistem Validasi Umum Duplikasi Nomor

#### 4.1 Implementasi Validasi Duplikasi

**Tujuan**: Memastikan tidak ada nomor surat ganda dalam sistem.

**Tingkat Validasi**:

| Level | Jenis Surat | Scope | Keterangan |
|---|---|---|---|
| 1 | Surat Keluar | Per kombinasi (Unit + Jenis + Subjenis + Tahun) | Nomor tidak boleh sama dalam scope ini |
| 2 | Keputusan Direksi | Per tahun | Nomor tidak boleh sama di tahun yang sama |
| 3 | Edaran Direksi | Per tahun | Nomor tidak boleh sama di tahun yang sama |

#### 4.2 SQL Query Validasi

**Validasi Duplikasi Surat Keluar**:

```sql
-- Check apakah nomor surat keluar sudah ada
SELECT COUNT(*) as count
FROM office
WHERE dari_tipe = ?
  AND jenis_surat_id = ?
  AND subjenis_surat_id = ?
  AND YEAR(tanggal_surat) = ?
  AND nomor_surat_surat_keluar = ?
  AND id != ? -- Exclude surat yang sedang diedit
  AND deleted_at IS NULL;

-- Jika count > 0, maka nomor sudah digunakan
```

**Validasi Duplikasi Keputusan Direksi**:

```sql
-- Check apakah nomor keputusan sudah ada di tahun tersebut
SELECT COUNT(*) as count
FROM office
WHERE YEAR(tanggal_surat) = ?
  AND nomor_surat_surat_keputusan_direksi = ?
  AND id != ? -- Exclude surat yang sedang diedit
  AND deleted_at IS NULL
  AND tipe_surat = 'KEPUTUSAN_DIREKSI';

-- Jika count > 0, maka nomor sudah digunakan
```

**Validasi Duplikasi Edaran Direksi**:

```sql
-- Check apakah nomor edaran sudah ada di tahun tersebut
SELECT COUNT(*) as count
FROM office
WHERE YEAR(tanggal_surat) = ?
  AND nomor_surat_surat_edaran_direksi = ?
  AND id != ? -- Exclude surat yang sedang diedit
  AND deleted_at IS NULL
  AND tipe_surat = 'EDARAN_DIREKSI';

-- Jika count > 0, maka nomor sudah digunakan
```

#### 4.3 Alert dan Error Message

**Error Message untuk Duplikasi**:

```
Surat Keluar:
  "Nomor surat OT.01.03.050 sudah digunakan pada tanggal 15 November 2025. 
   Silakan gunakan nomor lain atau ubah unit/jenis/subjenis surat."

Keputusan Direksi:
  "Nomor keputusan 025/KPTS-JPT/2025 sudah digunakan. 
   Silakan gunakan nomor berbeda untuk tahun ini."

Edaran Direksi:
  "Nomor edaran 012/SE-DIR/2025 sudah digunakan. 
   Silakan gunakan nomor berbeda untuk tahun ini."
```

---

## 4.6.3 Integrasi Sistem Penomoran

### 1. Flow Proses Pemberian Nomor Otomatis

**Alur Umum Pemberian Nomor:**

Ketika user membuat surat baru, sistem mengikuti alur berikut:

1. **User Memilih Jenis Surat**: User memilih tipe surat (Surat Keluar, Keputusan Direksi, atau Edaran Direksi)

2. **Sistem Menentukan Nomor Otomatis**: Berdasarkan jenis surat yang dipilih, sistem otomatis menentukan nomor:
   - Untuk Surat Keluar: Mengambil Unit, Jenis, Subjenis, dan menghitung urutan berikutnya
   - Untuk Keputusan/Edaran: Mengambil tahun saat ini dan menghitung urutan berikutnya di tahun tersebut

3. **Nomor Ditampilkan di Form (Draft)**:
   - Nomor ditampilkan dalam field read-only sebagai referensi
   - Nomor ini bersifat draft dan dapat berubah jika tanggal surat diubah

4. **User Dapat Mengaktifkan Backdate**: 
   - User dapat mencentang checkbox "Ubah nomor..." jika diperlukan
   - Jika checkbox dicentang, field input nomor baru menjadi visible
   - User dapat memodifikasi nomor sesuai aturan (hanya bagian tertentu yang dapat diubah)

5. **Validasi Duplikasi (Pre-save)**:
   - Sebelum surat disimpan final, sistem melakukan validasi:
     - Nomor urut tidak boleh sama dengan surat lain dalam scope yang sama
     - Untuk Surat Keluar: Scope adalah kombinasi Unit+Jenis+Subjenis+Tahun
     - Untuk Keputusan/Edaran: Scope adalah tahun yang sama
   - Jika ada duplikasi, sistem menampilkan error message dan reject penyimpanan
   - Jika valid, surat dapat disimpan dengan nomor final

6. **Penyimpanan**: Nomor final disimpan di database bersama data surat lainnya

### 2. User Interface untuk Backdate

**Deskripsi Umum:**

Fitur backdate diimplementasikan melalui antarmuka pengguna yang intuitif dengan mekanisme sebagai berikut:

#### 2.1 Komponen UI Untuk Surat Keluar

Untuk surat keluar, form harus menyediakan:

1. **Bidang Nomor Otomatis (Read-only)**
   - Menampilkan nomor surat yang auto-generated (contoh: OT.01.03.001)
   - Bidang ini tidak dapat diedit, hanya untuk referensi visual
   - Teks keterangan: "Nomor ini auto-generated berdasarkan Unit, Jenis, dan Subjenis Surat"

2. **Checkbox Aktivasi Backdate**
   - Labeled: "Ubah nomor urut surat (jika diperlukan)"
   - Ketika checkbox dicentang, section input nomor urut baru akan tampil
   - Ketika checkbox tidak dicentang, section input tersembunyi dan nomor kembali ke otomatis
   - Teks bantuan: "Centang ini jika Anda ingin mengubah angka urutan nomor surat"

3. **Input Nomor Urut Baru (Conditional)**
   - Hanya tampil ketika checkbox backdate dicentang
   - Type: numeric input dengan range 001-999
   - Placeholder: "Masukkan nomor urut baru"
   - Validasi real-time ketika nilai berubah

4. **Area Pesan Error (Conditional)**
   - Tampil di bawah input nomor urut jika ada error
   - Menampilkan pesan validasi (contoh: "Nomor OT.01.03.050 sudah digunakan")

5. **Bidang Nomor Final (Read-only)**
   - Menampilkan nomor surat hasil final (dengan atau tanpa backdate)
   - Bidang ini auto-update berdasarkan input user
   - Background highlight untuk menonjolkan hasil final
   - Format bold untuk visibility

#### 2.2 Komponen UI Untuk Keputusan Direksi

Untuk keputusan direksi, form harus menyediakan struktur serupa dengan variasi:

1. **Bidang Nomor Otomatis (Read-only)**
   - Menampilkan nomor keputusan yang auto-generated (contoh: 001/KPTS-JPT/2025)
   - Teks keterangan: "Nomor ini auto-generated berdasarkan tahun penerbitan"

2. **Checkbox Aktivasi Backdate**
   - Labeled: "Ubah nomor urut atau tahun keputusan (jika diperlukan)"
   - Ketika dicentang, section input nomor urut dan tahun tampil
   - Teks bantuan: "Centang ini jika Anda ingin mengubah nomor urut atau tahun keputusan"

3. **Input Nomor Urut Baru (Conditional)**
   - Hanya tampil ketika checkbox backdate dicentang
   - Type: numeric input dengan range 001-999
   - Label: "Nomor Urut (001-999)"

4. **Input Tahun Baru (Conditional)**
   - Hanya tampil ketika checkbox backdate dicentang
   - Type: numeric input dengan range 2000 - tahun sekarang + 5 tahun
   - Label: "Tahun (YYYY)"
   - Default value: tahun saat ini

5. **Area Pesan Error (Conditional)**
   - Tampil jika ada error validasi pada nomor urut atau tahun
   - Contoh pesan: "Nomor 025/KPTS-JPT/2025 sudah digunakan"

6. **Bidang Nomor Final (Read-only)**
   - Menampilkan nomor keputusan hasil final
   - Auto-update berdasarkan nomor urut dan tahun yang diinput
   - Format bold dan highlight untuk visibility

#### 2.3 Logika Interaksi UI

**Untuk Surat Keluar:**

1. Ketika user pertama kali membuka form, checkbox backdate tidak dicentang dan input nomor urut tersembunyi
2. Nomor otomatis ditampilkan dan nomor final menunjukkan nomor otomatis
3. Ketika user mencentang checkbox backdate:
   - Section input nomor urut baru menjadi visible
   - Input field menerima nilai numerik 001-999
   - Ketika user mengubah input nomor urut, sistem melakukan:
     - Validasi range (001-999)
     - Validasi duplikasi melalui API call ke server
     - Jika valid, nomor final otomatis update
     - Jika tidak valid, tampilkan pesan error dan jangan update nomor final
4. Ketika user unchecked checkbox backdate:
   - Section input tersembunyi
   - Input value direset
   - Nomor final kembali ke nomor otomatis

**Untuk Keputusan Direksi:**

1. Ketika user pertama kali membuka form, checkbox backdate tidak dicentang dan input nomor/tahun tersembunyi
2. Nomor otomatis ditampilkan dengan tahun saat ini
3. Ketika user mencentang checkbox backdate:
   - Section input nomor urut dan tahun menjadi visible
   - User dapat mengubah kedua field secara independent
   - Ketika user mengubah nomor urut atau tahun, sistem melakukan:
     - Validasi range untuk masing-masing field
     - Validasi duplikasi dengan kombinasi nomor urut dan tahun
     - Jika valid, nomor final otomatis update
     - Jika tidak valid, tampilkan pesan error untuk field yang bermasalah
4. Ketika user unchecked checkbox backdate:
   - Section input tersembunyi
   - Input values direset
   - Nomor final kembali ke nomor otomatis dengan tahun saat ini

**Validasi dan Error Handling:**

- Setiap kali user mengubah input backdate, sistem langsung melakukan validasi (tidak perlu menunggu submit)
- Pesan error ditampilkan secara real-time di bawah field yang bermasalah
- Nomor final hanya update jika semua validasi passed
- Jika ada error, nomor final tetap menampilkan nomor sebelumnya atau otomatis
- User dapat memperbaiki input hingga semua validasi passed

---

## 4.6.3 Integrasi Sistem Penomoran

---

## 4.6.4 Ringkasan Aturan Penomoran

### Tabel Ringkasan

| Aspek | Surat Keluar | Keputusan Direksi | Edaran Direksi |
|---|---|---|---|
| **Format** | OT.01.03.xxx | xxx/KPTS-JPT/yyyy | xxx/SE-DIR/yyyy |
| **Komponen Utama** | Unit.Jenis.Subjenis.Urut | Urut/Identifikasi/Tahun | Urut/Identifikasi/Tahun |
| **Scope Penghitung** | Per Unit+Jenis+Subjenis+Tahun | Per Tahun | Per Tahun |
| **Reset Policy** | Tahunan | Tahunan | Tahunan |
| **Dapat Dibackdate** | Nomor urut saja | Nomor urut & tahun | Nomor urut & tahun |
| **Validasi Utama** | Duplikasi per scope | Duplikasi per tahun | Duplikasi per tahun |
| **Max Nomor Urut** | 999 per scope | 999 per tahun | 999 per tahun |

---

## Kesimpulan

Sistem penomoran Jasa Marga dirancang dengan **fleksibilitas yang terkontrol**, memungkinkan:

✅ **Otomasi**: Nomor otomatis berdasarkan tipe surat
✅ **Validasi Ketat**: Mencegah duplikasi nomor
✅ **Backdate Terbatas**: Hanya komponen spesifik yang dapat diubah
✅ **Audit Trail**: Riwayat perubahan nomor tercatat
✅ **Kepatuhan Admin**: Sesuai dengan standar administrasi formal

Mekanisme ini memastikan setiap surat memiliki identitas unik, dapat dilacak, dan memenuhi persyaratan dokumentasi organisasi.

---

# 4.7 Template Surat: Struktur dan Pedoman

## Overview

Jasa Marga menggunakan tiga jenis template surat resmi, masing-masing dengan struktur, format, dan penggunaan yang spesifik. Bagian ini menjelaskan detail setiap template, termasuk komponen-komponen utama, format penulisan, dan panduan penggunaan.

---

## 4.7.1 Template Surat Keluar

### Definisi dan Tujuan

**Surat Keluar** adalah surat resmi yang dikeluarkan oleh unit/departemen kepada penerima eksternal maupun internal. Surat ini digunakan untuk:

- Komunikasi formal dengan mitra eksternal (vendor, pemerintah, organisasi lain)
- Pengajuan proposal, penawaran, atau permintaan informasi
- Permohonan, konfirmasi, atau pemberitahuan
- Laporan, hasil survey, atau dokumentasi teknis

**Karakteristik Umum:**
- Bersifat operasional dan administratif
- Dapat berasal dari berbagai unit/departemen
- Menggunakan penomoran dengan format `Unit.Jenis.Subjenis.Urut`
- Memiliki tanggal dan tanda tangan pengesah
- Dapat memiliki lampiran (dokumen pendukung, file, dll)

### Struktur Template Surat Keluar

Surat keluar terdiri dari bagian-bagian berikut dalam urutan top-to-bottom:

#### 1. **Header/Kop Surat**
- **Elemen yang ditampilkan:**
  - Logo Jasa Marga (centered, top)
  - Nama Perusahaan: "PT JASAMARGA PANDAAN TOL" atau unit terkait
  - Alamat lengkap kantor
  - Nomor telepon dan fax
  - Website (jika ada)
  - Garis pemisah horizontal

- **Spesifikasi:**
  - Ukuran font: konsisten dengan identitas visual perusahaan
  - Warna: sesuai brand guidelines Jasa Marga
  - Positioning: rata tengah untuk logo dan nama, rata kiri untuk alamat/kontak
  - Height: cukup untuk menampilkan semua elemen tanpa terlalu boros ruang

#### 2. **Nomor, Tempat, dan Tanggal Surat**
- **Posisi:** Atas kanan/kiri (tergantung standar internal)
- **Format:**
  ```
  Nomor    : OT.01.03.001
  Tempat   : Pandaan
  Tanggal  : 15 Desember 2025
  ```
- **Catatan:** 
  - Nomor mengikuti format penomoran surat keluar yang telah ditentukan
  - Tempat adalah lokasi kantor yang mengeluarkan surat
  - Tanggal menggunakan format: [Tanggal] [Bulan] [Tahun]

#### 3. **Tujuan Surat (Alamat Penerima)**
- **Posisi:** Bagian kiri, di bawah nomor/tanggal dengan jarak 1-2 baris
- **Format:**
  ```
  Kepada Yth. :
  [Nama/Instansi Penerima]
  [Alamat Lengkap Penerima]
  [Kota, Provinsi]
  ```
- **Spesifikasi:**
  - Label "Kepada Yth. :" (atau "Tujuan:") untuk penerima individual/jabatan
  - Gunakan "Kepada :" untuk organisasi/instansi
  - Nama penerima atau jabatan (tidak perlu nama jika sudah jelas)
  - Alamat lengkap jika pengiriman fisik diperlukan
  - Jarak dari kop surat: minimal 1,5 cm

#### 4. **Perihal/Subjek Surat**
- **Posisi:** Di bawah alamat penerima, dengan jarak 1-2 baris
- **Format:**
  ```
  Perihal : [Deskripsi singkat isi surat]
  ```
- **Spesifikasi:**
  - Singkat dan jelas (maksimal 1 baris, idealnya 5-10 kata)
  - Menceritakan isi utama surat tanpa terlalu detail
  - Contoh: "Penawaran Jasa Konsultasi Teknik Jalan Tol"
  - Contoh: "Permohonan Perpanjangan Kontrak Perawatan Aset"

#### 5. **Pembuka Surat (Salam Pembuka)**
- **Posisi:** Di bawah Perihal, dengan jarak 1-2 baris
- **Format:**
  ```
  Assalamu'alaikum Warahmatullahi Wabarakatuh
  
  Dengan hormat,
  ```
- **Spesifikasi:**
  - Salam islami atau salam formal standar
  - Diikuti dengan "Dengan hormat," atau "Salam hormat,"
  - Jarak: 1 baris kosong sebelum salam, 1 baris kosong setelah

#### 6. **Isi Surat (Body/Content)**
- **Posisi:** Di bawah pembuka
- **Struktur:**
  - **Paragraf Pembuka:** Penjelasan tujuan atau latar belakang surat (1-3 paragraf)
  - **Paragraf Inti:** Detail utama, permintaan, atau proposisi (1-3 paragraf)
  - **Paragraf Penutup:** Harapan, tindak lanjut, atau ajakan respons (1 paragraf)

- **Spesifikasi Penulisan:**
  - Gunakan bahasa formal dan profesional
  - Hindari singkatan yang tidak umum
  - Paragraf dipisahkan dengan jarak satu baris
  - Alignment: rata kiri (left align)
  - Font: standar yang mudah dibaca (Arial, Times New Roman, dll)
  - Size: 11-12 pt
  - Spasi: single atau 1.15 line spacing

- **Contoh Struktur Isi:**
  ```
  [Pembuka: Penjelasan singkat alasan/tujuan surat]
  
  [Inti 1: Detail pertama atau informasi utama]
  
  [Inti 2: Detail lanjutan atau permintaan spesifik]
  
  [Penutup: Harapan respons atau tindak lanjut]
  ```

#### 7. **Penutup Surat (Closing)**
- **Posisi:** Di bawah isi, dengan jarak 1 baris
- **Format:**
  ```
  Demikian disampaikan, atas perhatian dan kerja sama Bapak/Ibu, kami ucapkan terima kasih.
  
  Wassalamu'alaikum Warahmatullahi Wabarakatuh
  ```
- **Spesifikasi:**
  - Penutup formal yang menunjukkan kesantunan
  - Salam penutup (salam islami untuk formal, atau alternatif lain)
  - Jarak: 1 baris kosong sebelum penutup

#### 8. **Tanda Tangan dan Identitas Penandatangan**
- **Posisi:** Di bawah penutup, rata kanan
- **Format:**
  ```
  [Spasi untuk Tanda Tangan]
  
  [Nama Lengkap Penandatangan]
  [Jabatan Penandatangan]
  ```
- **Spesifikasi:**
  - Spasi untuk tanda tangan: minimal 3-4 baris
  - Nama lengkap ditulis tanpa singkatan
  - Jabatan penandatangan (direktur, manajer, kepala divisi, dll)
  - Jarak antara kata "Wassalamu'alaikum..." dan spasi tanda tangan: 2-3 baris kosong

#### 9. **Paraf Pengesahan (Opsional)**
- **Posisi:** Di antara isi dan tanda tangan (jika ada checkpoint approval)
- **Format:**
  ```
  Diverifikasi oleh: ________________
  [Nama Pemeriksa]
  ```
- **Catatan:** 
  - Komponen ini opsional, tergantung prosedur persetujuan
  - Dapat digunakan jika surat memerlukan validasi sebelum penandatanganan
  - Tidak selalu ditampilkan di template final

#### 10. **Lampiran (Jika ada)**
- **Posisi:** Di bawah tanda tangan
- **Format:**
  ```
  Lampiran :
  1. Proposal Teknik Jalan Tol (file: proposal_teknik_v2.pdf)
  2. Jadwal Pelaksanaan (file: jadwal_pelaksanaan.xlsx)
  3. Daftar Pengalaman Proyek (file: pengalaman_proyek.pdf)
  ```
- **Spesifikasi:**
  - Label "Lampiran :"
  - Nomor urut dengan format angka
  - Deskripsi singkat lampiran
  - Nama file (jika dalam format digital)
  - Jarak dari tanda tangan: 1-2 baris kosong
  - Hanya ditampilkan jika ada lampiran

#### 11. **Footer (Opsional)**
- **Posisi:** Bagian bawah halaman
- **Konten yang dapat ditampilkan:**
  - Halaman ke-X dari total halaman (untuk surat multi-halaman)
  - Tanggal cetak/terbit
  - Penomoran lain yang diperlukan
- **Spesifikasi:**
  - Font kecil (8-9 pt)
  - Rata tengah atau rata kanan
  - Terpisah dengan garis tipis dari body surat


### Panduan Penggunaan Surat Keluar

**Kapan Digunakan:**
- Komunikasi resmi dengan pihak eksternal (vendor, mitra, pemerintah)
- Pengajuan proposisi atau penawaran
- Permohonan atau permintaan informasi
- Laporan hasil pekerjaan atau survey
- Konfirmasi atau pemberitahuan penting

**Siapa yang Boleh Mengeluarkan:**
- Kepala Unit/Departemen atau yang diberi wewenang
- Direktur atau manajemen senior (untuk surat penting)
- Delegasi dari pihak yang berwenang

**Persyaratan Sebelum Penerbitan:**
- Isi surat sudah final dan telah dikonfirmasi oleh unit terkait
- Nomor surat sudah ditetapkan sesuai aturan penomoran
- Tanda tangan penandatangan sudah diperoleh (fisik atau digital)
- Dokumen pendukung/lampiran sudah lengkap
- Kualitas cetak/tampilan sesuai standar perusahaan

**Distribusi dan Penyimpanan:**
- Surat dicetak pada kertas official Jasa Marga (A4)
- Disimpan di arsip unit yang mengeluarkan
- Copy soft-file tersimpan di sistem dokumentasi digital
- Tracking delivery untuk surat yang dikirim

---

## 4.7.2 Template Surat Keputusan Direksi

### Definisi dan Tujuan

**Surat Keputusan Direksi** adalah dokumen resmi yang menetapkan kebijakan, keputusan, atau penetapan yang bersifat mengikat. Surat ini digunakan untuk:

- Menetapkan kebijakan organisasi yang bersifat permanen atau jangka panjang
- Persetujuan atau penolakan terhadap proposal/permohonan penting
- Penunjukan, pemberhentian, atau perubahan posisi/tanggung jawab
- Penetapan standar, prosedur, atau regulasi internal
- Keputusan bisnis strategis yang memerlukan dokumentasi formal

**Karakteristik Umum:**
- Bersifat formal dan mengikat
- Diterbitkan oleh Direksi/Pimpinan Utama
- Menggunakan penomoran dengan format `xxx/KPTS-JPT/yyyy`
- Memiliki konsideran/pertimbangan sebelum diktum
- Menggunakan bahasa hukum/formal yang ketat
- Memiliki daya laku jangka panjang (sampai dicabut/diubah)

### Struktur Template Surat Keputusan Direksi

Surat keputusan terdiri dari bagian-bagian berikut:

#### 1. **Header/Kop Surat**
- Identik dengan Surat Keluar (Logo, nama perusahaan, alamat, kontak)
- Spesifikasi: Format resmi, centered, profesional

#### 2. **Nomor dan Tanggal Keputusan**
- **Posisi:** Atas, rata tengah
- **Format:**
  ```
  KEPUTUSAN DIREKSI
  NOMOR 025/KPTS-JPT/2025
  
  TENTANG
  
  PENETAPAN KEBIJAKAN OPERASIONAL TAHUN 2025
  ```
- **Catatan:**
  - "KEPUTUSAN DIREKSI" sebagai label
  - Nomor sesuai format penomoran keputusan
  - "TENTANG" sebagai penghubung
  - Judul keputusan dalam HURUF BESAR, singkat dan jelas

#### 3. **Pembuka (Pembukaan Hukum)**
- **Format:**
  ```
  Dengan ini diumumkan kepada seluruh stakeholder bahwa,
  ```
- **Spesifikasi:**
  - Kalimat pembuka formal yang menunjukkan legalitas keputusan
  - Dapat berisi juga pemberitahuan kepada siapa keputusan ini berlaku

#### 4. **Konsideran/Pertimbangan**
- **Posisi:** Di bawah pembuka, dengan label "Pertimbangan:"
- **Format:**
  ```
  MENGINGAT:
  
  1. Bahwa untuk meningkatkan efisiensi operasional perusahaan, diperlukan 
     penetapan kebijakan yang jelas dan terarah;
  
  2. Bahwa berdasarkan hasil evaluasi tahun 2024, telah diidentifikasi 
     beberapa area yang memerlukan peningkatan kinerja;
  
  3. Bahwa untuk mencapai target strategis perusahaan tahun 2025, diperlukan 
     dukungan kebijakan operasional yang solid;
  ```
- **Spesifikasi:**
  - Dibuka dengan label "MENGINGAT:" atau "DENGAN MEMPERTIMBANGKAN:"
  - Setiap poin diberi nomor urut
  - Setiap poin dimulai dengan "Bahwa..."
  - Menyebutkan dasar hukum, pertimbangan faktual, dan justifikasi keputusan
  - Jarak antar poin: 1 baris kosong
  - Font: normal, tidak miring/bold

#### 5. **Diktum/Amar Keputusan**
- **Posisi:** Di bawah Konsideran
- **Format:**
  ```
  MEMUTUSKAN:
  
  Pertama   : Menetapkan Kebijakan Operasional PT Jasamarga Pandaan Tol 
              Tahun 2025 sebagaimana terlampir dalam Lampiran Keputusan ini.
  
  Kedua     : Kebijakan ini berlaku efektif sejak tanggal penetapan dan 
              mengikat seluruh unit kerja dalam organisasi PT Jasamarga 
              Pandaan Tol.
  
  Ketiga    : Direktur Keuangan dan SDM bertanggung jawab atas pelaksanaan 
              Keputusan ini.
  
  Keempat   : Keputusan ini berlaku sampai dengan dicabut dan/atau diubah 
              dengan Keputusan Direksi lainnya.
  ```
- **Spesifikasi:**
  - Dibuka dengan label "MEMUTUSKAN:" atau "MENETAPKAN:"
  - Setiap amar diberi nomor urut dengan kata bilangan (Pertama, Kedua, dll)
  - Menggunakan struktur formal: [Nomor] : [Isi Keputusan]
  - Jarak antar amar: 1 baris kosong
  - Font: normal, jelas, dan mudah dipahami
  - Amar dapat memuat tanggal berlaku, ketentuan detail, dan tanggung jawab implementasi

#### 6. **Penutup/Klausul Akhir**
- **Posisi:** Di bawah Diktum
- **Format:**
  ```
  Demikian Keputusan ini ditetapkan untuk dilaksanakan sebaik-baiknya oleh 
  semua pihak yang terkait.
  
  Ditetapkan di: Pandaan
  Pada tanggal : 15 Desember 2025
  ```
- **Spesifikasi:**
  - Pernyataan penutup yang menegaskan keputusan
  - Tempat dan tanggal penetapan
  - Jarak sebelum tempat/tanggal: 2-3 baris kosong

#### 7. **Tanda Tangan Direksi**
- **Posisi:** Bawah kanan, di bawah tempat/tanggal
- **Format:**
  ```
  Direksi PT Jasamarga Pandaan Tol

  [Spasi Tanda Tangan]

  Agung Widodo
  Direktur Utama
  ```
- **Spesifikasi:**
  - Label "Direksi PT Jasamarga Pandaan Tol"
  - Spasi untuk tanda tangan: 3-4 baris
  - Nama lengkap dan jabatan penandatangan
  - Tanda tangan harus original atau digital signature
  - Kop/meterai resmi (jika diperlukan)

#### 8. **Lampiran (Jika ada)**
- **Posisi:** Halaman berikutnya atau di bawah tanda tangan
- **Format:**
  ```
  LAMPIRAN KEPUTUSAN DIREKSI
  NOMOR 025/KPTS-JPT/2025
  
  KEBIJAKAN OPERASIONAL PT JASAMARGA PANDAAN TOL TAHUN 2025
  
  [Detail kebijakan, ketentuan, prosedur, dll]
  ```
- **Spesifikasi:**
  - Diberi judul "LAMPIRAN KEPUTUSAN DIREKSI"
  - Referensi nomor keputusan
  - Dapat berisi detail kebijakan, tabel, aturan, atau dokumen pendukung
  - Jika lampiran panjang, dapat disimpan sebagai dokumen terpisah

#### 9. **Nomor Halaman (untuk dokumen multi-halaman)**
- **Posisi:** Footer
- **Format:** "Halaman 1 dari 3" (jika ada lampiran)

### Panduan Penggunaan Surat Keputusan Direksi

**Kapan Digunakan:**
- Penetapan kebijakan organisasi yang bersifat mengikat
- Persetujuan strategis terhadap proposal atau program penting
- Penunjukan atau pemberhentian pejabat/posisi
- Penetapan standar, prosedur, atau regulasi internal
- Keputusan bisnis yang memerlukan dokumentasi formal dan legal

**Siapa yang Boleh Mengeluarkan:**
- Direksi/Pimpinan Utama Perusahaan
- Delegasi khusus dari Direksi (dalam kasus tertentu)
- Tidak dapat dikeluarkan oleh unit operasional biasa

**Persyaratan Sebelum Penerbitan:**
- Draft keputusan sudah direviu oleh bagian Hukum/Legal
- Pertimbangan dan diktum sudah final dan telah dikonfirmasi
- Nomor surat sudah ditetapkan sesuai aturan penomoran
- Tanda tangan Direksi sudah diperoleh
- Lampirannya lengkap dan jelas
- Distribusi ke departemen terkait sudah direncanakan

**Distribusi dan Penyimpanan:**
- Disimpan di Bagian Administrasi/Kesekretariatan Direksi
- Copies diberikan kepada unit-unit yang terkait
- Dipublikasikan melalui sistem informasi internal
- Diarsipkan dalam sistem manajemen dokumen
- Periode penyimpanan permanen (tidak ada jangka waktu, kecuali jika dicabut)

---

## 4.7.3 Template Surat Edaran Direksi

### Definisi dan Tujuan

**Surat Edaran Direksi** adalah surat resmi yang berisi instruksi, panduan, atau pemberitahuan dari pimpinan kepada seluruh atau sebagian unit dalam organisasi. Surat ini digunakan untuk:

- Memberikan instruksi atau panduan teknis implementasi
- Mengumumkan kebijakan atau keputusan yang sudah ditetapkan
- Memberikan arahan dan bimbingan operasional
- Mengingatkan kewajiban atau prosedur yang harus diikuti
- Merespons situasi urgen atau perubahan kondisi operasional
- Komunikasi informasional kepada seluruh staf

**Karakteristik Umum:**
- Bersifat lebih fleksibel dibanding Keputusan Direksi (dapat berupa instruksi)
- Sifat lebih operasional dan praktis
- Menggunakan penomoran dengan format `xxx/SE-DIR/yyyy`
- Tidak selalu memerlukan lampiran
- Dapat direvisi atau ditarik kembali lebih mudah
- Jangka waktu efektif dapat terbatas atau hingga dicabut

### Struktur Template Surat Edaran Direksi

Surat edaran terdiri dari bagian-bagian berikut:

#### 1. **Header/Kop Surat**
- Identik dengan template surat lainnya (Logo, nama perusahaan, alamat, kontak)
- Spesifikasi: Format resmi, centered, profesional

#### 2. **Nomor dan Judul Edaran**
- **Posisi:** Atas tengah
- **Format:**
  ```
  SURAT EDARAN DIREKSI
  NOMOR 012/SE-DIR/2025
  
  TENTANG
  
  INSTRUKSI PENGHEMATAN ENERGI DAN BIAYA OPERASIONAL
  ```
- **Catatan:**
  - "SURAT EDARAN DIREKSI" sebagai label
  - Nomor sesuai format penomoran edaran
  - Judul singkat, jelas, dalam HURUF BESAR
  - Fokus pada instruksi/arahan yang hendak disampaikan

#### 3. **Pembuka**
- **Format:**
  ```
  Sehubungan dengan kebutuhan penghematan biaya operasional dan efisiensi 
  energi di seluruh unit PT Jasamarga Pandaan Tol, dengan ini disampaikan 
  instruksi dan panduan pelaksanaan.
  ```
- **Spesifikasi:**
  - Paragraf pembuka yang menjelaskan latar belakang/alasan edaran
  - Dapat berisi konteks atau kondisi yang mendorong edaran
  - Format formal namun lebih casual dibanding Keputusan

#### 4. **Isi Edaran (Instruksi/Panduan)**
- **Posisi:** Di bawah pembuka
- **Format Umum:**
  ```
  Untuk itu, kepada seluruh Kepala Unit diminta untuk:
  
  1. Melakukan audit energi di masing-masing unit kerja dalam 2 minggu ke depan
  
  2. Mengidentifikasi perangkat yang tidak essential dan melakukan 
     penghematan penggunaan
  
  3. Melatih karyawan mengenai best practices penghematan energi
  
  4. Melaporkan hasil audit ke Bagian Administrasi Umum paling lambat 
     31 Desember 2025
  
  5. Mengimplementasikan rencana penghematan mulai 1 Januari 2026
  ```
- **Spesifikasi:**
  - Dapat berisi satu atau lebih bagian instruksi
  - Setiap poin diberi nomor jelas
  - Jika ada multiple pihak yang dituju, pisahkan dengan subjudul:
    ```
    A. UNTUK KEPALA UNIT:
       1. [Instruksi 1]
       2. [Instruksi 2]
    
    B. UNTUK KEPALA DEPARTEMEN OPERASI:
       1. [Instruksi 1]
       2. [Instruksi 2]
    ```
  - Gunakan bahasa imperatif/perintah yang jelas
  - Sertakan deadline/timeline jika ada
  - Jarak antar poin: normal (tidak perlu double space)

#### 5. **Harapan/Penutup**
- **Posisi:** Di bawah isi edaran
- **Format:**
  ```
  Demikian edaran ini kami sampaikan. Atas komitmen dan kepatuhan dalam 
  melaksanakan instruksi ini, kami ucapkan terima kasih.
  
  Apabila ada pertanyaan atau memerlukan klarifikasi, dapat menghubungi 
  Bagian Administrasi Umum, Ext. 205.
  ```
- **Spesifikasi:**
  - Pernyataan penutup yang formal namun friendly
  - Dapat berisi kontak person untuk pertanyaan
  - Jarak sebelum penutup: 1 baris kosong

#### 6. **Tanda Tangan Direksi**
- **Posisi:** Bawah kanan
- **Format:**
  ```
  Direksi PT Jasamarga Pandaan Tol

  [Spasi Tanda Tangan]

  Agung Widodo
  Direktur Utama
  ```
- **Spesifikasi:**
  - Label "Direksi PT Jasamarga Pandaan Tol"
  - Spasi untuk tanda tangan: 3-4 baris
  - Nama lengkap dan jabatan
  - Tanda tangan original atau digital

#### 7. **Tempat dan Tanggal Edaran**
- **Posisi:** Di bawah spasi tanda tangan
- **Format:**
  ```
  Pandaan, 15 Desember 2025
  ```
- **Spesifikasi:**
  - Format sederhana: [Tempat], [Tanggal Lengkap]
  - Ditempatkan setelah nama penandatangan

#### 8. **Catatan Distribusi (Opsional)**
- **Posisi:** Halaman terakhir
- **Format:**
  ```
  Distribusi:
  - Kepada seluruh Kepala Unit
  - Arsip Bagian Administrasi Umum
  ```
- **Spesifikasi:**
  - Daftar pihak-pihak yang menerima edaran
  - Membantu tracking penyebaran informasi
  - Opsional, tergantung kebijakan perusahaan

### Panduan Penggunaan Surat Edaran Direksi

**Kapan Digunakan:**
- Memberikan instruksi atau panduan implementasi dari kebijakan/keputusan
- Mengumumkan informasi penting kepada seluruh staf
- Memberikan arahan operasional yang memerlukan respons/aksi dari unit
- Mengingatkan prosedur atau kewajiban yang harus diikuti
- Merespons situasi urgen atau perubahan kondisi
- Komunikasi from-top-down kepada organisasi

**Siapa yang Boleh Mengeluarkan:**
- Direksi/Pimpinan Utama Perusahaan
- Dapat didelegasikan kepada Direktur tertentu (untuk edaran spesifik fungsi)

**Persyaratan Sebelum Penerbitan:**
- Isi edaran sudah final dan jelas
- Nomor edaran sudah ditetapkan sesuai aturan penomoran
- Tanda tangan penandatangan sudah diperoleh
- Daftar penerima sudah jelas (seluruh staf atau unit spesifik)
- Deadline/timeline instruksi sudah realistis dan jelas
- Format dan bahasa sudah dipastikan mudah dipahami

**Distribusi dan Penyimpanan:**
- Didistribusikan kepada semua penerima (seluruh staf atau unit target)
- Dapat didistribusikan via email, sistem informasi internal, dan/atau cetak
- Disimpan di Bagian Administrasi dan/atau Kesekretariatan Direksi
- Copy soft-file tersimpan di sistem manajemen dokumen
- Periode penyimpanan: minimal 3-5 tahun (tergantung content dan relevansi)
- Dapat ditarik kembali atau dibatalkan dengan edaran pencabutan jika diperlukan

---

## 4.7.4 Perbandingan Ketiga Template

### Tabel Perbandingan Karakteristik

| Aspek | Surat Keluar | Keputusan Direksi | Edaran Direksi |
|---|---|---|---|
| **Tujuan Utama** | Komunikasi eksternal/internal | Penetapan kebijakan mengikat | Instruksi/panduan operasional |
| **Penerbitan Dari** | Unit/Departemen | Direksi/Pimpinan Utama | Direksi/Direktur |
| **Format Nomor** | OT.01.03.xxx | xxx/KPTS-JPT/yyyy | xxx/SE-DIR/yyyy |
| **Bersifat Mengikat** | Tergantung isi | Ya, mengikat | Ya, mengikat |
| **Jangka Waktu Berlaku** | Temporary (sesuai kebutuhan) | Permanen (sampai dicabut) | Dapat terbatas atau permanen |
| **Memerlukan Konsideran** | Tidak | Ya, detail | Tidak, hanya pembuka |
| **Struktur Diktum** | Tidak ada | Ya, dengan amar bertingkat | Tidak, hanya instruksi |
| **Dapat Direvisi** | Jarang (dibuat baru jika perlu) | Dibuat keputusan baru untuk revisi | Dapat direvisi dengan edaran baru |
| **Distribusi Utama** | Eksternal/pihak tertentu | Seluruh organisasi | Seluruh organisasi atau unit target |
| **Level Formalitas** | Formal/Professional | Sangat Formal/Legal | Formal namun lebih operasional |
| **Lampiran Umum** | Dokumen pendukung/attachments | Detail kebijakan/prosedur | Jarang ada lampiran |

---

## 4.7.5 Best Practice dalam Penggunaan Template

### Untuk Semua Jenis Surat

1. **Konsistensi Format**
   - Gunakan font dan ukuran yang konsisten
   - Pastikan kop surat selalu tercetak jelas
   - Jarak margin dan spasi sudah sesuai standar

2. **Kejelasan Isi**
   - Gunakan bahasa formal, jelas, dan mudah dipahami
   - Hindari singkatan atau istilah yang tidak umum
   - Setiap poin harus mengandung informasi yang clear dan actionable

3. **Nomor Surat yang Tepat**
   - Pastikan nomor surat sudah ditetapkan sesuai aturan
   - Tidak ada duplikasi nomor dalam sistem
   - Nomor harus tercetak dengan jelas

4. **Tanda Tangan yang Valid**
   - Tanda tangan harus dari pihak yang berwenang
   - Jika menggunakan digital signature, pastikan sudah valid dan recognized
   - Tanda tangan harus legible dan jelas

5. **Penyimpanan dan Distribusi**
   - Setiap surat harus disimpan di arsip yang tepat
   - Soft-file harus tersimpan di sistem digital
   - Distribusi harus tercatat dan trackable

### Khusus untuk Surat Keluar

- Pastikan alamat penerima lengkap dan akurat
- Perihal surat harus singkat namun informatif
- Isi surat harus mengikuti struktur pembuka-inti-penutup yang jelas
- Lampiran harus relevan dan lengkap

### Khusus untuk Surat Keputusan Direksi

- Konsideran harus comprehensive dan jelas alasannya
- Diktum harus specific dan measurable (jika memungkinkan)
- Pastikan ada pihak yang bertanggung jawab atas implementasi
- Periode berlaku harus jelas (sampai dicabuk, atau jangka waktu tertentu)

### Khusus untuk Surat Edaran Direksi

- Instruksi harus jelas dan mudah diimplementasikan
- Deadline/timeline harus realistis
- Sertakan contact person untuk pertanyaan
- Gunakan bahasa yang friendly namun tetap formal dan authoritative
