<?php

/**
 * Server-side PDF Generation - NUCLEAR FIX for PARAF rendering
 */

session_start();
header('Content-Type: application/json; charset=utf-8');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['ok' => false, 'error' => 'Method not allowed']);
    exit;
}

if (empty($_SESSION)) {
    http_response_code(401);
    echo json_encode(['ok' => false, 'error' => 'Unauthorized']);
    exit;
}

$input = json_decode(file_get_contents('php://input'), true);

if (!isset($input['html']) || empty($input['html'])) {
    http_response_code(400);
    echo json_encode(['ok' => false, 'error' => 'No HTML content provided']);
    exit;
}

$htmlContent = $input['html'];
$filename = $input['filename'] ?? 'surat_' . time() . '.pdf';

if (!str_ends_with(strtolower($filename), '.pdf')) {
    $filename .= '.pdf';
}

require_once __DIR__ . '/../vendor/autoload.php';

try {
    $debugLog = __DIR__ . '/../letter/print/debug_' . time() . '.txt';
    file_put_contents($debugLog, "🔥 NUCLEAR FIX: Direct text replacement for PARAF\n");
    file_put_contents($debugLog, str_repeat("=", 80) . "\n", FILE_APPEND);

    // Extract PARAF value
    $parafValue = '';
    if (preg_match('/Paraf:\s*([A-Z][A-Z\/]*)/i', $htmlContent, $matches)) {
        $parafValue = trim($matches[1]);
        file_put_contents($debugLog, "✅ Extracted PARAF: '$parafValue'\n", FILE_APPEND);
    } else {
        file_put_contents($debugLog, "❌ Could not extract PARAF from HTML\n", FILE_APPEND);
        // Try to find [[PARAF]] placeholder - maybe it wasn't replaced
        if (strpos($htmlContent, '[[PARAF]]') !== false) {
            file_put_contents($debugLog, "⚠️ Found [[PARAF]] placeholder - it wasn't replaced by JavaScript!\n", FILE_APPEND);
        }
    }

    // ===== ULTIMATE SIMPLE FIX: Direct string replacement =====
    $renderHtml = $htmlContent;
    $parafInjected = false;

    if (!empty($parafValue)) {
        // SIMPLE FIX 1: Replace any span containing "Paraf: X" with text wrapped in <strong>
        // mPDF bug: single characters disappear if not wrapped in a tag
        $count1 = 0;
        $renderHtml = preg_replace(
            '/<span[^>]*>(Paraf:\s*)(' . preg_quote($parafValue, '/') . ')<\/span>/i',
            '$1<strong>$2</strong>',  // Wrap just the paraf value in <strong>
            $renderHtml,
            -1,
            $count1
        );
        file_put_contents($debugLog, "SIMPLE FIX 1: Wrapped PARAF value in <strong> ($count1 replacements)\n", FILE_APPEND);

        // SIMPLE FIX 1B: If there's bare "Paraf: I" (not in span), wrap the value
        $count1b = 0;
        $renderHtml = preg_replace(
            '/(\|\s*Paraf:\s*)(' . preg_quote($parafValue, '/') . ')(<\/div>)/i',
            '$1<strong>$2</strong>$3',  // Wrap the value in <strong>
            $renderHtml,
            -1,
            $count1b
        );
        file_put_contents($debugLog, "SIMPLE FIX 1B: Wrapped bare PARAF in <strong> ($count1b replacements)\n", FILE_APPEND);

        // SIMPLE FIX 2: If there's still a styled span, replace it
        $count2 = 0;
        $renderHtml = preg_replace(
            '/<span[^>]*style=["\'][^"\']*["\'][^>]*>(Paraf:\s*)(' . preg_quote($parafValue, '/') . ')<\/span>/i',
            '$1<strong>$2</strong>',
            $renderHtml,
            -1,
            $count2
        );
        file_put_contents($debugLog, "SIMPLE FIX 2: Wrapped styled span PARAF in <strong> ($count2 replacements)\n", FILE_APPEND);

        // SIMPLE FIX 3: Replace signature-paraf div with simple text in <strong>
        $count3 = 0;
        $renderHtml = preg_replace(
            '/<div[^>]*class=["\']signature-paraf["\'][^>]*>.*?<\/div>/is',
            '<div style="font-size:16px;margin-bottom:10mm;"><strong>' . $parafValue . '</strong></div>',
            $renderHtml,
            -1,
            $count3
        );
        file_put_contents($debugLog, "SIMPLE FIX 3: Replaced signature-paraf div ($count3 replacements)\n", FILE_APPEND);

        $parafInjected = ($count1 + $count1b + $count2 + $count3) > 0;
    }

    // Deep debug: Find and log the ACTUAL NOMOR section
    file_put_contents($debugLog, "\n🔍 Searching for NOMOR section:\n", FILE_APPEND);

    // Find the position of "NOMOR :" in the HTML
    $nomorPos = stripos($renderHtml, 'NOMOR :');
    if ($nomorPos !== false) {
        // Extract 300 characters around it
        $snippet = substr($renderHtml, max(0, $nomorPos - 50), 400);
        file_put_contents($debugLog, "FOUND NOMOR at position $nomorPos:\n", FILE_APPEND);
        file_put_contents($debugLog, $snippet . "\n", FILE_APPEND);
    } else {
        file_put_contents($debugLog, "❌ NOMOR : not found in HTML!\n", FILE_APPEND);
    }

    // Find any occurrence of "Paraf"
    file_put_contents($debugLog, "\n🔍 Searching for Paraf:\n", FILE_APPEND);
    $parafPos = stripos($renderHtml, 'Paraf');
    if ($parafPos !== false) {
        $snippet = substr($renderHtml, max(0, $parafPos - 50), 200);
        file_put_contents($debugLog, "FOUND Paraf at position $parafPos:\n", FILE_APPEND);
        file_put_contents($debugLog, $snippet . "\n", FILE_APPEND);
    } else {
        file_put_contents($debugLog, "❌ Paraf not found in HTML!\n", FILE_APPEND);
    }

    // Create mPDF
    $mpdf = new \Mpdf\Mpdf([
        'mode' => 'utf-8',
        'format' => 'A4',
        'margin_left' => 12,
        'margin_right' => 14,
        'margin_top' => 12,
        'margin_bottom' => 28,
        'default_font' => 'dejavusans',
        'tempDir' => sys_get_temp_dir() . '/mpdf',
    ]);

    $mpdf->SetTitle('Surat');
    $mpdf->SetAuthor('PT Jasamarga Pandaan Tol');
    $mpdf->shrink_tables_to_fit = 1;
    $mpdf->ignore_invalid_utf8 = true;

    // ===== CSS FIX: Extract stored CSS or fall back to fresh template =====
    // Strategy 1.A + 2.C: Store CSS with naskah, use stored CSS if available

    $cssToUse = '';
    $hasStoredCss = false;

    // Check if HTML contains <style> block (stored CSS from previous saves)
    if (preg_match('/<style[^>]*>([\s\S]*?)<\/style>/i', $renderHtml, $matches)) {
        $cssToUse = $matches[1];
        $hasStoredCss = true;
        file_put_contents($debugLog, "✅ Found stored CSS in saved naskah - will use this for consistency\n", FILE_APPEND);
        // Extract only the HTML body part (remove the <style> block for processing)
        $renderHtml = preg_replace('/<style[^>]*>[\s\S]*?<\/style>/i', '', $renderHtml);
    } else {
        file_put_contents($debugLog, "ℹ️ No stored CSS in naskah - using fresh template CSS as fallback\n", FILE_APPEND);
    }

    // Clean CSS to remove mPDF-incompatible features
    // 1. Remove :root block (contains CSS variable definitions)
    $cssToUse = preg_replace('/:root\s*\{(?:[^{}]|\{[^}]*\})*\}/s', '', $cssToUse);

    // 2. Remove @media blocks - mPDF doesn't handle media queries
    $cssToUse = preg_replace('/@media[^{]*\{(?:[^{}]|\{[^}]*\})*\}/s', '', $cssToUse);

    // 3. Remove @page blocks (handled by mPDF constructor options)
    $cssToUse = preg_replace('/@page\s*\{[^}]*\}/s', '', $cssToUse);

    // 4. Replace CSS variables with hardcoded values
    $cssVarMap = [
        'var(--page-w)' => '210mm',
        'var(--page-h)' => '297mm',
        'var(--pad-top)' => '12mm',
        'var(--pad-right)' => '14mm',
        'var(--pad-bottom)' => '42mm',
        'var(--pad-left)' => '12mm',
        'var(--content-h)' => '243mm',
        'var(--content-w)' => '182mm',
        'var(--slot-gap)' => '6mm',
        'var(--slot-h)' => '237mm',
    ];
    foreach ($cssVarMap as $var => $val) {
        $cssToUse = str_replace($var, $val, $cssToUse);
    }

    // 5. Replace calc() expressions with approximate values
    $cssToUse = preg_replace('/calc\s*\(\s*297mm\s*-\s*12mm\s*-\s*42mm\s*\)/i', '243mm', $cssToUse);
    $cssToUse = preg_replace('/calc\s*\(\s*210mm\s*-\s*(?:var\(--pad-left\)|12mm)\s*-\s*(?:var\(--pad-right\)|14mm)\s*\)/i', '182mm', $cssToUse);
    $cssToUse = preg_replace('/calc\s*\(\s*(?:var\(--content-h\)|243mm)\s*-\s*(?:var\(--slot-gap\)|6mm)\s*\)/i', '237mm', $cssToUse);
    $cssToUse = preg_replace('/calc\s*\([^)]*\)/i', 'auto', $cssToUse);

    // 6. Remove remaining var() references
    $cssToUse = preg_replace('/var\s*\([^)]*\)/i', 'auto', $cssToUse);

    // Rebuild HTML with cleaned CSS
    if (!empty($cssToUse)) {
        $renderHtml = '<style>' . $cssToUse . '</style>' . $renderHtml;
    }

    file_put_contents($debugLog, "✅ Cleaned CSS: removed :root, @media, @page, replaced variables and calc()\n", FILE_APPEND);
    file_put_contents($debugLog, ($hasStoredCss ? "✅" : "ℹ️") . " CSS applied to PDF (stored: " . ($hasStoredCss ? "YES" : "NO") . ")\n", FILE_APPEND);
    file_put_contents($debugLog, "\n📊 Final HTML stats:\n", FILE_APPEND);
    file_put_contents($debugLog, "HTML Length: " . strlen($renderHtml) . " bytes\n", FILE_APPEND);

    // Count pages worth of content (rough estimate)
    $pageCount = ceil(strlen($renderHtml) / 5000);
    file_put_contents($debugLog, "Estimated content pages: $pageCount\n", FILE_APPEND);

    // Sample the content
    if (strpos($renderHtml, '<body') !== false) {
        $bodyStart = strpos($renderHtml, '<body');
        $sample = substr($renderHtml, $bodyStart, 500);
        file_put_contents($debugLog, "\nHTML body sample:\n$sample\n", FILE_APPEND);
    }

    // Write the FIXED HTML to mPDF
    $mpdf->WriteHTML($renderHtml);

    // Output
    $outputDir = __DIR__ . '/../letter/print';
    if (!is_dir($outputDir)) mkdir($outputDir, 0775, true);

    $safeName = 'surat_' . time() . '.pdf';
    $outputPath = $outputDir . '/' . $safeName;
    $relativePath = 'letter/print/' . $safeName;

    $mpdf->Output($outputPath, \Mpdf\Output\Destination::FILE);
    file_put_contents($debugLog, "✅ PDF generated: $outputPath\n", FILE_APPEND);

    echo json_encode([
        'ok' => true,
        'path' => $relativePath,
        'filename' => $safeName,
        'paraf_value' => $parafValue,
        'paraf_injected' => $parafInjected
    ]);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['ok' => false, 'error' => $e->getMessage()]);
}
