<?php

/**
 * API Endpoints for Dynamic Classification Dropdowns
 * Supports hierarchical dropdown loading: Klasifikasi -> Jenis -> Subjenis
 * 
 * This file can be:
 * 1. Called directly as an API endpoint (outputs JSON)
 * 2. Included by other files to use the helper functions (getNextSequenceNumber, etc.)
 */

// Include database connection
require_once __DIR__ . '/../includes/koneksi.php';

// Check if this is an API request (either direct call or via wrapper like surat/api_classification.php)
$isApiRequest = isset($_GET['action']) && !empty($_GET['action']);

// Only execute API logic if there's an action parameter (API request)
if ($isApiRequest) {
    header('Content-Type: application/json');
    header('Access-Control-Allow-Origin: *');
    header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
    header('Access-Control-Allow-Headers: Content-Type');

    if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
        http_response_code(200);
        exit();
    }

    $action = $_GET['action'] ?? '';
    $response = ['success' => false, 'data' => [], 'message' => ''];

    try {
        switch ($action) {

            case 'get_tipe_surat':
                $response = getTipeSurat($conn);
                break;

            case 'get_klasifikasi':
                $response = getKlasifikasi($conn);
                break;

            case 'get_jenis_by_klasifikasi':
                $klasifikasiId = (int)($_GET['klasifikasi_id'] ?? 0);
                $response = getJenisByKlasifikasi($conn, $klasifikasiId);
                break;

            case 'get_subjenis_by_jenis':
                $jenisId = (int)($_GET['jenis_id'] ?? 0);
                $response = getSubjenisByJenis($conn, $jenisId);
                break;

            case 'generate_nomor_surat':
                $tipeSuratId = (int)($_GET['tipe_surat_id'] ?? 0);
                $klasifikasiKode = $_GET['klasifikasi_kode'] ?? '';
                $jenisKode = $_GET['jenis_kode'] ?? '';
                $subjenisId = (int)($_GET['subjenis_id'] ?? 0);
                $jabatan = $_GET['jabatan'] ?? '';
                $existingSequenceId = $_GET['existing_sequence_id'] ?? '';

                $response = generateNomorSurat($conn, $tipeSuratId, $klasifikasiKode, $jenisKode, $subjenisId, $jabatan, false, $existingSequenceId);
                break;

            case 'reserve_nomor_surat':
                $tipeSuratId = (int)($_GET['tipe_surat_id'] ?? 0);
                $klasifikasiKode = $_GET['klasifikasi_kode'] ?? '';
                $jenisKode = $_GET['jenis_kode'] ?? '';
                $subjenisId = (int)($_GET['subjenis_id'] ?? 0);
                $jabatan = $_GET['jabatan'] ?? '';

                $response = generateNomorSurat($conn, $tipeSuratId, $klasifikasiKode, $jenisKode, $subjenisId, $jabatan, true);
                break;

            case 'get_last_letter_number':
                $tipeSuratId = (int)($_GET['tipe_surat_id'] ?? 0);
                $response = getLastLetterNumber($conn, $tipeSuratId);
                break;

            case 'get_authorized_signers':
                $currentUserId = isset($_GET['current_user_id']) ? (int)$_GET['current_user_id'] : null;
                $response = getAuthorizedSigners($conn, $currentUserId);
                break;

            default:
                $response['message'] = 'Invalid action';
                http_response_code(400);
        }
    } catch (Exception $e) {
        $response['message'] = 'API Error: ' . $e->getMessage();
        $response['success'] = false;
        http_response_code(500);
        error_log("API Classification Error: " . $e->getMessage() . " on line " . $e->getLine());
    }

    // Ensure we always output JSON, never HTML
    ob_clean();
    echo json_encode($response, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit();
}
// End of API direct call logic

/**
 * Get all tipe surat
 */
function getTipeSurat($conn)
{
    $sql = "SELECT id, nama, kode FROM tipe_surat ORDER BY id";
    $result = $conn->query($sql);

    $data = [];
    if ($result && $result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $data[] = $row;
        }
    }

    return [
        'success' => true,
        'data' => $data,
        'message' => 'Tipe surat loaded successfully'
    ];
}

/**
 * Get all klasifikasi
 */
function getKlasifikasi($conn)
{
    $sql = "SELECT id, nama, kode, deskripsi FROM klasifikasi_arsip WHERE kode IS NOT NULL ORDER BY kode";
    $result = $conn->query($sql);

    $data = [];
    if ($result && $result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $data[] = $row;
        }
    }

    return [
        'success' => true,
        'data' => $data,
        'message' => 'Klasifikasi loaded successfully'
    ];
}

/**
 * Get jenis surat by klasifikasi
 */
function getJenisByKlasifikasi($conn, $klasifikasiId)
{
    if ($klasifikasiId <= 0) {
        return [
            'success' => false,
            'data' => [],
            'message' => 'Invalid klasifikasi ID'
        ];
    }

    $sql = "
        SELECT 
            js.id, 
            js.nama, 
            js.kode, 
            js.deskripsi,
            ka.kode as klasifikasi_kode,
            ka.nama as klasifikasi_nama
        FROM jenis_surat js
        INNER JOIN klasifikasi_arsip ka ON js.id_klasifikasi = ka.id
        WHERE js.id_klasifikasi = ?
        ORDER BY js.kode
    ";

    $stmt = $conn->prepare($sql);
    $stmt->bind_param('i', $klasifikasiId);
    $stmt->execute();
    $result = $stmt->get_result();

    $data = [];
    if ($result && $result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $data[] = $row;
        }
    }

    return [
        'success' => true,
        'data' => $data,
        'message' => 'Jenis surat loaded successfully'
    ];
}

/**
 * Get subjenis surat by jenis
 */
function getSubjenisByJenis($conn, $jenisId)
{
    if ($jenisId <= 0) {
        return [
            'success' => false,
            'data' => [],
            'message' => 'Invalid jenis ID'
        ];
    }

    $sql = "
        SELECT 
            ss.id,
            ss.nama,
            ss.kode,
            ss.deskripsi,
            js.kode as jenis_kode,
            js.nama as jenis_nama,
            ka.kode as klasifikasi_kode,
            ka.nama as klasifikasi_nama
        FROM subjenis_surat ss
        INNER JOIN jenis_surat js ON ss.id_jenis = js.id
        INNER JOIN klasifikasi_arsip ka ON ss.id_klasifikasi = ka.id
        WHERE ss.id_jenis = ?
        ORDER BY ss.kode
    ";

    $stmt = $conn->prepare($sql);
    $stmt->bind_param('i', $jenisId);
    $stmt->execute();
    $result = $stmt->get_result();

    $data = [];
    if ($result && $result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $data[] = $row;
        }
    }

    return [
        'success' => true,
        'data' => $data,
        'message' => 'Subjenis surat loaded successfully'
    ];
}

/**
 * Generate nomor surat based on tipe surat
 */
function generateNomorSurat($conn, $tipeSuratId, $klasifikasiKode, $jenisKode, $subjenisId, $jabatan, $reserve = false, $existingSequenceId = '')
{
    if ($tipeSuratId <= 0) {
        return [
            'success' => false,
            'data' => null,
            'message' => 'Invalid tipe surat ID'
        ];
    }

    // Get tipe surat info
    $stmt = $conn->prepare("SELECT kode, nama FROM tipe_surat WHERE id = ?");
    $stmt->bind_param('i', $tipeSuratId);
    $stmt->execute();
    $result = $stmt->get_result();
    $tipeSurat = $result->fetch_assoc();

    if (!$tipeSurat) {
        return [
            'success' => false,
            'data' => null,
            'message' => 'Tipe surat not found'
        ];
    }

    $nomorSurat = '';
    $currentYear = date('Y');

    switch ($tipeSurat['kode']) {
        case 'SK': // Surat Keluar
            if (empty($klasifikasiKode) || empty($jenisKode) || $subjenisId <= 0) {
                return [
                    'success' => false,
                    'data' => null,
                    'message' => 'Klasifikasi, Jenis, dan Subjenis harus dipilih untuk Surat Keluar'
                ];
            }

            // Get subjenis info
            $stmt = $conn->prepare("SELECT kode FROM subjenis_surat WHERE id = ?");
            $stmt->bind_param('i', $subjenisId);
            $stmt->execute();
            $result = $stmt->get_result();
            $subjenis = $result->fetch_assoc();

            if (!$subjenis) {
                return [
                    'success' => false,
                    'data' => null,
                    'message' => 'Subjenis surat not found'
                ];
            }

            // Get kode jabatan from JSON mapping
            $kodeJabatan = getKodeJabatan($jabatan);

            // Use existing sequence ID if provided (edit mode), otherwise get next sequence number
            if (!empty($existingSequenceId)) {
                $sequenceId = $existingSequenceId;
            } else {
                $sequenceId = getNextSequenceNumber($conn, $tipeSuratId, $reserve);
            }

            // Format Surat Keluar: [KodeJabatan]-[KodeKlasifikasi].[KodeJenis].[KodeSubjenis].[SequenceNumber]
            // Example: HCGA-MN.JPT.PR.01.07.25 or DKU.JPT.PL.02.03.80

            if ($kodeJabatan) {
                // Jabatan termasuk dalam mapping, gunakan format dengan kode jabatan
                $nomorSurat = "{$kodeJabatan}.{$klasifikasiKode}.{$jenisKode}.{$subjenis['kode']}.{$sequenceId}";
            } else {
                // Jabatan tidak ada dalam mapping, tanpa kode jabatan
                $nomorSurat = "{$klasifikasiKode}.{$jenisKode}.{$subjenis['kode']}.{$sequenceId}";
            }

            $sequenceNumber = $sequenceId;
            break;

        case 'KPTS': // Surat Keputusan Direksi
            // Use existing sequence ID if provided (edit mode), otherwise get next sequence number
            if (!empty($existingSequenceId)) {
                $sequenceNumber = $existingSequenceId;
            } else {
                $sequenceNumber = getNextSequenceNumber($conn, $tipeSuratId, $reserve);
            }
            $sequenceId = str_pad($sequenceNumber, 3, '0', STR_PAD_LEFT);
            $nomorSurat = "{$sequenceId}/KPTS-JPT/{$currentYear}";
            break;

        case 'SE': // Surat Edaran Direksi
            // Use existing sequence ID if provided (edit mode), otherwise get next sequence number
            if (!empty($existingSequenceId)) {
                $sequenceNumber = $existingSequenceId;
            } else {
                $sequenceNumber = getNextSequenceNumber($conn, $tipeSuratId, $reserve);
            }
            $sequenceId = str_pad($sequenceNumber, 3, '0', STR_PAD_LEFT);
            $nomorSurat = "{$sequenceId}/SE-DIR/{$currentYear}";
            break;

        default:
            return [
                'success' => false,
                'data' => null,
                'message' => 'Unknown tipe surat'
            ];
    }

    return [
        'success' => true,
        'data' => [
            'nomor_surat' => $nomorSurat,
            'tipe_surat' => $tipeSurat,
            'sequence_number' => $sequenceNumber
        ],
        'message' => 'Nomor surat generated successfully'
    ];
}

/**
 * Get kode jabatan from JSON mapping with alias support
 * Returns kode if jabatan exists in mapping, null otherwise
 */
function getKodeJabatan($jabatan)
{
    $jsonFile = __DIR__ . '/../data/kode_jabatan.json';

    if (!file_exists($jsonFile)) {
        return null;
    }

    $jsonContent = file_get_contents($jsonFile);
    $data = json_decode($jsonContent, true);

    if (!$data || !isset($data['kodeJabatan'])) {
        return null;
    }

    // Normalize jabatan for comparison (case-insensitive, trim spaces, remove extra spaces)
    $jabatanNormalized = strtolower(trim(preg_replace('/\s+/', ' ', $jabatan)));

    foreach ($data['kodeJabatan'] as $item) {
        // Check main jabatan
        $itemJabatanNormalized = strtolower(trim(preg_replace('/\s+/', ' ', $item['jabatan'])));

        if ($itemJabatanNormalized === $jabatanNormalized) {
            return $item['kode'];
        }

        // Check aliases if available
        if (isset($item['aliases']) && is_array($item['aliases'])) {
            foreach ($item['aliases'] as $alias) {
                $aliasNormalized = strtolower(trim(preg_replace('/\s+/', ' ', $alias)));

                if ($aliasNormalized === $jabatanNormalized) {
                    return $item['kode'];
                }
            }
        }
    }

    return null; // Jabatan not found in mapping
}

/**
 * Get next available sequence number for specific tipe_surat (independent per type)
 * @param mysqli $conn Database connection
 * @param int $tipeSuratId The tipe_surat ID (1=SK, 2=KPTS, 3=SE)
 * @return int Next sequence number for this specific letter type
 */
/**
 * Get next available sequence number for specific tipe_surat (independent per type, global without year)
 * @param mysqli $conn Database connection
 * @param int $tipeSuratId The tipe_surat ID (1=SK, 2=KPTS, 3=SE)
 * @param bool $reserve If true, persist the increment to database
 * @return int Next sequence number for this specific letter type
 */
function getNextSequenceNumber($conn, $tipeSuratId = null, $reserve = false)
{
    // If no tipe_surat specified, fall back to legacy behavior (global sequence from office table)
    if ($tipeSuratId === null) {
        $stmt = $conn->prepare("SELECT MAX(sequence_number) as max_seq FROM office");
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        return ($row['max_seq'] ?? 0) + 1;
    }

    // Get sequence from letter_sequences table (no year constraint - global per tipe_surat)
    $stmt = $conn->prepare("SELECT last_number FROM letter_sequences WHERE tipe_surat_id = ? FOR UPDATE");
    $stmt->bind_param('i', $tipeSuratId);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();

    if ($row) {
        // Calculate next sequence (current + 1)
        $nextSequence = $row['last_number'] + 1;

        // If caller asked to reserve/persist, update the database
        if ($reserve) {
            $updateStmt = $conn->prepare("UPDATE letter_sequences SET last_number = ? WHERE tipe_surat_id = ?");
            $updateStmt->bind_param('ii', $nextSequence, $tipeSuratId);
            $updateStmt->execute();
        }

        return $nextSequence;
    } else {
        // No sequence record found - this shouldn't happen if migration ran correctly
        // But handle it gracefully: start from 1
        $nextSequence = 1;

        if ($reserve) {
            $insertStmt = $conn->prepare("INSERT INTO letter_sequences (tipe_surat_id, last_number) VALUES (?, ?)");
            $insertStmt->bind_param('ii', $tipeSuratId, $nextSequence);
            $insertStmt->execute();
        }

        return $nextSequence;
    }
}

/**
 * Get next office ID for complete nomor surat prediction
 */
function getNextOfficeId($conn)
{
    // Get the next AUTO_INCREMENT value from office table
    $stmt = $conn->prepare("SELECT AUTO_INCREMENT FROM information_schema.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'office'");
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();

    return $row ? $row['AUTO_INCREMENT'] : 1;
}

/**
 * Get the last letter number for a specific tipe_surat
 * Used to display "Nomor Surat Sebelumnya" indicator
 */
function getLastLetterNumber($conn, $tipeSuratId)
{
    if ($tipeSuratId <= 0) {
        return [
            'success' => false,
            'data' => null,
            'message' => 'Invalid tipe surat ID'
        ];
    }

    // Get tipe surat info
    $stmt = $conn->prepare("SELECT kode, nama FROM tipe_surat WHERE id = ?");
    $stmt->bind_param('i', $tipeSuratId);
    $stmt->execute();
    $result = $stmt->get_result();
    $tipeSurat = $result->fetch_assoc();

    if (!$tipeSurat) {
        return [
            'success' => false,
            'data' => null,
            'message' => 'Tipe surat not found'
        ];
    }

    // Get the most recent letter of this type
    $stmt = $conn->prepare("
        SELECT o.nomor_surat, o.sequence_number, o.tanggal_surat 
        FROM office o 
        WHERE o.tipe_surat_id = ? 
        ORDER BY o.id DESC 
        LIMIT 1
    ");
    $stmt->bind_param('i', $tipeSuratId);
    $stmt->execute();
    $result = $stmt->get_result();
    $lastLetter = $result->fetch_assoc();

    // Get current sequence from letter_sequences table (global, no year constraint)
    $seqStmt = $conn->prepare("SELECT last_number FROM letter_sequences WHERE tipe_surat_id = ?");
    $seqStmt->bind_param('i', $tipeSuratId);
    $seqStmt->execute();
    $seqResult = $seqStmt->get_result();
    $seqRow = $seqResult->fetch_assoc();

    $currentSequence = $seqRow ? $seqRow['last_number'] : 0;

    return [
        'success' => true,
        'data' => [
            'tipe_surat' => $tipeSurat,
            'last_nomor_surat' => $lastLetter ? $lastLetter['nomor_surat'] : null,
            'last_sequence' => $lastLetter ? $lastLetter['sequence_number'] : 0,
            'current_sequence' => $currentSequence,
            'next_sequence' => $currentSequence + 1
        ],
        'message' => 'Last letter number retrieved'
    ];
}

/**
 * Get list of authorized signers (16 registered names)
 */
function getAuthorizedSigners($conn, $currentUserId = null)
{
    $stmt = $conn->prepare(" 
        SELECT a.id, a.user_id, a.nama, a.jabatan, a.approval_chain, u.username
        FROM authorized_signers a
        LEFT JOIN users u ON a.user_id = u.id
        WHERE a.is_active = 1
        ORDER BY a.id ASC
    ");
    $stmt->execute();
    $result = $stmt->get_result();

    $signers = [];
    while ($row = $result->fetch_assoc()) {
        $signers[] = [
            'id' => $row['id'],
            'user_id' => $row['user_id'],
            'nama' => $row['nama'],
            'jabatan' => $row['jabatan'],
            'approval_chain' => $row['approval_chain'],
            'username' => $row['username']
        ];
    }

    // If currentUserId provided, filter signers to those within the current user's approval chain (upward)
    if ($currentUserId) {
        // Find the current user's approval_chain if present
        $userChain = null;
        foreach ($signers as $s) {
            if ((int)$s['user_id'] === (int)$currentUserId) {
                $userChain = $s['approval_chain'];
                break;
            }
        }

        // If not found by user_id, attempt to fetch user's name and match within chains
        if (!$userChain) {
            $stmtUser = $conn->prepare("SELECT nama FROM users WHERE id = ? LIMIT 1");
            $stmtUser->bind_param('i', $currentUserId);
            $stmtUser->execute();
            $resUser = $stmtUser->get_result();
            $userRow = $resUser ? $resUser->fetch_assoc() : null;
            $stmtUser->close();
            if ($userRow && !empty($userRow['nama'])) {
                $userName = trim($userRow['nama']);
                // Try to find a signer that has this name in their approval_chain
                foreach ($signers as $s) {
                    if (!empty($s['approval_chain']) && strpos($s['approval_chain'], $userName) !== false) {
                        $userChain = $s['approval_chain'];
                        break;
                    }
                }
            }
        }

        if ($userChain) {
            // approval_chain uses '???' as separator
            $parts = array_map('trim', preg_split('/\?\?\?/', $userChain));
            // Keep unique and non-empty
            $allowedNames = array_values(array_filter(array_unique($parts)));

            $filtered = [];
            $isCurrentAuthorized = false;
            foreach ($signers as $s) {
                // Detect if current user is an authorized signer by user_id
                if ($currentUserId && (int)$s['user_id'] === (int)$currentUserId) {
                    $isCurrentAuthorized = true;
                }
                // Exclude the user itself from the dropdown
                if ($currentUserId && (int)$s['user_id'] === (int)$currentUserId) {
                    continue;
                }
                if (in_array(trim($s['nama']), $allowedNames, true)) {
                    $filtered[] = $s;
                }
            }

            return [
                'success' => true,
                'data' => $filtered,
                'message' => 'Authorized signers loaded for current user',
                'current_user_authorized' => $isCurrentAuthorized
            ];
        } else {
            // No chain found; return empty set to be safe
            return [
                'success' => true,
                'data' => [],
                'message' => 'No authorized signers for current user',
                'current_user_authorized' => false
            ];
        }
    }

    return [
        'success' => true,
        'data' => $signers,
        'message' => 'Authorized signers loaded'
    ];
}
