<?php

/**
 * API Endpoints for Approval System
 * Handles approval chain, paraf (initials), signatures, and comments
 */

// Suppress HTML error output for API - all errors must be JSON
ini_set('display_errors', 0);
error_reporting(E_ALL);

// Start output buffering to catch any accidental HTML output
ob_start();

try {
    require_once __DIR__ . '/../includes/koneksi.php';
    require_once __DIR__ . '/../vendor/autoload.php';
} catch (Throwable $e) {
    // Capture any loading errors
    error_log("Autoloader error: " . $e->getMessage());
    ob_end_clean();
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'System initialization error']);
    exit();
}

use chillerlan\QRCode\QRCode;
use chillerlan\QRCode\QROptions;

// Session check
session_start();
if (!isset($_SESSION['username'])) {
    ob_end_clean();
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit();
}

// Get current user
$username = $_SESSION['username'];
$stmt = $conn->prepare("SELECT id, tipe, nama FROM users WHERE username = ?");
$stmt->bind_param("s", $username);
$stmt->execute();
$currentUser = $stmt->get_result()->fetch_assoc();
$stmt->close();

if (!$currentUser) {
    ob_end_clean();
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'User not found']);
    exit();
}

$currentUserId = (int)$currentUser['id'];

header('Content-Type: application/json');

$action = $_GET['action'] ?? $_POST['action'] ?? '';
$response = ['success' => false, 'data' => [], 'message' => ''];

try {
    switch ($action) {
        case 'get_approval_status':
            // Get approval status for a letter
            $officeId = (int)($_GET['office_id'] ?? 0);
            $response = getApprovalStatus($conn, $officeId, $currentUserId);
            break;

        case 'get_approval_chain':
            // Get full approval chain for a letter
            $officeId = (int)($_GET['office_id'] ?? 0);
            $response = getApprovalChain($conn, $officeId);
            break;

        case 'can_approve':
            // Check if current user can approve this letter
            $officeId = (int)($_GET['office_id'] ?? 0);
            $response = canUserApprove($conn, $officeId, $currentUserId);
            break;

        case 'submit_approval':
            // Submit an approval (with signature and optional comment)
            if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
                $response['message'] = 'Method not allowed';
                break;
            }
            $officeId = (int)($_POST['office_id'] ?? 0);
            $comment = $_POST['comment'] ?? null;
            $signatureFile = $_FILES['signature'] ?? null;
            $response = submitApproval($conn, $officeId, $currentUserId, $signatureFile, $comment);
            break;

        case 'add_comment':
            // Add a comment to an approval (superior only)
            if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
                $response['message'] = 'Method not allowed';
                break;
            }
            $officeId = (int)($_POST['office_id'] ?? 0);
            $comment = $_POST['comment'] ?? '';
            $response = addApprovalComment($conn, $officeId, $currentUserId, $comment);
            break;

        case 'get_comments':
            // Get all comments for a letter
            $officeId = (int)($_GET['office_id'] ?? 0);
            $response = getApprovalComments($conn, $officeId);
            break;

        case 'get_user_initials':
            // Get initials for a user
            $userId = (int)($_GET['user_id'] ?? $currentUserId);
            $response = getUserInitials($conn, $userId);
            break;

        case 'initialize_approval_chain':
            // Initialize approval chain when letter is created
            if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
                $response['message'] = 'Method not allowed';
                break;
            }
            $officeId = (int)($_POST['office_id'] ?? 0);
            $signerId = (int)($_POST['signer_id'] ?? 0);
            $response = initializeApprovalChain($conn, $officeId, $currentUserId, $signerId);
            break;

        default:
            $response['message'] = 'Invalid action';
    }
} catch (Exception $e) {
    $response['message'] = 'Error: ' . $e->getMessage();
    error_log("Approval API Error: " . $e->getMessage() . " in " . $e->getFile() . " on line " . $e->getLine());
} catch (Error $e) {
    $response['message'] = 'Fatal Error: ' . $e->getMessage();
    error_log("Approval API Fatal Error: " . $e->getMessage() . " in " . $e->getFile() . " on line " . $e->getLine());
}

// Clear any accidental HTML output
ob_end_clean();
header('Content-Type: application/json');
echo json_encode($response, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
exit();

// ============================================================
// HELPER FUNCTIONS
// ============================================================

/**
 * Generate QR code with logo watermark for final approval
 */
function generateQRCodeWithLogo($verifyUrl, $qrCodePath, $logoPath)
{
    try {
        // Configure QR code options for PNG output
        $options = new QROptions([
            'version'          => QRCode::VERSION_AUTO,
            'outputType'       => QRCode::OUTPUT_IMAGE_PNG,
            'outputBase64'     => false,
            'eccLevel'         => QRCode::ECC_M,
            'scale'            => 15,
            'imageTransparent' => false,
            'imageBase64'      => false,
        ]);

        // Generate QR code and save as PNG
        $qrcode = new QRCode($options);
        $qrImageData = $qrcode->render($verifyUrl);
        $bytesWritten = file_put_contents($qrCodePath, $qrImageData);

        if ($bytesWritten === false) {
            throw new Exception("Failed to write QR code to disk at " . $qrCodePath);
        }

        // Add logo as watermark to QR code if logo exists
        if (file_exists($logoPath)) {
            // Load QR code image
            $qrImage = @imagecreatefrompng($qrCodePath);
            if ($qrImage === false) {
                error_log("Warning: Failed to load QR code image for watermarking at " . $qrCodePath . ", continuing without logo");
                return true;
            }

            // Enable alpha blending and save alpha for QR image
            imagealphablending($qrImage, true);
            imagesavealpha($qrImage, true);

            $qrWidth = imagesx($qrImage);
            $qrHeight = imagesy($qrImage);

            // Load logo with proper transparency support
            $logo = @imagecreatefrompng($logoPath);
            if ($logo === false) {
                imagedestroy($qrImage);
                error_log("Warning: Failed to load logo image at " . $logoPath . ", continuing without logo");
                return true;
            }

            // Preserve PNG transparency for logo
            imagealphablending($logo, true);
            imagesavealpha($logo, true);

            $logoWidth = imagesx($logo);
            $logoHeight = imagesy($logo);

            // Calculate logo size - 35% of QR size
            $logoSizeRatio = 0.35;
            $logoQrWidth = $qrWidth * $logoSizeRatio;
            $logoQrHeight = $logoQrWidth;

            // Calculate position to center logo
            $logoX = ($qrWidth - $logoQrWidth) / 2;
            $logoY = ($qrHeight - $logoQrHeight) / 2;

            // Create a temporary image for the logo with transparency
            $logoResized = imagecreatetruecolor($logoQrWidth, $logoQrHeight);

            // Enable alpha blending and save transparency
            imagealphablending($logoResized, false);
            imagesavealpha($logoResized, true);

            // Fill with transparent background
            $transparent = imagecolorallocatealpha($logoResized, 0, 0, 0, 127);
            imagefill($logoResized, 0, 0, $transparent);

            // Enable alpha blending for copying
            imagealphablending($logoResized, true);

            // Resize logo onto transparent background
            imagecopyresampled(
                $logoResized,
                $logo,
                0,
                0,
                0,
                0,
                $logoQrWidth,
                $logoQrHeight,
                $logoWidth,
                $logoHeight
            );

            // Use alpha-preserving merge
            imagecopymerge_alpha_approval(
                $qrImage,
                $logoResized,
                $logoX,
                $logoY,
                0,
                0,
                $logoQrWidth,
                $logoQrHeight,
                85 // 85% opacity
            );

            // Save the final QR code with watermark
            imagepng($qrImage, $qrCodePath);

            // Clean up
            imagedestroy($qrImage);
            imagedestroy($logo);
            imagedestroy($logoResized);
        } else {
            // No logo file found, just save the QR code without watermark
            error_log("Logo file not found at " . $logoPath . ", QR code saved without watermark");
        }
    } catch (Exception $logoError) {
        error_log("QR logo watermarking error: " . $logoError->getMessage() . ". QR code may have been saved without watermark.");
        // Don't throw - we already have a valid QR code without the logo
    }

    return true;
}

/**
 * Alpha-preserving image merge function for QR logo
 */
function imagecopymerge_alpha_approval($dst_im, $src_im, $dst_x, $dst_y, $src_x, $src_y, $src_w, $src_h, $pct)
{
    $pct /= 100;
    $w = imagesx($src_im);
    $h = imagesy($src_im);

    imagealphablending($src_im, false);

    $minalpha = 127;
    for ($x = 0; $x < $w; $x++) {
        for ($y = 0; $y < $h; $y++) {
            $alpha = (imagecolorat($src_im, $x, $y) >> 24) & 0xFF;
            if ($alpha < $minalpha) {
                $minalpha = $alpha;
            }
        }
    }

    for ($x = 0; $x < $w; $x++) {
        for ($y = 0; $y < $h; $y++) {
            $colorxy = imagecolorat($src_im, $x, $y);
            $alpha = ($colorxy >> 24) & 0xFF;

            if ($minalpha !== 127) {
                $alpha = 127 + 127 * $pct * ($alpha - 127) / (127 - $minalpha);
            } else {
                $alpha = $alpha * $pct;
            }

            $alphacolorxy = imagecolorallocatealpha($src_im, ($colorxy >> 16) & 0xFF, ($colorxy >> 8) & 0xFF, $colorxy & 0xFF, $alpha);

            if (!imagesetpixel($src_im, $x, $y, $alphacolorxy)) {
                return false;
            }
        }
    }

    imagecopy($dst_im, $src_im, $dst_x, $dst_y, $src_x, $src_y, $src_w, $src_h);
}

/**
 * Embed QR code into existing PDF
 */
function embedQRCodeIntoPDF($pdfPath, $qrCodePath, $nomorSurat)
{
    require_once(__DIR__ . '/../vendor/setasign/fpdf/fpdf.php');
    require_once(__DIR__ . '/../vendor/setasign/fpdi/src/autoload.php');

    $pdf = new \setasign\Fpdi\Fpdi();
    $pageCount = $pdf->setSourceFile($pdfPath);

    // Default QR placement settings
    $qrSize = 38; // mm size
    $signatureBlockLeft = 118; // mm from left edge
    $signatureBlockWidth = 72; // mm
    $xPos = $signatureBlockLeft + ($signatureBlockWidth - $qrSize) / 2;

    // Determine which page to place QR code on
    $qrPageNo = ($pageCount >= 2) ? 2 : 1;
    $qrYPos = 87;

    // Check for SE-DIR placement override
    $nomorSuratUpper = strtoupper(trim($nomorSurat ?? ''));
    if (strpos($nomorSuratUpper, '/SE-DIR') !== false) {
        $qrSize = 36;
        $signatureBlockLeft = 128;
        $signatureBlockWidth = 48;
        $qrYPos = 178;
        $xPos = $signatureBlockLeft + ($signatureBlockWidth - $qrSize) / 2;
        $qrPageNo = 1;
    }

    for ($pageNo = 1; $pageNo <= $pageCount; $pageNo++) {
        $templateId = $pdf->importPage($pageNo);
        $size = $pdf->getTemplateSize($templateId);

        $pdf->AddPage($size['orientation'], [$size['width'], $size['height']]);
        $pdf->useTemplate($templateId);

        // Add QR code to the designated page
        if ($pageNo == $qrPageNo) {
            $pdf->Image($qrCodePath, $xPos, $qrYPos, $qrSize, $qrSize, 'PNG');
        }
    }

    // Save the updated PDF (overwrite the original)
    $pdf->Output('F', $pdfPath);

    return true;
}

/**
 * Get the approval chain for a user based on authorized_signers
 * Returns array of user_ids from creator up to final signer
 */
function getApprovalChainForUser($conn, $signerId, $creatorUserId)
{
    // Get the approval chain string for the signer
    $stmt = $conn->prepare("SELECT user_id, nama, approval_chain FROM authorized_signers WHERE id = ? OR user_id = ?");
    $stmt->bind_param("ii", $signerId, $signerId);
    $stmt->execute();
    $signer = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if (!$signer || empty($signer['approval_chain'])) {
        return [];
    }

    // Parse approval chain (format: "Name1→Name2→Name3" or "Name1???Name2???Name3")
    $separator = strpos($signer['approval_chain'], '→') !== false ? '→' : '???';
    $chainNames = array_map('trim', explode($separator, $signer['approval_chain']));

    // Map names to user_ids
    $chain = [];
    foreach ($chainNames as $name) {
        $stmt = $conn->prepare("SELECT user_id FROM authorized_signers WHERE nama = ? AND is_active = 1");
        $stmt->bind_param("s", $name);
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();
        $stmt->close();

        if ($result && $result['user_id']) {
            $chain[] = (int)$result['user_id'];
        }
    }

    return $chain;
}

/**
 * Get approval status for a letter
 */
function getApprovalStatus($conn, $officeId, $currentUserId)
{
    if ($officeId <= 0) {
        return ['success' => false, 'message' => 'Invalid office ID'];
    }

    // Get letter info
    $stmt = $conn->prepare("SELECT id, dibuat_oleh, penandatangan_id, paraf, signature_path, is_validated, approval_completed FROM office WHERE id = ?");
    $stmt->bind_param("i", $officeId);
    $stmt->execute();
    $letter = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if (!$letter) {
        return ['success' => false, 'message' => 'Letter not found'];
    }

    // Get all approvals for this letter
    $stmt = $conn->prepare("
        SELECT up.*, u.nama as approver_name, i.inisial 
        FROM urutan_paraf up
        LEFT JOIN users u ON up.user_id = u.id
        LEFT JOIN inisial i ON up.user_id = i.user_id
        WHERE up.office_id = ?
        ORDER BY up.urutan ASC
    ");
    $stmt->bind_param("i", $officeId);
    $stmt->execute();
    $result = $stmt->get_result();

    $approvals = [];
    $lastApproved = 0;
    while ($row = $result->fetch_assoc()) {
        $approvals[] = $row;
        if ($row['approved_at'] !== null) {
            $lastApproved = (int)$row['urutan'];
        }
    }
    $stmt->close();

    // Determine if current user can approve
    $canApprove = false;
    $userPosition = null;
    foreach ($approvals as $approval) {
        if ((int)$approval['user_id'] === $currentUserId) {
            $userPosition = $approval;
            // User can approve if: their turn (urutan = lastApproved + 1) AND they haven't approved yet
            if ((int)$approval['urutan'] === ($lastApproved + 1) && $approval['approved_at'] === null) {
                $canApprove = true;
            }
            break;
        }
    }

    return [
        'success' => true,
        'data' => [
            'letter' => $letter,
            'approvals' => $approvals,
            'last_approved_step' => $lastApproved,
            'total_steps' => count($approvals),
            'is_complete' => (bool)$letter['approval_completed'],
            'can_approve' => $canApprove,
            'user_position' => $userPosition,
            'current_paraf' => $letter['paraf'],
            'current_signature' => $letter['signature_path']
        ]
    ];
}

/**
 * Get full approval chain for a letter
 */
function getApprovalChain($conn, $officeId)
{
    if ($officeId <= 0) {
        return ['success' => false, 'message' => 'Invalid office ID'];
    }

    $stmt = $conn->prepare("
        SELECT 
            up.id, up.urutan, up.approved_at, up.signature_path, up.inisial,
            u.id as user_id, u.nama, u.jabatan,
            CASE WHEN up.approved_at IS NOT NULL THEN 'approved' ELSE 'pending' END as status
        FROM urutan_paraf up
        LEFT JOIN users u ON up.user_id = u.id
        WHERE up.office_id = ?
        ORDER BY up.urutan ASC
    ");
    $stmt->bind_param("i", $officeId);
    $stmt->execute();
    $result = $stmt->get_result();

    $chain = [];
    while ($row = $result->fetch_assoc()) {
        $chain[] = $row;
    }
    $stmt->close();

    return [
        'success' => true,
        'data' => $chain
    ];
}

/**
 * Check if current user can approve this letter
 */
function canUserApprove($conn, $officeId, $currentUserId)
{
    $status = getApprovalStatus($conn, $officeId, $currentUserId);

    if (!$status['success']) {
        return $status;
    }

    return [
        'success' => true,
        'data' => [
            'can_approve' => $status['data']['can_approve'],
            'reason' => $status['data']['can_approve'] ? 'Your turn to approve' : 'Not your turn or already approved'
        ]
    ];
}

/**
 * Submit an approval
 */
function submitApproval($conn, $officeId, $currentUserId, $signatureFile = null, $comment = null)
{
    if ($officeId <= 0) {
        return ['success' => false, 'message' => 'Invalid office ID'];
    }

    // Check if user can approve
    $status = getApprovalStatus($conn, $officeId, $currentUserId);
    if (!$status['success'] || !$status['data']['can_approve']) {
        return ['success' => false, 'message' => 'You cannot approve this letter at this time'];
    }

    $userPosition = $status['data']['user_position'];
    if (!$userPosition) {
        return ['success' => false, 'message' => 'You are not in the approval chain for this letter'];
    }

    // Handle signature upload
    $signaturePath = null;
    if ($signatureFile && $signatureFile['error'] === UPLOAD_ERR_OK) {
        $signatureDir = __DIR__ . '/../surat/letter/signatures';
        if (!is_dir($signatureDir)) {
            mkdir($signatureDir, 0755, true);
        }

        $ext = strtolower(pathinfo($signatureFile['name'], PATHINFO_EXTENSION));
        if (!in_array($ext, ['png', 'jpg', 'jpeg', 'gif'])) {
            return ['success' => false, 'message' => 'Invalid signature file type'];
        }

        $filename = 'sig_' . $officeId . '_' . $currentUserId . '_' . time() . '.' . $ext;
        $signaturePath = 'surat/letter/signatures/' . $filename;

        if (!move_uploaded_file($signatureFile['tmp_name'], $signatureDir . '/' . $filename)) {
            return ['success' => false, 'message' => 'Failed to save signature'];
        }
    }

    // Get user's initials
    $stmt = $conn->prepare("SELECT inisial FROM inisial WHERE user_id = ? AND is_active = 1");
    $stmt->bind_param("i", $currentUserId);
    $stmt->execute();
    $inisialRow = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    $userInisial = $inisialRow ? $inisialRow['inisial'] : '';

    // Start transaction
    $conn->begin_transaction();

    try {
        // Update urutan_paraf
        $stmt = $conn->prepare("
            UPDATE urutan_paraf 
            SET approved_at = NOW(), signature_path = ?, inisial = ?, comment = ?
            WHERE office_id = ? AND user_id = ?
        ");
        $stmt->bind_param("sssii", $signaturePath, $userInisial, $comment, $officeId, $currentUserId);
        $stmt->execute();
        $stmt->close();

        // Build accumulated paraf string (in reverse order: latest first)
        $stmt = $conn->prepare("
            SELECT inisial FROM urutan_paraf 
            WHERE office_id = ? AND approved_at IS NOT NULL 
            ORDER BY urutan DESC
        ");
        $stmt->bind_param("i", $officeId);
        $stmt->execute();
        $result = $stmt->get_result();
        $initials = [];
        while ($row = $result->fetch_assoc()) {
            if (!empty($row['inisial'])) {
                $initials[] = $row['inisial'];
            }
        }
        $stmt->close();
        $parafString = implode('/', $initials);

        // Update office table with new paraf and signature (signature replaces previous)
        $stmt = $conn->prepare("UPDATE office SET paraf = ?, signature_path = ? WHERE id = ?");
        $stmt->bind_param("ssi", $parafString, $signaturePath, $officeId);
        $stmt->execute();
        $stmt->close();

        // Check if this was the final approval (penandatangan)
        $stmt = $conn->prepare("
            SELECT COUNT(*) as pending FROM urutan_paraf 
            WHERE office_id = ? AND approved_at IS NULL
        ");
        $stmt->bind_param("i", $officeId);
        $stmt->execute();
        $pendingCount = $stmt->get_result()->fetch_assoc()['pending'];
        $stmt->close();

        $qrCodePath = null;
        $verifyUrl = null;
        $token = null;

        if ($pendingCount == 0) {
            // All approvals complete - generate QR code and mark letter as validated

            // Get letter info for PDF embedding
            $stmt = $conn->prepare("SELECT nomor_surat, file_surat FROM office WHERE id = ?");
            $stmt->bind_param("i", $officeId);
            $stmt->execute();
            $letterInfo = $stmt->get_result()->fetch_assoc();
            $stmt->close();

            // Generate unique token for QR code
            $token = bin2hex(random_bytes(16));

            // Create QR code directory if not exists
            $qrDir = __DIR__ . '/../surat/letter/qrcodes';
            if (!is_dir($qrDir)) {
                mkdir($qrDir, 0755, true);
            }

            $qrCodeFilename = 'qr_' . $officeId . '_' . time() . '.png';
            $qrCodePath = 'surat/letter/qrcodes/' . $qrCodeFilename;
            $qrCodeFullPath = __DIR__ . '/../surat/letter/qrcodes/' . $qrCodeFilename;

            // Build verify URL
            $protocol = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? "https" : "http";
            $host = $_SERVER['HTTP_HOST'];
            $verifyUrl = $protocol . "://" . $host . "/jasa_marga/user/verify-letter.php?token=" . $token;

            // Generate QR code with logo
            $logoPath = __DIR__ . '/../img/logojpt-new/Logo QR New.png';
            try {
                generateQRCodeWithLogo($verifyUrl, $qrCodeFullPath, $logoPath);
            } catch (Exception $qrError) {
                // Log QR error but don't fail the approval
                error_log("QR Code generation error for office_id=$officeId: " . $qrError->getMessage());
            } catch (Error $qrFatalError) {
                // Log fatal error but don't fail the approval
                error_log("QR Code generation fatal error for office_id=$officeId: " . $qrFatalError->getMessage());
            }

            // Embed QR code into PDF if file exists
            if (!empty($letterInfo['file_surat'])) {
                $pdfPath = __DIR__ . '/../' . $letterInfo['file_surat'];
                if (file_exists($pdfPath)) {
                    try {
                        embedQRCodeIntoPDF($pdfPath, $qrCodeFullPath, $letterInfo['nomor_surat']);
                    } catch (Exception $pdfError) {
                        // Log PDF embedding error but don't fail the approval
                        error_log("PDF QR Code embedding error for office_id=$officeId: " . $pdfError->getMessage());
                    } catch (Error $pdfFatalError) {
                        // Log fatal error but don't fail the approval
                        error_log("PDF QR Code fatal error for office_id=$officeId: " . $pdfFatalError->getMessage());
                    }
                }
            }

            // Update office with QR code, token, and validation status
            $stmt = $conn->prepare("UPDATE office SET approval_completed = 1, is_validated = 1, qr_code = ?, validation_token = ?, validated_at = NOW() WHERE id = ?");
            $stmt->bind_param("ssi", $qrCodePath, $token, $officeId);
            $stmt->execute();
            $stmt->close();
        }

        $conn->commit();

        return [
            'success' => true,
            'message' => $pendingCount == 0 ? 'Final approval submitted. Letter validated and QR code generated.' : 'Approval submitted successfully',
            'data' => [
                'paraf' => $parafString,
                'signature_path' => $signaturePath,
                'is_complete' => ($pendingCount == 0),
                'qr_code_path' => $qrCodePath,
                'verify_url' => $verifyUrl
            ]
        ];
    } catch (Exception $e) {
        $conn->rollback();
        return ['success' => false, 'message' => 'Failed to submit approval: ' . $e->getMessage()];
    }
}

/**
 * Add a comment to a letter (superior in chain only)
 */
function addApprovalComment($conn, $officeId, $currentUserId, $comment)
{
    if ($officeId <= 0 || empty(trim($comment))) {
        return ['success' => false, 'message' => 'Invalid parameters'];
    }

    // Check if user is in approval chain and is superior (can comment on subordinates)
    $stmt = $conn->prepare("SELECT urutan FROM urutan_paraf WHERE office_id = ? AND user_id = ?");
    $stmt->bind_param("ii", $officeId, $currentUserId);
    $stmt->execute();
    $userPos = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if (!$userPos) {
        return ['success' => false, 'message' => 'You are not in the approval chain for this letter'];
    }

    // Insert comment
    $stmt = $conn->prepare("
        INSERT INTO approval_comments (office_id, user_id, comment) 
        VALUES (?, ?, ?)
    ");
    $stmt->bind_param("iis", $officeId, $currentUserId, $comment);

    if ($stmt->execute()) {
        $commentId = $conn->insert_id;
        $stmt->close();

        return [
            'success' => true,
            'message' => 'Comment added',
            'data' => ['id' => $commentId]
        ];
    }

    $stmt->close();
    return ['success' => false, 'message' => 'Failed to add comment'];
}

/**
 * Get all comments for a letter
 */
function getApprovalComments($conn, $officeId)
{
    if ($officeId <= 0) {
        return ['success' => false, 'message' => 'Invalid office ID'];
    }

    $stmt = $conn->prepare("
        SELECT ac.*, u.nama as commenter_name, u.jabatan as commenter_jabatan
        FROM approval_comments ac
        LEFT JOIN users u ON ac.user_id = u.id
        WHERE ac.office_id = ?
        ORDER BY ac.created_at DESC
    ");
    $stmt->bind_param("i", $officeId);
    $stmt->execute();
    $result = $stmt->get_result();

    $comments = [];
    while ($row = $result->fetch_assoc()) {
        $comments[] = $row;
    }
    $stmt->close();

    return [
        'success' => true,
        'data' => $comments
    ];
}

/**
 * Get user initials
 */
function getUserInitials($conn, $userId)
{
    $stmt = $conn->prepare("SELECT * FROM inisial WHERE user_id = ? AND is_active = 1");
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    return [
        'success' => true,
        'data' => $result
    ];
}

/**
 * Initialize approval chain when letter is created
 * Creates urutan_paraf entries based on authorized_signers.approval_chain
 */
function initializeApprovalChain($conn, $officeId, $creatorUserId, $signerId)
{
    if ($officeId <= 0) {
        return ['success' => false, 'message' => 'Invalid office ID'];
    }

    // Get the signer's info
    // First try to find by authorized_signers.id, then by user_id
    $stmt = $conn->prepare("
        SELECT id, user_id, nama
        FROM authorized_signers 
        WHERE (id = ? OR user_id = ?) AND is_active = 1
        LIMIT 1
    ");
    $stmt->bind_param("ii", $signerId, $signerId);
    $stmt->execute();
    $signer = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if (!$signer) {
        return ['success' => false, 'message' => 'Signer not found'];
    }

    // DEBUG: Log signer info
    error_log("DEBUG initializeApprovalChain - Office ID: $officeId, Creator ID: $creatorUserId, Signer ID: $signerId");
    error_log("DEBUG Signer found: ID=" . $signer['user_id'] . ", Name=" . $signer['nama']);

    // When a penandatangan is explicitly selected by the creator,
    // the approval chain should only require that one person's approval (0/1)
    // NOT their entire approval chain hierarchy
    $approvalChain = [
        [
            'user_id' => (int)$signer['user_id'],
            'nama' => $signer['nama']
        ]
    ];

    error_log("DEBUG Approval chain created with " . count($approvalChain) . " steps: " . json_encode($approvalChain));

    // Delete existing chain entries for this letter
    $stmt = $conn->prepare("DELETE FROM urutan_paraf WHERE office_id = ?");
    $stmt->bind_param("i", $officeId);
    $stmt->execute();
    error_log("DEBUG DELETE from urutan_paraf: office_id=$officeId, affected_rows=" . $stmt->affected_rows);
    $stmt->close();

    // Insert new chain (just the selected penandatangan)
    $urutan = 1;
    foreach ($approvalChain as $member) {
        $stmt = $conn->prepare("
            INSERT INTO urutan_paraf (office_id, user_id, urutan) 
            VALUES (?, ?, ?)
        ");
        $stmt->bind_param("iii", $officeId, $member['user_id'], $urutan);
        if ($stmt->execute()) {
            error_log("DEBUG Inserted urutan_paraf: office_id=$officeId, user_id=" . $member['user_id'] . ", urutan=$urutan");
        } else {
            error_log("DEBUG FAILED to insert urutan_paraf: " . $stmt->error);
        }
        $stmt->close();
        $urutan++;
    }

    // Verify what was inserted
    $verifyStmt = $conn->prepare("SELECT COUNT(*) as total FROM urutan_paraf WHERE office_id = ?");
    $verifyStmt->bind_param("i", $officeId);
    $verifyStmt->execute();
    $verifyResult = $verifyStmt->get_result()->fetch_assoc();
    $verifyStmt->close();
    error_log("DEBUG After insert, urutan_paraf count for office_id=$officeId: " . $verifyResult['total']);

    // Update office with penandatangan_id
    $signerUserId = (int)$signer['user_id'];
    $stmt = $conn->prepare("UPDATE office SET penandatangan_id = ? WHERE id = ?");
    $stmt->bind_param("ii", $signerUserId, $officeId);
    $stmt->execute();
    $stmt->close();

    return [
        'success' => true,
        'message' => 'Approval chain initialized',
        'data' => [
            'chain' => $approvalChain,
            'total_steps' => count($approvalChain)
        ]
    ];
}
