<?php
session_start();
include(__DIR__ . '/../includes/koneksi.php');
include(__DIR__ . '/../includes/functions.php');

if (!isset($_SESSION['username'])) {
    header('Location: ../login.php');
    exit();
}

// Cek apakah tipe user adalah admin
$username = $_SESSION['username'];
$query = "SELECT tipe, nama FROM users WHERE username = ?";
$stmt = $conn->prepare($query);
$stmt->bind_param("s", $username);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->fetch_assoc();

if (!$user || $user['tipe'] !== 'admin') {
    header('Location: ../login.php');
    exit();
}

date_default_timezone_set("Asia/Jakarta");
$hour = date("H");
$today = date('Y-m-d');
$nowTime = date('H:i');

// Hitung agenda aktif (hari ini dan akan datang)
$countQuery = "SELECT COUNT(*) as jumlah FROM agenda WHERE tanggal > ? OR (tanggal = ? AND waktu >= ?)";
$countStmt = $conn->prepare($countQuery);
$countStmt->bind_param("sss", $today, $today, $nowTime);
$countStmt->execute();
$countResult = $countStmt->get_result()->fetch_assoc();
$jumlahAgendaAktif = $countResult['jumlah'] ?? 0;

// Load tipe surat list for filter dropdown
$tipeSuratList = [];
if ($r = $conn->query("SELECT id, kode, nama FROM tipe_surat ORDER BY nama")) {
    $tipeSuratList = $r->fetch_all(MYSQLI_ASSOC);
    $r->free();
}

/**
 * Helper function to check if penandatangan has approved a letter
 * Admin can only approve after penandatangan (last in chain) has approved
 */
function isPenandatanganApproved($conn, $officeId)
{
    // Check if urutan_paraf exists for this letter
    $stmt = $conn->prepare("
        SELECT COUNT(*) as total,
               SUM(CASE WHEN approved_at IS NOT NULL THEN 1 ELSE 0 END) as approved_count
        FROM urutan_paraf 
        WHERE office_id = ?
    ");
    $stmt->bind_param("i", $officeId);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    // If no approval chain exists, check old system (legacy)
    if (!$result || $result['total'] == 0) {
        // Legacy system - check approval_completed flag
        $stmt = $conn->prepare("SELECT approval_completed FROM office WHERE id = ?");
        $stmt->bind_param("i", $officeId);
        $stmt->execute();
        $letter = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        return $letter && $letter['approval_completed'] == 1;
    }

    // New system - all approvals in chain must be complete
    return $result['total'] > 0 && $result['total'] == $result['approved_count'];
}

/**
 * Get approval chain status summary for a letter
 */
function getApprovalChainSummary($conn, $officeId)
{
    $stmt = $conn->prepare("
        SELECT up.*, u.nama as approver_name, i.inisial
        FROM urutan_paraf up
        LEFT JOIN users u ON up.user_id = u.id
        LEFT JOIN inisial i ON up.user_id = i.user_id AND i.is_active = 1
        WHERE up.office_id = ?
        ORDER BY up.urutan ASC
    ");
    $stmt->bind_param("i", $officeId);
    $stmt->execute();
    $result = $stmt->get_result();

    $chain = [];
    $parafParts = [];
    $totalSteps = 0;
    $completedSteps = 0;

    while ($row = $result->fetch_assoc()) {
        $chain[] = $row;
        $totalSteps++;
        if ($row['approved_at'] !== null) {
            $completedSteps++;
            if (!empty($row['inisial'])) {
                $parafParts[] = $row['inisial'];
            }
        }
    }
    $stmt->close();

    error_log("DEBUG getApprovalChainSummary - Office ID: $officeId, Total steps: $totalSteps, Completed: $completedSteps");
    foreach ($chain as $idx => $item) {
        error_log("DEBUG Chain step " . ($idx + 1) . ": user_id=" . $item['user_id'] . ", approver_name=" . $item['approver_name'] . ", urutan=" . $item['urutan'] . ", approved_at=" . $item['approved_at']);
    }

    return [
        'chain' => $chain,
        'total' => $totalSteps,
        'completed' => $completedSteps,
        'is_complete' => ($totalSteps > 0 && $totalSteps == $completedSteps),
        'paraf_display' => implode('/', $parafParts)
    ];
}
?>

<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <!-- Favicon -->
    <link href="img/baru/logojp.png" rel="icon">

    <title>Record Surat</title>

    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css">

    <style>
        .greeting {
            text-align: center;
            color: <?php echo $color;
                    ?>;
            animation: fadeIn 2s ease-in-out;
        }

        .icon {
            margin-top: 40px;
            font-size: 50px;
            margin-bottom: 0px;
            animation: bounce 1.5s infinite;
        }

        @keyframes bounce {

            0%,
            100% {
                transform: translateY(0);
            }

            50% {
                transform: translateY(-10px);
            }
        }

        .sidebar {
            background-color: #0c50e1;
            color: white;
            min-height: 100vh;
            padding: 15px;
        }

        .sidebar a {
            color: white;
            text-decoration: none;
            display: block;
            padding: 10px;
            margin: 5px 0;
            border-radius: 5px;
        }

        .sidebar a:hover,
        .sidebar a.active {
            background-color: #ffc107;
            color: #0047ab;
        }

        .btn-success {
            background-color: #ffc107;
            /* Kuning mencolok */
            border-color: #ffc107;
            color: #0047ab;
            font-weight: bold;
        }

        /* Warna untuk header tabel */
        .table thead {
            background-color: #0047ab;
            /* Biru Jasamarga */
            color: #0c50e1;
            text-transform: uppercase;
            font-weight: bold;
            color: #ffc107;
        }

        .table thead th {
            background-color: #0c50e1;
            /* Biru Jasamarga */
            color: #ffc107;
        }

        .sidebar-animated {
            background-color: #001eff;
            ;
            padding: 20px 10px;
            font-family: Arial, sans-serif;
            height: 100vh;
            width: 250px;
            transition: all 0.3s ease;
            max-height: 100vh;
            /* Limits sidebar height to viewport */
            overflow-y: auto;
            /* Enables vertical scroll */
            overflow-x: hidden;
            /* Hides horizontal scroll */
            padding-bottom: 20px;
            /* Adds extra padding to avoid cut-off */
            margin-bottom: 0;
            /* Ensures no unwanted margin at the bottom */
            scrollbar-width: thin;
        }

        /* Optional: Styling untuk scrollbar pada Chrome, Edge, dan Safari */
        .sidebar-animated::-webkit-scrollbar {
            width: 8px;
            /* Mengatur lebar scrollbar */
        }

        .sidebar-animated::-webkit-scrollbar-thumb {
            background-color: #e1b12c;
            /* Warna scrollbar */
            border-radius: 10px;
            /* Membuat sudut melengkung */
        }

        .sidebar-animated::-webkit-scrollbar-thumb:hover {
            background-color: #ffc107;
            /* Warna scrollbar saat di-hover */
        }

        .sidebar-animated a {
            color: #FFFFFFFF;
            text-decoration: none;
            font-size: 18px;
            display: flex;
            align-items: center;
            padding: 10px;
            border-radius: 5px;
            transition: background-color 0.3s ease, transform 0.3s ease;
        }

        .sidebar-animated a:hover {
            background-color: #ffc107;
            color: #fff;
            transform: translateX(5px);
        }

        .sidebar-animated a i {
            margin-right: 10px;
        }

        .sidebar-animated .submenu-toggle {
            cursor: pointer;
            transition: color 0.3s;
        }

        .sidebar-animated .submenu-toggle:hover .arrow-icon {
            transform: rotate(90deg);
            color: #001eff;
        }

        .sidebar-animated .submenu {
            padding-left: 1px;
            overflow: hidden;
            transition: max-height 0.4s ease;
        }

        .sidebar-animated .collapse.show {
            max-height: 300px;
            /* adjust as needed */
            transition: max-height 0.4s ease-in;
        }

        /* Terakhir Diedit cell: allow vertical scroll when content is long */
        td.td-edited,
        th.col-edited {
            max-height: 80px;
            overflow-y: auto;
            white-space: normal;
        }

        .arrow-icon {
            margin-left: auto;
            transition: transform 0.3s;
        }

        /* Hover and animation effects for submenu */
        .submenu a {
            font-size: 16px;
            color: #FFFFFFFF;
            transition: color 0.3s ease, transform 0.3s ease;
        }

        .submenu a:hover {
            color: #001eff;
            transform: translateX(5px);
        }

        /* Sidebar active link style */
        .sidebar-animated a.active {
            background-color: #ffc107;
            /* Warna latar belakang khusus untuk menu aktif */
            color: #fff;
            /* Warna teks untuk menu aktif */
            font-weight: bold;
        }

        .sidebar-animated a.active i {
            color: #fff;
            /* Warna ikon untuk menu aktif */
        }

        /* Navbar custom styling */
        .navbar-custom {
            background-color: #001eff;
            /* Same as sidebar background color */
            transition: background-color 0.3s ease;
            padding: 15px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            /* Adds a subtle shadow */
        }

        /* Logo and Sidebar Toggle */
        .navbar-custom .navbar-brand img {
            transition: transform 0.3s ease;
        }

        .navbar-custom .navbar-brand:hover img {
            transform: scale(1.05);
        }

        /* Toggle Sidebar button */
        .toggle-sidebar-btn {
            color: #fff;
            font-size: 1.3rem;
            cursor: pointer;
            margin-left: 10px;
            transition: color 0.3s ease;
        }

        .toggle-sidebar-btn:hover {
            color: #ffc107;
        }

        /* Animasi untuk kartu */
        @keyframes fadeInUp {
            0% {
                opacity: 0;
                transform: translateY(20px);
            }

            100% {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .highlight-card {
            border-radius: 15px;
            /* Membuat sudut melengkung */
            transition: transform 0.3s ease, box-shadow 0.3s ease;
            animation: fadeInUp 0.8s ease-in-out;
        }

        /* Hover efek */
        .highlight-card:hover {
            transform: translateY(-10px);
            box-shadow: 0 10px 20px rgba(0, 0, 0, 0.2);
        }

        /* Gaya untuk teks */
        .highlight-card h3 {
            font-size: 2.5rem;
            font-weight: bold;
            margin: 0;
        }

        .highlight-card p {
            font-size: 1.1rem;
            margin-top: 5px;
            opacity: 0.9;
        }

        /* Footer styling */
        .highlight-card .card-footer {
            background: rgba(0, 0, 0, 0.1);
            border-top: 1px solid rgba(255, 255, 255, 0.2);
        }

        .highlight-card .card-footer a {
            text-decoration: none;
            font-weight: bold;
            transition: color 0.3s ease;
        }

        .highlight-card .card-footer a:hover {
            color: #ffd700;
            /* Warna emas saat hover */
        }

        /* Animasi untuk ikon */
        .highlight-card i {
            animation: bounce 1.5s infinite;
        }

        /* Bounce animasi untuk ikon */
        @keyframes bounce {

            0%,
            20%,
            50%,
            80%,
            100% {
                transform: translateY(0);
            }

            40% {
                transform: translateY(-10px);
            }

            60% {
                transform: translateY(-5px);
            }
        }

        .pagination .page-link {
            color: #007bff;
        }

        .pagination .active .page-link {
            background-color: #007bff;
            border-color: #007bff;
            color: white;
        }


        .search-container-new {
            position: relative;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .search-box-new {
            display: flex;
            align-items: center;
            transition: all 0.3s ease;
        }

        .search-input-new {
            border-radius: 25px;
            padding: 8px 12px;
            border: 1px solid #ced4da;
            background-color: #ffc107;
            color: #001eff;
            transition: width 0.4s ease, box-shadow 0.3s ease;
            width: 184px;
        }

        .search-input-new::placeholder {
            color: #001eff;
            font-style: italic;
            opacity: 0.8;
        }

        .search-input-new:focus {
            width: 250px;
            background-color: #ffffff;
            border-color: #ffc107;
            box-shadow: 0 0 8px rgba(255, 193, 7, 0.6);
            /* Glow kuning */
            outline: none;
        }

        .search-btn-new {
            background-color: #001eff;
            color: #fff;
            border: none;
            border-radius: 20px;
            padding: 6px 20px;
            font-size: 14px;
            cursor: pointer;
            transition: background-color 0.3s ease, transform 0.3s ease;
            display: none;
        }

        .search-btn-new.show {
            display: inline-block;
        }
    </style>
</head>

<body>

    <?php if (isset($_GET['updated'])): ?>
        <script>
            Swal.fire({
                icon: 'success',
                title: 'Berhasil!',
                text: 'Surat berhasil diperbarui.',
                showConfirmButton: true
            });
        </script>

    <?php elseif (isset($_GET['deleted'])): ?>
        <script>
            Swal.fire({
                icon: 'success',
                title: 'Berhasil!',
                text: 'Surat berhasil dihapus.',
                showConfirmButton: true
            });
        </script>
    <?php endif; ?>

    <div class="d-flex">

        <!-- Sidebar -->
        <div class="sidebar">
            <a href="#"><img src="/jasa_marga/img/logojpt-new/Logo Header New.png" alt="#" style="width: 200px"></a>
            <a href="dashboard.php"><i class="fas fa-bars"></i> Dashboard</a>
            <a href="gaji.php"><i class="fas fa-money-bill-wave"></i> Gaji Pegawai</a>
            <a href="surat.php"><i class="fas fa-file-alt"></i> Surat Lainnya</a>

            <a href="#letterSubmenu" class="submenu-toggle" data-bs-toggle="collapse" data-bs-target="#letterSubmenu" aria-expanded="false">
                <i class="fas fa-envelope-open-text"></i> Surat
                <i class="fas fa-caret-down arrow-icon ml-1" id="locArrow"></i>
            </a>
            <div id="letterSubmenu" class="collapse submenu">
                <a href="create-letter.php"><i class="fas fa-pen-to-square"></i> Buat Surat</a>
                <a href="record-letter.php"><i class="fas fa-folder-open"></i> Record Surat</a>
                <a href="manajemen-surat.php"><i class="fas fa-cog"></i> Manajemen Data Surat</a>
                <a href="letter-order.php" class="<?= (basename($_SERVER['PHP_SELF']) === 'letter-order.php') ? 'active' : ''; ?>"><i class="fas fa-list-ol"></i> Urutan Surat</a>
                <a href="manage-inisial.php" class="<?= (basename($_SERVER['PHP_SELF']) === 'manage-inisial.php') ? 'active' : ''; ?>"><i class="fas fa-signature"></i> Manajemen Inisial</a>
            </div>
            <a href="#todolistSubmenu" class="submenu-toggle" data-bs-toggle="collapse"
                data-bs-target="#todolistSubmenu">
                <i class="fas fa-tasks"></i> To Do List Pekerjaan
                <i class="fas fa-caret-down arrow-icon ml-1" id="locArrow"></i>
            </a>
            <div id="todolistSubmenu" class="collapse submenu">
                <a href="todolist.php"><i class="fas fa-database"></i> Manajemen Data</a>
                <a href="record-todolist.php"><i class="fas fa-history"></i> Record To Do List</a>
            </div>

            <a href="#agendaSubmenu" class="submenu-toggle" data-bs-toggle="collapse" data-bs-target="#agendaSubmenu">
                <i class="fas fa-calendar-check"></i> Agenda Kerja
                <?php if ($jumlahAgendaAktif > 0): ?>
                    <span class="badge bg-danger ms-2"><?= $jumlahAgendaAktif ?></span>
                <?php endif; ?>
                <i class="fas fa-caret-down arrow-icon ml-1" id="agendaArrow"></i>
            </a>

            <div id="agendaSubmenu" class="collapse submenu">
                <a href="list-agenda.php">
                    <i class="fas fa-clock"></i> List
                    <?php if ($jumlahAgendaAktif > 0): ?>
                        <span class="badge bg-danger ms-1"><?= $jumlahAgendaAktif ?></span>
                    <?php endif; ?>
                </a>
                <a href="record-agenda.php">
                    <i class="fas fa-clipboard-list"></i> Record
                </a>
            </div>
            <a href="sub-unit-kerja.php"><i class="fas fa-sitemap"></i> Manajemen Unit Kerja</a>
            <a href="#systemSubmenu" class="submenu-toggle" data-bs-toggle="collapse" data-bs-target="#systemSubmenu">
                <i class="fas fa-newspaper"></i> Interface
                <i class="fas fa-caret-down arrow-icon ml-1" id="systemArrow"></i>
            </a>
            <div id="systemSubmenu" class="collapse submenu">
                <a href="kontak.php"><i class="fas fa-paper-plane"></i> Contact</a>
                <a href="footer.php"><i class="fas fa-shoe-prints"></i> Footer </a>
                <a href="header.php"><i class="fas fa-heading"></i> Header </a>
                <a href="home.php"><i class="fas fa-house-user"></i> Home </a>
                <a href="nav.php"><i class="fas fa-landmark"></i> Navbar </a>
                <a href="prof.php"><i class="fas fa-user-circle"></i> Profil </a>
                <a href="trav.php" class=""><i class="fas fa-font"></i> Travoy </a>
                <a href="jaringan.php" class=""><i class="fas fa-car"></i> Jaringan Tol</a>
                <a href="chatbot1.php" class=""><i class="fas fa-comments"></i> Chatbot </a>
                <a href="gratifi.php"><i class="fas fa-gift"></i> Gratifikasi</a>
            </div>
            <a href="#locSubmenu" class="submenu-toggle" data-bs-toggle="collapse" data-bs-target="#locSubmenu">
                <i class="fas fa-images" class="active"></i> Tampilan Gambar
                <i class="fas fa-caret-down arrow-icon ml-1" id="locArrow"></i>
            </a>
            <div id="locSubmenu" class="collapse submenu">
                <a href="berita.php"><i class="fas fa-newspaper"></i> Berita </a>
                <a href="sosial.php"><i class="fas fa-file"></i> Tanggung jawab </a>
                <a href="direk.php"><i class="fas fa-user-tie"></i> Direksi </a>
                <a href="komis.php"><i class="fas fa-users"></i> Komisaris </a>
                <a href="organisasi.php"><i class="fas fa-layer-group"></i> Organisasi </a>
                <a href="gerbang.php"><i class="fas fa-road"></i> Gerbang Tol </a>
            </div>
            <a href="logout.php"><i class="fas fa-sign-out-alt"></i> Logout</a>
        </div>

        <!-- Tabel Surat -->
        <div id="mainContent" class="container-fluid py-3">
            <div class="card shadow-sm border-0 rounded-3">
                <div class="card-header bg-primary text-white d-flex justify-content-between align-items-center">
                    <div class="d-flex align-items-center gap-3">
                        <h5 class="mb-0">Daftar Surat</h5>
                        <div class="search-container" style="position: relative; flex: 1; max-width: 300px;">
                            <div class="input-group input-group-sm">
                                <input type="text" id="searchInputNew" class="form-control" placeholder="Cari nomor, nama...">
                                <button class="btn btn-primary" id="searchButtonNew" type="button" style="display: none;">
                                    <i class="fas fa-search"></i> Cari
                                </button>
                            </div>
                        </div>
                        <div>
                            <select id="filterTipeSurat" class="form-select form-select-sm" aria-label="Filter Tipe Surat">
                                <option value="all" data-kode="ALL">Tampilkan Semua</option>
                                <?php foreach ($tipeSuratList as $t): ?>
                                    <option value="<?= (int)$t['id'] ?>" data-kode="<?= htmlspecialchars($t['kode']) ?>"><?= htmlspecialchars($t['nama']) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    <a href="create-letter.php" class="btn btn-warning btn-sm fw-bold">+ Buat Surat</a>
                </div>

                <div class="card-body px-0">
                    <div class="table-responsive">
                        <table class="table table-bordered table-hover mb-0 align-middle">
                            <thead class="table-light">
                                <tr class="text-center">
                                    <th style="width: 12%">Nomor Surat</th>
                                    <th style="width: 12%">Dibuat Oleh</th>
                                    <th style="width: 12%">Penandatangan</th>
                                    <th class="col-jenis" style="width: 14%">Jenis Surat</th>
                                    <th class="col-klasifikasi" style="width: 12%">Klasifikasi</th>
                                    <th class="col-edited" style="width: 14%">Terakhir Diedit</th>
                                    <th style="width: 8%">File</th>
                                    <th style="width: 16%">Aksi</th>
                                </tr>
                            </thead>

                            <tbody>
                                <?php
                                $sql = "
                                    SELECT 
                                        o.id, o.nomor_surat, o.created_at, o.file_surat,
                                        o.dari_tipe, o.dari_pribadi, o.qr_code, o.is_validated, o.tipe_surat_id,
                                        o.editor_history,
                                        ts.kode AS tipe_surat_kode, ts.nama AS tipe_surat_name,
                                        js.nama AS jenis_surat,
                                        ka.nama AS klasifikasi_arsip,
                                        creator.nama AS nama_pembuat,
                                        creator.jabatan AS jabatan_pembuat,
                                        signer.nama AS nama_penandatangan,
                                        signer.jabatan AS jabatan_penandatangan
                                FROM office o
                                LEFT JOIN jenis_surat js ON js.id = o.jenis_surat_id
                                LEFT JOIN tipe_surat ts ON ts.id = o.tipe_surat_id
                                LEFT JOIN klasifikasi_arsip ka ON ka.id = o.klasifikasi_arsip_id
                                LEFT JOIN users creator ON creator.id = o.dibuat_oleh
                                LEFT JOIN users signer ON signer.id = o.dari_pribadi
                                ORDER BY o.id DESC
                            ";

                                $res = $conn->query($sql);

                                if ($res && $res->num_rows > 0):
                                    while ($row = $res->fetch_assoc()):
                                        $createdAt = $row['created_at'];
                                        $createdFormatted = $createdAt;
                                        if (preg_match('/^\d{4}-\d{2}-\d{2}/', $createdAt)) {
                                            // Convert to Indonesian date format "3 Desember 2025 pukul 14:30"
                                            $timestamp = strtotime($createdAt);
                                            if ($timestamp !== false) {
                                                $bulanIndonesia = [
                                                    1 => 'Januari',
                                                    2 => 'Februari',
                                                    3 => 'Maret',
                                                    4 => 'April',
                                                    5 => 'Mei',
                                                    6 => 'Juni',
                                                    7 => 'Juli',
                                                    8 => 'Agustus',
                                                    9 => 'September',
                                                    10 => 'Oktober',
                                                    11 => 'November',
                                                    12 => 'Desember'
                                                ];
                                                $hari = date('j', $timestamp);
                                                $bulan = $bulanIndonesia[date('n', $timestamp)];
                                                $tahun = date('Y', $timestamp);
                                                $waktu = date('H:i', $timestamp);
                                                $createdFormatted = "$hari $bulan $tahun pukul $waktu";
                                            } else {
                                                [$datePart, $timePart] = explode(" ", $createdAt);
                                                [$y, $m, $d] = explode('-', $datePart);
                                                $createdFormatted = "$timePart $d-$m-$y";
                                            }
                                        }

                                        // Determine who created the letter
                                        if (!empty($row['nama_pembuat'])) {
                                            $dibuatOleh = $row['nama_pembuat'];
                                        } else {
                                            $dibuatOleh = 'Unknown User';
                                        }

                                        // Determine who signed the letter
                                        if (!empty($row['nama_penandatangan'])) {
                                            $penandatangan = $row['nama_penandatangan'];
                                        } else {
                                            $penandatangan = $dibuatOleh; // Fall back to creator
                                        }
                                ?>
                                        <tr data-tipe-id="<?= (int)$row['tipe_surat_id'] ?>" data-tipe-kode="<?= htmlspecialchars($row['tipe_surat_kode'] ?? '') ?>">
                                            <td class="text-center">
                                                <div class="fw-normal"><?= htmlspecialchars($row['nomor_surat']) ?></div>
                                                <small class="text-muted">Dibuat pada
                                                    <?= htmlspecialchars($createdFormatted) ?></small>
                                            </td>
                                            <td class="text-center">
                                                <div class="fw-bold"><?= htmlspecialchars($dibuatOleh) ?></div>
                                            </td>
                                            <td class="text-center">
                                                <div class="fw-bold"><?= htmlspecialchars($penandatangan) ?></div>
                                                <?php
                                                // Get approval chain status for this letter (read-only display)
                                                $chainStatus = getApprovalChainSummary($conn, (int)$row['id']);
                                                ?>

                                                <?php if ($row['is_validated'] != 1): ?>
                                                    <a href="verify-letter.php?id=<?= (int)$row['id'] ?>" target="_blank" rel="noopener noreferrer" class="btn btn-sm btn-primary fw-bold mt-2">
                                                        <i class="bi bi-eye-fill"></i> Check
                                                    </a>

                                                    <?php if ($chainStatus['total'] > 0): ?>
                                                        <!-- Show approval chain status (read-only for admin) -->
                                                        <div class="mt-2">
                                                            <span class="badge bg-<?= $chainStatus['is_complete'] ? 'success' : 'warning' ?> text-<?= $chainStatus['is_complete'] ? 'white' : 'dark' ?>">
                                                                <i class="bi bi-list-check"></i>
                                                                <?= $chainStatus['completed'] ?>/<?= $chainStatus['total'] ?> Approved
                                                            </span>
                                                            <?php if (!empty($chainStatus['paraf_display'])): ?>
                                                                <br><small class="text-success"><?= htmlspecialchars($chainStatus['paraf_display']) ?></small>
                                                            <?php endif; ?>
                                                        </div>

                                                        <?php if (!$chainStatus['is_complete']): ?>
                                                            <!-- Waiting for approval chain to complete -->
                                                            <span class="badge bg-secondary mt-1">
                                                                <i class="bi bi-hourglass-split"></i> Menunggu Approval
                                                            </span>
                                                        <?php else: ?>
                                                            <!-- Approval complete but not yet validated (QR pending) -->
                                                            <span class="badge bg-info mt-1">
                                                                <i class="bi bi-clock-history"></i> QR Code akan di-generate
                                                            </span>
                                                        <?php endif; ?>
                                                    <?php else: ?>
                                                        <!-- No approval chain exists -->
                                                        <span class="badge bg-secondary mt-1">
                                                            <i class="bi bi-exclamation-circle"></i> Belum ada approval chain
                                                        </span>
                                                    <?php endif; ?>
                                                <?php else: ?>
                                                    <div class="mt-2">
                                                        <span class="badge bg-success">
                                                            <i class="bi bi-check-circle-fill"></i> Approved
                                                        </span>
                                                        <?php if (!empty($chainStatus['paraf_display'])): ?>
                                                            <br><small class="text-success"><?= htmlspecialchars($chainStatus['paraf_display']) ?></small>
                                                        <?php endif; ?>
                                                    </div>
                                                <?php endif; ?>
                                            </td>
                                            <!-- kode jabatan removed from separate column to avoid misalignment -->
                                            <td class="col-jenis"><?= htmlspecialchars($row['jenis_surat']) ?></td>
                                            <td class="col-klasifikasi"><?= htmlspecialchars($row['klasifikasi_arsip']) ?></td>
                                            <td class="col-edited td-edited">
                                                <?php
                                                $editorHistory = [];
                                                if (!empty($row['editor_history'])) {
                                                    $maybe = json_decode($row['editor_history'], true);
                                                    if (is_array($maybe)) {
                                                        $editorHistory = $maybe;
                                                    } else {
                                                        // Fallback: comma-separated list
                                                        $parts = array_map('trim', explode(',', $row['editor_history']));
                                                        foreach ($parts as $p) {
                                                            if (empty($p)) continue;
                                                            $editorHistory[] = ['name' => $p, 'edited_at' => null];
                                                        }
                                                    }
                                                }
                                                if (!empty($editorHistory)) {
                                                    $last = end($editorHistory);
                                                    $lastWhen = $last['edited_at'] ? date('j M Y H:i', strtotime($last['edited_at'])) : '';
                                                    $lastName = htmlspecialchars($last['name'] ?? '');
                                                    echo '<div><small>Diedit terakhir: <strong>' . ($lastWhen ?: '-') . '</strong> oleh <strong>' . $lastName . '</strong></small></div>';
                                                } else {
                                                    echo '<div class="small text-muted">Belum diedit</div>';
                                                }
                                                ?>
                                            </td>
                                            <td class="text-center">
                                                <?php if (!empty($row['file_surat'])): ?>
                                                    <a href="/jasa_marga/<?= htmlspecialchars($row['file_surat']) ?>" target="_blank"
                                                        class="btn btn-sm btn-outline-success" title="Preview">
                                                        <i class="bi bi-eye-fill"></i>
                                                    </a>
                                                    <a href="/jasa_marga/<?= htmlspecialchars($row['file_surat']) ?>" download
                                                        class="btn btn-sm btn-outline-primary" title="Unduh">
                                                        <i class="bi bi-download"></i>
                                                    </a>
                                                <?php else: ?>
                                                    <span class="text-muted">-</span>
                                                <?php endif; ?>
                                            </td>
                                            <td class="text-center">
                                                <a href="edit-letter.php?id=<?= (int)$row['id'] ?>"
                                                    class="btn btn-sm btn-success fw-bold mb-1">
                                                    <i class="bi bi-pencil-square"></i> Edit
                                                </a>

                                                <a href="delete-letter.php?id=<?= (int)$row['id'] ?>"
                                                    class="btn btn-sm btn-danger fw-bold mb-1"
                                                    onclick="return confirm('Yakin ingin menghapus surat ini? Data tidak bisa dikembalikan.');">
                                                    <i class="bi bi-trash-fill"></i> Hapus
                                                </a>
                                            </td>
                                        </tr>

                                    <?php endwhile;
                                else: ?>

                                    <tr>
                                        <td colspan="7" class="text-center text-muted py-4">
                                            Belum ada data surat.
                                        </td>
                                    </tr>

                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>

    </div>

    <!-- JS -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <!-- Preview Modal -->
    <div class="modal fade" id="previewModal" tabindex="-1" aria-labelledby="previewModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="previewModalLabel">Preview Surat</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <dl class="row">
                        <dt class="col-sm-3">Nomor</dt>
                        <dd class="col-sm-9" id="previewNomor"></dd>
                        <dt class="col-sm-3">Pembuat</dt>
                        <dd class="col-sm-9" id="previewPembuat"></dd>
                        <dt class="col-sm-3">Kode Jabatan</dt>
                        <dd class="col-sm-9" id="previewKode"></dd>
                        <dt class="col-sm-3">Perihal</dt>
                        <dd class="col-sm-9" id="previewPerihal"></dd>
                        <dt class="col-sm-3">File</dt>
                        <dd class="col-sm-9" id="previewFile"></dd>
                    </dl>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const bsModal = new bootstrap.Modal(document.getElementById('previewModal'));
            document.querySelectorAll('.btn-preview').forEach(btn => {
                btn.addEventListener('click', function() {
                    document.getElementById('previewNomor').textContent = this.dataset.nomor || '';
                    document.getElementById('previewPembuat').textContent = this.dataset.pembuat || '';
                    document.getElementById('previewKode').textContent = this.dataset.kode || '';
                    document.getElementById('previewPerihal').textContent = this.dataset.perihal || '';
                    const file = this.dataset.file || '';
                    if (file) document.getElementById('previewFile').innerHTML = `<a href="${file}" target="_blank">Open File</a>`;
                    else document.getElementById('previewFile').textContent = 'Tidak ada';
                    bsModal.show();
                });
            });
        });
    </script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

    <script>
        const searchInputNew = document.getElementById('searchInputNew');
        const searchButtonNew = document.getElementById('searchButtonNew');

        // Event listener untuk memperlihatkan tombol ketika input diklik
        if (searchInputNew && searchButtonNew) {
            searchInputNew.addEventListener('focus', function() {
                searchButtonNew.classList.add('show'); // Tampilkan tombol
                searchInputNew.placeholder = ''; // Hilangkan placeholder
            });

            // Event listener untuk menyembunyikan tombol ketika input kosong
            searchInputNew.addEventListener('blur', function() {
                if (!searchInputNew.value.trim()) {
                    searchButtonNew.classList.remove('show'); // Sembunyikan tombol
                    searchInputNew.placeholder = 'Cari nomor, nama...'; // Kembalikan placeholder
                }
            });

            // Search functionality
            function performSearch() {
                const searchTerm = searchInputNew.value.toLowerCase().trim();
                const rows = document.querySelectorAll('table tbody tr');
                let foundCount = 0;

                if (!searchTerm) {
                    rows.forEach(row => row.style.display = '');
                    const noResultMsg = document.querySelector('.no-results-msg');
                    if (noResultMsg) noResultMsg.remove();
                    return;
                }

                rows.forEach(row => {
                    const nomorSurat = row.querySelector('td:nth-child(1)')?.textContent.toLowerCase() || '';
                    const dibuatOleh = row.querySelector('td:nth-child(2)')?.textContent.toLowerCase() || '';
                    const penandatangan = row.querySelector('td:nth-child(3)')?.textContent.toLowerCase() || '';
                    const jenisSurat = row.querySelector('td:nth-child(4)')?.textContent.toLowerCase() || '';

                    if (nomorSurat.includes(searchTerm) || dibuatOleh.includes(searchTerm) ||
                        penandatangan.includes(searchTerm) || jenisSurat.includes(searchTerm)) {
                        row.style.display = '';
                        foundCount++;
                    } else {
                        row.style.display = 'none';
                    }
                });

                const tbody = document.querySelector('table tbody');
                let noResultMsg = tbody.querySelector('.no-results-msg');
                if (foundCount === 0 && !noResultMsg) {
                    noResultMsg = document.createElement('tr');
                    noResultMsg.className = 'no-results-msg';
                    noResultMsg.innerHTML = `<td colspan="8" class="text-center text-muted p-3">Tidak ada hasil untuk "${searchTerm}"</td>`;
                    tbody.appendChild(noResultMsg);
                } else if (foundCount > 0 && noResultMsg) {
                    noResultMsg.remove();
                }
            }

            // Add event listeners for search
            searchInputNew.addEventListener('keypress', function(e) {
                if (e.key === 'Enter') {
                    performSearch();
                }
            });

            searchButtonNew.addEventListener('click', function() {
                performSearch();
            });
        }

        // Only add delete handlers if modal exists
        const deleteUserModal = document.getElementById('deleteUserModal');
        if (deleteUserModal) {
            document.querySelectorAll('.btn-delete').forEach(button => {
                button.addEventListener('click', function() {
                    document.getElementById('deleteUserId').value = this.dataset.id;
                    new bootstrap.Modal(deleteUserModal).show();
                });
            });
        }

        // Only add edit handlers if modal exists
        const editUserModalEl = document.getElementById('editUserModal');
        if (editUserModalEl) {
            document.querySelectorAll('.btn-edit').forEach(button => {
                button.addEventListener('click', function() {
                    document.getElementById('editUserId').value = this.dataset.id;
                    document.getElementById('editUsername').value = this.dataset.username;
                    document.getElementById('editNama').value = this.dataset.nama;
                    document.getElementById('editEmail').value = this.dataset.email;
                    document.getElementById('editTipe').value = this.dataset.tipe;
                    document.getElementById('editJabatan').value = this.dataset.jabatan;
                    document.getElementById('editGender').value = this.dataset.gender;
                    document.getElementById('editSubUnit').value = this.dataset.sub_unit_id;

                    // Reset dan centang akses
                    document.getElementById('aksesBerita').checked = false;
                    document.getElementById('aksesSosial').checked = false;
                    document.getElementById('aksesGaji').checked = false;
                    document.getElementById('aksesSurat').checked = false;
                    document.getElementById('aksesAgenda').checked = false;

                    try {
                        const akses = JSON.parse(this.dataset.access);
                        if (akses.berita === 'on') document.getElementById('aksesBerita').checked = true;
                        if (akses.sosial === 'on') document.getElementById('aksesSosial').checked = true;
                        if (akses.gaji === 'on') document.getElementById('aksesGaji').checked = true;
                        if (akses.surat === 'on') document.getElementById('aksesSurat').checked = true;
                        if (akses.agenda === 'on') document.getElementById('aksesAgenda').checked = true;
                        if (akses.todolist === 'on') document.getElementById('aksesTodolist').checked =
                            true;
                    } catch (e) {
                        console.error("Gagal parse access_modul");
                    }

                    new bootstrap.Modal(editUserModalEl).show();
                });
            });
        }

        document.addEventListener('DOMContentLoaded', function() {
            // Only setup modal if it exists
            const editModalEl = document.getElementById('editUserModal');
            if (editModalEl) {
                var editButtons = document.querySelectorAll('.btn-edit');
                var modal = new bootstrap.Modal(editModalEl);

                editButtons.forEach(function(button) {
                    button.addEventListener('click', function() {
                        var userId = this.getAttribute('data-id');
                        var username = this.getAttribute('data-username');
                        document.getElementById('editUserId').value = userId;
                        document.getElementById('editUsername').value = username;
                        modal.show();
                    });
                });
            }

            // Note: Admin approval functionality has been removed
            // Approval is now handled entirely by the user approval chain
        });

        // Filter by tipe surat on client-side
        document.addEventListener('DOMContentLoaded', function() {
            const filterSelect = document.getElementById('filterTipeSurat');
            if (!filterSelect) return;

            function updateFilter() {
                const selectedId = filterSelect.value;
                const selectedOption = filterSelect.options[filterSelect.selectedIndex];
                const selectedKode = (selectedOption?.dataset?.kode || '').toUpperCase();

                const rows = Array.from(document.querySelectorAll('#mainContent table tbody tr')).filter(r => !!r.dataset.tipeId);
                let visibleCount = 0;

                rows.forEach(row => {
                    if (selectedId === 'all' || row.dataset.tipeId === selectedId) {
                        row.style.display = '';
                        visibleCount++;
                    } else {
                        row.style.display = 'none';
                    }
                });

                // Show/hide jenis & klasifikasi columns based on selected code (SK -> show, otherwise hide)
                const showJenis = (selectedKode === 'SK' || selectedId === 'all');
                document.querySelectorAll('.col-jenis').forEach(el => el.style.display = showJenis ? '' : 'none');
                document.querySelectorAll('.col-klasifikasi').forEach(el => el.style.display = showJenis ? '' : 'none');

                // Show custom no-data row when needed
                const existingNoData = document.getElementById('filterNoDataRow');
                if (visibleCount === 0) {
                    // Compute colspan (if jenis/klasifikasi hidden -> 5 columns, else 7)
                    const colspan = showJenis ? 7 : 5;
                    if (existingNoData) {
                        existingNoData.querySelector('td').setAttribute('colspan', colspan);
                        existingNoData.style.display = '';
                    } else {
                        const tbody = document.querySelector('#mainContent table tbody');
                        const tr = document.createElement('tr');
                        tr.id = 'filterNoDataRow';
                        tr.innerHTML = `<td colspan="${colspan}" class="text-center text-muted py-4">Tidak ada surat sesuai filter.</td>`;
                        tbody.appendChild(tr);
                    }
                } else {
                    if (existingNoData) existingNoData.style.display = 'none';
                }
            }

            filterSelect.addEventListener('change', updateFilter);
            // Initialize with current value
            updateFilter();
        });

        // Signature Canvas Functionality
        const canvas = document.getElementById('signatureCanvas');
        const ctx = canvas.getContext('2d');
        let isDrawing = false;
        let lastX = 0;
        let lastY = 0;

        // Set canvas size
        canvas.width = 600;
        canvas.height = 200;

        canvas.addEventListener('mousedown', startDrawing);
        canvas.addEventListener('mousemove', draw);
        canvas.addEventListener('mouseup', stopDrawing);
        canvas.addEventListener('mouseout', stopDrawing);

        // Touch events for mobile
        canvas.addEventListener('touchstart', handleTouch);
        canvas.addEventListener('touchmove', handleTouch);
        canvas.addEventListener('touchend', stopDrawing);

        function startDrawing(e) {
            isDrawing = true;
            [lastX, lastY] = getMousePos(e);
        }

        function draw(e) {
            if (!isDrawing) return;

            const [currentX, currentY] = getMousePos(e);

            ctx.beginPath();
            ctx.moveTo(lastX, lastY);
            ctx.lineTo(currentX, currentY);
            ctx.strokeStyle = '#000';
            ctx.lineWidth = 2;
            ctx.lineCap = 'round';
            ctx.stroke();

            [lastX, lastY] = [currentX, currentY];
        }

        function stopDrawing() {
            isDrawing = false;
        }

        function getMousePos(e) {
            const rect = canvas.getBoundingClientRect();
            const scaleX = canvas.width / rect.width;
            const scaleY = canvas.height / rect.height;

            return [
                (e.clientX - rect.left) * scaleX,
                (e.clientY - rect.top) * scaleY
            ];
        }

        function handleTouch(e) {
            e.preventDefault();
            const touch = e.touches[0];
            const mouseEvent = new MouseEvent(e.type === 'touchstart' ? 'mousedown' :
                e.type === 'touchmove' ? 'mousemove' : 'mouseup', {
                    clientX: touch.clientX,
                    clientY: touch.clientY
                });
            canvas.dispatchEvent(mouseEvent);
        }

        // Clear signature button
        document.getElementById('clearSignature').addEventListener('click', function() {
            ctx.clearRect(0, 0, canvas.width, canvas.height);
        });

        // Handle signature file upload
        document.getElementById('signatureFile').addEventListener('change', function(e) {
            const file = e.target.files[0];
            const preview = document.getElementById('signaturePreview');
            const previewImg = document.getElementById('signaturePreviewImg');

            if (file && file.size <= 5 * 1024 * 1024) { // 5MB limit
                const reader = new FileReader();
                reader.onload = function(e) {
                    previewImg.src = e.target.result;
                    preview.style.display = 'block';
                    // Clear canvas when file is uploaded
                    ctx.clearRect(0, 0, canvas.width, canvas.height);
                };
                reader.readAsDataURL(file);
            } else if (file) {
                alert('File terlalu besar! Maksimal 5MB.');
                this.value = '';
            } else {
                preview.style.display = 'none';
            }
        });
    </script>

</body>

</html>