# Approval Button Issue Analysis - "Menunggu giliran" (Waiting Turn)

## Issue Description
When editing an approved letter, the approval button changes to "Menunggu giliran" (Waiting turn) and users cannot approve again.

---

## KEY FINDINGS

### 1. **Button Text Display Location: "Menunggu giliran"**

**File:** `user/record-letter.php`  
**Line:** 602

```php
<?php elseif ($isUserInChain): ?>
    <!-- User is in chain but not their turn -->
    <span class="badge bg-secondary me-1">Menunggu giliran</span>
    <button type="button" class="btn btn-sm btn-outline-info btn-view-chain"
        data-id="<?= (int)$row['id'] ?>"
        data-nomor="<?= htmlspecialchars($row['nomor_surat']) ?>">
        <i class="fas fa-list-ol"></i> Detail
    </button>
```

**Context:** This badge is shown in the list view (record-letter.php) when:
- User is in the approval chain (`$isUserInChain` = true)
- BUT it's NOT their turn to approve

---

### 2. **Critical Approval Logic - canApprove Determination**

**File:** `user/edit-letter.php`  
**Line:** 184

```php
$canApprove = ($userUrutan === $lastApprovedStep + 1) && ($userStepInChain['approved_at'] === null);
```

**The Problem:** This is checking TWO conditions:
1. **`$userUrutan === $lastApprovedStep + 1`** - Current user's step number equals last approved step + 1 (their turn)
2. **`$userStepInChain['approved_at'] === null`** - User hasn't already approved this letter

### Key Variables Context (Lines 136-190):

```php
// Line 136-137: Initialize
$canApprove = false;
$isFinalApprover = false;

// Line 155-165: Calculate lastApprovedStep
$lastApprovedStep = 0;
// ... in loop ...
if ($row['approved_at'] !== null) {
    $lastApprovedStep = (int)$row['urutan'];
}

// Line 181-185: THE CRITICAL LOGIC
// Check if user can approve (their turn)
if ($userStepInChain) {
    $userUrutan = (int)$userStepInChain['urutan'];
    $canApprove = ($userUrutan === $lastApprovedStep + 1) && ($userStepInChain['approved_at'] === null);
    $isFinalApprover = ($userUrutan === $totalSteps);
```

---

### 3. **Approval Chain Status Function**

**File:** `user/record-letter.php`  
**Lines:** 80-145

```php
function getLetterApprovalChainStatus($conn, $officeId)
{
    $result = [
        'chain' => [],
        'current_step' => 0,
        'total_steps' => 0,
        'is_complete' => false,
        'paraf_display' => '',
        'next_approver_id' => null,
        'approved_by' => []
    ];

    // ... code fetches urutan_paraf records ...
    
    // Line 115-123: Compute current_step and track approvals
    $lastApproved = 0;
    while ($row = $res->fetch_assoc()) {
        $result['chain'][] = $row;
        $result['total_steps']++;

        if ($row['approved_at'] !== null) {
            $lastApproved = (int)$row['urutan'];
            $result['current_step'] = $lastApproved;
            $result['approved_by'][] = (int)$row['user_id'];
        } else if ($result['next_approver_id'] === null) {
            $result['next_approver_id'] = (int)$row['user_id'];  // Next in line
        }
    }
```

---

### 4. **Helper Function: canUserApproveLetter**

**File:** `user/record-letter.php`  
**Lines:** 195-199

```php
/**
 * Check if user can approve a specific letter
 */
function canUserApproveLetter($approvalStatus, $userId)
{
    return $approvalStatus['next_approver_id'] === $userId;
}
```

**Logic:** User can approve if they are the **next_approver_id** (which is set to the first person in the chain who hasn't approved yet)

---

### 5. **Helper Function: isUserInApprovalChain**

**File:** `user/record-letter.php`  
**Lines:** 202-210

```php
/**
 * Check if user is in the approval chain for a letter
 */
function isUserInApprovalChain($approvalStatus, $userId)
{
    foreach ($approvalStatus['chain'] as $step) {
        if ((int)$step['user_id'] === $userId) {
            return true;
        }
    }
    return false;
}
```

---

### 6. **Approval Mode Button in Edit Page**

**File:** `user/edit-letter.php`  
**Line:** 1744-1747

```php
<?php if ($approvalMode): ?>
    <button type="button" id="btnApprove" class="btn btn-warning fw-bold">
        <i class="fas fa-check-circle"></i> <?= $isFinalApprover ? 'Approve & Generate QR' : 'Approve' ?>
    </button>
```

**Context:** This button is only shown in approval mode (`?mode=approval`), and the text changes based on whether user is final approver.

---

### 7. **JavaScript Event Handler for Approval Button**

**File:** `user/edit-letter.php`  
**Lines:** 4231-4280

```php
if (btnApprove && isApprovalMode) {
    btnApprove.addEventListener('click', async function(e) {
        e.preventDefault();

        const fileInput = document.getElementById('fileSuratPath');

        // Check if PDF has been generated
        if (!fileInput || !fileInput.value.trim()) {
            Swal.fire({
                icon: 'warning',
                title: 'PDF Belum Di-generate',
                html: 'Harap klik <strong>"Cetak & Upload PDF"</strong> terlebih dahulu sebelum melakukan approval.',
                confirmButtonColor: '#ffc107'
            });
            return;
        }

        if (isFinalApprover) {
            // Final approver - show signature modal first
            showSignatureModal();
        } else {
            // Non-final approver - confirm and submit approval directly
            const result = await Swal.fire({
                title: 'Konfirmasi Approval',
                html: `<p>Anda akan menyetujui surat ini.</p>
                       <p>Paraf Anda (<strong>${approvalInfo.user_inisial || '?'}</strong>) akan ditambahkan ke surat.</p>`,
                icon: 'question',
                showCancelButton: true,
                confirmButtonText: '<i class="fas fa-check"></i> Approve',
                cancelButtonText: 'Batal',
                confirmButtonColor: '#28a745',
                cancelButtonColor: '#6c757d'
            });

            if (result.isConfirmed) {
                await submitApproval(null);
            }
        }
    });
}
```

**KEY POINT:** Button is only enabled if `btnApprove` exists AND `isApprovalMode` is true.

---

### 8. **Access Control Logic for Approval Mode**

**File:** `user/edit-letter.php`  
**Lines:** 208-216

```php
// If user is not in approval chain or not their turn, redirect with error
if (!$canApprove) {
    $_SESSION['error_msg'] = 'Anda tidak memiliki akses untuk melakukan approval surat ini saat ini.';
    header('Location: record-letter.php');
    exit();
}
```

**CRITICAL:** If `$canApprove` is false, user is REDIRECTED AWAY entirely!

---

## FLOW SUMMARY

### When Accessing a Letter in Approval Mode (`?mode=approval`):

1. **PHP Backend (edit-letter.php, lines 136-216):**
   - Gets all approvers from `urutan_paraf` table
   - Finds `$lastApprovedStep` (highest urutan with approved_at != NULL)
   - Finds `$userStepInChain` (current user's row in chain)
   - Calculates `$canApprove = ($userUrutan === $lastApprovedStep + 1) && ($userStepInChain['approved_at'] === null)`
   - If `!$canApprove` → REDIRECT TO record-letter.php immediately (line 210)

2. **If $canApprove is TRUE:**
   - User sees approval mode banner (line 1415)
   - Button `btnApprove` is displayed (line 1744-1747)
   - JavaScript listens for click and calls `submitApproval()` (lines 4231-4280)

3. **In List View (record-letter.php, line 602):**
   - If user is in chain but `next_approver_id !== $userId` → Shows "Menunggu giliran" badge
   - No approval button shown, only "Detail" button

---

## THE ISSUE IDENTIFIED

### Problem: User Already Approved, But Trying to Approve Again

**Root Cause:** Once a user approves a letter, their `approved_at` field in `urutan_paraf` is set to a timestamp (NOT NULL).

**Next check fails:**
```php
$canApprove = ($userUrutan === $lastApprovedStep + 1) && ($userStepInChain['approved_at'] === null);
                                                       ^
                                                   This becomes FALSE
                                                   because user already approved
```

**Result:** 
- Line 210: User is redirected from edit-letter.php
- Line 602 in record-letter.php: Shows "Menunggu giliran" badge instead of approval button
- User cannot approve again (by design - they already have)

---

## VERIFICATION QUERY

To check if this is the issue:

```sql
SELECT 
    up.id,
    up.office_id,
    up.user_id,
    up.urutan,
    up.approved_at,
    u.nama,
    u.id as current_user_check
FROM urutan_paraf up
LEFT JOIN users u ON up.user_id = u.id
WHERE up.office_id = [LETTER_ID]
ORDER BY up.urutan ASC;
```

If current user's row has `approved_at` NOT NULL, they've already approved and cannot approve again.

---

## Lines Summary Table

| Feature | File | Lines | Description |
|---------|------|-------|-------------|
| canApprove Logic | edit-letter.php | 184 | Core approval eligibility check |
| lastApprovedStep | edit-letter.php | 155-165 | Finds how far approval has progressed |
| Access Control | edit-letter.php | 208-216 | Redirects if canApprove=false |
| Approve Button HTML | edit-letter.php | 1744-1747 | Button rendered only in approval mode |
| Button Event Handler | edit-letter.php | 4231-4280 | JavaScript click listener |
| "Menunggu giliran" Badge | record-letter.php | 602 | Shows in list view when not user's turn |
| getLetterApprovalChainStatus | record-letter.php | 80-145 | Gets approval status of letter |
| canUserApproveLetter | record-letter.php | 195-199 | Checks if user is next approver |
| isUserInApprovalChain | record-letter.php | 202-210 | Checks if user is in chain at all |

